/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef EPIC_H
#define EPIC_H
/* * * * * * * * * * * * * epic.h  * * * * * * * * * * * * * * * * * * * * * * 
 *                                                                           *
 *       Timothy E. Dowling                                                  *
 *                                                                           *
 *       Header file for the EPIC atmospheric model                          *
 *                                                                           *
 *       This file provides a glossary of terms for the compiler.            *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef _XOPEN_SOURCE
#  define _XOPEN_SOURCE
#  define _XOPEN_SOURCE_EXTENDED 1
#  ifdef sun4
     /* To get definitions of u_short, etc., needed for socket.h, etc. */
#    define __EXTENSIONS__
#  endif
#endif

#include <stdio.h> 
#include <stdlib.h>   
#include <errno.h>    
#include <math.h>    
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <float.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <netcdf.h>

#include "avs_epic.h"
#include "epic_io_macros.h"

#if defined(EPIC_MPI)
#  include "mpi.h"
#  include "mpg.h"
#endif

/**** Turn on to print messages for debugging purposes:
#define DEBUG_MILESTONE ;
*****/
#define DEBUG_MILESTONE fprintf(stderr,"\n node %2d %s DEBUG_MILESTONE %2d\n", \
                                       IAMNODE,dbmsname,++idbms);
/*
 *  There are two horizontal momentum advection schemes available:
 *
 *  AL81 stands for Arakawa and Lamb, 1981. This scheme conserves total
 *  energy and total enstrophy.
 *
 *  HA90 stands for Hsu and Arakawa, 1990.  This scheme conserves total
 *  energy and total enstrophy for the case of nondivergent mass flux.
 *  It is appropriate for the massless-layer approach of
 *  handling the intersection of the theta surfaces with the solid 
 *  bottom of a terrestrial atmosphere.
 *
 *  Define SW_SCHEME to be AL81 or HA90:
 */
#define SW_SCHEME HA90
/*
 *  Choose whether or not to use the Hsu and Arakawa predictor-corrector 
 *  for the horizontal continuity terms (TRUE or FALSE):
 */
#define HSU_PREDICT_CORRECT TRUE

#if defined(EPIC_AVS)
#  include <avs/avs.h>
#  include <avs/flow.h>
#  include <avs/field.h>
#  include <avs/geom.h>
#  include <avs/avs_math.h>
#  include <avs/colormap.h>
  /* 
   * AVS memory allocation debugging. See AVS Developer's Guide, Advanced Topics.
   * To use this utility, set the following environment variables:
   *   AVS_MEM_CHECK   1
   *   AVS_MEM_HISTORY 1
   *   AVS_MEM_VERBOSE 1, 2, or 3
   */
#  define  MEM_DEFS_ENABLE 1
#endif

/*
 * Logical:
 */
#define TRUE    1
#define FALSE   0
#define SUCCESS TRUE
#define FAILURE FALSE
/*
 * Mathematical:
 */
#if !defined(M_E)
#  define M_E         2.7182818284590452354
#  define M_LOG2E     1.4426950408889634074
#  define M_LOG10E    0.43429448190325182765
#  define M_LN2       0.69314718055994530942
#  define M_LN10      2.30258509299404568402
#  define M_PI        3.14159265358979323846
#  define M_PI_2      1.57079632679489661923
#  define M_PI_4      0.78539816339744830962
#  define M_1_PI      0.31830988618379067154
#  define M_2_PI      0.63661977236758134308
#  define M_2_SQRTPI  1.12837916709551257390
#  define M_SQRT2     1.41421356237309504880
#  define M_SQRT1_2   0.70710678118654752440
#endif

#define DEG (M_PI/180.)

#define SQR(a)   ((a)*(a))
#define MAX(a,b) (((a) > (b)) ? (a) : (b))
#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#define NINT(a)  (((a) >  0.) ? (int)((a)+.5) : (int)((a)-.5))
#define SIGN(x)  (((double)(x) == 0.) ? 0. : ((double)(x)/fabs(x)))

/* 
 * Complex notation: 
 */
typedef struct{
  double
    x,y;
} complex;

#if defined(EPIC_MPI)
  /* Defined in mpispec_init(): */
  MPI_Datatype EPIC_MPI_COMPLEX;
#endif

/*
 * NOTE: For LINUX, with -D_BSD_SOURCE cabs() is defined, such that
 * a type-mismatch error occurs if we call cabsolute() below cabs().
 */

complex cnum(     double   x, double   y);
complex cmult(    complex z1, complex z2);
complex cadd(     complex z1, complex z2);
complex csub(     complex z1, complex z2);
complex cexp(     complex z);
double  cabsolute(complex z);
double  creal(    complex z);
double  cimag(    complex z);
 
/*
 * Physical:
 */
#define K_B    1.38066e-23   /* Boltzmann constant, J/K */
#define R_GAS  8.31451e+3    /* Universal gas constant, J/K/kmol */
#define YEAR   31536000      /* secs/year */

/*
 * Bookkeeping:
 */
#define NODE0 0
#define PACKET_SIZE 1024
#define N_STR 256


#define SKIP_BC2D   0
#define APPLY_BC2D  1

/* Filter is applied poleward of LAT0 [deg]: */
#define LAT0 45.

/* 
 * Prognostic variable states: 
 */
#define CHEM_OFF       0
#define CHEM_ACTIVE    1
#define CHEM_PASSIVE  -1

#define BY_NUMBER 1
#define BY_MASS   2

/*
 * Read, write modes:
 */
#define DONT_BCAST -1
/*
 * i/o types: 
 */
#define VIA_SOCKET   1
#define VIA_FILE     2
#define VIA_STDIO    3
/* 
 * portion types (see next):
 */
#define INIT_DATA       1
#define NONINIT_DATA    2
#define ALL_DATA        3
#define UVP_DATA        4
/*
 * Add 1000 to portion type to
 * signal model exit after current i/o:
 */
#define INIT_EXIT    1001
#define NONINIT_EXIT 1002
#define ALL_EXIT     1003
#define UVP_EXIT     1004

#define ONEDIM   1
#define TWODIM   2
#define THREEDIM 3
#define FOURDIM  4

/* 
 * Initialization types: 
 */
#define INIT_TP   0
#define INIT_T    1
#define INIT_QM   2

/* 
 * Vertical wind profile choices: 
 */
#define UZ_GALILEO   0
#define UZ_CZS       1   /* RAUL */
#define UZ_GIERASCH  2
#define UZ_CONSTANT  3 
#define UZ_LINEAR    4  
#define UZ_USER      5  

/* 
 * Layer spacing choices: 
 */
#define SPACING_SW       -1
#define SPACING_THETA     0
#define SPACING_LOGTHETA  1
#define SPACING_LOGP      2

/*
 * Misc argument types:
 */
#define PASSING_THETA 0
#define PASSING_T     1

/* 
 * Time-index pointers: 
 */      
int 
  IT_ZERO,IT_MINUS1,IT_MINUS2;

/*
 *  Horizontal (shallow-water) schemes:
 */
#define AL81 1
#define HA90 2

#define FILE_STR 256

/*
 *  Structures: 
 */
/* NOTE: these numbers must be put in by hand in avs_epic.h:                    */
#define GEOM_STR     16   /* geometry string length                             */
#define MAX_NU_ORDER  6   /* grad^MAX_NU_ORDER is highest-order viscosity       */
#define TOPDIM        3   /* 3 for x,y,z                                        */

typedef struct {
  char
    geometry[GEOM_STR];
  double
    data_version,
    globe_lonbot,
    globe_lontop,
    globe_latbot,
    globe_lattop;
  char
    f_plane_map[GEOM_STR];
  double
    f_plane_lat0,
    f_plane_half_width;
  int   
    dt,                   /* timestep, s                                        */
    cfl_dt,               /* CFL timestep                                       */
    nk,                   /* number of vertical layers                          */
    nj,                   /* number of grid points in latitude                  */
    ni,                   /* number of grid points in longitude                 */
    wrap[TOPDIM],         /* periodicity flags                                  */
    pad[TOPDIM],          /* boundary pad widths                                */
    jlo,                  /* lower index for j, 0 for globe geometry            */
    jfirst,               /* used to handle C-grid for q and v variables        */
    klast_active,         /* terrestrial planets have active k = nk             */
    we_num_nodes,         /* number of nodes on this computer                   */
    they_num_nodes;       /* number of nodes on other computer                  */
  double 
    dln,                  /* longitudinal grid spacing, deg                     */
    dlt,                  /* latitudinal  grid spacing, deg                     */
    theta_bot,            /* theta for bottom of model                          */
    theta_top,            /* theta for top of model                             */
    press0,               /* reference pressure                                 */
    mont0;                /* reference Montgomery potential                     */
  int
    k_sponge,             /* range of sponge layers                             */
    init_type,            /* indicates intialiation: 0=T(p), 1=T(y,p), etc.     */
    newt_cool_on,         /* 1 enables Newtonian cooling                        */
    thermal_conduct_on;   /* 1 enables thermal conduction                       */
  char
    eos[8];               /* equation of state: "ideal", "virial"               */
  int    
    aux_a,                /* for any use                                        */
    aux_b,                /* for any use                                        */
    aux_c;                /* for any use                                        */
  double 
    aux_fa,               /* for any use                                        */
    aux_fb,               /* for any use                                        */
    aux_fc;               /* for any use                                        */
  /* 
   * The following assume nk <= 200.  
   * Their size is hardwired for convenience.  
   * Analogous memory is found in avs_epic.h.
   */
  double
    theta[401],           /* theta array,  assumes nk <= 200                    */ 
    rgas[201],            /* rgas  array                                        */ 
    kappa[201],           /* kappa array                                        */
    p_avg[201],           /* p_avg array                                        */
    qmin[201],            /* array of qmin's                                    */
    qmax[201],            /* array of qmax's                                    */
    tmin[201],            /* array of tmin's                                    */
    tmax[201];            /* array of tmax's                                    */
  double
    *f,                   /* Coriolis parameter, 1/s                            */
    *m,                   /* longitudinal map factor 1/r/dln/DEG                */
    *n,                   /* latitudinal  map factor 1/R/dlt/DEG                */
    *mn,                  /* m*n                                                */
    *lat,                 /* latitude,  deg                                     */
    *lon;                 /* longitude, deg                                     */
  int   
    is_spole,             /* south pole flag                                    */
    is_npole;             /* north pole flag                                    */
  double 
    *t_sponge_inv,        /* Rayleigh damping time for sponge at top            */
    hasten,               /* artificial speedup factor in relaxation processes  */
    prandtl,              /* Ratio t_rad/t_drag                                 */
    nu[MAX_NU_ORDER+1];   /* viscosity coefficients                             */
  int   
    itback,               /* num. of timesteps bet. backups to disk (same file) */
    itsave,               /* num. of timesteps bet. outputs to disk (diff file) */
    itout;                /* num. of timesteps bet. outputs to host             */
  unsigned long 
    itime;                /* index for main time-incrementing loop              */
  char
    view_host[80];        /* machine running EPIC_RECEIVE                       */
} gridspec;

/*
 * NOTE: Declare grid structure globally:
 */
gridspec 
  grid;

/*
 * NOTE: Declare planet-structure pointer globally:
 */
#if !defined(EPIC_VIEW)
planetspec
  *planet;
#endif

/*
 * NOTE: Declare domain-structure globally:
 */
#if !defined(EPIC_VIEW)
domainspec
 *domain;
#endif

/*
 * Set indices for prognostic variables:
 */
#define NO_INDEX           -1
/* Core prognostic variables: */
#define U_INDEX             0
#define V_INDEX             1
#define P_INDEX             2

/* Para hydrogen fraction: */
#define FPARA_INDEX         3 

/* Basic humidities: */
#define FIRST_HUMIDITY      4

#define H_2O_INDEX          4
#define NH_3_INDEX          5
#define H_2S_INDEX          6
#define CH_4_INDEX          7
#define CO_2_INDEX          8
#define DUST_INDEX          9

/* Chemical-product humidities: */
#define FIRST_CHEM_PRODUCT 10

#define NH_4SH_INDEX       10
#define O_3_INDEX          11
#define AEROSOL_INDEX      12

#define LAST_HUMIDITY      12

/* Maximum number of prognostic variables:  */
/* 
 * NOTE: If MAX_NVARS is changed then the corresponding numbers
 *       in epic/include/avs_epic.h must be updated.
 */
#define MAX_NVARS     (LAST_HUMIDITY+1) 
#define DRY_AIR_INDEX MAX_NVARS

#define CHEM_NM_SZ 64

typedef struct{
  int
    time[2];              /* time[0] is secs, time[1] is years                  */
  int
    nvars,                /* number of carried prognostic variables             */
    chem_on[MAX_NVARS+1], /* 0 implies chemical not carried                     */
    index[MAX_NVARS];     /* actual memory index for carried variables          */
  int
    nc_chemid[MAX_NVARS],
    nc_tendid[MAX_NVARS],
    nc_dimids[MAX_NVARS][FOURDIM],
    nc_coorid[MAX_NVARS][FOURDIM],
    nc_u_spinupid,
    nc_surface_gzid;
  char 
    chem_name[ MAX_NVARS+1][CHEM_NM_SZ],
    tend_name[ MAX_NVARS+1][CHEM_NM_SZ],
    chem_units[MAX_NVARS+1][CHEM_NM_SZ],
    tend_units[MAX_NVARS+1][CHEM_NM_SZ];
  double
    time_fp_bar;          /* Nominal = 3.e+8*1000.*100. [sec*1bar, mks]         */
  double  
    *vector[2*MAX_NVARS], /* prognostic variables and tendencies                */
    *p1,                  /* pressure in layer                                  */
    *t,*t2,               /* diagnostic: temperature in layer, interface        */
    *rho,*rho2,           /* diagnostic: density in layer, interface            */
    *u_spinup,            /* Rayleigh drag zonal-wind profile (k,j,i)           */
    *surface_gz;          /* Solid surface elevation, multiplied by g           */

#if defined(EPIC_AVS)
  AVSfield_double
    *field;               /*  prognostic variables                              */
#endif
} variables;
/*
 * NOTE: Declare var structure globally:
 */
variables
  var;

/*
 * Element defined in epic_funcs_diag.c by including chemical_elements.h.
 */
#define LAST_ATOMIC_NUMBER 103

typedef struct {
  int 
    atomic_number;
  char
    symbol[4],
    name[16];
  double
    molar_mass,
    solar_abundance;
} chem_element;

#define TIME ((double)var.time[0]+(double)var.time[1]*(double)YEAR)

typedef struct{
  double
    *vector[3];
#if defined(EPIC_AVS)
  AVSfield_double
    *field;
#endif
} diagnostics;
/*
 * NOTE: Declare diag structure globally:
 */
diagnostics
  diag;

#if defined(EPIC_MPI)
typedef struct{
  int 
    iamnode,
    nproc,
    nelem2d,
    nelem3d,
    ndim,
    jfirst,
    is_npole,
    is_spole;
  int
    npad[TOPDIM],
    wrap[TOPDIM],
    nstart[TOPDIM],
    nend[TOPDIM],
    dimlen[TOPDIM],
    mylo[TOPDIM],
    myhi[TOPDIM],
    arraydims[TOPDIM],
    nprocs[TOPDIM];
  MPI_Comm 
    comm,
    comm_cart,
    comm_kj;
} mpispec;
mpispec para;
#endif

#if defined(EPIC_VIEW)
typedef struct {
  char 
    *color_input;
  int
    layer,
    height_scale,
    downsize,
    light_north,
    wind,
    wind_scale,
    polar;
  AVSfield_double
    *color;
  double
    *color_min,
    *color_max;
  AVScolormap
    *colormap;
} listspec;
#endif

/*
 * Default-parameter structures:
 */
typedef struct {
  char  
    geometry[GEOM_STR],
    system_id[8],
    f_plane_map[GEOM_STR],
    eos[8];
  int 
    nk,nj,ni,dt,
    init_type,
    add_optional_var,
    chem_on[MAX_NVARS],
    uz_type,spacing_type,
    geostr_init,u_init,
    mq_ztyp,mq_ptyp,mq_maxit,
    mq_manual,mq_checkit,mq_qmrepeat,mq_jzero;
   double
     globe_lonbot,  /* keep globe_lonbot as first double */
     globe_lontop,globe_latbot,globe_lattop,
     f_plane_lat0,f_plane_half_width,
     ptop,pbot,
     prandtl,hasten,
     mole_fraction[MAX_NVARS],
     nu[MAX_NU_ORDER+1],
     time_fp,
     mq_c_q,mq_pfact,mq_x;
} init_defaultspec;

typedef struct {
  char
    infile[ N_STR],
    outfile[N_STR];
} change_defaultspec;

/* 
 * Lower threshold for h = -(1/g)dp/dtheta.
 */
#define H_TINY   (1.e-10)

/* 
 * (Ortho-para hydrogen equilibration time)*(p = 1000 mbar) in mks.
 * Nominal value has been (3.e+8*1000.*100.).
 */

#if defined(EPIC_MPI)
#  define IAMNODE    (para.iamnode)
#  define ILO        (para.mylo[0])
#  define JLO        (para.mylo[1])
#  define KLO        (para.mylo[2])
#  define IHI        (para.myhi[0])
#  define JHI        (para.myhi[1])
#  define KHI        (para.myhi[2])
#  define IPAD       (para.npad[0])
#  define JPAD       (para.npad[1])
#  define KPAD       (para.npad[2])
#  define JFIRST     (para.jfirst)
#  define IS_SPOLE   (para.is_spole)
#  define IS_NPOLE   (para.is_npole)
#else
#  define IAMNODE    0
#  define ILO        1
#  define JLO        (grid.jlo)
#  define KLO        1
#  define IHI        (grid.ni)
#  define JHI        (grid.nj)
#  define KHI        (grid.nk)
#  define IPAD       (grid.pad[0])
#  define JPAD       (grid.pad[1])
#  define KPAD       (grid.pad[2])
#  define JFIRST     (grid.jfirst)
#  define IS_NPOLE   (grid.is_npole)
#  define IS_SPOLE   (grid.is_spole)
#endif

#define KLAST_ACTIVE (grid.klast_active)

#define IADIM        (IHI-ILO+1+2*IPAD)
#define JADIM        (JHI-JLO+1+2*JPAD)
#define KADIM        (KHI-KLO+1+2*KPAD)
#define NELEM2D      (IADIM*JADIM)
#define NELEM3D      (NELEM2D*KADIM)

/* Initialization choices for M from q: */
#define SORPTOP  1
#define SORUTOP  2
#define SORPANDU 3
#define FILUP    4
#define FILDWN   5
#define FILLFT   6

   /*
    * Declare globally the following shift integers to speed up multidimensional 
    * array referencing. Set Ishift = ILO-IPAD, etc. in var_read().
    */
int
  Ishift,Jshift,Kshift,
  Iadim,Jadim,Kadim,
  Nelem2d,Nelem3d,
  Shift2d,Shift3d,Shiftkj;
 
   /*
    *  Memory allocation for the prognostic variables is in the form of
    *  an AVS field if AVS is available, otherwise it is in the form of a vector:
    */
#if !defined(EPIC_AVS)
#  define VAR(in,k,j,i)    (var.vector[in]           )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define U(k,j,i)         (var.vector[U_INDEX]      )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define V(k,j,i)         (var.vector[V_INDEX]      )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define P(k,j,i)         (var.vector[P_INDEX]      )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define FPARA(k,j,i)     (var.vector[FPARA_INDEX]  )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define H_2O(k,j,i)      (var.vector[H_2O_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define NH_4SH(k,j,i)    (var.vector[NH_4SH_INDEX] )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define NH_3(k,j,i)      (var.vector[NH_3_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define H_2S(k,j,i)      (var.vector[H_2S_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define CH_4(k,j,i)      (var.vector[CH_4_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define CO_2(k,j,i)      (var.vector[CO_2_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define O_3(k,j,i)       (var.vector[O_3_INDEX]    )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define AEROSOL(k,j,i)   (var.vector[AEROSOL_INDEX])[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#  define DUST(k,j,i)      (var.vector[DUST_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]

#  define EP_FLUX_Y(k,j)   (diag.vector[0])[j+(k)*Jadim-Shiftkj]
#  define EP_FLUX_Z(k,j)   (diag.vector[1])[j+(k)*Jadim-Shiftkj]
#  define EP_FLUX_DIV(k,j) (diag.vector[2])[j+(k)*Jadim-Shiftkj]
#else
#  define FIELD(field,n,k,j,i) (((field)->data)[n+(field)->veclen*(((field)->dimensions[0]* \
                                ((field)->dimensions[1]*(k-Kshift)+(j-Jshift)))+(i-Ishift))])
#  define VAR(in,k,j,i)    FIELD(var.field,var.index[in],           k,j,i)
#  define U(k,j,i)         FIELD(var.field,var.index[U_INDEX],      k,j,i)
#  define V(k,j,i)         FIELD(var.field,var.index[V_INDEX],      k,j,i)
#  define P(k,j,i)         FIELD(var.field,var.index[P_INDEX],      k,j,i)
#  define FPARA(k,j,i)     FIELD(var.field,var.index[FPARA_INDEX],  k,j,i)
#  define H_2O(k,j,i)      FIELD(var.field,var.index[H_2O_INDEX],   k,j,i)
#  define NH_4SH(k,j,i)    FIELD(var.field,var.index[NH_4SH_INDEX], k,j,i)
#  define NH_3(k,j,i)      FIELD(var.field,var.index[NH_3_INDEX],   k,j,i)
#  define H_2S(k,j,i)      FIELD(var.field,var.index[H_2S_INDEX],   k,j,i)
#  define CH_4(k,j,i)      FIELD(var.field,var.index[CH_4_INDEX],   k,j,i)
#  define CO_2(k,j,i)      FIELD(var.field,var.index[CO_2_INDEX],   k,j,i)
#  define O_3(k,j,i)       FIELD(var.field,var.index[O_3_INDEX],    k,j,i)
#  define AEROSOL(k,j,i)   FIELD(var.field,var.index[AEROSOL_INDEX],k,j,i)
#  define DUST(k,j,i)      FIELD(var.field,var.index[DUST_INDEX],   k,j,i)

#  define EP_FLUX_Y(k,j)   FIELD(diag.field,0,k,j,Ishift)
#  define EP_FLUX_Z(k,j)   FIELD(diag.field,1,k,j,Ishift)
#  define EP_FLUX_DIV(k,j) FIELD(diag.field,2,k,j,Ishift)
  /* 
   * Make the indices of the AVS macros consistent with EPIC macros:
   */
#  define E3D(field,k,j,i)      I3D(field,i-Ishift,j-Jshift,k-Kshift)
#  define COORD_X(field,i,j,k) \
           (((field)->points)[ ((field)->dimensions[0] * \
                                   ((field)->dimensions[1] * \
                                    (k) + (j))) + (i) ])
#  define COORD_Y(field,i,j,k) \
           (((field)->points)[ ((field)->dimensions[0] * \
                                   (((field)->dimensions[1] * \
                                    ((field)->dimensions[2] + (k))) + (j))) + (i) ])
#  define COORD_Z(field,i,j,k) \
           (((field)->points)[ ((field)->dimensions[0] * \
                                   (((field)->dimensions[1] * \
                                    ((field)->dimensions[2]*2 + (k))) + (j))) + (i) ])
#  define ECOORD_X(field,k,j,i) COORD_X(field,i-Ishift,j-Jshift,k-Kshift) 
#  define ECOORD_Y(field,k,j,i) COORD_Y(field,i-Ishift,j-Jshift,k-Kshift) 
#  define ECOORD_Z(field,k,j,i) COORD_Z(field,i-Ishift,j-Jshift,k-Kshift) 
#endif

    /*
     *  Memory for tendencies and secondary variables is in the form of a vector.
     *  The tendency for p is dhdt, where h = -(1/g)dp/dtheta.
     *  Mass tendencies are h-weighted.
     */
#define DVARDT(index,it,k,j,i) \
    (var.vector[MAX_NVARS+index]        )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DUDT(it,k,j,i)         \
    (var.vector[MAX_NVARS+U_INDEX]      )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DVDT(it,k,j,i)         \
    (var.vector[MAX_NVARS+V_INDEX]      )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHDT(it,k,j,i)         \
    (var.vector[MAX_NVARS+P_INDEX]      )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHFPARADT(it,k,j,i)    \
    (var.vector[MAX_NVARS+FPARA_INDEX]  )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHH_2ODT(it,k,j,i)     \
    (var.vector[MAX_NVARS+H_2O_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHNH_4SHDT(it,k,j,i)   \
    (var.vector[MAX_NVARS+NH_4SH_INDEX] )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHNH_3DT(it,k,j,i)     \
    (var.vector[MAX_NVARS+NH_3_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DH_2SDT(it,k,j,i)      \
    (var.vector[MAX_NVARS+H_2S_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHCH_4DT(it,k,j,i)     \
    (var.vector[MAX_NVARS+CH_4_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHCO_2DT(it,k,j,i)     \
    (var.vector[MAX_NVARS+CO_2_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHO_3DT(it,k,j,i)     \
    (var.vector[MAX_NVARS+O_3_INDEX]    )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHAEROSOL(it,k,j,i)     \
    (var.vector[MAX_NVARS+AEROSOL_INDEX])[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]
#define DHDUSTDT(it,k,j,i)       \
    (var.vector[MAX_NVARS+DUST_INDEX]   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d+(it)*Nelem3d]

#define SURFACE_GZ(j,i) (var.surface_gz)[i+(j)*Iadim-Shift2d]
#define P1(k,j,i)       (var.p1  )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define T(k,j,i)        (var.t   )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define T2(k,j,i)       (var.t2  )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define T3(k,j,i)       (var.t3  )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define RHO(k,j,i)      (var.rho )[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define RHO2(k,j,i)     (var.rho2)[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define U_SPINUP(k,j,i) (var.u_spinup)[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define UAMP(k,j)       uamp[j+(k)*Jadim-Shiftkj]

#define UH(j,i)      uh[     i+(j)*Iadim-Shift2d]
#define VH(j,i)      vh[     i+(j)*Iadim-Shift2d]
#define Q(j,i)       q[      i+(j)*Iadim-Shift2d]
#define KIN(j,i)     kin[    i+(j)*Iadim-Shift2d]
#define H(j,i)       h[      i+(j)*Iadim-Shift2d]
#define H_PT(j,i)    h_pt[   i+(j)*Iadim-Shift2d]
#define DH_PTDT(j,i) dh_ptdt[i+(j)*Iadim-Shift2d]

#define RH(j,i)      rh[     i+(j)*Iadim-Shift2d]
#define UM(j,i)    um[   i+(j)*Iadim-Shift2d]
#define UP(j,i)    up[   i+(j)*Iadim-Shift2d]
#define U2D(j,i)   u2d[  i+(j)*Iadim-Shift2d]
#define VM(j,i)    vm[   i+(j)*Iadim-Shift2d]
#define VP(j,i)    vp[   i+(j)*Iadim-Shift2d]
#define V2D(j,i)   v2d  [i+(j)*Iadim-Shift2d]

#define A(j,i)        a[      i+(j)*Iadim-Shift2d]
#define FF(j,i)       ff[     i+(j)*Iadim-Shift2d]
#define A_PRED(j,i)   a_pred[ i+(j)*Iadim-Shift2d]
#define A_DIFF1(j,i)  a_diff1[i+(j)*Iadim-Shift2d]
#define A_DIFF2(j,i)  a_diff2[i+(j)*Iadim-Shift2d]
#define AAA(j,i)      aaa[    i+(j)*Iadim-Shift2d]
#define A_OLD(j,i)    a_old[  i+(j)*Iadim-Shift2d]

#define HEAT(j,i)  heat[ i+(j)*Iadim-Shift2d]
#define MONT(j,i)  mont[ i+(j)*Iadim-Shift2d]
#define EXNER(j,i) exner[i+(j)*Iadim-Shift2d]
#define WH1(j,i)   wh1[  i+(j)*Iadim-Shift2d]
#define WH0(j,i)   wh0[  i+(j)*Iadim-Shift2d]

#define MONT3D(k,j,i) mont3d[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define FFGIBB(j,i)   ffgibb[i+(j)*Iadim-Shift2d]

#define LAP1(j,i)       lap1[i+(j)*Iadim-Shift2d]
#define LAP2(j,i)       lap2[i+(j)*Iadim-Shift2d]
#define LAPH(j,i)       laph[i+(j)*Iadim-Shift2d]

#define LAPT(j,i)       lapt[i+(j)*Iadim-Shift2d]

#define LOWPASS_H(j,i) lowpass_h[i+(j)*Iadim-Shift2d]
#define LOWPASS_V(j,i) lowpass_v[i+(j)*Iadim-Shift2d]

#define BUFF3D(k,j,i) buff3d[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define BUFFJI(j,i)   buffji[i+(j)*Iadim-Shift2d]
#define BUFFJ(j)      buffj[j-Jshift]
#define BUFFI(i)      buffi[i-Ishift]

#define Q3D(k,j,i)     q3d[      i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define THERM(k,j,i)   therm[    i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define COLOR(k,j,i)   color[    i+(j)*Iadim+(k)*Nelem2d-Shift3d]
#define LOG_PRESS(j,i) log_press[i+(j)*Iadim-Shift2d]

#define BC2D(pt,index,k) bc_ji(pt,index,k)

/*
 * Set BSWAP, for byte-order swapping.
 */
#define DO_SWAP   TRUE
#define DONT_SWAP FALSE
#if defined(decalpha) || defined(decmips) || defined(ncube2) || defined(LINUX)
#  define BSWAP DO_SWAP
#else
#  define BSWAP DONT_SWAP
#endif

/*
 * Function prototypes:
 */

#if defined(EPIC_VIEW)
void make_arrays(planetspec *planet,
                 domainspec *domain);
#else
void make_arrays(void);

void set_name_lists(void);

long Read(int    node,
          void  *buf,
          int    size,
          int    nitems,
          int    swap,
          FILE  *io,
          int    fd,
          int    bcast);

long Write(int    node,
           void  *buf,
           int    size,
           int    nitems,
           int    swap,
           FILE  *io,
           int    fd);

void read_array(int      dim,
                int      io_type,
                char    *name,
                FILE    *io,
                int      fd);

void write_array(int      dim,
                 int      io_type,
                 char    *name,
                 FILE    *io,
                 int      fd);

void var_read(planetspec *planet,
              char       *infile,
              int         socket_port,
              int         io_type,
              int         portion);

void var_write(planetspec *planet,
               char       *outfile,
               int         socket_port,
               int         io_type,
               int         portion);
#endif

void free_arrays(planetspec *planet);

void lookup_netcdf(char   *infile,
                   int    *nc_id,
                   int    *ngatts,
                   char ***gattname,
                   int    *nvars,
                   char ***varname);

void define_netcdf(planetspec *planet,
                   char       *outfile,
                   int        *nc_id);

void timestep(void);

double get_h(planetspec *planet,
             int         kk,
             int         J,
             int         I);

double get_p(planetspec *planet,
             int         index,
             int         kk,
             int         J,
             int         I);

int get_index(char *chem_name);

double *get_pointer(char *name);

double get_mole_fraction(planetspec *planet,
                         int         index,
                         int         kk,
                         int         J,
                         int         I);

double get_chem(planetspec *planet,
                int         index,
                int         kk,
                int         J,
                int         I); 

double get_brunt2(planetspec *planet,
                  int         kk,
                  int         J,
                  int         I);
		  
double return_cp(planetspec *planet,
                 double      fp,
                 double      p,
                 double      temp);

void store_temp_dens_p(planetspec *planet,
                       int         jlo,
                       int         jhi,
                       int         ilo,
                       int         ihi);

int cfl_dt(planetspec *planet,
           double     *dx0);

double interface_value(planetspec *planet,
                       int         index,
                       int         K,
                       int         J,
                       int         I,
                       int        *flag);

void bcast_char(int   node,
                char *str,
                int   num);

void bcast_int(int  node,
               int *val,
               int  num);

void bcast_double(int     node,
                  double *val,
                  int     num);

void print_zonal_info(planetspec *planet,
                      double     *q);

void print_vertical_column(planetspec *planet,
                           int         J,
                           int         I);

void print_humidity_column(planetspec *planet,
                           char       *outfile_str,
                           int         index,
                           char       *humidity_str,
                           double     *humidity,
                           double     *cloud_density,
                           int        J,
                           int        I);

void write_ascii(planetspec *planet,
                 int         io_type,
                 char       *datfile);

int read_t_vs_p(planetspec       *planet,
                double            ptop,
                double            pbot,
                int              *ceiling_tp,
                int              *floor_tp,
                double           *pdat,
                double           *tdat,
                int               portion);

void create_socket(int *sock,
                   int *socket_port);

int accept_socket(int socket_port);

int connect_socket(char *hostname,
                   int   socket_port);

double t_yp(double            p, 
            double            latr, 
            int               mode,
            init_defaultspec *def);

double fp_yp(double p,
             double latr,
             int    mode);         

double sech2(double);

double find_root(double (*func)(double),
                 double x1,
                 double x2,
                 double xacc);

void hsu_predict_correct(double *h,
                         int     K,
                         double *ab);

void horizontal_divergence_terms(double *uh,
                                 double *vh,
                                 int     K);

void mont_geostrophic(planetspec *planet,
                      double     *mont,
                      double      mont_south,
                      int         K);

void mont_nk(planetspec *planet,
             double     *mont);

void mixing_ratio(int     chem_index,
                  double *buffji,
                  int     K,
                  int     bc2d);

void parse_chem_name(char    *chem_name,
                     int     *num_elements,
                     char ***symbols,
                     int   **counts);

double solar_fraction(char *chem,
                      int   type,
                      char *min_element);

void relative_humidity(planetspec *planet,
                       int         chem_index,
                       double     *buffji,
                       int         K);

void set_u_balanced(double *mont,
                    double *ffgibb,
                    int     K);

void set_p_balanced(planetspec *planet,
                    double     *mont3d);

double low_mode(int J);

double kcond(planetspec *planet,
             double      temperature);

void laplacian(planetspec *planet,
               int         K,
               double     *uu,   double *vv, 
               double     *lp1,  double *lp2,
               double     *buff1,double *buff2,
               double    (*kcond)(planetspec *,double));

void potential_vorticity(planetspec *planet,
                         double     *q,
                         int         k,
                         int         yes_h);

void angular_momentum(planetspec *planet,
                      double     *buffji,
                      int         k);

void heating(planetspec *planet, 
             double     *heat,
             int         k);

double t_rad(planetspec *planet,
             double      pressure);

double temp_eq(planetspec *planet,
               double      lat,
               double      press,
               double      time);

double b_vir(char   *chem_a,
             char   *chem_b,
             double  temperature);

double b1_vir(char  *chem_a,
              char  *chem_b,
              double temperature);

double b2_vir(char  *chem_a,
              char  *chem_b,
              double temperature);

double sum_xx(planetspec *planet,
              double    (*bfunc)(char *,
                                 char *,
                                 double),
              double      temperature);

double avg_molar_mass(planetspec *planet,
                      int         kk,
                      int         j,
                      int         i);

double molar_mass(planetspec *planet,
                  char       *chem_name);

double min_richardson(planetspec *planet,
                      int        *kmin,
                      int        *jmin,
                      int        *imin);

double *extract_scalar(int      index,
                       int      k,
                       double  *buff);

void  insert_scalar(int     index,
                    int     k,
                    double *buff);

void scdswap(char *arr, 
             int   nbytes, 
             int   cnt);

void thermo_setup(planetspec *planet,
                  double     *cpr);

double return_temp(planetspec *planet,
                   double      fp,
                   double      p,
                   double      theta);

double return_dens(planetspec *planet,
                   double      fp,
                   double      p,
                   double      theta,
                   double      mu,
                   int         temp_type);

double return_theta(planetspec *planet,
                    double      fp,
                    double      p,
                    double      temperature,
                    double     *theta_ortho,
                    double     *theta_para);

double return_press(planetspec *planet,
                    double      fp,
                    double      temperature,
                    double      theta);

double return_enthalpy(planetspec *planet,
                       double      fp,
                       double      pressure,
                       double      temperature,
                       double     *fgibb,
                       double     *fpe,
                       double     *uoup);

double return_sat_vapor_p(char   *chem,
                          double  temperature);

double galileo_u(double pressure);

void filter(void);

void rain(planetspec *planet,
                    double     *heat,
                    int         K);

void print_rh(planetspec *planet,
               char      *outfile);

void print_sh(planetspec *planet,
               char      *outfile);

void print_pv(planetspec *planet,
               char      *outfile);

void print_p(planetspec *planet,
             char       *outfile);

void methane_mass(planetspec *planet,
                  char       *outfile);


/* memory allocation function */
double 
  *dvector(int nl, int nh);
void  
  free_dvector(double *m, int nl, int nh);

double input_double(char [N_STR], double);
float  input_float( char [N_STR], float);
int    input_int(   char [N_STR], int);
void   input_string(char [N_STR], char [N_STR], char *);

int time_mod(int *time,
             int  step);

void bc_ji(double *pt,
           int     index,
           int     k);

    /*
     *  The following functions are adapted from 
     *  Numerical Recipes in C:
     */
int    spline(double *, double *, int, double, double, double *);
double splint(double, double *, double *, double *, double);
double linint(double, double *, double *, double *, double);
void   four1( double [],unsigned long,int);
void   realft(double [],unsigned long,int);
void   vclr(double *, int, int);

#if defined(EPIC_AVS)
void grid_to_domain(domainspec *domain,
                    int         direction);
#endif

#if defined(EPIC_SINGLE) 
void bc_ji(double *pt,
           int     index,
           int     k);

void uvp_out(planetspec *planet);

void write_ep(planetspec *planet);

#elif defined(EPIC_MPI)
void mpispec_init(void);

#elif defined(EPIC_VIEW)
void hsv_to_rgb(double  h, double  s, double  v,
                double *r, double *g, double *b);

void make_layers_list(planetspec    *planet,
                      GEOMedit_list *list,
                      listspec      *info);

void make_sphere_list(planetspec    *planet,
                      GEOMedit_list *list,
                      listspec      *info);

void thermodynamic(char   *color_input,
                   double *therm);
#endif

/* * * * * * * * * *  end of epic.h  * * * * * * * * * * * * * * * * * * * * */ 
#endif

