/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef EPIC_SW_SCHEMES_H
#define EPIC_SW_SCHEMES_H
/* * * * * * * * * * * * *  epic_sw_schemes.h  * * * * * * * * * * * * * * * * 
 *                                                                           *
 *  Timothy E. Dowling                                                       *
 *                                                                           *
 *  There are two horizontal (shallow-water) advection schemes available:    *
 *                                                                           *
 *  AL81 stands for Arakawa and Lamb, 1981, "A potential enstrophy and       *
 *  energy conserving scheme for the shallow water equations,"               *
 *  Monthly Weather Review, 109: 18-36.                                      *
 *  This scheme conserves total energy and total enstrophy in general.       *                                            *
 *                                                                           *
 *  HA90 stands for Hsu and Arakawa, 1990, "Numerical modeling               *
 *  of the atmosphere with an isentropic vertical coordinate,"               *
 *  Monthly Weather Review, 118: 1933-1959.                                  *
 *  This scheme conserves total energy and total enstrophy for the case      *
 *  of nondivergent mass flux.  It is uniquely appropriate for the           *
 *  massless-layer approach of handling the intersection of the theta        *
 *  surfaces with the solid bottom of a terrestrial atmosphere.              *                                           *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#if SW_SCHEME == AL81
#  define AL_U (2.*Q(J+1,I+1)+Q(J+1,I)+2.*Q(J,I)+Q(J,I+1))
#  define BE_U (Q(J+1,I)+2.*Q(J+1,I-1)+Q(J,I-1)+2.*Q(J,I))
#  define GA_U (2.*Q(J+1,I)+Q(J+1,I-1)+2.*Q(J,I-1)+Q(J,I))
#  define DE_U (Q(J+1,I+1)+2.*Q(J+1,I)+Q(J,I)+2.*Q(J,I+1))
#  define EP1  (Q(J+1,I+1)+Q(J+1,I)-Q(J,I)-Q(J,I+1))
#  define EP2  (Q(J+1,I)+Q(J+1,I-1)-Q(J,I-1)-Q(J,I))
#  define AL_V (2.*Q(J,I+1)+Q(J,I)+2.*Q(J-1,I)+Q(J-1,I+1))
#  define BE_V (Q(J,I+1)+2.*Q(J,I)+Q(J-1,I)+2.*Q(J-1,I+1))
#  define GA_V (2.*Q(J+1,I+1)+Q(J+1,I)+2.*Q(J,I)+Q(J,I+1))
#  define DE_V (Q(J+1,I+1)+2.*Q(J+1,I)+Q(J,I)+2.*Q(J,I+1))
#  define PH1  (-Q(J+1,I+1)+Q(J+1,I)+Q(J,I)-Q(J,I+1))
#  define PH2  (-Q(J,I+1)+Q(J,I)+Q(J-1,I)-Q(J-1,I+1))
#  define SW_COEF (1./24.)
#elif SW_SCHEME == HA90
#  define AL_U (Q(J,I)+Q(J+1,I)+Q(J+1,I+1))
#  define BE_U (Q(J+1,I-1)+Q(J+1,I)+Q(J,I))
#  define GA_U (Q(J,I-1)+Q(J,I)+Q(J+1,I))
#  define DE_U (Q(J,I)+Q(J+1,I)+Q(J,I+1))
#  define EP1  0.
#  define EP2  0.
#  define AL_V (Q(J-1,I)+Q(J,I)+Q(J,I+1))
#  define BE_V (Q(J,I)+Q(J,I+1)+Q(J-1,I+1))
#  define GA_V (Q(J,I)+Q(J,I+1)+Q(J+1,I+1))
#  define DE_V (Q(J,I)+Q(J+1,I)+Q(J,I+1))
#  define PH1  0.
#  define PH2  0.
#  define SW_COEF (1./12.)
#endif

/* * * * * * * * * * end of epic_sw_schemes.h  * * * * * * * * * * * * * * * */ 
#endif
