/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * m_epic_functions() * * * * * * * * * * * * * * * * * */

#include <epic.h>

/*====================== mpispec_init() ======================================*/ 

/*
 *  Sets up mpi bookkeeping.  
 *  The mpispec structure para is defined globally in epic.h.
 */

void mpispec_init(void)
{
  int   
    i,dim;
  int
    idbms=0;
  char
    dbmsname[]="mpispec_init";

#if defined(EPIC_MPI)
  /* Define complex data type: */
  MPI_Type_contiguous(2,MPI_DOUBLE,&EPIC_MPI_COMPLEX);
  MPI_Type_commit(&EPIC_MPI_COMPLEX);
#endif

  /* 
   * Determine grid decompositions.
   *
   * Dimensions: zonal = 0, meridional = 1, vertical = 2 
   */
  para.nstart[0] = 1;
  /* set para.nstart[1] below */
  para.nstart[2] = 1;

  para.nend[0] = grid.ni;
  para.nend[1] = grid.nj;
  para.nend[2] = grid.nk;

  for (i = 0; i < TOPDIM; i++) {
    para.npad[i] = grid.pad[i];
    para.wrap[i] = grid.wrap[i];
  }

  /* set para.nprocs[0] below */
  para.nprocs[1] = para.nproc;
  para.nprocs[2] = 1;
  if (strcmp(grid.geometry,"globe") == 0) {
    /* do not decompose in zonal direction because of FFT filter */
    para.nprocs[0] = 1;
    /* 
     * J = 0 is an interior row (except for q or v, 
     * but it is harmless to treat them the same). 
     */
    para.nstart[ 1] = 0;
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
    /* The i-direction is not ready to be decomposed in the code. */
      para.nprocs[0] = 1;
      para.nstart[1] = 1;
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      /* do not decompose in zonal direction because of FFT filter */
      para.nprocs[0] = 1;
      para.nstart[1] = 0;
    }
    else {
      fprintf(stderr,"mpispec_init: Unrecognized f_plane_map \n");
      exit(1);
    }
  }
  else {
    fprintf(stderr,"m_epic_functions: Unrecognized geometry \n");
    exit(1);
  }

  for (dim = 0; dim < TOPDIM; dim++) {
    para.dimlen[dim] = para.nend[dim]-para.nstart[dim]+1;
  }

  MPG_Cart_decomp(para.comm,2,&(para.dimlen[1]),
                  &(para.nprocs[1]),&(para.wrap[1]),&(para.npad[1]),
                  &(para.mylo[1]),&(para.myhi[1]),&(para.arraydims[1]),&para.comm_kj);

  MPG_Cart_decomp(para.comm,TOPDIM,para.dimlen,
                  para.nprocs,para.wrap,para.npad,
                  para.mylo,para.myhi,para.arraydims,&para.comm_cart);

  MPI_Comm_rank(para.comm_cart,&para.iamnode);
  MPI_Comm_size(para.comm_cart,&para.nproc);
  grid.we_num_nodes   = para.nproc;
  grid.they_num_nodes = 1;
  para.ndim           = NINT(log((double)para.nproc)/log(2.));

  if (para.nproc > grid.nj) {
    fprintf(stderr,"This model must be run on <= %d processors\n", grid.nj);
    exit(0);
  }

  for (dim = 0; dim < TOPDIM; dim++) {
    /* Shift array endpoints */
    para.mylo[dim] += para.nstart[dim];
    para.myhi[dim] += para.nstart[dim];
  }

  para.nelem2d = para.arraydims[0]*para.arraydims[1];
  para.nelem3d = para.nelem2d*para.arraydims[2];

  /*
   *  The parameter jfirst is used to handle the staggered C-grid for q and v 
   *  variables in the globe geometry.
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    if (JLO == para.nstart[1]) {
      para.jfirst   = 1;
      if (grid.globe_latbot == -90.) {
        para.is_spole = 1;
      }
      else {
        para.is_spole = 0;
      }
    }
    else {
      para.is_spole = 0;
      para.jfirst   = JLO;
    }
    if (JHI == para.nend[1]) {
      if (grid.globe_lattop == 90.) {
        para.is_npole = 1;
      }
      else {
        para.is_npole = 0;
      }
    }
    else {
      para.is_npole = 0;
    }
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    para.is_spole = 0;
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      para.is_npole = 0;
      para.jfirst   = JLO;
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      if (JLO == para.nstart[1]) {
        para.jfirst   = 1;
      }
      else {
        para.jfirst   = JLO;
      }

      if (JHI == para.nend[1]) {
        para.is_npole = 1;
      }
      else {
        para.is_npole = 0;
      }
    }
  }

}

/*====================== end of mpispec_init() ====================================*/

/*====================== bc_ji() ==================================================*/

void bc_ji(double *pt,
           int     index,
           int     k)
{
  /*
   * This bc_ji() function is a wrapper function, useful in case we ever need 
   * to augment MPG_Cart_edgeexch().
   */

  MPG_Cart_edgeexch(para.comm_cart,2,para.dimlen,para.npad,MPI_DOUBLE,
                    pt-(IPAD+JPAD*Iadim+KPAD*Nelem2d)+(k-Kshift)*Nelem2d);

  return;
}

/*====================== end of bc_ji() ===========================================*/

/* * * * * * * * * * * * end of m_epic_functions() * * * * * * * * * * * * * * * * */
