/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * epic_change.c * * * * * * * * * * * * * * * * * 
 *                                                                 *
 *       Makes changes to parameters in epic.nc                    *
 *                                                                 *
 *       Use -ascii to write physical data to an ascii file.       *
 *       Use "-spots spots.dat" to add vortices.                   *
 *       Use -recursive to transform all the ncfiles in the        * 
 *                      current directory to ascii files with      *
 *                      ".dat" extension                           *
 *                                                                 *
 *       Timothy E. Dowling                                        *
 *       Expansion in zonal direction by R. LeBeau.                *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>
#include <stdio.h>
#include <string.h>

extern chem_element
  *Element;

/*
 * Function prototypes:
 */
void add_spots(planetspec *planet,
               char       *spots_file);

void read_defaults(change_defaultspec  *def);

void write_defaults(change_defaultspec *def);


/*======================= main() =====================================*/

int main(int   argc,
         char *argv[])
{
/*
 *  NOTE: structures planet, grid, var declared globally in epic.h.
 */
  char   
    spots_file[FILE_STR], /*  added-spot locations and sizes         */
    sflag[16],            /*  string to hold command-line flags      */
    infile[FILE_STR],
    outfile[FILE_STR],
    listfile[FILE_STR],
    datfile[FILE_STR],
    nc_nombre[FILE_STR];
  int    
    K,J,I,
    nk,nj,ni,new_ni,
    dt_cfl,
    flag,count,index,
    sw    = FALSE,
    ascii = FALSE,
    spots = FALSE,
    recursive = FALSE;        /* enables recursive ncfiles to ascii file conversion */
  long
    offset;               /*  used in make_arrays()                  */
  double  
    mult,uspinmin,uspinmax,
    cpr,
    dx0,dt,hyper,
    lat,rln,rlt,m_45,n_45,
    max_nu[MAX_NU_ORDER+1];
  change_defaultspec
    defaults;
  FILE   
    *input,              /*  input data file                            */
    *output,             /*  output data file, same format as input     */
    *nc_list;            /*  file containig the names of the ncfiles to be reduced */

  declare_copyright();

  /* 
   * Interpret command-line arguments: 
   */
  /* Start with defaults: */
  sprintf(spots_file,"none");
  if (argc > 1) {
    /* Read flags: */
    for (count = 1; count < argc; count++) {
      sscanf(argv[count],"%s",sflag);
      if (strcmp(sflag,"-spots") == 0) {
        sscanf(argv[++count],"%s",spots_file);
        spots = TRUE;
      }
      else if (strcmp(sflag,"-help") == 0 ||
               strcmp(sflag,"-h")    == 0) {
        /* Print help, exit: */
        system("more "EPIC_PATH"/help/epic_change.help");
        exit(1);
      }
      else if (strcmp(sflag,"-ascii") == 0) {
        ascii = TRUE;
      }
      else if (strcmp(sflag,"-recursive") == 0) {
        recursive = TRUE;
      }
      else {
        fprintf(stderr,"Unrecognized change command-line flag: %s \n",sflag);
        exit(1);
      }
    }
  }

  /* Allocate memory */
  if((planet=( planetspec *)malloc(sizeof( planetspec))) == 0) {
    fprintf(stderr,"epic_change: Error allocating space for planetspec \n");
    exit(0);
  }

#if defined(EPIC_MPI)
  MPI_Init(&argc,&argv);
  para.comm = MPI_COMM_WORLD;
  MPI_Errhandler_set(para.comm,MPI_ERRORS_RETURN);
  MPI_Comm_rank(para.comm,&para.iamnode);
  MPI_Comm_size(para.comm,&para.nproc);
  para.ndim = NINT(log((double)para.nproc)/log(2.));;
#endif

  /*
   *  Read in default parameter settings:
   */
  read_defaults(&defaults);

  if (recursive) {    /****** COMIENZO bucle recursive or not   ********/
    
      system("ls *.nc > ncfiles.txt");
      nc_list = fopen("ncfiles.txt","r"); 
      
      /* if (nc_list == NULL) {
       *     puts("El fichero de nombres ncfiles.txt no ha sido creado");
       *     exit(1);
       *}
       */
     
       while (fscanf(nc_list,"%s",nc_nombre) != EOF) {
             /*
              * Read epic.nc to determine model size, allocate memory for arrays,
              * and read in the rest of the data:
              */
             strcpy(infile,nc_nombre);
             var_read(planet,infile,0,VIA_FILE,INIT_DATA);
             make_arrays();
             var_read(planet,infile,0,VIA_FILE,NONINIT_DATA);
             if (strcmp(planet->name,"sw") == 0) {
                sw = TRUE;
             }

             if (!sw) {
                 /*
                  * Set up P. Gierasch's thermodynamics subroutines:
                  */
             thermo_setup(planet,&cpr);
             }

             /* Store heavily used diagnostic data: */
             fprintf(stderr,"\nCalculating and storing heavily used diagnostic variables...\n");
             store_temp_dens_p(planet,JLO,JHI,ILO,IHI);
            /* strcpy(nc_nombre,strtok(nc_nombre,":")); */
             strcpy(nc_nombre,strtok(nc_nombre,"."));
             strcat(nc_nombre,".dat");      /*anyado la exension .dat al nombre del fichero nc*/
             strcpy(datfile,nc_nombre);    
             write_ascii(planet,VIA_FILE,datfile);
             free_arrays(planet);
      }
      fclose(nc_list);
      system("rm ncfiles.txt");
      fprintf(stderr,"\nAll .nc_files in the current directory\nconverted to ascii files\n");
      exit(1);
  }
  else {    /********* -spots or -ascii (non recursive) **********/ 
  /*
   * Read epic.nc to determine model size, allocate memory for arrays,
   * and read in the rest of the data:
   */
  input_string("Input file [netCDF format]\n",defaults.infile,infile);
  var_read(planet,infile,0,VIA_FILE,INIT_DATA);
  make_arrays();
  var_read(planet,infile,0,VIA_FILE,NONINIT_DATA);
  if (strcmp(planet->name,"sw") == 0) {
    sw = TRUE;
  }

  if (!sw) {
      /*
       * Set up P. Gierasch's thermodynamics subroutines:
       */
    thermo_setup(planet,&cpr);
  }

  /* Store heavily used diagnostic data: */
  if (ascii || spots) {
    fprintf(stderr,"\nCalculating and storing heavily used diagnostic variables...\n");
    store_temp_dens_p(planet,JLO,JHI,ILO,IHI);
  }
  else {
    /* Only need JLO,ILO column */
    store_temp_dens_p(planet,JLO,JLO,ILO,ILO);
  }

  if (ascii) {
    /*
     * With -ascii flag set, just output physical data to ascii file
     * and exit.
     */
    input_string("Output data file [ascii format]\n",">",datfile);
    write_ascii(planet,VIA_FILE,datfile); 
    free_arrays(planet);
    exit(1);
  }
  }      /********* FIN bucle recursive or not  ************/

  nk = grid.nk;
  nj = grid.nj;
  ni = grid.ni;

  /* Calculate CFL dt: */
  dt_cfl = cfl_dt(planet,&dx0);

  /*
   * Compute hyperviscosity coefficients.
   */
  dt = (double)(grid.dt);
  max_nu[0] = (1./3.  )/dt;
  max_nu[2] = (1./30. )*dx0*dx0/dt;
  max_nu[4] = (1./240.)*pow(dx0,4.)/dt;
  max_nu[6] = (1./800.)*pow(dx0,6.)/dt;


  if (IAMNODE == 0) {
    fprintf(stderr,"\n");
    fprintf(stderr,"              Version: %3.1f\n",grid.data_version);
    /* Capitalize name for printing */
    planet->name[0] = (char)toupper((int)planet->name[0]);
    fprintf(stderr,"               System: %s \n",planet->name);
    /* Return name to lower case */
    planet->name[0] = (char)tolower((int)planet->name[0]);
    fprintf(stderr,"             Geometry: %s \n",grid.geometry);
    fprintf(stderr,"            Lon range: %-5.1f to %-5.1f deg\n",
           grid.lon[2*1],grid.lon[2*(grid.ni+1)]);
    fprintf(stderr,"            Lat range: %-5.1f to %-5.1f deg \n",
           grid.lat[2*(grid.jlo)],grid.lat[2*(grid.nj+1)]);
    fprintf(stderr,"\n");
    fprintf(stderr,"                 Time: %d yrs %d secs \n",
            var.time[1],var.time[0]);
    fprintf(stderr,"             Timestep: dt = %d s, CFL dt = %d s\n",
                    grid.dt,dt_cfl);
    fprintf(stderr,"                 Size: nk = %d, nj = %d, ni = %d \n",
           grid.nk,grid.nj,grid.ni);
    /* 
     * Print chemical list: 
     */
    for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
      if (var.chem_on[index] == CHEM_OFF) {
        ;
      }
      else if (var.chem_on[index] == CHEM_ACTIVE) {
        fprintf(stderr,"%21s: active \n",var.chem_name[index]);
      }
      else if (var.chem_on[index] == CHEM_PASSIVE) {
        fprintf(stderr,"%21s: passive \n",var.chem_name[index]);
      }
      else {
        fprintf(stderr,"Error: epic_change: unrecognized chem_on %d \n",
                        var.chem_on[index]);
        exit(1);
      }
    }
    /*
     * Print special chemical information:
     */
    if (var.chem_on[FPARA_INDEX] == CHEM_ACTIVE) {
      fprintf(stderr," para conversion time: t_p(1bar) = %8.1e s \n",
              var.time_fp_bar/1.e+5);
    }
    fprintf(stderr,"    Equation of state: %s \n",grid.eos);
    if (grid.newt_cool_on == TRUE) {
      fprintf(stderr,"    Newtonian cooling: on \n");
      if (grid.prandtl == 0.) {
        fprintf(stderr,"        Rayleigh drag: off \n");
      } 
      else {
        fprintf(stderr,"        Rayleigh drag: on, trad/tdrag = %4.2f \n",grid.prandtl);
      }
    }
    else {
      fprintf(stderr,"    Newtonian cooling: off \n");
      if (grid.prandtl >= 1.e+20) {
        fprintf(stderr,"        Rayleigh drag: off \n");
      }
      else {
        fprintf(stderr,"        Rayleigh drag: on, tdrag = %8.2e s \n",
                       1./grid.prandtl);
      }
    }
    if (grid.thermal_conduct_on == TRUE) {
      fprintf(stderr," Thermal conductivity: on \n");
    }
    else {
      fprintf(stderr," Thermal conductivity: off \n");
    }
    fprintf(stderr,"                nu[2]: %4.2f (%9.3e m^2/s)\n",
           grid.nu[2]/max_nu[2],grid.nu[2]);
    fprintf(stderr,"                nu[4]: %4.2f (%9.3e m^4/s)\n",
           grid.nu[4]/max_nu[4],grid.nu[4]);
    fprintf(stderr,"                nu[6]: %4.2f (%9.3e m^6/s)\n",
           grid.nu[6]/max_nu[6],grid.nu[6]);
    fprintf(stderr,"        hasten factor: %-4.1f \n",grid.hasten);

    /* 
     *  Print out vertical information:
     */
    print_vertical_column(planet,JLO,ILO);
  }

      /*
       *  Change parameters:
       */
  grid.dt = input_int("\nInput timestep\n", grid.dt);
  if (grid.newt_cool_on == TRUE) {
    grid.prandtl = input_double("Input Prandtl number [t_rad/t_drag]\n",
                                 grid.prandtl);
  }

  /*
   * Recompute hyperviscosity coefficients.
   */
  dt = (double)(grid.dt);
  max_nu[0] = (1./3.  )/dt;
  max_nu[2] = (1./30. )*dx0*dx0/dt;
  max_nu[4] = (1./240.)*pow(dx0,4.)/dt;
  max_nu[6] = (1./800.)*pow(dx0,6.)/dt;

  hyper = MIN(1.,grid.nu[2]/max_nu[2]);
  grid.nu[2] = max_nu[2]*input_double("Input nu[2] (fraction of max) \n",hyper);
  hyper = MIN(1.,grid.nu[4]/max_nu[4]);
  grid.nu[4] = max_nu[4]*input_double("Input nu[4] (fraction of max) \n",hyper);
  hyper = MIN(1.,grid.nu[6]/max_nu[6]);
  grid.nu[6] = max_nu[6]*input_double("Input nu[6] (fraction of max) \n",hyper);
  
  /* 
   * Modify spinup wind profile:
   */
  mult = 1.;
  mult = input_double("Input multiplier for spinup wind [1=no change]: \n",mult);
  uspinmin =  1.0e8;
  uspinmax = -1.0e8;
  if (mult != 1.) {
    if (IAMNODE == 0) {
      fprintf(stderr,"\nU_SPINUP multiplied by a factor of %f\n",mult);
    }
    for (K = 1; K <= nk; K++) {
      for (J = 1; J <= nj; J++) {
        for (I = 1; I <= ni; I++) {
          U_SPINUP(K,J,I) *= mult;
          if (U_SPINUP(K,J,I) < uspinmin) uspinmin = U_SPINUP(K,J,I);
          if (U_SPINUP(K,J,I) > uspinmax) uspinmax = U_SPINUP(K,J,I);
        }
      }
      BC2D(&(U_SPINUP(K,JLO,ILO)),NO_INDEX,K);
    }
    fprintf(stderr," %f < U_SPINUP < %f \n",uspinmin,uspinmax);
  }

  /*
   * Add vortices if requested:
   */
  if (spots) {
    add_spots(planet,spots_file);
  }

  input_string("Name of output file\n",defaults.outfile,outfile);
  var_write(planet,outfile,0,VIA_FILE,ALL_DATA);

  /* Write defaults file: */
  write_defaults(&defaults);

  free_arrays(planet);

  return 0;
}

/*======================= end of main() =====================================*/

/*======================= add_spots() =======================================*/

/*
 * Add gaussian spots based on parameters in spots_file. Use geostrophy
 * to balance u,v,p.
 * 
 * NOTE: Not MPI ready. 
 */

void add_spots(planetspec *planet,
               char       *spots_file)
{
  int
    K,J,I,
    nspots= 0,
    ispot,
    sw    = FALSE;
  double
    rr,rr1,xspot,yspot,zspot,fspot,
    exner,d_th0,theta,fpara,
    *mont,*mont3d,*ampspot,
    *lonspot,*latspot,*pspot,
    *aspot,*bspot,*cspot_up,*cspot_down,*nshape,
     factor;
  char
    *char_pt,
    buffer[FILE_STR],
    err_str[128];
  FILE
    *spots;

  if (strcmp(spots_file,"none") == 0) {
    /* Return if there is nothing to do: */
    return;
  }

  if (strcmp(planet->name,"sw") == 0) {
    /* Set shallow-water state to true: */
    sw = TRUE;
  }

  if (IAMNODE == 0) {
    /* Read vortex description file: */
    spots = fopen(spots_file,"r");
    if (!spots) {
      sprintf(err_str,"Error: add_spots: unable to open vortex file %s \n",spots_file);
      perror(err_str);
      exit(1);
    }
    /* Read number of vortices: */
    while (nspots == 0) {
      fgets(buffer,FILE_STR,spots);
      char_pt = strchr(buffer,':');
      if (char_pt) {
        sscanf(char_pt+1,"%d",&nspots);
      }
    }
  }

  /* 
   * Allocate memory:
   */
  lonspot = dvector(0,nspots-1);
  latspot = dvector(0,nspots-1);
  pspot   = dvector(0,nspots-1);
  aspot   = dvector(0,nspots-1);
  bspot   = dvector(0,nspots-1);
  cspot_up= dvector(0,nspots-1);
  cspot_down=dvector(0,nspots-1);
  ampspot = dvector(0,nspots-1);
  nshape  = dvector(0,nspots-1);
  mont    = dvector(0,Nelem2d-1);
  mont3d  = dvector(0,Nelem3d-1);

  /* 
   * Read in vortex information: 
   */
  fprintf(stderr,"\n");
  fgets(buffer,FILE_STR,spots);
  for (ispot = 0; ispot < nspots; ispot++) {
    fscanf(spots,"%lf %lf %lf %lf %lf %lf %lf %lf %lf",
           lonspot+ispot,latspot+ispot,pspot+ispot,
           aspot+ispot,bspot+ispot,cspot_up+ispot,cspot_down+ispot,ampspot+ispot,
           nshape+ispot);

    fprintf(stderr,"nshape[%d]=%f \n",ispot,nshape[ispot]);
    fprintf(stderr,"cspot_up[%d]=%f \n",ispot,cspot_up[ispot]);
    fprintf(stderr,"cspot_down[%d]=%f \n",ispot,cspot_down[ispot]);

    /* Convert pspot from mbar to mks: */
    pspot[ispot] *= 100.;
    /* 
     * Convert ampspot to amp for mont.  The factor named 'factor' 
     * should make the maximum spot velocity close to the input 
     * ampspot[m/s] for a gaussian spot mont.
     */
    fspot           = 2.*planet->omega*sin(latspot[ispot]*DEG);
    factor          = 1.-1./nshape[ispot];
    factor          = exp(factor)/(nshape[ispot]*pow(factor,factor));
    ampspot[ispot] *= factor*bspot[ispot]*DEG*planet->re*fabs(fspot);
    fgets(buffer,FILE_STR,spots);
  }
  if (IAMNODE == 0) {
    fclose(spots);
  }

  /* 
   * Calculate unmodified mont3d:
   */
  if (IAMNODE == NODE0) {
    fprintf(stderr,"\n Calculating mont prior to adding spots");
  }
  K = grid.nk;
  mont_nk(planet,mont3d+(K-Kshift)*Nelem2d);

  for (K = grid.nk-1; K >= 1; K--) {
    /* 
     * Calculate K-plane of mont3d from unmodified pressures
     * using hydrostatic balance:
     */
    theta = grid.theta[2*K+1];
    if (K > 1) {
      d_th0 = grid.theta[2*K]-grid.theta[2*K+2];
    }
    else {
      /* special delta-theta in top layer: */
      d_th0 = grid.theta[2*K]-grid.theta[2*K+2];
    }
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        if (sw) {
          /* kappa = 1. */
          exner = (planet->cp)*P(K,J,I)/grid.press0;
        }
        else {
          /* Calculate exner for the bottom of current layer: */
          exner = (planet->cp)*T2(K,J,I)/theta;
        }
        MONT3D(K,J,I) = MONT3D(K+1,J,I)+exner*d_th0;
      }
    }
    BC2D(&(MONT3D(KLO,JLO,ILO)),NO_INDEX,K);
    if (IAMNODE == NODE0) {
      fprintf(stderr,".");
    }
  }
  
  if (IAMNODE == NODE0) {
    fprintf(stderr,"done.\n");
  }

  /* 
   * Calculate 2D mont for vortices, and modify u and v:
   */
  if (IAMNODE == NODE0) {
    fprintf(stderr," Calculating mont for spots, modifying u,v");
  }
  for (K = KLO; K < KHI; K++) {
    /* Calculate 2D mont for vortices: */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        MONT(J,I) = 0.;
        for (ispot = 0; ispot < nspots; ispot++) {
          /* Account for periodicity in x-direction: */
          xspot  = fabs(grid.lon[2*I+1]-lonspot[ispot]);
          xspot  = (xspot < 180.) ? xspot : (360.-xspot);
          xspot /= aspot[ispot];
          yspot  = (grid.lat[2*J+1]-latspot[ispot])/bspot[ispot];
	  rr     = xspot*xspot+yspot*yspot;
          rr     = pow(rr,nshape[ispot]*0.5);
          if (P1(K,J,I) <= pspot[ispot]){
          	zspot  = log(P1(K,J,I)/pspot[ispot])/cspot_up[ispot];
	  }
          else{
                zspot  = log(P1(K,J,I)/pspot[ispot])/cspot_down[ispot];
          }

          rr1    = zspot*zspot; 
      
          /* skirt underflows: */
          if (rr < 10.) {
             MONT(J,I) += ampspot[ispot]*exp(-rr)*exp(-rr1);
          }
        }
      }
    }
    BC2D(&(MONT(JLO,ILO)),NO_INDEX,1);

    /* Form modified mont3d (add vortices): */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        MONT3D(K,J,I) += MONT(J,I);
      }
    }
    /* No need to apply BC2D here. */

    /* 
     * Modify u and v (|lat| > 10.) assuming geostrophic balance: 
     */
    for (J = JLO+1; J <= JHI-1; J++) {
      if (fabs(grid.lat[2*J+1]) > 10.) {
        for (I = ILO; I <= IHI; I++) {
          U(K,J,I) -= ( MONT(J+1,I-1)+MONT(J+1,I)
                       -MONT(J-1,I-1)-MONT(J-1,I) )*
                      .25*grid.n[2*J+1]/grid.f[2*J+1];
        }
      }
    }
    BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);
    for (J = JFIRST; J <= JHI; J++) {
      if (fabs(grid.lat[2*J]) > 10.) {
        for (I = ILO; I <= IHI; I++) {
          V(K,J,I) += ( MONT(J,I+1)+MONT(J-1,I+1)
                       -MONT(J,I-1)-MONT(J-1,I-1) )*
                      .25*grid.m[2*J]/grid.f[2*J];
        }
      }
    }
    BC2D(&(V(KLO,JLO,ILO)),V_INDEX,K);
    if (IAMNODE == NODE0) {
      fprintf(stderr,".");
    }
  }
  
  if (IAMNODE == NODE0) {
    fprintf(stderr,"done.\n");
  }

  /* 
   * Calculate new p from modified mont3d via hydrostatic balance.
   * The function set_p_balanced() is defined in epic_init_funcs.c.
   */
  if (IAMNODE == NODE0) {
    fprintf(stderr," Calculating new p from modified 3D mont.");
  }
  set_p_balanced(planet,mont3d);

  if (IAMNODE == NODE0) {
    for (K = 1; K < KHI; K++) {
      fprintf(stderr,".");
    }
    fprintf(stderr,"done.\n\n");
  }

  /* Free allocated memory: */
  free_dvector(mont3d, 0,Nelem3d-1);
  free_dvector(mont,   0,Nelem2d-1);
  free_dvector(nshape, 0,nspots-1);
  free_dvector(ampspot,0,nspots-1);
  free_dvector(cspot_down,  0,nspots-1);
  free_dvector(cspot_up,  0,nspots-1);
  free_dvector(bspot,  0,nspots-1);
  free_dvector(aspot,  0,nspots-1);
  free_dvector(pspot,  0,nspots-1);
  free_dvector(latspot,0,nspots-1);
  free_dvector(lonspot,0,nspots-1);

  return;
}

/*======================= end of add_spots() ================================*/

/*======================= read_defaults() ===================================*/

/*
 * NOTE: To use the READ* macros in epic_io_macros.h, dummy
 *       io and fd variables are declared.
 */

void read_defaults(change_defaultspec *def) 
{
  int
    nc_id,nc_err,
    fd,
    io_type=VIA_FILE,
    node   =NODE0;
  static char
    **gattname=NULL,
    **varname =NULL;
  static int
    ngatts=0,
    nvars =0;
  nc_type
    the_nc_type;
  FILE
    *io;

  lookup_netcdf(EPIC_PATH"/tmp/change_defaults.nc",
                &nc_id,&ngatts,&gattname,&nvars,&varname);

  strcpy(def->infile,"epic.nc");
    READC(NODE0,*&,def,->,infile, N_STR);
  strcpy(def->outfile,  "epic.nc");
    READC(NODE0,*&,def,->,outfile,N_STR);

  return;
}

/*======================= end of read_defaults() ============================*/

/*======================= write_defaults() ==================================*/

/*
 * NOTE: To use the WRITE* macros in epic_io_macros.h, dummy
 *       io and fd variables are declared.
 */

void write_defaults(change_defaultspec *def)
{
  int
    index,
    nc_id,nc_err,
    fd,
    io_type=VIA_FILE,
    node   =NODE0;
  nc_type
    the_nc_type;
  FILE
    *io;

  nc_err = nc_create(EPIC_PATH"/tmp/change_defaults.nc",NC_CLOBBER,&nc_id);

  WRITEC(NODE0,*&,def,->,infile, N_STR);
  WRITEC(NODE0,*&,def,->,outfile,N_STR);

  nc_close(nc_id);

  return;
}

/*======================= end of write_defaults() ===========================*/


/* * * * * * * * * * * * end of epic_change.c * * * * * * * * * * * * * * * * */

