/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * *  epic_funcs_diag.c  * * * * * * * * * * * * * * * * * * * *
 *                                                                           *
 *       Timothy E. Dowling                                                  *
 *                                                                           *
 *       Functions that calculate values of diagnostic variables.            *
 *       This file includes the following:                                   *
 *                                                                           *
 *           get_h(),get_p()                                                 *
 *           get_index(),get_pointer()                                       *
 *           get_mole_fraction()                                             *
 *           get_chem()                                                      *
 *           get_brunt2()                                                    *
 *           return_cp()                                                     *
 *           store_temp_dens_p()                                             *
 *           kcond()                                                         *
 *           laplacian()                                                     *
 *           potential_vorticity()                                           *
 *           angular_momentum()                                              *
 *           mont_geostrophic()                                              *
 *           mont_nk()                                                       *
 *           mixing_ratio()                                                  *
 *           relative_humidity()                                             *
 *           cfl_dt()                                                        *
 *           interface_value()                                               *
 *           time_mod()                                                      *
 *           galileo_u()                                                     *
 *           b_vir(),b1_vir(),b2_vir()                                       *
 *           sum_xx()                                                        *
 *           avg_molar_mass()                                                *
 *           molar_mass()                                                    *
 *           parse_chem_name()                                               *
 *           solar_fraction()                                                *
 *           min_richardson()                                                *
 *                                                                           *
 *           The following are C interface functions that call               *
 *           Fortran thermodynamics subroutines:                             *
 *                                                                           *
 *           thermo_setup()                                                  *
 *           return_temp()                                                   *
 *           return_dens()                                                   *
 *           return_theta()                                                  *
 *           return_press()                                                  *
 *           return_enthalpy()                                               *
 *           return_sat_vapor_p()                                            *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>
#include <chemical_elements.h>

/*======================= get_h() ===========================================*/

/*
 * Calculates h = -1/g dp/dtheta for kk = 2*k (doubled to handle half integers).
 */

double get_h(planetspec *planet,
             int         kk,
             int         J,
             int         I)
{
  int
    K;
  double 
    h,
    d_th1_inv,
    g_inv;

  g_inv = 1./planet->g;

  if (kk%2 == 0) {
    /* Layer value for h */
    K = kk/2;
    if (K == 1) {
      /* h in top layer has a special definition */
      d_th1_inv = g_inv/(grid.theta[2*K]-grid.theta[2*K+1]);
      h = P(K,J,I)*d_th1_inv;
    }
    else if (strcmp(planet->class,"gas-giant") == 0 &&
             K == grid.nk) {
      /* h in gas-giant deep layer is set to 1 */
      h = 1.;
    }
    else {
      d_th1_inv = g_inv/(grid.theta[2*K-1]-grid.theta[2*K+1]);
      h = (P(K,J,I)-P(K-1,J,I))*d_th1_inv;
    }
  }
  else {
    /* Interface value for h. K refers to layer underneath interface. */
    K = (kk+1)/2;
    if (K <= KLAST_ACTIVE) {
      d_th1_inv = g_inv/(grid.theta[2*(K-1)]-grid.theta[2*K]);
      h = (P1(K,J,I)-P1(K-1,J,I))*d_th1_inv;
    }
    else {
      /* h at bottom interface has special definition */
      d_th1_inv = g_inv/(grid.theta[2*(K-1)]-grid.theta[2*K-1]);
      h = (P(K-1,J,I)-P1(K-1,J,I))*d_th1_inv;
    }
  }

  /* 
   * Check for nonpositive h: 
   */
  if (h <= 0.) {
    fprintf(stderr,"Warning: get_h: H(%3.1f,%3d,%3d) = %9.2e, lat = %5.1f, lon = %6.1f\n",
                    (double)kk*.5,J,I,h,grid.lat[2*J+1],grid.lon[2*I+1]);
  }

  return h;
}

/*======================= end of get_h() =====================================*/

/*======================= get_p() ============================================*/

double get_p(planetspec *planet,
             int         index,
             int         kk,
             int         J,
             int         I) 
{
  int
    K;
  double
    pressure,
    partial_pressure,
    kappa_inv,kappap1;

  /*
   * Check for legal bounds on kk:
   */
  if (kk < 1) {
    fprintf(stderr,"Error: get_p(): kk = %d < 1\n",kk);
    exit(1);
  }
  else if (kk > 2*grid.nk) {
    fprintf(stderr,"Error: get_p(): kk = %d > 2*nk = %d\n",kk,2*grid.nk);
    exit(1);
  }
  else if (kk == 2*grid.nk) {
    /* 
     * Set pressure in deep layer equal to pressure
     * at interface between atmosphere and interior.
     *
     * NOTE: We do not treat a call for pressure in layer nk
     *       as an error, to allow t_rad(planet,pressure) in
     *       the calculation of Rayleigh drag for k = nk to work.
     */
    kk = 2*grid.nk-1;
  }

  if (kk == 1) {
    /* Top of top layer has pressure = 0. */
    pressure = 0.;
    return pressure;
  }
  else if (kk%2 == 0) {
    /*
     * Inside-the-layer pressure has a complicated definition.
     * See Hsu and Arakawa 1990 (5.43).
     */
    K         = kk/2;
    kappap1   = grid.kappa[K]+1.;
    kappa_inv = 1./(kappap1-1.);
    if (K == 1) {
      /* top of top layer has p = 0 */
      pressure = P(K,J,I)*pow(kappap1,-kappa_inv);
    }
    else {
      pressure = pow((pow(P(K,J,I),kappap1)-pow(P(K-1,J,I),kappap1))/
                     (kappap1*(P(K,J,I)-P(K-1,J,I))),kappa_inv);
    }
  }
  else {
    /* Interface value: */
    K        = (kk-1)/2;
    pressure = P(K,J,I);
  }

  /* 
   * Depending on index, return total pressure or partial pressure:
   */
  if (index == P_INDEX) {
    return pressure;
  }
  else if (index >= FIRST_HUMIDITY && index <= LAST_HUMIDITY) {
    partial_pressure = pressure*get_mole_fraction(planet,index,kk,J,I);
    return partial_pressure;
  }
  else {
    fprintf(stderr,"Error: get_p(): index = %d unknown \n",index);
    exit(1);
  }
}

/*======================= end of get_p() =====================================*/

/*======================= get_index() ========================================*/

/*
 * NOTE: The index value MAX_NVARS is context dependent. For
 *       array storage, var.vector[MAX_NVARS] is the tendency variable
 *       associated with the prognostic variable var.vector[0] (i.e., U_INDEX).  
 *       However, we also use MAX_NVARS to indicate DRY_AIR_INDEX.
 *
 *       To help prevent confusion, this function does not return
 *       a positive index value unless chem_name is a prognostic variable.
 */

int get_index(char *chem_name)
{
  int
    index;

  for (index = 0; index < MAX_NVARS; index++) {
    if (strcmp(chem_name,var.chem_name[index]) == 0) {
      return index;
    }
  }
    
  return NO_INDEX;
}

/*======================= end of get_index() =================================*/

/*======================= get_pointer() ======================================*/

double *get_pointer(char *name)
{
  int
    index;

  if (strcmp(name,"t") == 0) {
    return var.t;
  }
  else if (strcmp(name,"t2") == 0) {
    return var.t2;
  }
  else if (strcmp(name,"rho") == 0) {
    return var.rho;
  }
  else if (strcmp(name,"rho2") == 0) {
    return var.rho2;
  }
  else if (strcmp(name,"u_spinup") == 0) {
    return var.u_spinup;
  }
  else if (strcmp(name,"surface_gz") == 0) {
    return var.surface_gz;
  }

  for (index = 0; index < MAX_NVARS; index++) {
    if (strcmp(name,var.chem_name[index]) == 0) {
      return (var.vector[index]);
    }
    else if (strcmp(name,var.tend_name[index]) == 0) {
      return (var.vector[MAX_NVARS+index]);
    }
  }
  
  /* 
   * Default case: 
   */
  return NULL;
}

/*======================= end of get_pointer() ===============================*/

/*======================= get_mole_fraction() ================================*/

double get_mole_fraction(planetspec *planet,
                         int         chem_index,
                         int         kk,
                         int         J,
                         int         I)
{
  int
    index;
  static int
    initialized=0;
  double
    x,n_tot,chem;
  static double
    *n;

  if (!initialized) {
    /* Allocate memory for number array: */
    n           = dvector(FIRST_HUMIDITY,DRY_AIR_INDEX);
    initialized = 1;
  }

  if (!var.chem_on[chem_index]) {
    /* Return 0. if chemical not activated: */
    return 0.;
  }
  
  n_tot = 0.;
  for (index = FIRST_HUMIDITY; index <= DRY_AIR_INDEX; index++) {
    if (var.chem_on[index]) {
      chem      = get_chem(planet,index,kk,J,I);
      n[index]  = chem/molar_mass(planet,var.chem_name[index]);
      n_tot    += n[index];
/*      fprintf(stderr,"n[methane] = %8.2e   n_tot = %8.2e \n",n[index],n_tot);
*/
    }
  }
  
  if (n_tot > 0.) {
    x = n[chem_index]/n_tot;
  }
  else {
    fprintf(stderr,"Error: get_mole_fraction(): n_tot = %8.2e \n",n_tot);
    exit(1);
  }

  return x;
}

/*======================= end of get_mole_fraction() =========================*/

/*======================= get_chem() =========================================*/

/*
 * NOTE: The flag argument for interface_value(), which is used to make
 *       exact the case fpara = fpe, is not returned here.
 *
 */

double get_chem(planetspec *planet,
                int         index,
                int         kk,
                int         J,
                int         I)
{
  int
    ii,K,
    flag;
  double
    chem = 0.;

  if (index == P_INDEX) {
    chem = get_p(planet,P_INDEX,kk,J,I);
  }
  else if (index == DRY_AIR_INDEX) {
    chem = 1.;
    for (ii = FIRST_HUMIDITY; ii <= LAST_HUMIDITY; ii++) {
      chem -= get_chem(planet,ii,kk,J,I);
    }
  }
  else if (var.chem_on[index]) {
    if (kk%2 == 0) {
      /* Layer value for chem: */
      K    = kk/2;
      chem = VAR(index,K,J,I);
    }
    else {
      /* Interface value for chem: */
      K    = (kk-1)/2;
      chem = interface_value(planet,index,K,J,I,&flag);
    }
  }
  else {
    if (index == FPARA_INDEX) {
      /* High-temperature ("normal") value for fpara: */
      chem = .25;
    }
    else {
      chem = 0.;
    }
  }

  return chem; 
}

/*======================= end of get_chem() ==================================*/

/*======================= get_brunt2() =======================================*/

/*
 * A.P. Showman, 8/31/99.
 * See notes dated 8/31/99.
 *
 * Calculates and returns the squared Brunt-Vaisala (buoyancy) frequency at
 * position kk/2,J,I.  The formula used holds for any equation of state. It
 * incorporates a dry adiabatic lapse rate assuming no chemical reactions or
 * condensation. The environmental density structure takes into account the 
 * effects of vertical gradients of molar mass and entropy as well as 
 * compressibility. 
 *
 * The equation used is
 *
 *   N^2 = g^2{-[drho/dp]_T+(T/rho^2 cp)([drho/dT]_p)^2+Drho/Dp}      (1)
 *
 * where [drho/dx]_y is the partial derivative of rho with respect to x 
 * at const y, and Drho/Dp is the total derivative of rho along the 
 * environmental profile. This form is more accurate than 
 *
 *   N^2 = (g/theta)dtheta/dz                                         (2)
 *
 * which assumes the ideal gas law with no molar mass gradients and only
 * works for the traditional definition of theta.  EPIC uses a more 
 * general mean theta for hydrogen (ortho and para) such that (2) does
 * not yield the correct value of N^2.
 */

double get_brunt2(planetspec *planet,
                  int         kk,
                  int         J,
                  int         I)
{
  int
    K;
  double
    brunt2,        /* squared Brunt-Vaisala frequency, 1/s^2      */
    mu,            /* molar mass                                  */
    pressure,
    temperature,
    density,
    fpara,
    deltap,
    deltaT,
    drho_dp_T,     /* partial deriv of rho w/r to p at const T       */
    drho_dT_p,     /* partial deriv of rho w/r to T at const p       */
    Drho_Dp,       /* total deriv of environmental rho profile wrt p */
    cp,            /* specific heat at constant pressure             */
    g;

  /*
   * Limit range of kk. 
   * Assume answer is constant outside valid range.
   */
  kk = MAX(kk,2*KLO);
  kk = MIN(kk,2*KHI);

  K = kk/2;
  g = planet->g;

  if (kk%2 == 0) {
    /* 
     * Get values in layer:
     */
    pressure    = P1(  K,J,I);  
    temperature = T (  K,J,I);  
    density     = RHO(K,J,I);  
  }
  else {
    /* 
     * Get values at interface:
     */
    pressure    = P(   K,J,I);  
    temperature = T2(  K,J,I);  
    density     = RHO2(K,J,I);  
  }
  fpara = get_chem(planet,FPARA_INDEX,kk,J,I);
  cp    = return_cp(planet,fpara,pressure,temperature);
  mu    = avg_molar_mass(planet,kk,J,I);

  deltap = 0.001*pressure;
  deltaT = 0.001*temperature;

  drho_dp_T = (return_dens(planet,fpara,pressure+deltap,temperature,mu,PASSING_T) 
              -return_dens(planet,fpara,pressure-deltap,temperature,mu,PASSING_T))/
              (2.*deltap);

  drho_dT_p = (return_dens(planet,fpara,pressure,temperature+deltaT,mu,PASSING_T)
              -return_dens(planet,fpara,pressure,temperature-deltaT,mu,PASSING_T))/
              (2.*deltaT);

  if (kk%2 == 0) {
    if (K == 1) {
      Drho_Dp = (RHO2(K,J,I)-RHO(K,J,I))/
                (P(   K,J,I)-P1( K,J,I));
    }
    else {
      Drho_Dp = (RHO2(K,J,I)-RHO2(K-1,J,I))/
                (P(   K,J,I)-P(   K-1,J,I));
    }
  }
  else {
    Drho_Dp = (RHO(K+1,J,I)-RHO(K,J,I))/
              (P1(  K+1,J,I)-P1(  K,J,I));
  }

  brunt2 = g*g*(Drho_Dp-drho_dp_T
                +temperature/(cp*density*density)*drho_dT_p*drho_dT_p);

  return brunt2;
}

/*======================= end of get_brunt2() ================================*/

/*======================= return_cp() ========================================*/

/*  
 * A.P. Showman, 8/31/99.
 *
 * Return the specific heat at constant pressure. This is calculated as a 
 * derivative of the enthalpy, cp = (denthalpy/dT)_p, a partial derivative  
 * at constant p. All other state variables, such as fpara, water amount, etc,
 * are also to be held constant.  By using this method, cp will be correct
 * even if there are changes to the thermodynamics in return_enthalpy().
 * 
 * NOTE: thermo_setup() must have already been called at initialization.
 */

double return_cp(planetspec *planet,
                 double      fp,
                 double      p,
                 double      temp)
{
  double
    cp,
    deltaT,     
    fgibb,fpe,uoup;    

  deltaT = 0.001*temp;

  cp = (return_enthalpy(planet,fp,p,temp+deltaT,&fgibb,&fpe,&uoup)
       -return_enthalpy(planet,fp,p,temp-deltaT,&fgibb,&fpe,&uoup))/
       (2.*deltaT);

  return cp;
}

/*======================= end of return_cp() =================================*/


/*======================= store_temp_dens_p() ================================*/

/*
 * P1 refers to the layer value of pressure.
 * T refers to the layer value of temperature; T2 refers to the 
 * layer's bottom-interface value.
 */

void store_temp_dens_p(planetspec *planet,
                       int         jlo,
                       int         jhi,
                       int         ilo,
                       int         ihi) 
{
  int
    K,J,I,
    kk;
  double
    theta,
    fpara,
    pressure,
    mu;
  int
    idbms=0;
  char
    dbmsname[]="store_temp_dens_p";

  for (K = KLO; K <= KHI; K++) {
    /* 
     * Layer values: 
     */
    kk    = 2*K;
    theta = grid.theta[kk];
    for (J = jlo; J <= jhi; J++) {
      for (I = ilo; I <= ihi; I++) {
        fpara       = get_chem(planet,FPARA_INDEX,kk,J,I);
        pressure    = get_chem(planet,P_INDEX,    kk,J,I);
        P1(  K,J,I) = pressure;
        T(   K,J,I) = return_temp(planet,fpara,pressure,theta);
        mu          = avg_molar_mass(planet,kk,J,I);
        RHO( K,J,I) = return_dens(planet,fpara,pressure,T(K,J,I),mu,PASSING_T);
        /* 
         * Check for NaNs: 
         */
        if (isnan(P1( K,J,I))) fprintf(stderr,"  P1(%3d,%3d,%3d) = NaN ",K,J,I);
        if (isnan(T(  K,J,I))) fprintf(stderr,"   T(%3d,%3d,%3d) = NaN ",K,J,I);
        if (isnan(RHO(K,J,I))) fprintf(stderr," RHO(%3d,%3d,%3d) = NaN ",K,J,I);
      }
    }
    BC2D(&( P1(KLO,JLO,ILO)),NO_INDEX,K);
    BC2D(&(  T(KLO,JLO,ILO)),NO_INDEX,K);
    BC2D(&(RHO(KLO,JLO,ILO)),NO_INDEX,K);

    /* 
     * Lower interface values: 
     */
    if (K <= KLAST_ACTIVE) {
      kk    = 2*K+1;
      theta = grid.theta[kk];
      for (J = jlo; J <= jhi; J++) {
        for (I = ilo; I <= ihi; I++) {
          fpara       = get_chem(planet,FPARA_INDEX,kk,J,I);
          pressure    = P(K,J,I);
          T2(  K,J,I) = return_temp(planet,fpara,pressure,theta);
          mu          = avg_molar_mass(planet,kk,J,I);
          RHO2(K,J,I) = return_dens(planet,fpara,pressure,T2(K,J,I),mu,PASSING_T);
          if (isnan(T2(  K,J,I))) fprintf(stderr,"  T2(%3d,%3d,%3d) = NaN ",K,J,I);
          if (isnan(RHO2(K,J,I))) fprintf(stderr,"RHO2(%3d,%3d,%3d) = NaN ",K,J,I);
        }
      }
    }
    else {
      for (J = jlo; J <= jhi; J++) {
        for (I = ilo; I <= ihi; I++) {
          T2(  K,J,I) = T(  K,J,I);
          RHO2(K,J,I) = RHO(K,J,I);
        }
      }
    }
    BC2D(&(  T2(KLO,JLO,ILO)),NO_INDEX,K);
    BC2D(&(RHO2(KLO,JLO,ILO)),NO_INDEX,K);
  }

  return;
}

/*======================= end of store_temp_dens_p() =========================*/

/*======================= kcond() ============================================*/

/*
 * Calculates the thermal conductivity in the form: k = A*T^s.
 *
 * Note that k has units of J * sec^-1 * m^-1 * K^-(1+s).
 * Another common unit is the thermal diffusivity kappa = k/(rho*cp).
 * Kappa typically has units of m^2 * s^-1.
 */

double kcond(planetspec *planet,
             double      temperature)
{
  static int
    initialized=0;
  static double
    acond,
    scond;
  double
    k;

  if (!initialized) {
    if (strcmp(planet->name,"triton") == 0) {
      /* See Yelle, Lunine, and Hunten (1991, Icarus 89, 347-58). */
      acond = 2.6e-5;
      scond = 1.3;
    }
    else {
      fprintf(stderr,"Error: heating: kcond() not defined for %s \n",
                     planet->name);
      exit(1);
    }
    initialized = 1;
  }

  k = acond*pow(temperature,scond);

  return k;
}

/*==================== end of kcond() ========================================*/

/*==================== laplacian() ===========================================*/

#define UU(j,i)   uu[i+(j)*Iadim-Shift2d]
#define VV(j,i)   vv[i+(j)*Iadim-Shift2d]
#define HH(j,i)   hh[i+(j)*Iadim-Shift2d]
#define ZE(j,i)   ze[i+(j)*Iadim-Shift2d]
#define DI(j,i)   di[i+(j)*Iadim-Shift2d]
#define GH1(j,i) gh1[i+(j)*Iadim-Shift2d]
#define GH2(j,i) gh2[i+(j)*Iadim-Shift2d]
#define LP1(j,i) lp1[i+(j)*Iadim-Shift2d]
#define LP2(j,i) lp2[i+(j)*Iadim-Shift2d]
#define LPH(j,i) lph[i+(j)*Iadim-Shift2d]

void laplacian(planetspec *planet,
               int         K,
               double     *uu,   double *vv, 
               double     *lp1,  double *lp2,
               double     *buff1,double *buff2,
               double    (*kcond)(planetspec *,double))

      /*
       *  Calculates the 2D Laplacian of the vector (u,v) for layer K.
       *  Assumes u and v are staggered on the C-grid in
       *  spheroidal coordinates.
       *
       *  If vv == NULL, assumes uu points to an h-grid variable
       *  and returns the scalar laplacian in lp1.
       *
       *  If kcond != NULL, multiply by temperature-dependent
       *  diffusion coefficient, (*kcond)(planet,temperature), before
       *  taking divergence.
       *
       *  Pointers to memory for two working JI-plane buffers
       *  are passed in as buff1 and buff2.
       */
{
  int 
    J,I;
  double 
    zeta,
    h_edge,
    m_2j,n_2j,m_2jp1,n_2jp1,
    m_2jm1_inv,m_2jp1_inv,n_2jp1_inv,
    m_2j_inv,m_2jp2_inv,mn_2jp1;
  double
    pressure,temperature,fpara,theta;
  double
    *hh,*lph,
    *ze,*di,
    *gh1,*gh2;

  /* Zero working buffers: */
  memset(buff1,0,Nelem2d*sizeof(double));
  memset(buff2,0,Nelem2d*sizeof(double));

  if (uu != NULL && vv != NULL) {
    /* 
     * Laplacian on (u,v)-grid vector.
     *
     * Align pointers with 2D working arrays:
     */
    ze = buff1;
    di = buff2; 
    /*
     * Calculate vorticity:
     */
    for (J = JFIRST; J <= JHI; J++) {
      m_2j        =    (grid.m)[2*J  ];
      n_2j        =    (grid.n)[2*J  ];
      m_2jm1_inv  = 1./(grid.m)[2*J-1];
      m_2jp1_inv  = 1./(grid.m)[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        ZE(J,I) =  m_2j*( (VV(J,I) - VV(J,I-1)) +
                   n_2j*(UU(J-1,I)*m_2jm1_inv-UU(J,I)*m_2jp1_inv) );
      }
    }
    if (strcmp(grid.geometry,"f-plane")  == 0 &&
        strcmp(grid.f_plane_map,"polar") == 0) {
      if (JLO == 0) {
        /* Apply channel boundary condition to ze */
        /* Just take zeta = 0. */
        J = JLO;
        zeta = 0.;
        for (I = ILO; I <= IHI; I++) {
          ZE(J,I) = zeta;
        }
      }
      if (IS_NPOLE) {
        /* Calculate vorticity at the north pole: */
        zeta = 0.;
        for (I = ILO; I <= IHI; I++) {
          zeta += UU(grid.nj, I);
        }
        zeta *= (grid.mn)[2*(grid.nj+1)]/((grid.m)[2*grid.nj+1]*(double)(grid.ni));
        for (I = ILO; I <= IHI; I++) {
          ZE(grid.nj+1,I) = zeta;
        }
      }
    }
    else if (strcmp(grid.geometry,"globe") == 0) {
      if (grid.globe_latbot == -90.) {
        /* Calculate vorticity at the south pole: */
        if (IS_SPOLE) {
          zeta = 0.;
          for (I = ILO; I <= IHI; I++) {
            zeta  -= UU(0,I);  /*  Beware of southern circulation sign. */
          }
          zeta  *= (grid.mn)[0]/((grid.m[1])*(double)(grid.ni)); 
          for (I = ILO; I <= IHI; I++) {
            ZE(0,I) = zeta;
          }
        }
      }
      else {
        if (JLO == 0) {
          /* Apply channel boundary condition to ze */
          /* Just take zeta = 0. */
          zeta = 0.;
          for (I = ILO; I <= IHI; I++) {
            ZE(0,I) = zeta;
          }
        }
      }
      if (grid.globe_lattop == 90.) {
        /* Calculate vorticity at the north pole: */
        if (IS_NPOLE) {
          zeta = 0.;
          for (I = ILO; I <= IHI; I++) {
            zeta += UU(grid.nj, I);
          }
          zeta *= (grid.mn)[2*(grid.nj+1)]/((grid.m)[2*grid.nj+1]*(double)(grid.ni));
          for (I = ILO; I <= IHI; I++) {
            ZE(grid.nj+1,I) = zeta;
          }
        }
      }
      else {
        if (JHI == grid.nj) {
          /* Apply channel boundary condition to ze */
          /* Just take zeta = 0. */
          zeta = 0.;
          for (I = ILO; I <= IHI; I++) {
            ZE(grid.nj+1,I) = zeta;
          }
        }
      }
    }
    BC2D(&(ZE(JLO,ILO)),NO_INDEX,1);

    /*
     *  Calculate divergence:
     */
    for (J = JLO; J <= JHI; J++) {
      m_2jp1     = (grid.m)[2*J+1];
      n_2jp1     = (grid.n)[2*J+1];
      n_2jp1_inv = 1./n_2jp1;
      mn_2jp1    = (grid.mn)[2*J+1];
      if (J == grid.jlo && IS_SPOLE) {
        m_2j_inv = 0.;
      }
      else {
        m_2j_inv   = 1./(grid.m)[2*J  ];
      }
      if (J == grid.nj && IS_NPOLE) {
        m_2jp2_inv = 0.;
      }
      else {
        m_2jp2_inv = 1./(grid.m)[2*J+2];
      }
      for (I = ILO; I <= IHI; I++) {
        /* NOTE: mn != m*n at the pole, because the area is triangular. */
        DI(J,I) = mn_2jp1*( (UU(J,I+1)*n_2jp1_inv-UU(J,I)*n_2jp1_inv)
                           +(VV(J+1,I)*m_2jp2_inv-VV(J,I)*m_2j_inv  ) );
      }
    }
    BC2D(&(DI(JLO,ILO)),NO_INDEX,1);

    /*
     *  Calculate (lp1,lp2) = Laplacian(u,v):
     */
    for (J = JLO; J <= JHI; J++) {
      m_2jp1 = (grid.m)[2*J+1];
      n_2jp1 = (grid.n)[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        LP1(J,I) = -n_2jp1*(ZE(J+1,I)-ZE(J,I))
                   +m_2jp1*(DI(J,I)-DI(J,I-1));
      }
    }
    BC2D(&(LP1(JLO,ILO)),NO_INDEX,1);

    for (J = JFIRST; J <= JHI; J++) {
      m_2j = (grid.m)[2*J];
      n_2j = (grid.n)[2*J];
      for (I = ILO; I <= IHI; I++) {
        LP2(J,I) =  m_2j*(ZE(J,I+1)-ZE(J,I))
                   +n_2j*(DI(J,I)-DI(J-1,I));
      }
    }
    BC2D(&(LP2(JLO,ILO)),NO_INDEX,1);

    if (kcond != NULL) {
      fprintf(stderr,"Warning: laplacian(): kcond not yet implemented for (u,v)\n");
    }
  }
  else if (uu != NULL && vv == NULL) {
    /* 
     * Laplacian on h-grid scalar.
     * 
     * Align pointers: 
     */
    hh  = uu;
    lph = lp1;
    gh1 = buff1;
    gh2 = buff2;
    /* Compute grad h = (gh1,gh2) */
    for (J = JLO; J <= JHI; J++) {
      m_2jp1 = (grid.m)[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        GH1(J,I) = m_2jp1*(HH(J,I)-HH(J,I-1));    
      }
    }
    /* update gh1 edges below */
    for (J = JFIRST; J <= JHI; J++) {
      n_2j = (grid.n)[2*J];
      for (I = ILO; I <= IHI; I++) {
        GH2(J,I) = n_2j*(HH(J,I)-HH(J-1,I));
      }
    }

    /* 
     * Fill in gh2 for top and bottom channel boundaries. 
     * NOTE: Code only valid if i-direction is not cut. 
     */
    if (JLO == grid.jlo && !IS_SPOLE) {
      /* southern edge */
      h_edge = 0.;
      for (I = ILO; I <= IHI; I++) {
        h_edge += HH(JLO,I);
      }
      h_edge /= (IHI-ILO+1);
      n_2j = (grid.n)[2*JLO];
      for (I = ILO-IPAD; I <= IHI+IPAD; I++) {
        GH2(JLO,I) = n_2j*(HH(JLO,I)-h_edge);
      }
    }
    if (JHI == grid.nj && !IS_NPOLE) {
      /* northern edge */
      h_edge = 0.;
      for (I = ILO; I <= IHI; I++) {
        h_edge += HH(JHI,I);
      }
      h_edge /= (IHI-ILO+1);
      n_2j = (grid.n)[2*(JHI+1)];
      for (I = ILO-IPAD; I <= IHI+IPAD; I++) {
        GH2(JHI+1,I) = n_2j*(h_edge-HH(JHI,I));
      }
    }
    /* update gh2 edges below */

    if (kcond) {
      /* 
       * Multiply by temperature-dependent diffusion coefficient:
       */
      theta = grid.theta[2*K];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          fpara       = .5*(get_chem(planet,FPARA_INDEX,2*K,J,I  )+
                            get_chem(planet,FPARA_INDEX,2*K,J,I-1));
          pressure    = .5*(P1(K,J,I)+P1(K,J,I-1));
          temperature = return_temp(planet,fpara,pressure,theta);
          GH1(J,I)   *= (*kcond)(planet,temperature);
        }
      }
      for (J = JFIRST; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          fpara       = .5*(get_chem(planet,FPARA_INDEX,2*K,J,I  )+
                            get_chem(planet,FPARA_INDEX,2*K,J-1,I));
          pressure    = .5*(P1(K,J,I)+P1(K,J-1,I));
          temperature = return_temp(planet,fpara,pressure,theta);
          GH2(J,I)   *= (*kcond)(planet,temperature);
        }
      }
      /* 
       * Handle top and bottom channel boundaries. 
       * NOTE: Code only valid if i-direction is not cut. 
       */
      if (JLO == grid.jlo && !IS_SPOLE) {
        /* southern edge */
        for (I = ILO-IPAD; I <= IHI+IPAD; I++) {
          GH2(JLO,I) *= (*kcond)(planet,T(K,JLO,ILO));
        }
      }
      if (JHI == grid.nj && !IS_NPOLE) {
        /* northern edge */
        for (I = ILO-IPAD; I <= IHI+IPAD; I++) {
          GH2(JHI+1,I) *= (*kcond)(planet,T(K,JHI,ILO));
        }
      }
    }

    /* Update edges for gh1, gh2: */
    BC2D(&(GH1(JLO,ILO)),NO_INDEX,1);
    BC2D(&(GH2(JLO,ILO)),NO_INDEX,1);

    /* Compute lap h */
    for (J = JLO; J <= JHI; J++) {
      m_2jp1 = (grid.m)[2*J+1];
      n_2jp1 = (grid.n)[2*J+1];
      if (J == grid.jlo && IS_SPOLE) {
        m_2j_inv = 0.;
      }
      else {
        m_2j_inv = 1./(grid.m)[2*J];
      }
      if (J == grid.nj && IS_NPOLE) {
        m_2jp2_inv = 0.;
      }
      else {
        m_2jp2_inv = 1./(grid.m)[2*J+2];
      }
      for (I = ILO; I <= IHI; I++) {
        LPH(J,I) = m_2jp1*( (GH1(J,I+1)-GH1(J,I))
                    +n_2jp1*(GH2(J+1,I)*m_2jp2_inv-GH2(J,I)*m_2j_inv) );
      }
    }
    BC2D(&(LPH(JLO,ILO)),NO_INDEX,1);
  }
  else {
    fprintf(stderr,"Error in laplacian(); unrecognized input arguments \n");
  }

  return;
}

/*==================== end of laplacian() ====================================*/

/*==================== potential_vorticity() =================================*/

/*
 * Calculates potential vorticity for C-grid in 2D plane denoted by K.
 * If yes_h does not equal 1, only calculate absolute vorticity (zeta+f).
 */

void potential_vorticity(planetspec *planet,
                         double     *q,
                         int         K,
                         int         yes_h)
{
  int
    J,I;
  static int
    initialized=0;
  static double
    *h;
  double
    f_2j,m_2j,n_2j,mn_q,
    m_2jm1_inv,m_2jp1_inv,
    mn_2jm1_inv,mn_2jp1_inv,
    ze,zetabot,zetatop,h_q,q_pole,qbot,qtop;

  if (!initialized) {
    h           = dvector(0,Nelem2d-1);
    initialized = 1;
  }

  if (yes_h == 1) {
    /* Calculate h */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        H(J,I) = get_h(planet,2*K,J,I);
      }
    }
    BC2D(&(H(JLO,ILO)),NO_INDEX,1);
  }
  else {
    /* Set h to 1 */
    for (J = JLO-JPAD; J <= JHI+JPAD; J++) {
      for (I = ILO-IPAD; I <= IHI+IPAD; I++) {
        H(J,I) = 1.;
      }
    }
  }
  
  /* Calculate interior points */ 
  for (J = JFIRST; J <= JHI; J++) {
    f_2j        =    (grid.f )[2*J  ];
    m_2j        =    (grid.m )[2*J  ];
    n_2j        =    (grid.n )[2*J  ];
    m_2jm1_inv  = 1./(grid.m )[2*J-1];
    m_2jp1_inv  = 1./(grid.m )[2*J+1];
    mn_2jm1_inv = 1./(grid.mn)[2*J-1];
    mn_2jp1_inv = 1./(grid.mn)[2*J+1];
    mn_q        = .5/(mn_2jm1_inv+mn_2jp1_inv);
    for (I = ILO; I <= IHI; I++) {
      ze  =  m_2j*( (V(K,J,I)-V(K,J,I-1)) +
             n_2j*(U(K,J-1,I)*m_2jm1_inv-U(K,J,I)*m_2jp1_inv) );
      h_q = ( (H(J,  I)+H(J,  I-1))*mn_2jp1_inv  
             +(H(J-1,I)+H(J-1,I-1))*mn_2jm1_inv )*mn_q;
      Q(J,I) = (ze+f_2j)/h_q;
    }
  }
  if (strcmp(grid.geometry,"f-plane")  == 0 &&
      strcmp(grid.f_plane_map,"polar") == 0) {
    if (JLO == 0) {
      /*  Apply channel boundary condition for q: */
      J = 1;
      m_2j        =    (grid.m )[2*J  ];
      n_2j        =    (grid.n )[2*J  ];
      m_2jm1_inv  = 1./(grid.m )[2*J-1];
      m_2jp1_inv  = 1./(grid.m )[2*J+1];
      /* Calculate average zeta in next row */
      /* 
       * NOTE: this global sum is fine so long as the zonal direction is
       * not decomposed.
       */
      zetabot = 0.;
      h_q     = 0.;
      for (I = ILO; I <= IHI; I++) {
        zetabot +=  m_2j*( (V(K,J,I)-V(K,J,I-1)) +
                    n_2j*(U(K,J-1,I)*m_2jm1_inv-U(K,J,I)*m_2jp1_inv) );
        h_q += H(0,I);
      }
      J = 0;
      zetabot /= (IHI-ILO+1);
      h_q     /= (IHI-ILO+1);
      qbot     = (zetabot+(grid.f)[2*J])/h_q;
      for (I = ILO; I <= IHI; I++) {
        Q(J,I) = qbot;
      }
    }
    if (IS_NPOLE) {
      /* Calculate "north pole" q */
      ze  = 0.;
      h_q = 0.;
      for (I = ILO; I <= IHI; I++) {
        ze  += U(K,grid.nj,I);
        h_q += H(grid.nj,I);
      }
      ze     *= (grid.mn)[2*(grid.nj+1)]/((grid.m)[2*grid.nj+1]*(double)(grid.ni));
      f_2j    = (grid.f)[2*(grid.nj+1)];
      h_q    /= (double)(grid.ni);
      q_pole  = (ze+f_2j)/h_q;
      for (I = ILO; I <= IHI; I++) {
        Q(grid.nj+1,I) = q_pole;
      }
    }
  }
  else if (strcmp(grid.geometry,"globe") == 0) {
    if (grid.globe_latbot == -90.) {
      if (IS_SPOLE) {
        /* Calculate q at the south pole: */
        ze  = 0.;
        h_q = 0.;
        for (I = ILO; I <= IHI; I++) {
          ze  -= U(K,0,I);  /*  Beware of southern circulation sign. */
          h_q += H(0,I);
        }
        ze   *= (grid.mn)[0]/((grid.m[1])*(double)(grid.ni)); 
        f_2j  = (grid.f)[0];
        h_q    /= (double)(grid.ni);
        q_pole  = (ze+f_2j)/h_q;
        for (I = ILO; I <= IHI; I++) {
          Q(0,I) = q_pole;
        }
      }
    }
    else {
      if (JLO == 0) {
        /*  Apply channel boundary condition for q: */
        if (grid.globe_latbot == 0.) {
          /* special case at equator */
          J = 0;
          qbot = 0.;
        }
        else {
          /* Calculate average zeta in next row */
          J = 1;
          m_2j        =    (grid.m )[2*J  ];
          n_2j        =    (grid.n )[2*J  ];
          m_2jm1_inv  = 1./(grid.m )[2*J-1];
          m_2jp1_inv  = 1./(grid.m )[2*J+1];
          /* 
           * NOTE: this global sum is fine so long as the zonal direction is
           * not decomposed.
           */
          zetabot = 0.;
          h_q     = 0.;
          for (I = ILO; I <= IHI; I++) {
            zetabot +=  m_2j*( (V(K,J,I)-V(K,J,I-1)) +
                        n_2j*(U(K,J-1,I)*m_2jm1_inv-U(K,J,I)*m_2jp1_inv) );
            h_q += H(0,I);
          }
          J = 0;
          zetabot /= (IHI-ILO+1);
          h_q     /= (IHI-ILO+1);
          qbot     = (zetabot+(grid.f)[2*J])/h_q;
        }
        for (I = ILO; I <= IHI; I++) {
          Q(J,I) = qbot;
        }
      }
    }
    if (grid.globe_lattop == 90.) {
      if (IS_NPOLE) {
        /* Calculate q at the north pole: */
        ze  = 0.;
        h_q = 0.;
        for (I = ILO; I <= IHI; I++) {
          ze  += U(K,grid.nj,I);
          h_q += H(grid.nj,I);
        }
        ze     *= (grid.mn)[2*(grid.nj+1)]/((grid.m)[2*grid.nj+1]*(double)(grid.ni));
        f_2j    = (grid.f)[2*(grid.nj+1)];
        h_q    /= (double)(grid.ni);
        q_pole  = (ze+f_2j)/h_q;
        for (I = ILO; I <= IHI; I++) {
          Q(grid.nj+1,I) = q_pole;
        }
      }
    }
    else {
      if (JHI == grid.nj) {
        /*  Apply channel boundary condition for q: */
        if (grid.globe_lattop == 0.) {
          /* special case at equator */
          J = grid.nj+1;
          qtop = 0.;
        }
        else {
          /* Calculate average zeta in next row */
          J = grid.nj;
          m_2j        =    (grid.m )[2*J  ];
          n_2j        =    (grid.n )[2*J  ];
          m_2jm1_inv  = 1./(grid.m )[2*J-1];
          m_2jp1_inv  = 1./(grid.m )[2*J+1];
          /* 
           * NOTE: this global sum is fine so long as the zonal direction is
           * not decomposed.
           */
          zetatop = 0.;
          h_q     = 0.;
          for (I = ILO; I <= IHI; I++) {
            zetatop +=  m_2j*( (V(K,J,I)-V(K,J,I-1)) +
                        n_2j*(U(K,J-1,I)*m_2jm1_inv-U(K,J,I)*m_2jp1_inv) );
            h_q += H(grid.nj,I);
          }
          J = grid.nj+1;
          zetatop /= (IHI-ILO+1);
          h_q     /= (IHI-ILO+1);
          qtop     = (zetatop+(grid.f)[2*J])/h_q;
        }
        for (I = ILO; I <= IHI; I++) {
          Q(J,I) = qtop;
        }
      }
    }
  }
  BC2D(&(Q(JLO,ILO)),NO_INDEX,1);

  return;
}

/*======================= end of potential_vorticity() ==========================*/

/*======================= angular_momentum() ====================================*/

/* Calculates absolute zonal angular momentum for layer K */

void angular_momentum(planetspec *planet,
                      double     *buffji,
                      int         K)
{
  int
    J,I;
  double
    rln,omega,lat,re,rp;

  omega = planet->omega;

  for (J = JLO; J <= JHI; J++) {
    lat = grid.lat[2*J+1]*DEG;
    rln  = 1./(grid.m[2*J+1]*grid.dln*DEG);
    for (I = ILO; I <= IHI; I++) {
      BUFFJI(J,I) = rln*(U(K,J,I)+omega*rln);
    }
  }
  BC2D(&(BUFFJI(JLO,ILO)),NO_INDEX,1);

  return;
}

/*======================= end of angular_momentum() =============================*/

/*======================= mont_geostrophic() ====================================*/

#include <epic_sw_schemes.h>

void mont_geostrophic(planetspec *planet,
                      double     *mont,
                      double      mont_south,
                      int         K) 
/*
 *  Integrates the geostrophic-balance equation to calculate mont(J,I)
 *  for layer K.  The southern boundary value is input as mont_south.
 *
 *  NOTE: Currently assumes zonal symmetry. 
 *       (Must invert an elliptic operator to generalize.)
 *
 *  We use one 2D plane to enable the call to potential_vorticity().  
 *  This call is made in order to keep the vorticity code in one place.
 */
{
  int  
    J,I;
  static int 
    initialized=0;
  static double 
    *tmp,
    *q,
    *uh,
    *quh,
    *bern,
    *sendbuf;
  double
    kin,
    kin_south,
    fpara,pressure,temperature,theta,
    fgibb,fpe,uoup;

  if(!initialized) {
    initialized=1;
    tmp  = dvector(0,Nelem2d-1);
    uh   = dvector(0,Jadim-1);
    /* quh, bern are global spanning */
    quh     = dvector(0,grid.nj+1);
    bern    = dvector(0,grid.nj+1);
    sendbuf = dvector(0,grid.nj+1);
  }

  /* 
   * Calculate "uh," where h = 1.
   * In the absence of a "BC1D" function, start J loop at JFIRST-1.
   */
  I = ILO;
  for (J = JFIRST-1; J <= JHI; J++) {
    uh[J-Jshift] = U(K,J,I)/(grid.n)[2*J+1];
  }

  /* Calculate "q," which is zeta+f here */
  q = tmp;
  potential_vorticity(NULL,q,K,0);

  /* Calculate local quh*dy */
  memset(quh,0,(grid.nj+1+1)*sizeof(double));
  I = ILO;
  for (J = JFIRST; J <= JHI; J++) {
    /* Don't multiply by grid.n[2*J] to leave in dy factor: */
    quh[J] = (GA_V*uh[J  -Jshift]+DE_V*uh[J  -Jshift]
             +AL_V*uh[J-1-Jshift]+BE_V*uh[J-1-Jshift])*SW_COEF;
  }
  if (var.chem_on[FPARA_INDEX]) {
    /*
     * Include extra fpara term as part of quh:
     */
    theta = grid.theta[2*K];
    for (J = JFIRST; J <= JHI; J++) {
      /* Need to average quantities onto v grid: */
      fpara       = .5*(FPARA(K,J-1,I)+FPARA(K,J,I));
      /* 
       * NOTE: For the case when the pressure field has not yet been initialized,
       * rely on P1 being initialized by grid.p_avg[K] in make_arrays() for
       * an approximate pressure.
       */
      pressure    = .5*(P1(   K,J-1,I)+P1(   K,J,I));
      temperature = return_temp(planet,fpara,pressure,theta);
      return_enthalpy(planet,fpara,pressure,temperature,&fgibb,&fpe,&uoup);
      /* Don't multiply by grid.n[2*J] to leave in dy factor: */
      quh[J] += fgibb*(FPARA(K,J,I)-FPARA(K,J-1,I));
    }
  }
#if defined(EPIC_MPI)
  /*
   *  Fill in global-spanning quh for each node:
   */ 
  memcpy(sendbuf,quh,(grid.nj+1+1)*sizeof(double));
  MPI_Allreduce(sendbuf,quh,grid.nj+1+1,MPI_DOUBLE,MPI_SUM,para.comm);
#endif

  /* Broadcast kin for southernmost edge of model: */
  kin_south = 0.;
  if (JLO == grid.jlo) {
    kin_south = .5*U(K,JLO,ILO)*U(K,JLO,ILO);
  }
#if defined(EPIC_MPI)
  sendbuf[0] = kin_south;
  MPI_Allreduce(sendbuf,&kin_south,1,MPI_DOUBLE,MPI_SUM,para.comm);
#endif

  /*
   *  Calculate bern by integrating -quh*dy.
   *  For convenience, this global-spanning integral is calculated redundantly 
   *  on every node.
   */
  bern[0] = mont_south+kin_south;
  for (J = 1; J <= grid.nj; J++) { 
    /* the dy factor is already in quh */
    bern[J] = bern[J-1]-quh[J];
  }

  /*
   * Calculate mont:
   */
  for (J = JLO; J <= JHI; J++) {
    kin = (U(K,J,ILO+1)*U(K,J,ILO+1)+
           U(K,J,ILO  )*U(K,J,ILO  ))*.25;
    for (I = ILO; I <= IHI; I++) {
      MONT(J,I) = bern[J]-kin;
    }
  }
  BC2D(&(MONT(JLO,ILO)),NO_INDEX,1);

  return;
}

/*======================= end of mont_geostrophic() ============================*/

/*======================= mont_nk() ============================================*/

void mont_nk(planetspec *planet,
             double     *mont)

/*
 * For gas-giant planets, calls mont_geostrophic with k = nk.
 * For terrestrial planets, uses surface_gz to get mont_nk.
 */
{
  int
    K,J,I;
  double
    cp_thetas,
    fgibb,fpe,uoup;

  K = grid.nk;

  if (strcmp(planet->class,"gas-giant") == 0) {
    mont_geostrophic(planet,mont,grid.mont0,K);
  }
  else if (strcmp(planet->class,"terrestrial") == 0) {
    cp_thetas = planet->cp*grid.theta[2*K]/grid.theta[2*K+1];
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        /* 
         * Hydrostatic balance in the form:
         * dM/dtheta = Cp T/theta, M = H+gz, H = U+P/rho (enthalpy), 
         * does not assume the ideal-gas equation of state.
         */
        MONT(J,I) = SURFACE_GZ(J,I)+cp_thetas*T2(K,J,I);
      }
    }
    BC2D(&(MONT(JLO,ILO)),NO_INDEX,1);
  }
  else {
    fprintf(stderr,"Error in mont_nk(), planet->class = %s unrecognized.\n",
                    planet->class);
    exit(1);
  }

  return;
}

/*======================= end of mont_nk() =====================================*/

/*======================= mixing_ratio() =======================================*/

/*
 * Converts specific humidity (ratio of vapor density to total density)
 * to mixing ratio (ratio of vapor density to dry-air density).
 */

void mixing_ratio(int     chem_index,
                  double *buffji,
                  int     K,
                  int     bc2d)
{
  int
    J,I,
    index;
  double
    dry_air;

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      dry_air = 1.;
      for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
        if (var.chem_on[index]) {
          dry_air -= VAR(index,K,J,I);
        }
      }
      if (chem_index == NO_INDEX) {
        BUFFJI(J,I) /= dry_air;
      }
      else {
        BUFFJI(J,I) = VAR(chem_index,K,J,I)/dry_air;
      }
    }
  }

  if (bc2d == APPLY_BC2D) {
    BC2D(&(BUFFJI(JLO,ILO)),NO_INDEX,1);
  }

  return;
}

/*======================= end of mixing_ratio() ================================*/

/*======================= relative_humidity() ==================================*/

/*
 * Chemicals are carried as specific humidity (ratio of vapor
 * density to total density). Relative humidity is defined as the
 * mixing ratio (ratio of vapor density to dry-air density) divided by the 
 * saturation mixing ratio.
 */

#define SATURATION(j,i) saturation[i+(j)*Iadim-Shift2d]

void relative_humidity(planetspec *planet,
                       int         chem_index,
                       double     *buffji,
                       int         K)
{
  int
    J,I,
    kk,
    index;
  static int
    initialized=0;
  char
    chem_name[8];
  double
    theta,
    fpara, 
    temperature,
    pressure,
    mu,
    z_comp;
  static double
   *saturation;

  if (!initialized) {
    /* Allocate memory for saturation values: */
    saturation  = dvector(0,Nelem2d-1);
    initialized = 1;
  }

  kk = 2*K;

  /* Compute saturation specific humidity: */
  theta = grid.theta[2*K];
  strcpy(chem_name,var.chem_name[chem_index]);
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      pressure    = P1(K,J,I);
      temperature = T( K,J,I);
      /* 
       * Use equation of state to calculate 
       * saturation density times temperature: 
       */
      SATURATION(J,I) = return_sat_vapor_p(chem_name,temperature)*
                        molar_mass(planet,chem_name)/R_GAS;
      if (strcmp(grid.eos,"virial") == 0) {
        /* 
         * Make non-ideal equation of state correction:
         */
        z_comp           = 1.+b_vir(chem_name,chem_name,temperature)*pressure;
        SATURATION(J,I) /= z_comp;
      }
      /* Finish calculation of saturation specific humidity: */
      SATURATION(J,I) /= RHO(K,J,I)*temperature;
    }
  }
  /* No need to apply BC2D here */

  /* Convert to saturation mixing ratio: */
  mixing_ratio(NO_INDEX,saturation,K,SKIP_BC2D);

  /* Convert specific humidity to mixing ratio: */
  mixing_ratio(chem_index,buffji,K,SKIP_BC2D);

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      BUFFJI(J,I) /= SATURATION(J,I);
    }
  }
  BC2D(&(BUFFJI(JLO,ILO)),NO_INDEX,1);

  return;
}

/*======================= end of relative_humidity() ===========================*/

/*======================= cfl_dt() =============================================*/

/* 
 * Estimate CFL timestep for numerical stability.
 * Use sound speed as an upper bound on gravity-wave speed.
 * Writes dx0 to *pdx0.
 */

int cfl_dt(planetspec *planet,
           double     *pdx0)
{
  int
    K,J,I,
    cfl_dt,
    itmp,
    min_cfl_dt=INT_MAX,
    sw=0;
  double
    lat0,rln,rlt,m0,n0,
    dx,dx0,rgas,kappa,
    mn_2jp1,mn_2j_inv,
    mn_2jp2_inv,
    fpara,pressure,temperature,
    kinetic,speed,cs;

  if (strcmp(planet->name,"sw") == 0) {
    sw = 1;
  }

  /*
   * Set nominal spacing dx0:
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    lat0 = LAT0*DEG;  
    rln  = planet->re/sqrt( 1.+ pow(planet->rp/planet->re*tan(lat0),2.) );
    rlt  = rln/( cos(lat0)*( pow(sin(lat0),2.) +
                pow(planet->re/planet->rp*cos(lat0),2.) ) );
    m0   = 1./(rln*grid.dln*DEG);
    n0   = 1./(rlt*grid.dlt*DEG);
  }
  else if (strcmp(grid.geometry,"f-plane")  == 0) { 
    if (strcmp(grid.f_plane_map,"polar") == 0) {
      rln = .5*grid.f_plane_half_width;
      m0  = 1./(rln*grid.dln*DEG);
      n0  = 1./(grid.f_plane_half_width/grid.nj);
    }
    else {
      n0 = m0 = 1./(grid.f_plane_half_width/grid.ni);
    }
  }
  else {
    fprintf(stderr,"Error in epic_timestep: Unrecognized grid.geometry \n");
    exit(1);
  }
  dx0 = sqrt(1./(m0*m0+n0*n0));
  /* Write dx0 to *pdx0: */
  *pdx0 = dx0;

  for (K = KLO; K <= KLAST_ACTIVE; K++) {
    rgas  = grid.rgas[K];
    kappa = grid.kappa[K];
    for (J = JLO; J <= JHI; J++) {
      mn_2jp1     =    (grid.mn)[2*J+1];
      mn_2j_inv   = 1./(grid.mn)[2*J  ];
      mn_2jp2_inv = 1./(grid.mn)[2*J+2];
      for (I = ILO; I <= IHI; I++) {
        if (sw) {
          /* For sw, p holds gh */
          cs = sqrt(P(K,J,I));
        }
        else {
          /* calculate speed of sound: */
          cs = sqrt(rgas*T(K,J,I)/(1.-kappa));
        }
        dx = sqrt(1./((grid.m[2*J+1])*(grid.m[2*J+1])
                     +(grid.n[2*J+1])*(grid.n[2*J+1])));
        /* 
         * For polar grids, freeze dx poleward of lat = lat0.
         */
        dx = MAX(dx,dx0);
        kinetic = (      U(K,J,  I+1)*U(K,J,  I+1)+
                         U(K,J,  I  )*U(K,J,  I  )
               +mn_2jp1*(V(K,J+1,I  )*V(K,J+1,I  )*mn_2jp2_inv
                        +V(K,J,  I  )*V(K,J,  I  )*mn_2j_inv  ) )*.25;
        speed = sqrt(2.*kinetic);
        /* CFL timestep */
        cfl_dt = (int)(.5*dx/(cs+speed));
        if (cfl_dt < min_cfl_dt) {
          min_cfl_dt = cfl_dt;
        }
      }
    }
  }
  /*
   * Sanity bound:
   */
  if (min_cfl_dt < 0 || min_cfl_dt == INT_MAX) {
    min_cfl_dt = 0;
  }

#if defined(EPIC_MPI)
  /* determine global minimum */
  itmp = min_cfl_dt;
  MPI_Allreduce(&itmp,&min_cfl_dt,1,MPI_INT,MPI_MIN,para.comm);
#endif

  return min_cfl_dt;
}

/*======================= end of cfl_dt() ======================================*/

/*======================= interface_value() ====================================*/

/*
 * Calculates mass-weighted value of variable denoted by index on the bottom
 * interface of layer K.
 *
 */

double interface_value(planetspec *planet,
                       int         index,
                       int         K,
                       int         J,
                       int         I,
                       int        *flag)
{
  int
    ik,KK,fpe_flag;
  double
    value,v[2],h[2],
    d_th1_inv;
  double
    temp,fgibb,fpe,uoup,fpara;

  *flag    = 0;
  fpe_flag = 0;

  if (K < 1 || K > KLAST_ACTIVE) {
    fprintf(stderr,"error: interface_value() called for K = %d (nk = %d) \n",
                    K,KHI);
    exit(1);
  }

  if (K == KLAST_ACTIVE) {
    /* For bottom interface, take k = nk layer value (gas-giant or terrestrial): */
#if defined(EPIC_AVS) 
    value = FIELD(var.field,var.index[index],grid.nk,J,I);
#else
    value = (var.vector[index])[I+(J)*Iadim+(grid.nk)*Nelem2d-Shift3d];
#endif
  }
  else {
    for (ik = 0; ik <= 1; ik++) {
      KK = K+ik;
      if (KK == 1) {
        /* h in top layer has a special definition */
        d_th1_inv = 1./(grid.theta[2*KK]-grid.theta[2*KK+1]);
        h[ik] = P(KK,J,I)*d_th1_inv;
      }
      else {
        d_th1_inv = 1./(grid.theta[2*KK-1]-grid.theta[2*KK+1]);
        h[ik] = (P(KK,J,I)-P(KK-1,J,I))*d_th1_inv;
      }
#if defined(EPIC_AVS)
      v[ik] = FIELD(var.field,var.index[index],KK,J,I);
#else
      v[ik] = (var.vector[index])[I+(J)*Iadim+(KK)*Nelem2d-Shift3d];
#endif
      if (index == FPARA_INDEX) {
        /* Subtract fpe: */
        return_enthalpy(planet,FPARA(KK,J,I),0,T(KK,J,I),&fgibb,&fpe,&uoup);
        if (v[ik] == fpe) {
          fpe_flag += 1;
        }
        v[ik] -= fpe;
      }
    }
    value = (h[0]*v[0]+h[1]*v[1])/(h[0]+h[1]);
    if (index == FPARA_INDEX) {
      /* 
       * Add fpe back in.
       *
       * We are confronted with a Catch-22 here: we need
       * the interface_value of fpara to get the interface temperature
       * to calculate the interface fpe, in order to add it back in and get
       * the interface_value of fpara.  We will make the convention that
       * the fpara that generates the interface temperature for fpe is the 
       * straight average.
       */
      fpara = .5*(get_chem(planet,FPARA_INDEX,2*K,J,I)+
                  get_chem(planet,FPARA_INDEX,2*(K+1),J,I));
      temp  = return_temp(planet,fpara,P(K,J,I),grid.theta[2*K+1]);
      return_enthalpy(planet,fpara,0,temp,&fgibb,&fpe,&uoup);
      value += fpe;
      if (fpe_flag == 2) {
        /* Set flag = 1 to signal fp = fpe: */
        *flag = 1;
      }
    }
  }

  return value;
}

/*======================= end of interface_value() ============================*/

/*======================= time_mod() ==========================================*/

/* 
 * Return integer remainder of time/step.
 * Avoids directly forming YEAR*time.years to stay below INT_MAX.
 */

int time_mod(int *time,
             int  step) {

  int
    ans;

  ans = (time[0]%step+(YEAR%step)*(time[1]%step))%step;

  return ans;
}

/*====================== end of time_mod() ===================================*/

/*====================== galileo_u() =========================================*/

double galileo_u(double pressure) 
{
  char   
    header[N_STR],
    infile[N_STR];
  int
    kk;
  static int
    initialized=0,
    nup;
  double
    log_p,
    u,
    p_up_d;
  static double
    *p_up_dat,
    *u_up_dat,
    *u_up_y2dat;
  FILE
    *u_vs_p;

  if (!initialized) {
    if (IAMNODE == 0) {
      sprintf(infile,EPIC_PATH"/data/jupiter/u_vs_p.jupiter");
      u_vs_p = fopen(infile,"r");
      for (kk = 0; kk < 7; kk++) {
        fgets(header,100,u_vs_p); 
      }
      /* input number of data points */
      fscanf(u_vs_p, "%d", &nup); 
      for (kk = 0; kk < 4; kk++) {
        fgets(header,100,u_vs_p); 
      }
    }
#if defined(EPIC_MPI)
    MPI_Bcast(&nup,1,MPI_INT,NODE0,para.comm);
#endif
    /* Allocate memory: */
    p_up_dat   = dvector(0,nup-1);
    u_up_dat   = dvector(0,nup-1);
    u_up_y2dat = dvector(0,nup-1);

    if (IAMNODE == 0) { 
      /* in order of increasing p */
      for (kk = 0; kk < nup;  kk++) {  
        fscanf(u_vs_p, "%*lf %*lf %lf %lf",(p_up_dat+kk),(u_up_dat+kk));
        /* convert from bar to mks */
        p_up_dat[kk] *= 1.e+5;
      }
      fclose(u_vs_p);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(p_up_dat,nup,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(u_up_dat,nup,MPI_DOUBLE,NODE0,para.comm);
#endif

    for (kk = 0; kk < nup; kk++) {
      /* spline on log p (not neg log p) */
      p_up_dat[kk] = log(p_up_dat[kk]);  
    }
    spline(p_up_dat,u_up_dat,nup,1.e+30,1.e+30,u_up_y2dat);

    initialized = 1;
  }
  /* End of initialization. */

  /*
   *  Interpolate to get zonal wind:
   */
  log_p = log(pressure);
  if (log_p >= p_up_dat[112]) {
    /* Set u = u(16.429 bar) for large p (tangent to data): */
    u = u_up_dat[112];
  }
  else if (log_p <= p_up_dat[5]) {
    /* 
     * Use a 10% drop in u per scale height,
     * which is motivated by thermal-wind results.
     */
    u = u_up_dat[5]*pow(pressure/exp(p_up_dat[5]),log(10./9.));
  }
  else {
    kk = find_place_in_table(nup,p_up_dat,&log_p,&p_up_d);
    u  = splint(log_p,p_up_dat+kk,u_up_dat+kk,u_up_y2dat+kk,p_up_d);
  }

  return u;
}

/*====================== end of galileo_u() ==================================*/

/*====================== b_vir() =============================================*/

/*
 * Returns 2nd virial coefficient B_{ab} as a function of temperature.
 * For example, b_vir("H_2","H_2",temperature) returns B for pure molecular
 * hydrogen, H_2, whereas b_vir("H_2","He",temperature) returns the cross-term  
 * B for H_2+He. Data are taken from "The virial coefficients of pure gases
 * and mixtures," by J.H. Dymond and E.B. Smith (1980, Oxford), and converted
 * for a pressure expansion in mks units.
 */

#define MAX_CHEM_PAIRS 8

double b_vir(char   *chem_a,
             char   *chem_b,
             double  temperature) 
{
  char
    header[N_STR],
    infile[N_STR],
    chem_ab[16];
  /* Memory for up to MAX_CHEM_PAIRS different chemical pairs, 8 characters each: */
  static char
    list[MAX_CHEM_PAIRS][16];
  int
    i,j;
  static int
    ndat[MAX_CHEM_PAIRS],
    count=0;
  double 
    b,t_d;
  static double
    *tdat[MAX_CHEM_PAIRS],
    *bdat[MAX_CHEM_PAIRS],
    *b_y2[MAX_CHEM_PAIRS],
    first_dbdt[MAX_CHEM_PAIRS],
    last_dbdt[ MAX_CHEM_PAIRS];
  FILE
    *input;

  /* 
   * Determine list index for chemical pair: 
   */
  sprintf(chem_ab,"%s%s",chem_a,chem_b);
  i = 0;
  while (strcmp(chem_ab,list[i]) != 0 && i < count && i < MAX_CHEM_PAIRS) {
    i++;
  }

  if (i >= MAX_CHEM_PAIRS) {
    fprintf(stderr,"Error: b_vir() exceeded MAX_CHEM_PAIRS = %d\n",MAX_CHEM_PAIRS);
    exit(1);
  }
  else if (i == count) {
    /* 
     * Chemical pair not on list. 
     * Add to list and input data. 
     */
    if (IAMNODE == 0) {
      sprintf(list[count],"%s",chem_ab);
      if (strcmp(chem_a,chem_b) == 0) {
        sprintf(infile,EPIC_PATH"/data/chemistry/virial/b_vs_t.%s",chem_a);
        input = fopen(infile,"r");
        if (!input) {
          fprintf(stderr,"Warning: b_vir() cannot find %s \n",infile);
          fprintf(stderr,"         Defaulting to ideal equation of state for %s. \n",chem_a);
        }
      }
      else {
        sprintf(infile,EPIC_PATH"/data/chemistry/virial/b_vs_t.%sx%s",chem_a,chem_b);
        input = fopen(infile,"r");
        if (!input) {
          /* Try the names reversed: */
          sprintf(infile,EPIC_PATH"/data/chemistry/virial/b_vs_t.%sx%s",chem_b,chem_a);
          input = fopen(infile,"r");
          if (!input) {
            fprintf(stderr,"Warning: b_vir() cannot find %s \n",infile);
            fprintf(stderr,"         Defaulting to ideal equation of state for %s.\n",chem_ab);
          }
        }
      }
      if (input) {
        /* Skip over header: */
        for (j = 0; j < 6; j++) {
          fgets(header,100,input);  
        }
        /* Input number of data points: */
        fscanf(input,"%d",ndat+count); 
        /* Allocate memory: */
        tdat[count] = dvector(0,ndat[count]-1);
        bdat[count] = dvector(0,ndat[count]-1);
        b_y2[count] = dvector(0,ndat[count]-1);

        /* Input B(T): */
        for (j = 0; j < ndat[count]; j++) {
          fscanf(input,"%lf %lf %*lf",tdat[count]+j,bdat[count]+j);
          /* Convert for pressure expansion in mks units: */
          bdat[count][j] /= 1.e+3*R_GAS*tdat[count][j];
        }
        fclose(input);
      }
      else {
        /*
         * When a data file is not available, default to ideal equation of state
         * by setting set bdat's to zero:
         */
        ndat[count] = 3;
        tdat[count] = dvector(0,ndat[count]-1);
        bdat[count] = dvector(0,ndat[count]-1);
        b_y2[count] = dvector(0,ndat[count]-1);
        for (j = 0; j < ndat[count]; j++) {
          tdat[count][j] = (double)(j+1)*100.;
          bdat[count][j] = 0.;
        }
      }
      count++;
    }
#if defined(EPIC_MPI)
    MPI_Bcast(&count,                   1,MPI_INT,   NODE0,para.comm);
    MPI_Bcast(list[count-1],            8,MPI_CHAR,  NODE0,para.comm);
    MPI_Bcast(ndat+(count-1),           1,MPI_INT,   NODE0,para.comm);
    MPI_Bcast(tdat[count-1],ndat[count-1],MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(bdat[count-1],ndat[count-1],MPI_DOUBLE,NODE0,para.comm);
#endif
    /* Set endpoint slopes: */
    first_dbdt[count-1] = 
          (bdat[count-1][1]-bdat[count-1][0])/
          (tdat[count-1][1]-tdat[count-1][0]);
    last_dbdt[count-1]  = 
          (bdat[count-1][ndat[count-1]-1]-bdat[count-1][ndat[count-1]-2])/
          (tdat[count-1][ndat[count-1]-1]-tdat[count-1][ndat[count-1]-2]);
    /* Prepare for cubic-spline interpolation: */
    spline(tdat[count-1],bdat[count-1],ndat[count-1],
           first_dbdt[count-1],last_dbdt[count-1],b_y2[count-1]);
  }

  /* 
   * Main function evaluation:
   */

  /* Use cubic-spline interpolation: */
  if (temperature <= tdat[i][0]) {
    /* At or before start of table. */
    b = bdat[i][0]+first_dbdt[i]*(temperature-tdat[i][0]);
  }
  else if (temperature >= tdat[i][ndat[i]-1]) {
    /* At or after end of table. */
    b = bdat[i][ndat[i]-1]+last_dbdt[i]*(temperature-tdat[i][ndat[i]-1]);
  }
  else {
    j = find_place_in_table(ndat[i],tdat[i],&temperature,&t_d);
    b = splint(temperature,tdat[i]+j,bdat[i]+j,b_y2[i]+j,t_d);
  }

  return b;
}

/*====================== end of b_vir() ======================================*/

/*====================== b1_vir() ============================================*/

/*
 * Returns B1 = T dB/dT.
 */
double b1_vir(char  *chem_a,
              char  *chem_b,
              double temperature)
{
  double
    b1,tt;
  static double
    dt=1.;

  tt = temperature/dt;
  b1 = (b_vir(chem_a,chem_b,temperature+.5*dt)-
        b_vir(chem_a,chem_b,temperature-.5*dt))*tt;

  return b1;
}

/*====================== end of b1_vir() =====================================*/

/*====================== b2_vir() ============================================*/

/*
 * Returns B2 = T^2 (d/dT)^2 B.
 */
double b2_vir(char  *chem_a,
              char  *chem_b,
              double temperature)
{
  double
    b2,tt;
  static double
    dt=1.;

  tt = temperature/dt;
  b2 = (    b_vir(chem_a,chem_b,temperature+dt)
        -2.*b_vir(chem_a,chem_b,temperature   )
           +b_vir(chem_a,chem_b,temperature-dt))*tt*tt;

  return b2;
}

/*====================== end of b2_vir() =====================================*/

/*====================== sum_xx() ============================================*/
/*
 * Returns sum of 2nd virial coefficient, or related function,
 * with quadratic mole-fraction weighting appropriate to specified planet.
 */
double sum_xx(planetspec *planet,
              double    (*b_func)(char *,
                                  char *,
                                  double),
              double      temperature)
{
  static int
    initialized=0;
  double
    b_sum,x_sum;
  static double
    x_H_2,x_He;

  if (strcmp(grid.eos,"ideal") == 0) {
    /*
     * B = 0 for ideal case:
     */
    b_sum = 0.;
  }
  else if (strcmp(grid.eos,"virial") == 0) {
    if (strcmp(planet->class,"gas-giant") == 0) {
      if (!initialized) {
        /* NOTE: currently including only H_2+He: */
        x_sum  = planet->x_h2+planet->x_he;
        x_H_2  = planet->x_h2/x_sum;
        x_He   = planet->x_he/x_sum;
        initialized = 1;
      }
      b_sum =    (*b_func)("H_2","H_2",temperature)*x_H_2*x_H_2
             +2.*(*b_func)("H_2","He", temperature)*x_H_2*x_He
                +(*b_func)("He", "He", temperature)*x_He *x_He;
    }
    else if (strcmp(planet->name,"venus") == 0 ||
             strcmp(planet->name,"mars")  == 0) {
      b_sum = (*b_func)("CO_2","CO_2",temperature);
    }
    else if (strcmp(planet->name,"titan") == 0) {
      b_sum = (*b_func)("N_2","N_2",temperature);
    }
    else if (strcmp(planet->name,"earth") == 0) {
      b_sum = (*b_func)("N_2","N_2",temperature);
    }
    else {
      /* Default to ideal case. */
      if (!initialized) {
        if (IAMNODE == 0) {
          fprintf(stderr,"Warning: sum_xx(): equation of state = %s not defined for %s, "
                         "using ideal equation of state.\n",grid.eos,planet->name);
        }
        initialized = 1;
      }
      b_sum = 0.;
    }
  }
  else {
    fprintf(stderr,"Unrecognized equation of state = %s in sum_xx()\n",grid.eos);
    exit(1);
  }

  return b_sum;
}

/*====================== end of sum_xx() =====================================*/

/*====================== avg_molar_mass() ====================================*/

/*
 * Computes average molar mass at position kk/2,j,i.
 */

double avg_molar_mass(planetspec *planet,
                      int         kk,
                      int         j,
                      int         i) 
{
  int
    index;
  double
    mu=0.;

  for (index = FIRST_HUMIDITY; index <= DRY_AIR_INDEX; index++) {
    if (var.chem_on[index]) {
      mu += get_mole_fraction(planet,index,kk,j,i)*
            molar_mass(planet,var.chem_name[index]);
    }
  }

  return mu;
}

/*====================== end of avg_molar_mass() =============================*/

/*====================== molar_mass() ========================================*/

/*
 * Returns the molar mass (molecular weight) for the chemical name.
 * Units are kg/kmol, which is the same as g/mol.
 */
double molar_mass(planetspec *planet,
                  char       *chem_name)
{
  double 
    mu;
  int
    i,ii;
  static int
    num_elements=0,
    *counts     =NULL;
  static char
    **symbols   =NULL;

  /* Special cases: */
  if (strcmp(chem_name,"dry_air") == 0) {
    if (planet) {
      mu = R_GAS/planet->rgas;
    }
    else {
      fprintf(stderr,"Error: molar_mass(): planet not defined \n");
    }
  }
  else if (strcmp(chem_name,"dust") == 0) {
    /*
     * NOTE: Place holder until we get a viable dust model.
     * The book Mars (1992) mentions montmorillonite (clay)
     * and basalt as primary components of Martian dust.
     * The specific gravity (SG = rho/rho_H_2O) of montmorillonite
     * averages 2.3-3.  Assume SG = 3.0 and assume
     * mu = SG*mu_H_2O (a bad assumption).
     */
    mu = 3.*molar_mass(planet,"H_2O");
  }
  else if (strcmp(chem_name,"aerosol") == 0) {
    /*
     * NOTE: Place holder until we get a viable aerosol model.
     * Assume P_4.
     */
    mu = molar_mass(planet,"P_4");
  }
  else {
    parse_chem_name(chem_name,&num_elements,&symbols,&counts);
    mu = 0.;
    for (i = 0; i < num_elements; i++) {
      ii = 1;
      /* Identify element */
      while(strcmp(Element[ii].symbol,symbols[i]) != 0) {
        ii++;
      };
      mu += counts[i]*(Element[ii].molar_mass);
    }
  }

  return mu;
}

/*====================== end of molar_mass() =================================*/

/*====================== parse_chem_name() ===================================*/

/*
 * Takes a chemical-name string (e.g., "NH_4SH") and returns  
 * the number of distinct elements, their symbols, and how many atoms 
 * of each are present.  Reallocates the necessary memory to hold this
 * information.
 *
 * NOTE: num_elements and **symbols shoud be declared static in the calling 
 *       function, with their input values equal to the last call, 
 *       in order properly reallocate memory.
 *
 */
void parse_chem_name(char   *chem_name,
                     int    *num_elements,
                     char ***symbols,
                     int   **counts)
{
  int
    num_caps,
    i,ii,iii;
  char
    *ptr,
    subscript[4],
    format[4];

  /*
   * Free previous memory:
   */
  for (i = 0; i < *num_elements; i++) {
    free((*symbols)[i]);
  }

  /*
   * Count number of capital letters to determine working array sizes:
   */
  num_caps = 0;
  ptr      = chem_name;
  while (*ptr != '\0') {
    if (isupper(*ptr)) num_caps++;
    ptr++;
  }

  /*
   * Reallocate memory:
   */
  *counts  = (int   *)realloc(*counts, num_caps*sizeof(int   ));
  *symbols = (char **)realloc(*symbols,num_caps*sizeof(char *));
  for (i = 0; i < num_caps; i++) {
    (*symbols)[i] = (char *)calloc(4,sizeof(char));
  }

  /*
   * Determine symbols and counts:
   */
  i   = 0;
  ptr = chem_name;
  while (*ptr != '\0') {
    if (isupper(*ptr)) {
      if (islower(*(ptr+1))) {
        /* Element symbol has two letters: */
        strncpy((*symbols)[i],ptr,2);
        ptr+=2;
      }
      else {
        /* Element symbol has one letter: */
        strncpy((*symbols)[i],ptr,1);
        ptr+=1;
      }
      if (*ptr == '_') {
        subscript[0] = '\0';
        /* Determine subscript's number of places: */
        ii = 0;
        while(isdigit(*(++ptr))) {
          ii++;
        }
        if (ii == 0) {
          fprintf(stderr,"Error: parse_chem_name(): \"_\" not followed by digit: %s \n",
                          chem_name);
          exit(1);
        }
        else {
          sprintf(format,"\%%dd",ii);
          sscanf(ptr-ii,format,*counts+i);
        }
      }
      else {
        (*counts)[i] = 1;
      }
      i++;
    }
    else {
      ptr++;
    }
  }
  /*
   * Trim arrays to refer to distinct elements:
   */
  *num_elements = num_caps;
  for (i = 0; i < num_caps; i++) {
    for (ii = i+1; ii < num_caps; ii++) {
      if ((*counts)[ii] > 0 && strcmp((*symbols)[i],(*symbols)[ii]) == 0) {
        (*counts)[i ] += (*counts)[ii];
        (*counts)[ii]  = 0;
        (*num_elements)--;
      }
    }
  }
  /* Remove zero entries by shifting: */
  for (i = 0; i < num_caps; i++) {
    if ((*counts)[i] == 0) {
      for (ii = i; ii < num_caps-1; ii++) {
        (*counts)[ii] = (*counts)[ii+1];
        strcpy((*symbols)[ii],(*symbols)[ii+1]);
      }
      (*counts)[num_caps-1] = 0;
    }
  }
  /* Trim allocated memory */
  for (i = (*num_elements); i < num_caps; i++) {
    free((*symbols)[i]);
  }
  *counts  = (int   *)realloc(*counts, (*num_elements)*sizeof(int   ));
  *symbols = (char **)realloc(*symbols,(*num_elements)*sizeof(char *));

  return;
}

/*====================== end of parse_chem_name() ============================*/

/*====================== solar_fraction() ====================================*/

/*
 * Returns solar mixing ratio of the least-abundant element in
 * the given chemical name (divided by its stochiometric count).
 * Choices for the type argument: BY_NUMBER, BY_MASS.
 * The character string min_element should be 4 bytes.
 */

double solar_fraction(char *chem,
                      int   type,
                      char *min_element)
{
  int
    i,ii,ii_min,
    min_count;
  double
    ratio,
    min_abundance,
    abundance,tmp;
  static int
    initialized =0;
  static int
    num_elements=0,
    *counts     =NULL;
  static double
    total_number=0.,
    total_mass  =0.;
  static char
    **symbols   =NULL;

  if (!initialized) {
    /* Add up total number and total mass: */
    for (i = 1; i <= LAST_ATOMIC_NUMBER; i++) {
      abundance     = pow(10.,Element[i].solar_abundance);
      total_number += abundance;
      total_mass   += abundance*(Element[i].molar_mass);
    }

    initialized = 1;
  }

  /* Check for null string: */
  if (chem == NULL || *chem == '\0') {
    return 0.;
  }

  parse_chem_name(chem,&num_elements,&symbols,&counts);
  
  /*
   * Return ratio = 0. if num_elements is zero.
   */
  if (num_elements == 0) {
    ratio = 0.;
    return ratio;
  }

  /*
   * Find abundance of least-abundant element in chem:
   */
  min_abundance = Element[1].solar_abundance;
  min_count     = 1;
  ii_min        =-1;
  for (ii = 0; ii < num_elements; ii++) {
    /* Identify element */
    for (i = 1; i <= LAST_ATOMIC_NUMBER; i++) {
      if (strcmp(Element[i].symbol,symbols[ii]) == 0) {
        tmp = Element[i].solar_abundance;
        if (tmp <= min_abundance) {
          min_abundance = tmp;
          ii_min        = ii;
        }
        break;
      }
    }
  }

  /* Sanity check on ii_min: */
  if (ii_min < 0) {
    fprintf(stderr,"Error: solar_fraction(): ii_min < 0 \n");
    exit(1);
  }

  min_count = counts[ii_min];
  strcpy(min_element,symbols[ii_min]);

  abundance = pow(10.,min_abundance)/min_count;

  if (type == BY_NUMBER) {
    ratio = abundance/total_number;
  }
  else if (type == BY_MASS) {
    ratio = abundance*molar_mass(NULL,chem)/total_mass;
  }
  else {
    fprintf(stderr,"Error: solar_fraction(): Unknown type %d \n",type);
    exit(1);
  }

  return ratio;
}

/*====================== end of solar_fraction() =============================*/

/*====================== min_richardson() ====================================*/
        
/*
 * Determine minimum Richardson number. 
 * See Drazin and Reid, 1981, (44.19).
 *
 * Ri = -g*drho/dz / (rho (du/dz)^2 + 4.*u*du/dz*drho/dz)
 *
 *    = -(gh*dlnrho/dtheta) / (rho*(du/dtheta)(du/dtheta+4.*u*dlnrho/dtheta))
 *
 * Calculate Ri on the layer interfaces for the h-grid.
 *
 */
 
double min_richardson(planetspec *planet,
                      int        *k_min,
                      int        *j_min,
                      int        *i_min)
{
  int
    K,J,I,
    kk;
  double
    g,gh,
    u,dudth,
    rho,lnrho1,lnrho2,dlnrhodth,
    dth_inv,
    min_k,min_j,min_i,
    Ri_inv,max_Ri_inv,
    min_Ri;

  g          = planet->g;
  max_Ri_inv = 0.;

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      /* 
       * Do K-loop as the inner loop to reuse previous lnrho calculation. 
       * Skip K = 1 since this is an unusual layer.
       */
      K      = 2;
      lnrho1 = log(RHO(K,J,I));

      for (K = 2; K < KLAST_ACTIVE; K++) {
        kk        = 2*K+1;
        dth_inv   = 1./(grid.theta[2*K]-grid.theta[2*(K+1)]);
        gh        = g*get_h(planet,kk,J,I);

        /* Average u onto h-grid: */
        u         = 0.5*(get_chem(planet,U_INDEX,kk,J,I  )
                        +get_chem(planet,U_INDEX,kk,J,I+1));
        dudth     = 0.5*(U(K,J,I  )-U(K+1,J,I  )
                        +U(K,J,I+1)-U(K+1,J,I+1))*dth_inv;
        lnrho2    = lnrho1;
        rho       = RHO2(K,J,I);
        lnrho1    = log(RHO(K+1,J,I));
        dlnrhodth = (lnrho2-lnrho1)*dth_inv;

        /* Calculate inverse Ri since dudth can be zero. */
        Ri_inv     = -rho*dudth*(dudth+4.*u*dlnrhodth)/(gh*dlnrhodth);
        if (Ri_inv > max_Ri_inv) {
          max_Ri_inv = Ri_inv;
          min_k      = K;
          min_j      = J;
          min_i      = I;
        }
      }
    }
  }

  if (max_Ri_inv == 0.) {
    /* Trigger reporting "infinity" */
    min_Ri = DBL_MAX;
  }
  else {
    min_Ri = 1./max_Ri_inv;
  }

  *k_min = min_k;
  *j_min = min_j;
  *i_min = min_i;

  return min_Ri;
}

/*====================== end of min_richardson() =============================*/

/*====================== thermo_setup() ======================================*/

/*
 * This function initializes the Fortran thermodynamic functions.
 */
void thermo_setup(planetspec *planet,
                  double     *cpr)
{

/*
 * NOTE: LINUX appends one underscore for Fortran subroutine names without
 * an underscore (setup -> setup_) and two underscores for those with an
 * underscore (get_enthalpy -> get_enthalpy__).
 */

#if defined(ncube2)
    SETUP(  &(planet->x_h2),&(planet->x_he),&(planet->x_3),cpr);
#elif defined(hpux)
    setup(  &(planet->x_h2),&(planet->x_he),&(planet->x_3),cpr);
#elif defined(LINUX)
    setup_(&(planet->x_h2),&(planet->x_he),&(planet->x_3),cpr);
#else
    setup_( &(planet->x_h2),&(planet->x_he),&(planet->x_3),cpr);
#endif

  return;
}

/*====================== end of thermo_setup() ===============================*/

/*====================== return_temp() =======================================*/

#define MAXIT_RETURN_TEMP 60

/*
 * NOTE:  The calling program must initialize this function with a call
 *        to thermo_setup().
 */

double return_temp(planetspec *planet,
                   double      fp,
                   double      p,
                   double      theta)
{
  double
    b,b1,kappa,tmp,
    temperature;

#if defined(ncube2) 
  GET_TEMPERATURE(  &fp,&p,&theta,&temperature);
#elif defined(hpux) 
  get_temperature(  &fp,&p,&theta,&temperature);
#elif defined(LINUX)
  get_temperature__(&fp,&p,&theta,&temperature);
#else
  get_temperature_( &fp,&p,&theta,&temperature); 
#endif

  if (strcmp(grid.eos,"ideal") == 0) {
    return temperature;
  }
  else if (strcmp(grid.eos,"virial") == 0) {
    /*
     * Iterate to get temperature that satisfies
     * theta-return_theta(temperature) = 0.
     * Use Ridder's method to find root, Numerical Recipes in C, 
     * 2nd ed., p.358.
     */
    int
      j;
    double
      ans,fh,fl,fm,fnew,
      s,xh,xl,xm,xnew,xacc,x1,x2,
      theta_ortho,theta_para;

    xacc = temperature*1.e-9;
    x1   = temperature*0.9;
    x2   = temperature*1.1;
    fl   = theta-return_theta(planet,fp,p,x1,&theta_ortho,&theta_para);
    fh   = theta-return_theta(planet,fp,p,x2,&theta_ortho,&theta_para);
    if ((fl > 0. && fh < 0.) || (fl < 0. && fh > 0.)) {
      xl  = x1;
      xh  = x2;
      ans = DBL_MAX;
      for (j = 0; j < MAXIT_RETURN_TEMP; j++) {
        xm = 0.5*(xl+xh);
        fm = theta-return_theta(planet,fp,p,xm,&theta_ortho,&theta_para);
        s  = sqrt(fm*fm-fl*fh);
        if (s == 0.) {
          return ans;
        }
        xnew = xm+(xm-xl)*((fl > fh ? 1. : -1.)*fm/s);
        if (fabs(xnew-ans) <= xacc) {
          return ans;
        }
        ans  = xnew;
        fnew = theta-return_theta(planet,fp,p,ans,&theta_ortho,&theta_para);
        if (fnew == 0.) {
          return ans;
        }
        if ((fnew > 0. ? fabs(fm) : -fabs(fm)) != fm) {
          xl = xm;
          fl = fm;
          xh = ans;
          fh = fnew;
        }
        else if ((fnew > 0. ? fabs(fl) : -fabs(fl)) != fl) {
          xh = ans;
          fh = fnew;
        }
        else if ((fnew > 0. ? fabs(fh) : -fabs(fh)) != fh) {
          xl = ans;
          fl = fnew;
        }
        if (fabs(xh-xl) <= xacc) {
          return ans;
        }
      }
      fprintf(stderr,"Warning: return_temp(): exceeded MAXIT_RETURN_TEMP = %d \n",
                     MAXIT_RETURN_TEMP);
      return ans;
    }
    else {
      if (fl == 0.) {
        return x1;
      }
      if (fh == 0.) {
        return x2;
      }
      fprintf(stderr,"Error: return_temp(): not bracketed: fl,fh = %f %f \n",fl,fh);
      exit(1);
    }
  }

  return temperature;
}

/*======================= end of return_temp() ==============================*/

/*======================= return_dens() =====================================*/

double return_dens(planetspec *planet, 
                   double      fp,
                   double      p,
                   double      theta,
                   double      mu,
                   int         temp_type)
{
  double 
    temperature,
    density,
    b,z_comp;

  if (temp_type == PASSING_THETA) {
    temperature = return_temp(planet,fp,p,theta);
  }
  else if (temp_type == PASSING_T) {
    temperature = theta;
  }
  else {
    fprintf(stderr,"Error: return_dens(): unknown temp_type = %d \n",temp_type);
    exit(1);
  }

  density = p*mu/(R_GAS*temperature);

  if (strcmp(grid.eos,"virial") == 0) {
    /* 
     * Make non-ideal equation of state correction:
     */
    b        = sum_xx(planet,b_vir,temperature);
    z_comp   = 1.+b*p;
    density /= z_comp;
  }

  return density;
}

/*======================= end of return_dens() ==============================*/

/*======================= return_theta() ====================================*/

double return_theta(planetspec *planet,
                    double      fp,
                    double      p,
                    double      temperature,
                    double     *theta_ortho,
                    double     *theta_para)
{
  /*
   * NOTE:  The calling program must initialize this function with a call
   *        to thermo_setup().
   */
  double
    b,b1,kappa,tmp,
    theta;

#if defined(ncube2)
  GET_THETA(  &fp,&p,&temperature,&theta,theta_ortho,theta_para);
#elif defined(hpux)
  get_theta(  &fp,&p,&temperature,&theta,theta_ortho,theta_para);
#elif defined(LINUX)
  get_theta__(&fp,&p,&temperature,&theta,theta_ortho,theta_para);
#else
  get_theta_( &fp,&p,&temperature,&theta,theta_ortho,theta_para);
#endif

  if (strcmp(grid.eos,"virial") == 0) {
    /* 
     * Make non-ideal equation of state corrections:
     */
    kappa         = planet->kappa;
    b             = sum_xx(planet,b_vir, temperature);
    b1            = sum_xx(planet,b1_vir,temperature);
    tmp           = exp(-p*(b+b1)*kappa);
    theta        *= tmp;

    kappa         = planet->kappa*R_GAS/(2.016*planet->rgas);
    b             = b_vir( "H_2","H_2",temperature);
    b1            = b1_vir("H_2","H_2",temperature);
    tmp           = exp(-p*(b+b1)*kappa);
    *theta_ortho *= tmp;
    *theta_para  *= tmp;
  }

  return theta;
}

/*======================= end of return_theta() =============================*/

/*======================= return_press() ====================================*/

#define MAXIT_RETURN_PRESS 60

/*
 * NOTE:  The calling program must initialize this function with a call
 *        to thermo_setup().
 */

double return_press(planetspec *planet,
                    double      fp,
                    double      temperature,
                    double      theta)
{
  int
    j;
  double
    press,
    theta_ortho,
    theta_para;
  double
    ans,fh,fl,fm,fnew,
    s,xh,xl,xm,xnew,xacc,x1,x2;

  /* Sanity checks: */
  if (fp <= 0.) {
    fprintf(stderr,"\nError: return_press(): fp = %e <= 0.\n",fp);
    exit(1);
  }
  if (temperature <= 0.) {
    fprintf(stderr,"\nError: return_press(): temperature = %e <= 0.\n",temperature);
    exit(1);
  }
  if (theta <= 0.) {
    fprintf(stderr,"\nError: return_press(): theta = %e <= 0.\n",theta);
    exit(1);
  }

  /* First guess: */
  press = grid.press0*pow(temperature/theta,1./planet->kappa);

  /*
   * Iterate to get pressure that satisfies
   * theta-return_theta(pressure) = 0.
   * Use Ridder's method to find root, Numerical Recipes in C, 
   * 2nd ed., p.358.
   */

  xacc = press*1.e-9;
  x1   = press*0.01;
  x2   = press*100.;
  fl   = theta-return_theta(planet,fp,x1,temperature,&theta_ortho,&theta_para);
  fh   = theta-return_theta(planet,fp,x2,temperature,&theta_ortho,&theta_para);

  if ((fl > 0. && fh < 0.) || (fl < 0. && fh > 0.)) {
    xl  = x1;
    xh  = x2;
    ans = DBL_MAX;
    for (j = 0; j < MAXIT_RETURN_PRESS; j++) {
      xm = 0.5*(xl+xh);
      fm = theta-return_theta(planet,fp,xm,temperature,&theta_ortho,&theta_para);
      s  = sqrt(fm*fm-fl*fh);
      if (s == 0.) {
        return ans;
      }
      xnew = xm+(xm-xl)*((fl > fh ? 1. : -1.)*fm/s);
      if (fabs(xnew-ans) <= xacc) {
        return ans;
      }
      ans  = xnew;
      fnew = theta-return_theta(planet,fp,ans,temperature,&theta_ortho,&theta_para);
      if (fnew == 0.) {
        return ans;
      }
      if ((fnew > 0. ? fabs(fm) : -fabs(fm)) != fm) {
        xl = xm;
        fl = fm;
        xh = ans;
        fh = fnew;
      }
      else if ((fnew > 0. ? fabs(fl) : -fabs(fl)) != fl) {
        xh = ans;
        fh = fnew;
      }
      else if ((fnew > 0. ? fabs(fh) : -fabs(fh)) != fh) {
        xl = ans;
        fl = fnew;
      }
      if (fabs(xh-xl) <= xacc) {
        return ans;
      }
    }
    fprintf(stderr,"Warning: return_press(): exceeded MAXIT_RETURN_PRESS = %d \n",
                   MAXIT_RETURN_PRESS);
    return ans;
  }
  else {
    if (fl == 0.) {
      return x1;
    }
    if (fh == 0.) {
      return x2;
    }
    fprintf(stderr,"\nError: return_press(): not bracketed: fl,fh = %f %f \n",fl,fh);
    exit(1);
  }

  return press;
}

/*======================= end of return_press() =============================*/

/*================= enthalpy_condensation() ===================================*/

/*
 *  P. Stratman
 *  Returns the enthalpy of condensation of a given condensate at a 
 *  given temnperature in mks units.
 */

double enthalpy_condensation(char   *chem,
			     double  temperature)
{
  char
    header[N_STR],
    infile[N_STR];
  int
    index,j,i;
  static int
    ndat[MAX_NVARS],
    initialized=0;
  double
    enth,t_d;
  static double
    *tdat[MAX_NVARS],
    *enthdat[MAX_NVARS],
    *enth_y2[MAX_NVARS],
    first_denthdt[MAX_NVARS],
    last_denthdt[MAX_NVARS];
  FILE
    *enth_vs_t;
  /*
   * Initialization:
   */
  if (!initialized) 
  {
    /*
     * Loop through known condensables and set up data table splines:
     */
    for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
      if (IAMNODE == NODE0) 
      {
	sprintf(infile,EPIC_PATH"/data/chemistry/enthalpy/condensation/enth_vs_t.%s",
		var.chem_name[index]);
	enth_vs_t = fopen(infile,"r");
	if (!enth_vs_t) {
	  fprintf(stderr,"Error: enthalpy_condensation(): failed to open %s.\n",infile);
	  exit(1);
	}
	/* Skip over header: */
	for (j = 0; j < 6; j++) {
	  fgets(header,100,enth_vs_t);
	}
	/* Input number of data points: */
	fscanf(enth_vs_t,"%d",ndat+index);
      }

#if defined(EPIC_MPI)
      MPI_Bcast(ndat+index,1,MPI_INT,NODE0,para.comm);
#endif

      /* Allocate memory: */
      tdat[   index] = dvector(0,ndat[index]-1);
      enthdat[index] = dvector(0,ndat[index]-1);
      enth_y2[index] = dvector(0,ndat[index]-1);

      if (IAMNODE == NODE0) {
	/* Input enth(T): */
	for (j = 0; j < ndat[index]; j++) 
	{
	  fscanf(enth_vs_t,"%lf %lf",tdat[index]+j,enthdat[index]+j);
	}
	fclose(enth_vs_t);
      }

#if defined(EPIC_MPI)
      MPI_Bcast(   tdat[index],ndat[index],MPI_DOUBLE,NODE0,para.comm);
      MPI_Bcast(enthdat[index],ndat[index],MPI_DOUBLE,NODE0,para.comm);
#endif

    /* Calculate cubic spline coefficients: */

    first_denthdt[index] = 
       (enthdat[index][1]-enthdat[index][0])/
       (tdat[index][1]-tdat[index][0]);
    last_denthdt [index] = 
       (enthdat[index][ndat[index]-1]-enthdat[index][ndat[index]-2])/
       (tdat[index][ndat[index]-1]-tdat[index][ndat[index]-2]);

    }

    initialized = 1;

  }

  /* End of initialization. */
  
  index = get_index(chem);

  /* Apply cubic spline: */

  if (temperature <= tdat[index][0]) {

   /* at or before start of table */

    enth = enthdat[index][0]+first_denthdt[index]*(temperature-tdat[index][0]);
  }

  else if (temperature >= tdat[index][ndat[index]-1]) {

   /* at or after end of table   */

    enth = enthdat[index][ndat[index]-1]+last_denthdt[index]*(temperature-tdat[index] [ndat[index] - 1]);
  }

  else {

    j    = find_place_in_table(ndat[index], tdat[index], &temperature, &t_d);
    enth = linint(temperature, tdat[index]+j,
                                  enthdat[index]+j, enth_y2[index]+j, t_d);
  }

  return enth;
}

/*================= end of enthalpy_condensation() ==============================*/

/*======================= return_enthalpy() =================================*/

double return_enthalpy(planetspec *planet,
                       double      fp,
                       double      pressure,
                       double      temperature,
                       double     *fgibb,
                       double     *fpe,
                       double     *uoup)
{
  /*
   * NOTE:  The calling program must initialize this function with a call
   *        to thermo_setup().
   */
  double
    b1,
    rgas,
    enthalpy;

#if defined(ncube2)
  GET_ENTHALPY(  &fp,&temperature,&enthalpy,fgibb,fpe,uoup);
#elif defined(hpux)
  get_enthalpy(  &fp,&temperature,&enthalpy,fgibb,fpe,uoup);
#elif defined(LINUX)
  get_enthalpy__(&fp,&temperature,&enthalpy,fgibb,fpe,uoup);
#else
  get_enthalpy_( &fp,&temperature,&enthalpy,fgibb,fpe,uoup);
#endif

  if (strcmp(grid.eos,"virial") == 0) {
    /* 
     * Make non-ideal equation of state corrections.
     * The quantities fgibb and uoup are differences between ortho and para
     * hydrogen.  Since we are presently not distinguishing these in the
     * non-ideal equation of state, we make no corrections to fgibb and uoup.
     */
    b1        = sum_xx(planet,b1_vir,temperature);
    enthalpy -= pressure*temperature*b1;
  }

  rgas      = planet->rgas;
  enthalpy *= rgas;
  *fgibb   *= rgas;
  *uoup    *= rgas;

  return enthalpy;
}

/*======================= end of return_enthalpy() ==========================*/

/*======================= return_sat_vapor_p() ==============================*/

double return_sat_vapor_p(char     *chem,
                          double    temperature)
{
  double
    sat_vapor_p;
  static int
    warn_once=0;

#if defined(ncube2)
  if (strcmp(chem,"H_2O") == 0) {
    PSH2O(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_3") == 0) {
    PSNH3(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CH_4") == 0) {
    PSCH4(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"H_2S") == 0) {
    PSH2S(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_4SH") == 0) {
    PSNH4SH(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"PH_3") == 0) {
    PSPH3(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"GeH_4") == 0) {
    PSGEH4(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CO_2") == 0) {
    PSCO2(&temperature,&sat_vapor_p);
  }
  else {
    fprintf(stderr,"Error: return_sat_vapor_p(): unrecognized chem %s\n",chem);
    exit(1);
  }
#elif defined(hpux)
  if (strcmp(chem,"H_2O") == 0) {
    psh2o(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_3") == 0) {
    psnh3(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CH_4") == 0) {
    psch4(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"H_2S") == 0) {
    psh2s(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_4SH") == 0) {
    psnh4sh(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"PH_3") == 0) {
    psph3(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"GeH_4") == 0) {
    psgeh4(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CO_2") == 0) {
    psco2(&temperature,&sat_vapor_p);
  }
  else {
    fprintf(stderr,"Error: return_sat_vapor_p(): unrecognized chem %s\n",chem);
    exit(1);
  }
#elif defined(LINUX)
  /*
   * NOTE: LINUX appends one underscore for Fortran subroutine names without
   * an underscore (setup -> setup_) and two underscores for those with an
   * underscore (get_enthalpy -> get_enthalpy__).
   */
  if (strcmp(chem,"H_2O") == 0) {
    psh2o_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_3") == 0) {
    psnh3_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CH_4") == 0) {
    psch4_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"H_2S") == 0) {
    psh2s_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_4SH") == 0) {
    psnh4sh_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"PH_3") == 0) {
    psph3_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"GeH_4") == 0) {
    psgeh4_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CO_2") == 0) {
    psco2_(&temperature,&sat_vapor_p);
  }
  else {
    fprintf(stderr,"Error: return_sat_vapor_p(): unrecognized chem %s\n",chem);
    exit(1);
  }
#else
  if (strcmp(chem,"H_2O") == 0) {
    psh2o_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_3") == 0) {
    psnh3_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CH_4") == 0) {
    psch4_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"H_2S") == 0) {
    psh2s_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"NH_4SH") == 0) {
    psnh4sh_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"PH_3") == 0) {
    psph3_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"GeH_4") == 0) {
    psgeh4_(&temperature,&sat_vapor_p);
  }
  else if (strcmp(chem,"CO_2") == 0) {
    psco2_(&temperature,&sat_vapor_p);
  }
  else {
    fprintf(stderr,"Error: return_sat_vapor_p(): unrecognized chem %s\n",chem);
    exit(1);
  }
#endif

 /*
  * Convert pressure from cgs to mks: 
  */
  sat_vapor_p *= .1;

  if (strcmp(grid.eos,"virial") == 0) {
    /* 
     * NOTE: non ideal case not implemented.
     */
    if (!warn_once) {
      warn_once = 1;
      if (IAMNODE == NODE0) {
        fprintf(stderr,"Warning: return_sat_vapor_p(): using ideal e.o.s. value \n");
      }
    }
  }


  return sat_vapor_p;
}

/*======================= end of return_sat_vapor_p() =======================*/

/* * * * * * * * * * * end of epic_funcs_diag.c  * * * * * * * * * * * * * * */









