/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 2000 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * *  epic_funcs_io.c  * * * * * * * * * * * * * * * * 
 *                                                                 *
 *       Timothy E. Dowling                                        *
 *                                                                 *
 *       Functions for input and output.                           *
 *       This file includes the following:                         *
 *                                                                 *
 *           Read(),Write()                                        *
 *           read_array(),write_array()                            *
 *           var_read(),var_write()                                *
 *           lookup_netcdf(),define_netcdf()                       *
 *           bcast_char(),bcast_int(),bcast_double()               *
 *           write_ascii()                                         *
 *           read_t_vs_p()                                         *
 *           read_external_heating()                               *
 *           create_socket()                                       *
 *           accept_socket()                                       *
 *           connect_socket()                                      *
 *           input_double(),input_float()                          *
 *           input_int(),input_string()                            *
 *           input_sw_arrays()                                     *
 *           print_zonal_info()                                    *
 *           print_vertical_column()                               *
 *           print_humidity_column()                               *
 *           scdswap()                                             *
 *                                                                 *
 *           declare_copyright()                                   *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

/*======================= Read() ===========================================*/

/*
 * The specified node reads from a file or socket, 
 * and returns the number of bytes read.
 *
 * NOTE: If broadcast >= 0, this function must be called on all processors, 
 * that is, outside "if (IAMNODE == node) {}" blocks, in which case the
 * data are broadcast from node = broadcast.
 *
 * If swap == DO_SWAP, the byte order is swapped using scdswap().
 *
 */ 

long Read(int    node,
          void  *buf,
          int    size,
          int    nitems,
          int    swap,
          FILE  *io,
          int    fd,
          int    broadcast)
{
  long
    num_bytes,
    nbytes=0L,
    nitems_read=0L;
  int  
    rval,
    ip;
  long
    num_packets,
    last_size,
    bytes,
    cbytes;
  static char 
    buffer[PACKET_SIZE];

  if (IAMNODE == node) {
    num_bytes = (long)(size*nitems);

    if (io) {
      /* Use FILE pointer when available. */
      nbytes = (long)size*fread(buf,size,nitems,io);
    }
    else {
      /* Divide data into PACKET_SIZE elements */
      num_packets = (num_bytes-1)/PACKET_SIZE+1;
      last_size   = num_bytes-(num_packets-1)*PACKET_SIZE;
      for (ip = 1; ip <= num_packets; ip++) {
        cbytes = 0;
        if (ip < num_packets) {
          bytes = PACKET_SIZE;
        }
        else {
          bytes = last_size;
        }
        memset(buffer,0,PACKET_SIZE);
        do {
          rval = read(fd,buffer+cbytes,bytes-cbytes);
          if (rval < 0) {
            perror("Read()");
            exit(1);
          }   
          else if (rval == 0) {
            fprintf(stderr,"\nError: Read(): EOF encountered\n");
            exit(1);
          }  
          else {
            cbytes += rval;
          }
        } while (cbytes < bytes); 
        if (cbytes != bytes) {
          fprintf(stderr,"cbytes != bytes\n");
          exit(1);
        } 
        memcpy((char *)buf+nbytes,buffer,cbytes);
        nbytes += cbytes;
      }
    }

    if (nbytes != num_bytes) {
      perror("Number of bytes read != number requested \n");
      fprintf(stderr,"%d != %d \n\n",nbytes,num_bytes);
      fprintf(stderr,"size=%d nitems=%d \n",size,nitems);
      exit(1);
    }

    if (swap == DO_SWAP) {
      /* swap the byte order */
      scdswap((char *)buf,size,nitems);
    }
  }


#if defined(EPIC_MPI)
  if (broadcast >= 0) {
    /* Broadcast from node = broadcast */
    MPI_Bcast(&nbytes, 1,MPI_LONG,broadcast,para.comm);
    MPI_Bcast(buf,nbytes,MPI_BYTE,broadcast,para.comm);
  }
#endif

  return nbytes;
}

/*======================= end of Read() =====================================*/

/*======================= Write() ===========================================*/

/*
 * The specified node writes to a file or socket, 
 * and returns the number of bytes written.
 *
 * If swap == DO_SWAP, the byte order is swapped using scdswap().
 */ 

long Write(int    node,
           void  *buf,
           int    size,
           int    nitems,
           int    swap,
           FILE  *io,
           int    fd)
{
  long
    num_bytes,
    nbytes=0L,
    nitems_write=0L;
  long
    num_packets,
    last_size,
    bytes,
    cbytes;
  int 
    ip;

  /* Return if not specified node: */
  if (IAMNODE != node) {
    return nbytes;
  }

  if (swap == DO_SWAP) {
    /* Swap the byte order in place before writing: */
    scdswap((char *)buf,size,nitems);
  }

  num_bytes = (long)(size*nitems);

  if (io) {
    /* Use FILE pointer when available. */    
    nbytes = (long)size*fwrite(buf,size,nitems,io);
  }
  else {
    /* Divide into PACKET_SIZE elements */
    num_packets = (num_bytes-1)/PACKET_SIZE+1;
    last_size   = num_bytes-(num_packets-1)*PACKET_SIZE;
    for (ip = 1; ip <= num_packets; ip++) {
      cbytes = 0;
      if (ip < num_packets) {
        bytes = PACKET_SIZE;
      }
      else {
        bytes = last_size;
      }
      cbytes  = (long)write(fd,(char *)buf+nbytes,bytes);
      nbytes += cbytes;
    }
  }

  if (nbytes != num_bytes) {
    perror("Number of bytes written != number requested");
    fprintf(stderr,"%d != %d \n\n",nbytes,num_bytes);
    exit(1);
  }

  if (swap == DO_SWAP) {
    /* Restore the byte order: */
    scdswap((char *)buf,size,nitems);
  }

  return nbytes;
}

/*======================= end of Write() ====================================*/

/*======================= read_array() ======================================*/

/*
 * NOTE: Does not apply boundary conditions.
 *
 * Input nc_id through fd for io_type == VIA_FILE
 */

void read_array(int    dim,
                int    io_type,
                char  *name,
                FILE  *io,
                int    fd)
{
  int
    k,j,i,
    index,
    nk,nj,ni,
    jlohi[2],
    it,iitt[2],
    khi,itlo,ithi,
    al=FOURDIM-dim;
  int
    nc_id=fd,
    nc_varid,
    nc_err;
  size_t
    nc_start[FOURDIM], 
    nc_count[FOURDIM];
  char
    chtmp;
  double
    *buffji,
    *buffer,
    *ptr_buff;

  nk = grid.nk;
  nj = grid.nj;
  ni = grid.ni;

  if (io_type == VIA_FILE) {
    /*
     * Get the netCDF variable ID:
     */
    nc_err = nc_inq_varid(nc_id,name,&nc_varid);
    if (nc_err != NC_NOERR) {
      return;
    }
    nc_count[NETCDF_IT_INDEX] = 1;  
    nc_count[NETCDF_K_INDEX ] = 1; 
    nc_count[NETCDF_J_INDEX ] = 1;  
    nc_count[NETCDF_I_INDEX ] = ni; 
    nc_start[NETCDF_I_INDEX ] = 0;
  }

  if (io_type == VIA_SOCKET) {
    /* Get the node's j range */
    Read(IAMNODE,jlohi,sizeof(int),2,BSWAP,io,fd,DONT_BCAST);
  }
  else { 
    /* Assign j range */
    jlohi[0] = JLO;
    jlohi[1] = JHI;
  }

  itlo = 0;
  if (dim == TWODIM) {
    ithi    = itlo;
    khi     = 1;
    iitt[0] = 0;
    iitt[1] = 0;
  }
  else if (dim == THREEDIM) {
    ithi    = itlo;
    khi     = nk;
    iitt[0] = 0;
    iitt[1] = 0;
  }
  else if (dim == FOURDIM) {
    ithi    = itlo+1;
    khi     = nk;
    iitt[0] = IT_MINUS2;
    iitt[1] = IT_MINUS1;
  }
  else {
    fprintf(stderr,"Error in read_array, dim = %d not recognized. \n", dim);
    exit(1);
  }

  index = get_index(name);

#if defined(EPIC_AVS)
  /* Allocate buffer memory: */
  buffer = dvector(0,ni-1);
#endif

  for (it = itlo; it <= ithi; it++) {
    nc_start[NETCDF_IT_INDEX] = it-itlo;
    for (k = 1; k <= khi; k++) {
      nc_start[NETCDF_K_INDEX] = k-1;
#if defined(EPIC_AVS)
      if (index >= 0 && index < MAX_NVARS) { 
        /* 
         * Stored as AVSfield.
         * Each i is processed separately because of the stride 
         * in the AVSfield memory. 
         */
        for (j = jlohi[0]; j <= jlohi[1]; j++) {
          if (io_type == VIA_FILE) {
            nc_start[NETCDF_J_INDEX] = j-grid.jlo;
            nc_err = nc_get_vara_double(nc_id,nc_varid,nc_start+al,nc_count+al,buffer);
            if (nc_err != NC_NOERR) {
              fprintf(stderr,"Warning: read_array(): nc_get_vara_double(), %s\n",
                              nc_strerror(nc_err));
            }
          }
          else {
            Read(IAMNODE,buffer,sizeof(double),ni,BSWAP,io,fd,DONT_BCAST);
          }
          for (i = 1; i <= ni; i++) {
            FIELD(var.field,var.index[index],k,j,i) = buffer[i-1];
          }
        }
      }
      else {
#endif
        /*  
         * Stored as ordinary vector.
         *
         * Align buffji to beginning of JI plane : 
         */
        buffji = get_pointer(name)+iitt[it]*Nelem3d+(k-Kshift)*Nelem2d;
        for (j = jlohi[0]; j <= jlohi[1]; j++) {
          ptr_buff = &BUFFJI(j,ILO);
          if (io_type == VIA_FILE) {
            nc_start[NETCDF_J_INDEX] = j-grid.jlo;
            nc_err = nc_get_vara_double(nc_id,nc_varid,nc_start+al,nc_count+al,ptr_buff);
            if (nc_err != NC_NOERR) {
              fprintf(stderr,"Warning: read_array(): nc_get_vara_double(), %s\n",
                              nc_strerror(nc_err));
            }
          }
          else {
            Read(IAMNODE,ptr_buff,sizeof(double),ni,BSWAP,io,fd,DONT_BCAST);
          }
        }
#if defined(EPIC_AVS)
      }
#endif
    }
  }

#if defined(EPIC_AVS)
  /* Free allocated memory: */
  free_dvector(buffer,0,ni-1);
#endif

  if (io_type == VIA_SOCKET) {
    /* Write a confirmation byte back before returning: */
    chtmp = 'y';
    Write(IAMNODE,&chtmp,sizeof(char),1,FALSE,io,fd);
  }
  
  return;
}

/*======================= end of read_array() ===============================*/

/*======================= write_array() =====================================*/

/*
 * Input nc_id through fd for io_type == VIA_FILE
 */

void write_array(int    dim,
                 int    io_type,
                 char  *name,
                 FILE  *io,
                 int    fd)
{
  int
    k,j,i,
    index,
    nk,nj,ni,
    jlohi[2],
    it,iitt[2],
    khi,itlo,ithi,
    al=FOURDIM-dim;
  int
    nc_id=fd,
    nc_varid,
    nc_err;
  size_t
    nc_start[FOURDIM],
    nc_count[FOURDIM];
  char
    chtmp;
  double
     tmp,
    *buffji,
    *buffer,
    *ptr_buff;

  nk = grid.nk;
  nj = grid.nj;
  ni = grid.ni;

  itlo = 0;
  if (dim == TWODIM) {
    ithi    = itlo;
    khi     = 1;
    iitt[0] = 0;
    iitt[1] = 0;
  }
  else if (dim == THREEDIM) {
    ithi    = itlo;
    khi     = nk;
    iitt[0] = 0;
    iitt[1] = 0;
  }
  else if (dim == FOURDIM) {
    ithi    = itlo+1;
    khi     = nk;
    iitt[0] = IT_MINUS2;
    iitt[1] = IT_MINUS1;
  }
  else {
    fprintf(stderr,"Error: write_array(), dim = %d not recognized. \n", dim);
    exit(1);
  }

  /* Assign j range */
  jlohi[0] = JLO;
  jlohi[1] = JHI;
  if (io_type == VIA_SOCKET) {
    /* Send the node's j range */
    Write(IAMNODE,jlohi,sizeof(int),2,BSWAP,io,fd);
  }

  if (io_type == VIA_FILE) {
    /*
     * Get the netCDF variable ID:
     */
    nc_err = nc_inq_varid(nc_id,name,&nc_varid);
    if (nc_err != NC_NOERR) {
      fprintf(stderr,"Error: write_array(): nc_varid=%d, %s, nc_inq_varid() failed for %s\n",
                      nc_varid,nc_strerror(nc_err),name);
      exit(1);
    }
    nc_count[NETCDF_IT_INDEX] = 1;  
    nc_count[NETCDF_K_INDEX ] = 1; 
    nc_count[NETCDF_J_INDEX ] = 1;  
    nc_count[NETCDF_I_INDEX ] = ni; 
    nc_start[NETCDF_I_INDEX ] = 0;
  }

  index = get_index(name);

#if defined(EPIC_AVS)
  /* Allocate buffer memory: */
  buffer = dvector(0,ni-1);
#endif

  for (it = itlo; it <= ithi; it++) {
    nc_start[NETCDF_IT_INDEX] = it-itlo;
    for (k = 1; k <= khi; k++) {
      nc_start[NETCDF_K_INDEX] = k-1;
#if defined(EPIC_AVS) 
      if (index >= 0 && index < MAX_NVARS) {
        /* Stored as AVSfield. */
        for (j = jlohi[0]; j <= jlohi[1]; j++) {
          /* 
           * Each i is processed separately because of the stride 
           * in the AVSfield memory. 
           */
          for (i = 1; i <= ni; i++) {
            buffer[i-1] = FIELD(var.field,var.index[index],k,j,i);
          }
          if (io_type == VIA_FILE) {
            nc_start[NETCDF_J_INDEX] = j-grid.jlo;
            nc_err = nc_put_vara_double(nc_id,nc_varid,nc_start+al,nc_count+al,buffer);
            if (nc_err != NC_NOERR) {
              fprintf(stderr,"Error: write_array(): nc_put_vara_double(), %s\n",
                              nc_strerror(nc_err));
              exit(1);
            }
          }
          else {
            Write(IAMNODE,buffer,sizeof(double),ni,BSWAP,io,fd);
          }
        }
      }
      else {
#endif
        /*  
         * Stored as ordinary vector.
         *
         * Align buffji to beginning of JI plane: 
         */
        buffji = get_pointer(name)+iitt[it]*Nelem3d+(k-Kshift)*Nelem2d;
        for (j = jlohi[0]; j <= jlohi[1]; j++) {
          ptr_buff = &BUFFJI(j,ILO);
          if (io_type == VIA_FILE) {
            nc_start[NETCDF_J_INDEX] = j-grid.jlo;
            nc_err = nc_put_vara_double(nc_id,nc_varid,nc_start+al,nc_count+al,ptr_buff);
            if (nc_err != NC_NOERR) {
              fprintf(stderr,"\nError: write_array(): nc_put_vara_double(), %s, %s\n",
                              name,nc_strerror(nc_err));
              exit(1);
            }
          }
          else {
            Write(IAMNODE,ptr_buff,sizeof(double),ni,BSWAP,io,fd);
          }
        }
#if defined(EPIC_AVS)
      }
#endif
    }
  }

#if defined(EPIC_AVS)
  /* Free allocated memory: */
  free_dvector(buffer,0,ni-1);
#endif

  if (io_type == VIA_SOCKET) {
    /* Read a confirmation byte before returning: */
    Read(IAMNODE,&chtmp,sizeof(char),1,DONT_SWAP,io,fd,DONT_BCAST);
  }

  return;
}

/*======================= end of write_array() ==============================*/

/*======================= var_read() ========================================*/

/*
 *  Reads in the variables u,v,p, etc.
 */

void var_read(planetspec *planet,
              char       *infile,
              int         socket_port,
              int         io_type,
              int         portion)
{
  int 
    k,
    node,
    num_nodes,
    index,
    itback,itsave,itout,
    fd,
    exit_now=FALSE;
  static int
    nk;
  char
    chtmp;
  static char
    **gattname=NULL,
    **varname =NULL;
  static int
    ngatts=0,
    nvars =0;
  int
    nc_err,nc_id;
  nc_type
    the_nc_type;
  FILE 
    *io;

  if (portion > 1000) {
    /* 
     * Signal to exit after this read. 
     * Reset portion to normal value.
     */
    exit_now  = TRUE;
    portion  -= 1000;
  }

  if (io_type == VIA_FILE) {
    /*
     * NOTE: Call lookup_netcdf() from all nodes so that
     *       its calls to MPI_Bcast() will work.
     */
    lookup_netcdf(infile,&nc_id,&ngatts,&gattname,&nvars,&varname);
  }

  if (IAMNODE == 0) {
    if (io_type == VIA_STDIO) {
      io = stdin;
      fd = STDIN_FILENO;
    }
    else if (io_type == VIA_SOCKET) {
      io = NULL;
      fd = accept_socket(socket_port);
    }
    else if (io_type == VIA_FILE) {
      io = NULL;
      fd = nc_id;
      if (portion == INIT_DATA) {
        fprintf(stderr,"Reading INIT_DATA from %s ",infile);
      }
      else if (portion == ALL_DATA) {
        fprintf(stderr,"Reading ALL_DATA from %s ",infile);
      }
      else if (portion == UVP_DATA) {
        fprintf(stderr,"Reading UVP_DATA from %s ",infile);
      }
      else if (portion == NONINIT_DATA) {
        fprintf(stderr,"Reading NONINIT_DATA from %s ",infile);
      }
      fflush(stderr);
    }
  }

  if (portion == INIT_DATA || portion == ALL_DATA) {
    /*
     * Read in unchanging data that is common on all nodes.
     */
    READC(NODE0,*&,planet,->,name,              16);
    READC(NODE0,*&,planet,->,class,             16);
    READD(NODE0,&,planet,->,re,                 1);
    READD(NODE0,&,planet,->,rp,                 1);
    READD(NODE0,&,planet,->,omega,              1);
    READD(NODE0,&,planet,->,cp,                 1);
    READD(NODE0,&,planet,->,rgas,               1);
    READD(NODE0,&,planet,->,kappa,              1);
    READD(NODE0,&,planet,->,g,                  1);
    READD(NODE0,&,planet,->,x_he,               1);
    READD(NODE0,&,planet,->,x_h2,               1);
    READD(NODE0,&,planet,->,x_3,                1);
    READD(NODE0,&,planet,->,a,                  1);
    READD(NODE0,&,planet,->,e,                  1);
    READD(NODE0,&,planet,->,orbit_period,       1);

    READD(NODE0,&,grid,.,data_version,          1);
    READC(NODE0,*&,grid,.,geometry,       GEOM_STR);
    READD(NODE0,&,grid,.,globe_lonbot,          1);
    READD(NODE0,&,grid,.,globe_lontop,          1);
    READD(NODE0,&,grid,.,globe_latbot,          1);
    READD(NODE0,&,grid,.,globe_lattop,          1);
    READC(NODE0,*&,grid,.,f_plane_map,    GEOM_STR);
    READD(NODE0,&,grid,.,f_plane_lat0,          1);
    READD(NODE0,&,grid,.,f_plane_half_width,    1);
    READI(NODE0,&,grid,.,nk,                    1);
    READI(NODE0,&,grid,.,nj,                    1);
    READI(NODE0,&,grid,.,ni,                    1);
    READD(NODE0,&,grid,.,dln,                   1);
    READD(NODE0,&,grid,.,dlt,                   1);
    READD(NODE0,&,grid,.,theta_bot,             1);
    READD(NODE0,&,grid,.,theta_top,             1);
    READD(NODE0,&,grid,.,press0,                1);
    READD(NODE0,&,grid,.,mont0,                 1);
    READI(NODE0,&,grid,.,k_sponge,              1);
    READI(NODE0,&,grid,.,init_type,             1);
    READI(NODE0,&,grid,.,newt_cool_on,          1);
    READI(NODE0,&,grid,.,thermal_conduct_on,    1);
    READC(NODE0,*&,grid,.,eos,                   8);
    READI(NODE0,*&,var,.,chem_on,      MAX_NVARS+1);
    if (io_type != VIA_FILE) {
      for (index = 0; index <= MAX_NVARS; index++) {
        Read(NODE0,var.chem_name[index],sizeof(char),CHEM_NM_SZ,
             DONT_SWAP,io,fd,NODE0);
        Read(NODE0,var.tend_name[index],sizeof(char),CHEM_NM_SZ+8,
             DONT_SWAP,io,fd,NODE0);
      }
    }
    READD(NODE0,&,var,.,time_fp_bar,            1);
    READI(NODE0,*&,grid,.,wrap,             TOPDIM);
    READI(NODE0,*&,grid,.,pad,              TOPDIM);
    READI(NODE0,&,grid,.,jlo,                   1);
    READI(NODE0,&,grid,.,jfirst,                1);
    READI(NODE0,&,grid,.,klast_active,          1);
    READD(NODE0,&,grid,.,prandtl,               1);
    READD(NODE0,*&,grid,.,nu,       MAX_NU_ORDER+1);
    READD(NODE0,&,grid,.,hasten,                1);
 
    nk = grid.nk;

    READD(NODE0,*&,grid,.,theta,          2*(nk+1));
    READD(NODE0,*&,grid,.,rgas,             (nk+1));
    READD(NODE0,*&,grid,.,kappa,            (nk+1));
    READD(NODE0,*&,grid,.,p_avg,            (nk+1));
    READD(NODE0,*&,grid,.,qmin,             (nk+1));
    READD(NODE0,*&,grid,.,qmax,             (nk+1));
    READD(NODE0,*&,grid,.,tmin,             (nk+1));
    READD(NODE0,*&,grid,.,tmax,             (nk+1));

    if (io_type == VIA_SOCKET) {
      /* Read number of nodes running model */
      Read(NODE0,&(grid.they_num_nodes),sizeof(int),1,BSWAP,io,fd,NODE0);
    }
  }

  if (portion != NONINIT_DATA) {
    /* The following parameters may change with time: */
    READI(NODE0,&,grid,.,dt,                    1);
    READI(NODE0,&,grid,.,cfl_dt,                1);
    READI(NODE0,&,grid,.,aux_a,                 1);
    READI(NODE0,&,grid,.,aux_b,                 1);
    READI(NODE0,&,grid,.,aux_c,                 1);
    READD(NODE0,&,grid,.,aux_fa,                1);
    READD(NODE0,&,grid,.,aux_fb,                1);
    READD(NODE0,&,grid,.,aux_fc,                1);
    READI(NODE0,*&,var,.,time,                   2);
  }

  /* 
   * Seam between INIT_DATA and NONINIT_DATA 
   */

  if (portion == INIT_DATA) {
    if (IAMNODE == 0) {
      if (io_type == VIA_FILE) {
        nc_close(nc_id);
        fprintf(stderr,"100%%\n");
      }
      else if (io_type == VIA_SOCKET) {
        /* Write a confirmation byte back before closing: */
        chtmp = 'z';
        Write(IAMNODE,&chtmp,sizeof(char),1,DONT_SWAP,io,fd);
        close(fd);
      }
    }
    return;
  }

  if (IAMNODE == 0) {
    if (io_type == VIA_FILE) {
      fprintf(stderr,"  0%%");
      fflush(stderr);
    }
  }

  if (io_type == VIA_SOCKET) {
    /* 
     * NOTE: This computer is assumed to have only one node.
     * Loop over nodes on other computer to get all the information 
     * from one node before moving on to the next. 
     */
    num_nodes = grid.they_num_nodes;
  }
  else {
    /* 
     * Loop over nodes on this computer; read all information 
     * into this node before moving on to the next. 
     */
    num_nodes = grid.we_num_nodes;
  }

  /* 
   * Store current values of grid.itback, grid.itsave, grid.itout. 
   * These are sent upstream for a socket connection.
   *
   * If portion > 1000 such that exit_now == TRUE,
   * set itout = -1 to signal the model that it is time to exit.
   */
  itback = grid.itback;
  itsave = grid.itsave;
  itout  = grid.itout;

  if (exit_now == TRUE) {
    /* Set the signal for the model to exit after writing data: */
    itout = -1;
  }

  for (node = 0; node < num_nodes; node++) {

#if defined(EPIC_MPI)
    /* Synchronize processors: */
    MPI_Barrier(para.comm);
#endif

    if (IAMNODE == node || io_type == VIA_SOCKET) {
      if (io_type == VIA_FILE) {
        fprintf(stderr,"\b\b\b\b%3d%%",(int)(100.*(double)node/num_nodes));
        fflush(stderr);
      }
      if (node > 0) {
        if (io_type == VIA_STDIO) {
          io = stdin;
          fd = STDIN_FILENO;
        }
        else if (io_type == VIA_SOCKET) {
          io = NULL;
          fd = accept_socket(socket_port);
        }
        else if (io_type == VIA_FILE) {
          /* Open file */
          nc_err = nc_open(infile,NC_NOWRITE,&nc_id);
          io = NULL;
          fd = nc_id;
          if (nc_err != NC_NOERR) {
            fprintf(stderr,"Cannot find input file %s \n",infile);
            exit(1);
          }
        }
      }

      if (strcmp(planet->class,"terrestrial") == 0) {
        /*
         * Read surface geopotential.
         */
        read_array(TWODIM,io_type,"surface_gz",io,fd);
      }

      /*
       * Read spinup zonal-wind profile.  
       */
      read_array(THREEDIM,io_type,"u_spinup",io,fd);

      /* 
       * Read u,v,p and chemicals: 
       */
      for (index = 0; index < MAX_NVARS; index++) {
        if (var.chem_on[index]) {
          read_array(THREEDIM,io_type,var.chem_name[index],io,fd);
        }
      }
  
      if (portion == ALL_DATA || portion == NONINIT_DATA) {
        /* Read in tendencies:*/
        for (index = 0; index < MAX_NVARS; index++) {
          if (var.chem_on[index]) {
            read_array(FOURDIM,io_type,var.tend_name[index],io,fd);
          }
        }
      }
      if (io_type == VIA_FILE) {
        nc_close(nc_id);
      }
      else if (io_type == VIA_SOCKET) {
        /* Write useful information back before closing: */
        Write(IAMNODE,&itback,sizeof(int),1,BSWAP,io,fd);
        Write(IAMNODE,&itsave,sizeof(int),1,BSWAP,io,fd);
        Write(IAMNODE,&itout, sizeof(int),1,BSWAP,io,fd);
        close(fd);
      }
    }
  }

  /*
   * Apply lateral boundary conditions:
   */
  if(strcmp(planet->class,"terrestrial") == 0) {
    BC2D(&(SURFACE_GZ(JLO,ILO)),NO_INDEX,1);
  }
  for (k = 1; k <= grid.nk; k++) {
    BC2D(&(U_SPINUP(KLO,JLO,ILO)),NO_INDEX,k);
    for (index = 0; index < MAX_NVARS; index++) {
      if (var.chem_on[index]) {
        BC2D(&(VAR(index,KLO,JLO,ILO)),index,k);
        if (portion == ALL_DATA || portion == NONINIT_DATA) {
          BC2D(&(DVARDT(index,IT_MINUS2,KLO,JLO,ILO)),NO_INDEX,k);
          BC2D(&(DVARDT(index,IT_MINUS1,KLO,JLO,ILO)),NO_INDEX,k);
        }
      }
    }
  }

  if (IAMNODE == 0) {
    if (io_type == VIA_FILE) {
      fprintf(stderr,"\b\b\b\b%3d%%\n",100);
    }
  }

  return;
}

/*======================= end of var_read() =================================*/

/*======================= var_write() =======================================*/

/*
 *  Writes the variables u,v,p,etc.
 */

void var_write(planetspec *planet,
               char       *outfile,
               int         socket_port,
               int         io_type,
               int         portion)
{
  int 
    k,nk,
    node,
    num_nodes,
    index,
    fd,
    iret;
  static int
    sock_init=0;
  char
    chtmp;
  int
    nc_err,nc_id;
  nc_type
    the_nc_type;
  FILE 
    *io;
  char 
    namestr[32];

  if (IAMNODE == 0) {
    if (io_type == VIA_STDIO) {
      io = stdout;
      fd = STDOUT_FILENO;
    }
    else if (io_type == VIA_SOCKET) {
      io = NULL;
      fd = connect_socket(grid.view_host,socket_port);
    }
    else if (io_type == VIA_FILE) {
      /*
       * Define variables and attributes for netCDF file:
       */
      define_netcdf(planet,outfile,&nc_id);
      io = NULL;
      fd = nc_id;
      if (portion == INIT_DATA) {
        fprintf(stderr,"Writing INIT_DATA to %s ",outfile);
      }
      else if (portion == ALL_DATA) {
        fprintf(stderr,"Writing ALL_DATA to %s ",outfile);
      }
      else if (portion == UVP_DATA) {
        fprintf(stderr,"Writing UVP_DATA to %s ",outfile);
      }
      else if (portion == NONINIT_DATA) {
        fprintf(stderr,"Writing NONINIT_DATA to %s ",outfile);
      }
      fflush(stderr);
    } 
  }

  if (portion == INIT_DATA || portion == ALL_DATA) {
    /*
     * Write unchanging data that is common on all nodes.
     */
    if (io_type != 0) {
      WRITEC(NODE0,*&,planet,->,name,              16);
      WRITEC(NODE0,*&,planet,->,class,             16);
      WRITED(NODE0,&,planet,->,re,                 1);
      WRITED(NODE0,&,planet,->,rp,                 1);
      WRITED(NODE0,&,planet,->,omega,              1);
      WRITED(NODE0,&,planet,->,cp,                 1);
      WRITED(NODE0,&,planet,->,rgas,               1);
      WRITED(NODE0,&,planet,->,kappa,              1);
      WRITED(NODE0,&,planet,->,g,                  1);
      WRITED(NODE0,&,planet,->,x_he,               1);
      WRITED(NODE0,&,planet,->,x_h2,               1);
      WRITED(NODE0,&,planet,->,x_3,                1);
      WRITED(NODE0,&,planet,->,a,                  1);
      WRITED(NODE0,&,planet,->,e,                  1);
      WRITED(NODE0,&,planet,->,orbit_period,       1);

      WRITED(NODE0,&,grid,.,data_version,          1);
      WRITEC(NODE0,*&,grid,.,geometry,       GEOM_STR);
      WRITED(NODE0,&,grid,.,globe_lonbot,          1);
      WRITED(NODE0,&,grid,.,globe_lontop,          1);
      WRITED(NODE0,&,grid,.,globe_latbot,          1);
      WRITED(NODE0,&,grid,.,globe_lattop,          1);
      WRITEC(NODE0,*&,grid,.,f_plane_map,    GEOM_STR);       
      WRITED(NODE0,&,grid,.,f_plane_lat0,          1);
      WRITED(NODE0,&,grid,.,f_plane_half_width,    1);
      WRITEI(NODE0,&,grid,.,nk,                    1);
      WRITEI(NODE0,&,grid,.,nj,                    1);
      WRITEI(NODE0,&,grid,.,ni,                    1);
      WRITED(NODE0,&,grid,.,dln,                   1);
      WRITED(NODE0,&,grid,.,dlt,                   1);
      WRITED(NODE0,&,grid,.,theta_bot,             1);
      WRITED(NODE0,&,grid,.,theta_top,             1);
      WRITED(NODE0,&,grid,.,press0,                1);
      WRITED(NODE0,&,grid,.,mont0,                 1);
      WRITEI(NODE0,&,grid,.,k_sponge,              1);
      WRITEI(NODE0,&,grid,.,init_type,             1);
      WRITEI(NODE0,&,grid,.,newt_cool_on,          1);
      WRITEI(NODE0,&,grid,.,thermal_conduct_on,    1);
      WRITEC(NODE0,*&,grid,.,eos,                   8);
      WRITEI(NODE0,*&,var,.,chem_on,      MAX_NVARS+1);
      if (io_type != VIA_FILE) {
        for (index = 0; index <= MAX_NVARS; index++) {
          Write(NODE0,var.chem_name[index],sizeof(char),CHEM_NM_SZ,
                DONT_SWAP,io,fd);
          Write(NODE0,var.tend_name[index],sizeof(char),CHEM_NM_SZ+8,
                DONT_SWAP,io,fd);
        }
      }
      WRITED(NODE0,&,var,.,time_fp_bar,            1);
      WRITEI(NODE0,*&,grid,.,wrap,             TOPDIM);
      WRITEI(NODE0,*&,grid,.,pad,              TOPDIM);
      WRITEI(NODE0,&,grid,.,jlo,                   1);
      WRITEI(NODE0,&,grid,.,jfirst,                1);
      WRITEI(NODE0,&,grid,.,klast_active,          1);
      WRITED(NODE0,&,grid,.,prandtl,               1);
      WRITED(NODE0,*&,grid,.,nu,       MAX_NU_ORDER+1);
      WRITED(NODE0,&,grid,.,hasten,                1);

      nk = grid.nk;

      WRITED(NODE0,*&,grid,.,theta,          2*(nk+1));
      WRITED(NODE0,*&,grid,.,rgas,             (nk+1));
      WRITED(NODE0,*&,grid,.,kappa,            (nk+1));
      WRITED(NODE0,*&,grid,.,p_avg,            (nk+1));
      WRITED(NODE0,*&,grid,.,qmin,             (nk+1));
      WRITED(NODE0,*&,grid,.,qmax,             (nk+1));
      WRITED(NODE0,*&,grid,.,tmin,             (nk+1));
      WRITED(NODE0,*&,grid,.,tmax,             (nk+1));
    }

    if (io_type == VIA_SOCKET) {
      /* Write number of nodes running model */
      Write(NODE0,&(grid.we_num_nodes),sizeof(int),1,BSWAP,io,fd);
    }
  }

  if (portion != NONINIT_DATA) {
    /*
     * The following parameters can change with time:
     */
    WRITEI(NODE0,&,grid,.,dt,    1);
    WRITEI(NODE0,&,grid,.,cfl_dt,1);
    WRITEI(NODE0,&,grid,.,aux_a, 1);
    WRITEI(NODE0,&,grid,.,aux_b, 1);
    WRITEI(NODE0,&,grid,.,aux_c, 1);
    WRITED(NODE0,&,grid,.,aux_fa,1);
    WRITED(NODE0,&,grid,.,aux_fb,1);
    WRITED(NODE0,&,grid,.,aux_fc,1);
    WRITEI(NODE0,*&,var,.,time,   2);
  }

  /* Seam between INIT_DATA and NONINIT_DATA */

  if (portion == INIT_DATA) {
    if (IAMNODE == 0) {
      if (io_type == VIA_FILE) {
        nc_close(nc_id);
        fprintf(stderr,"100%%\n");
      }
      else if (io_type == VIA_SOCKET) {
        /* Read a confirmation byte before closing: */
        Read(IAMNODE,&chtmp,sizeof(char),1,DONT_SWAP,io,fd,DONT_BCAST);
        close(fd);
      }
    }
    return;
  }
  else {
    if (IAMNODE == 0) {
      if (io_type == VIA_FILE) {
        /*
         * Leave define mode for netCDF file:
         */
        nc_err = nc_enddef(nc_id);
        if (nc_err != NC_NOERR) {
          fprintf(stderr,"Error: var_write(): nc_enddef(), %s\n",
                  nc_strerror(nc_err));
        }
      }
    }
  }

  /* 
   * Loop over nodes to get all the information off one node before
   * moving on to the next. 
   */

  if (IAMNODE == 0) {
    if (io_type == VIA_FILE) {
      fprintf(stderr,"  0%%");
      fflush(stderr);
    }
  }

  for (node = 0; node < grid.we_num_nodes; node++) {

#if defined(EPIC_MPI)
    /* Synchronize processors: */
    MPI_Barrier(para.comm);
#endif

    if (node == IAMNODE) {
      if (io_type == VIA_FILE) {
        fprintf(stderr,"\b\b\b\b%3d%%",(int)(100.*(double)node/grid.we_num_nodes));
        fflush(stderr);
      }
      if (node > 0) {
        /* Open io,fd */
        if (io_type == VIA_STDIO) {
          io = stdout;
          fd = STDOUT_FILENO;
        }
        else if (io_type == VIA_SOCKET) {
          io = NULL;
          fd = connect_socket(grid.view_host,socket_port);
        }
        else if (io_type == VIA_FILE) {
          /* open file */
          nc_err = nc_open(outfile,NC_WRITE,&nc_id);
          if (nc_err != NC_NOERR) {
            fprintf(stderr,"Cannot open output file %s \n",outfile);
            exit(1);
          }
          io = NULL;
          fd = nc_id;
        }
      }

      if (strcmp(planet->class,"terrestrial") == 0) {
        /* 
         * Write surface geopotential.
         */
        write_array(TWODIM,io_type,"surface_gz",io,fd);
      }

      /*
       * Write spinup zonal-wind profile.  
       */
      write_array(THREEDIM,io_type,"u_spinup",io,fd);

      /* Write u,v,p and chemicals: */
      for (index = 0; index < MAX_NVARS; index++) {
        if (var.chem_on[index]) {
          write_array(THREEDIM,io_type,var.chem_name[index],io,fd);
        }
      }

      if (portion == ALL_DATA || portion == NONINIT_DATA) {
        /* Write tendencies: */
        for (index = 0; index < MAX_NVARS; index++) {
          if (var.chem_on[index]) {
            write_array(FOURDIM,io_type,var.tend_name[index],io,fd);
          }
        }
      }

      if (io_type == VIA_FILE) {
        nc_close(nc_id);
      }
      else if (io_type == VIA_SOCKET) {
        /* Read useful information before closing: */
        Read(IAMNODE,&(grid.itback),sizeof(int),1,BSWAP,io,fd,DONT_BCAST);
        Read(IAMNODE,&(grid.itsave),sizeof(int),1,BSWAP,io,fd,DONT_BCAST);
        Read(IAMNODE,&(grid.itout), sizeof(int),1,BSWAP,io,fd,DONT_BCAST);
        close(fd);
      }
    }
  }

  if (IAMNODE == 0) {
    if (io_type == VIA_FILE) {
      fprintf(stderr,"\b\b\b\b%3d%%\n",100);
    }
  }

  if (grid.itout == -1) {
    /* 
     * Time to exit: 
     */
    free_arrays(planet);
    free(planet);

#if defined(EPIC_MPI)
    MPI_Finalize();
#endif

    exit(1);
  }

  return;

}

/*======================= end of var_write() ==================================*/

/*======================= lookup_netcdf() =====================================*/

/*
 * Read numbers and names of global attributes and variables 
 * contained in infile, which must be in netCDF format.
 *
 * NOTE: ngatts,**gattname,nvars,**varname should be declared static in the calling
 *       function, with their input values equal to the last call, in order to
 *       properly reallocate memory.
 */

void lookup_netcdf(char   *infile,
                   int    *nc_id,
                   int    *ngatts,
                   char ***gattname,
                   int    *nvars,
                   char ***varname)
{
  int
    i,
    ndims,unlimdimid,
    nc_err;

  /*
   * Free previous memory:
   */
  for (i = 0; i < *ngatts; i++) {
    free((*gattname)[i]);
  }
  for (i = 0; i < *nvars; i++) {
    free((*varname)[i]);
  }

  if (IAMNODE == NODE0) {
    nc_err = nc_open(infile,NC_NOWRITE,nc_id);
    if (nc_err != NC_NOERR &&
        !strstr(infile,"_defaults.nc")) {
      fprintf(stderr,"Error: lookup_netcdf(): %s, %s \n",
              nc_strerror(nc_err),infile);
      exit(1);
    }
    nc_inq(*nc_id,&ndims,nvars,ngatts,&unlimdimid);
  }

#if defined(EPIC_MPI)
  MPI_Bcast(ngatts,1,MPI_INT,NODE0,para.comm);
  MPI_Bcast(nvars, 1,MPI_INT,NODE0,para.comm);
#endif

  /*
   * Reallocate memory for character arrays. 
   * Look up and store names.
   */
  *gattname = (char **)realloc(*gattname,(*ngatts)*sizeof(char *));
  for (i = 0; i < *ngatts; i++) {
    (*gattname)[i] = (char *)calloc(NC_MAX_NAME,sizeof(char));
    if (IAMNODE == NODE0) {
      nc_inq_attname(*nc_id,NC_GLOBAL,i,(*gattname)[i]);
    }

#if defined (EPIC_MPI)
    MPI_Bcast((*gattname)[i],NC_MAX_NAME,MPI_CHAR,NODE0,para.comm);
#endif 

  }

  *varname = (char **)realloc(*varname,(*nvars)*sizeof(char *));
  for (i = 0; i < *nvars; i++) {
    (*varname)[i] = (char *)calloc(NC_MAX_NAME,sizeof(char));
    if (IAMNODE == NODE0) {
      nc_inq_varname(*nc_id,i,(*varname)[i]);
    }

#if defined (EPIC_MPI)
    MPI_Bcast((*varname)[i],NC_MAX_NAME,MPI_CHAR,NODE0,para.comm);
#endif

  } 

  return;
}

/*======================= end of lookup_netcdf() ==============================*/

/*======================= define_netcdf() =====================================*/

/*
 *  Define variables and attributes for netCDF file.
 */

void define_netcdf(planetspec *planet,
                   char       *outfile,
                   int        *nc_id)
{
  int
    n,nc_err,
    kdim,jdim,idim,itdim,
    K,J,I,
    it,itmp;
  size_t
    index[1];
  char
    title[32],
    coord_name[CHEM_NM_SZ+8];

  if (IAMNODE != NODE0) {
    return;
  }

  /*
   * Enter define mode:
   */
  nc_err = nc_create(outfile,NC_CLOBBER,nc_id);
  if (nc_err != NC_NOERR) {
    fprintf(stderr,"Cannot write outfile file %s \n",outfile);
    exit(1);
  }

  sprintf(title,"EPIC Model Version %4.2f",grid.data_version);
  nc_put_att_text(*nc_id,NC_GLOBAL,"title",strlen(title)+1,title);

  kdim  = grid.nk;
  jdim  = grid.nj-grid.jlo+1;
  idim  = grid.ni;
  itdim = 2;

  for (n = 0; n < MAX_NVARS; n++) {
    if (var.chem_on[n]) {
      /* 
       * lon (I direction): 
       */
      sprintf(coord_name,"lon_%s",var.chem_name[n]);
      nc_def_dim(*nc_id,coord_name,idim,&(var.nc_dimids[n][NETCDF_I_INDEX]));
      nc_def_var(*nc_id,coord_name,NC_DOUBLE,ONEDIM,
                 &(var.nc_dimids[n][NETCDF_I_INDEX]),&(var.nc_coorid[n][NETCDF_I_INDEX]));
      nc_put_att_text(*nc_id,var.nc_coorid[n][NETCDF_I_INDEX],"units",
                      strlen("degrees_east")+1,"degrees_east");
      /* 
       * lat (J direction):
       */
      sprintf(coord_name,"lat_%s",var.chem_name[n]);
      nc_def_dim(*nc_id,coord_name,jdim,&(var.nc_dimids[n][NETCDF_J_INDEX]));
      nc_def_var(*nc_id,coord_name,NC_DOUBLE,ONEDIM,
                 &(var.nc_dimids[n][NETCDF_J_INDEX]),&(var.nc_coorid[n][NETCDF_J_INDEX]));
      nc_put_att_text(*nc_id,var.nc_coorid[n][NETCDF_J_INDEX],"units",
                      strlen("degrees_north")+1,"degrees_north");
      nc_put_att_text(*nc_id,var.nc_coorid[n][NETCDF_J_INDEX],"mapping",
                      strlen("planetographic")+1,"planetographic");
      /* 
       * theta (i.e., potential temperature; K direction):
       */
      sprintf(coord_name,"theta_%s",var.chem_name[n]);
      nc_def_dim(*nc_id,coord_name,kdim,&(var.nc_dimids[n][NETCDF_K_INDEX]));
      nc_def_var(*nc_id,coord_name,NC_DOUBLE,ONEDIM,
                 &(var.nc_dimids[n][NETCDF_K_INDEX]),&(var.nc_coorid[n][NETCDF_K_INDEX]));
      nc_put_att_text(*nc_id,var.nc_coorid[n][NETCDF_K_INDEX],"units",
                      strlen("degrees_k")+1,"degrees_k");
      /*
       * time_index (2 values needed to record past tendency fields):
       */
      sprintf(coord_name,"time_index_%s",var.chem_name[n]);
      nc_def_dim(*nc_id,coord_name,itdim,&(var.nc_dimids[n][NETCDF_IT_INDEX])); 
      nc_def_var(*nc_id,coord_name,NC_INT,ONEDIM,
                 &(var.nc_dimids[n][NETCDF_IT_INDEX]),&(var.nc_coorid[n][NETCDF_IT_INDEX]));
      nc_put_att_text(*nc_id,var.nc_coorid[n][NETCDF_IT_INDEX],"units",
                      strlen("count")+1,"count");
      /* 
       * n-th prognostic variable: 
       */
      nc_def_var(*nc_id,var.chem_name[n],NC_DOUBLE,THREEDIM,
                 &(var.nc_dimids[n][NETCDF_K_INDEX]),&(var.nc_chemid[n]));
      nc_put_att_text(*nc_id,var.nc_chemid[n],"units",
                      strlen(var.chem_units[n])+1,var.chem_units[n]); 
      /*
       * Associated tendencies:
       */
      nc_def_var(*nc_id,var.tend_name[n],NC_DOUBLE,FOURDIM,
                 &(var.nc_dimids[n][NETCDF_IT_INDEX]),&(var.nc_tendid[n]));
      nc_put_att_text(*nc_id,var.nc_tendid[n],"units",
                      strlen(var.tend_units[n])+1,var.tend_units[n]);
    }
  }
  /*
   * Define Rayleigh drag zonal-wind profile:
   */
  nc_def_var(*nc_id,"u_spinup",NC_DOUBLE,THREEDIM,
             &(var.nc_dimids[U_INDEX][NETCDF_K_INDEX]),&(var.nc_u_spinupid));
  nc_put_att_text(*nc_id,var.nc_u_spinupid,"units",
                  strlen("m/s")+1,"m/s");

  if (strcmp(planet->class,"terrestrial") == 0) {
    /* 
     * Define surface geoid:
     */
    nc_def_var(*nc_id,"surface_gz",NC_DOUBLE,TWODIM,
               &(var.nc_dimids[P_INDEX][NETCDF_J_INDEX]),&(var.nc_surface_gzid));
    nc_put_att_text(*nc_id,var.nc_surface_gzid,"units",
                    strlen("J/kg")+1,"J/kg");
  }
  /*
   * Leave define mode:
   */
  nc_enddef(*nc_id);

  /*
   * Assign values to coordinates, taking into account the EPIC model's 
   * staggered C-grid.
   */
  /*
   * lon:
   */
  for (I = 1; I <= grid.ni; I++) {
    index[0] = I-1;
    for (n = 0; n < MAX_NVARS; n++) {
      if (var.chem_on[n]) {
        if (n == U_INDEX) {
          nc_put_var1_double(*nc_id,var.nc_coorid[n][NETCDF_I_INDEX],
                             index,&(grid.lon[2*I  ]));
        }
        else {
          nc_put_var1_double(*nc_id,var.nc_coorid[n][NETCDF_I_INDEX],
                             index,&(grid.lon[2*I+1]));
        }
      }
    }
  }
  /*
   * lat:
   */
  for (J = grid.jlo; J <= grid.nj; J++) {
    index[0] = J-grid.jlo;
    for (n = 0; n < MAX_NVARS; n++) {
      if (var.chem_on[n]) {
        if (n == V_INDEX) {
          nc_put_var1_double(*nc_id,var.nc_coorid[n][NETCDF_J_INDEX],
                             index,&(grid.lat[2*J  ]));
        }
        else {
          nc_put_var1_double(*nc_id,var.nc_coorid[n][NETCDF_J_INDEX],
                             index,&(grid.lat[2*J+1]));
        } 
      }
    }
  }
  /*
   * theta:
   */
  for (K = 1; K <= grid.nk; K++) {
    index[0] = K-1;
    for (n = 0; n < MAX_NVARS; n++) {
      if (var.chem_on[n]) {
        if (n == P_INDEX) {
          nc_put_var1_double(*nc_id,var.nc_coorid[n][NETCDF_K_INDEX],
                             index,&(grid.theta[2*K+1]));
        }
        else {
          nc_put_var1_double(*nc_id,var.nc_coorid[n][NETCDF_K_INDEX],
                             index,&(grid.theta[2*K  ]));
        }
      }
    }
  }
  /*
   * time_index:
   */
  for (it = 0; it < itdim; it++) {
    index[0] = it;
    itmp     = it-itdim;
    for (n = 0; n < MAX_NVARS; n++) {
      if (var.chem_on[n]) {
        nc_put_var1_int(*nc_id,var.nc_coorid[n][NETCDF_IT_INDEX],
                        index,&itmp);
      }
    }
  }

  /*
   * Put back into define mode before returning:
   */
  nc_err = nc_redef(*nc_id);
  if (nc_err != NC_NOERR) {
    fprintf(stderr,"Error: define_netcdf(): nc_redef(), %s\n",
            nc_strerror(nc_err));
    exit(1);
  }

  return;
}

/*======================= end of define_netcdf() ==============================*/

/*======================= bcast_char() ========================================*/

void bcast_char(int   node,
                char *str,
                int   num)
{

#if defined(EPIC_MPI)
  MPI_Bcast(str,num,MPI_CHAR,node,para.comm);
#endif


  return;
}

/*======================= end of bcast_char() =================================*/

/*======================= bcast_int() =========================================*/

void bcast_int(int  node,
               int *val,
               int  num)
{

#if defined(EPIC_MPI)
  MPI_Bcast(val,num,MPI_INT,node,para.comm);
#endif

  return;
}

/*======================= end of bcast_int() ==================================*/

/*======================= bcast_double() ======================================*/

void bcast_double(int     node,
                  double *val,
                  int     num)
{

#if defined(EPIC_MPI)
  MPI_Bcast(val,num,MPI_DOUBLE,node,para.comm);
#endif

  return;
}

/*======================= end of bcast_double() ===============================*/

/*======================= write_ascii() =======================================*/

/*
 * NOTE: Only implemented for a single-processor computer.
 */

void write_ascii(planetspec *planet,
                 int         io_type,
                 char       *datfile) {
  int
    K,J,I,
    sw=0;
  double
     theta,theta2,d_th0,
     lat,lon,
     avgu,avgv,
     temperature,
     pressure,
     fpara,
     fgibb,fpe,uoup,exner,
    *q,
    *mont,
    *rh,
    *h;
  FILE
    *io;

  if (io_type == VIA_FILE) {
     io = fopen(datfile,"w");
  }
  else if (io_type == VIA_STDIO) {
     io = stdout;
  }
  else { 
    fprintf(stderr,"Error in write_ascii(), io_type = %d not recognized.\n",io_type);
    exit(1);
  }

  if (strcmp(planet->name,"sw") == 0) {
    sw = 1;
  }

  /* Allocate memory: */
  q    = dvector(0,Nelem2d-1);
  mont = dvector(0,Nelem2d-1);
  rh   = dvector(0,Nelem2d-1);
  h    = dvector(0,Nelem2d-1);

  fprintf(io,"\n");
  fprintf(io," Output from EPIC model in ascii format. \n");
  fprintf(io," All variables are averaged onto the h-grid. \n");
  fprintf(io," Latitude is planetographic (glat), longitude is eastward (elon).\n");
  fprintf(io," The units of potential vorticity are PVU = 10^-6 m^2 K/s/kg. \n");
  fprintf(io," planet: %s \n",planet->name);
  fprintf(io,"  nk    nj    ni   years        secs \n");
  fprintf(io," %3d  %4d  %4d     %3d    %8d \n",
          grid.nk,grid.nj,grid.ni,var.time[1],var.time[0]);
  fprintf(io,"\n");
  fprintf(io," k-range    j-range    i-range \n");
  fprintf(io," %3d %3d    %3d %3d    %3d %3d \n",KHI-1,KLO+1,JLO,JHI,ILO,IHI);
  fprintf(io,"\n");
  fprintf(io," theta[K]  glat[deg]  elon[deg]  u[m/s]  v[m/s]  p[mbar]    T[K]     q[PVU]      M[m^2/s^2]   h\n");
  mont_nk(planet,mont);
  for (K = KHI-1; K > KLO; K--) {
    theta = grid.theta[2*K];
    potential_vorticity(planet,q,K,1);
    if (var.chem_on[NH_3_INDEX]) {
      /*
       * Hardwiring relative humidity to be NH_3.
       */
      relative_humidity(planet,NH_3_INDEX,rh,K);
    }
    theta2 = grid.theta[2*K+1];
    if (K > 1) {
      d_th0 = grid.theta[2*K]-grid.theta[2*K+2];
    }
    else {
      /* special delta-thetas in top layer: */
      d_th0 = grid.theta[2*K]-grid.theta[2*K+2];
    }
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        temperature = T2(K,J,I);
        if (sw) {
          /* kappa = 1. */
          exner = planet->cp*P(K,J,I)/grid.press0;
        }
        else {
          exner = planet->cp*temperature/theta2;
        }

        if (var.chem_on[FPARA_INDEX]) {
          /* add extra term for fpara thermodynamics */
          fpara = get_chem(planet,FPARA_INDEX,2*K+1,J,I);
          return_enthalpy(planet,fpara,P(K,J,I),temperature,&fgibb,&fpe,&uoup);
          MONT(J,I) -= fgibb*(FPARA(K,J,I)-FPARA(K+1,J,I));
        }
        MONT(J,I) += exner*d_th0;
      }
    }
    BC2D(&(MONT(JLO,ILO)),NO_INDEX,1);

      /* Calculate h */

    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        H(J,I) = get_h(planet,2*K,J,I);
      }
    }
    BC2D(&(H(JLO,ILO)),NO_INDEX,1);  

    for (J = JLO; J <= JHI; J++) {
      lat = grid.lat[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        lon         = grid.lon[2*I+1];
        avgu        = .5*(U(K,J,I)+U(K,J,  I+1));
        avgv        = .5*(V(K,J,I)+V(K,J+1,I  ));
        pressure    = P1(K,J,I);
        temperature = T( K,J,I);
        fprintf(io,"%7.1e      %5.1f     %6.1f   %7.2e %7.2e %8.2e %7.1e  %12.5e %12.5e %12.5e\n",
               theta,lat,lon,avgu,avgv,
               pressure*1.e-2,temperature,Q(J,I)*1.e+6,MONT(J,I),H(J,I));
      }
    }
  }

  if (io_type == VIA_FILE) {
    fclose(io);
  }

  /* Free allocated memory: */
  free_dvector(rh,  0,Nelem2d-1);
  free_dvector(mont,0,Nelem2d-1);
  free_dvector(q,   0,Nelem2d-1);
  free_dvector(h,   0,Nelem2d-1);
  return;
}

/*======================= end of write_ascii() ================================*/

/*======================= read_t_vs_p() =======================================*/

int read_t_vs_p(planetspec       *planet,
                double            ptop,
                double            pbot,
                int              *ceiling_tp,
                int              *floor_tp,
                double           *pdat,
                double           *tdat,
                int               portion)
{
  char
    infile[FILE_STR],
    header[N_STR];
  int
    kk,ntp;
  double
    p1,t1;
  FILE
    *t_vs_p;
  /* 
   * The following are part of DEBUG_MILESTONE statements: 
   */
  int
    idbms=0;
  char
    dbmsname[]="read_t_vs_p";

  if (IAMNODE == 0) {
    if (strcmp(planet->name,"triton") == 0) {
      /* For Triton, use grid.aux_a to flag which T(p) data is used */
      if (portion == INIT_DATA) {
        grid.aux_a = input_int("Use Strobel[0] or Olkin[1] T(p)? \n",0);
      }
      if (grid.aux_a == 0) {
        sprintf(infile,EPIC_PATH"/data/%s/t_vs_p.%s.strobel",planet->name,planet->name);
      }
      else if (grid.aux_a == 1) {
        sprintf(infile,EPIC_PATH"/data/%s/t_vs_p.%s.olkin",  planet->name,planet->name);
      }
      else {
        fprintf(stderr,"Error: unrecognized T(p) flag in initial for Triton. \n");
        exit(1);
      }
    }
    else {
      sprintf(infile,EPIC_PATH"/data/%s/t_vs_p.%s",planet->name,planet->name);
    }
    /* Open t_vs_p file */
    t_vs_p = fopen(infile,"r");
    if (!t_vs_p) {
      fprintf(stderr,"Error: read_t_vs_p(): Cannot open file %s \n",infile);
      exit(1);
    }
  }

  /* Skip over 6-line header: */
  if (IAMNODE == 0) {
    for (kk = 0; kk < 6; kk++) {
      fgets(header,128,t_vs_p);  
    }
    /* input number of data points */
    fscanf(t_vs_p,"%d",&ntp); 
  }

#if defined(EPIC_MPI)
  MPI_Bcast(&ntp,1,MPI_INT,NODE0,para.comm);
#endif
 
  if (portion == INIT_DATA) {
    if (IAMNODE == NODE0) {
      fclose(t_vs_p);
    }
    return ntp;
  }

  if (IAMNODE == 0) {
    if (ceiling_tp != NULL) *ceiling_tp = ntp;
    if (floor_tp   != NULL) *floor_tp   = 0;
    /* Store in order of increasing theta */
    for (kk = ntp-1; kk >= 0; kk--) {  
      fscanf(t_vs_p,"%lf %lf",&p1,&t1);
      /* convert from mbar to mks */
      pdat[kk] = 100.*p1; 
      tdat[kk] = t1;
      if (ceiling_tp != NULL && pdat[kk] < ptop) {
        (*ceiling_tp)--;
      }
      if (floor_tp != NULL && pdat[kk] > pbot) {
        (*floor_tp)++;
      }
    }
    fclose(t_vs_p);

    if (ceiling_tp != NULL && floor_tp != NULL) {
      if (grid.init_type == INIT_T) {
        /* 
         * Adjust *floor_tp and *ceiling_tp 
         * to keep p in range of t_vs_yp data.
         * Use trial and error.
         */
        *floor_tp   = input_int("Increase floor_tp? \n",  *floor_tp);
        *ceiling_tp = input_int("Decrease ceiling_tp? \n",*ceiling_tp);
      }
    }
  }

#if defined(EPIC_MPI)
  if (ceiling_tp != NULL && floor_tp != NULL) {
    MPI_Bcast(floor_tp,  1,MPI_INT,NODE0,para.comm);
    MPI_Bcast(ceiling_tp,1,MPI_INT,NODE0,para.comm);
  }
  MPI_Bcast(pdat,ntp,MPI_DOUBLE,NODE0,para.comm);
  MPI_Bcast(tdat,ntp,MPI_DOUBLE,NODE0,para.comm);
#endif

  return ntp;
}

/*======================= end of read_t_vs_p() ================================*/

/*======================= read_external_heating() =============================*/
/*Danie&Raul 25 March 2004*/
void read_external_heating(planetspec *planet,
                           double     *ext_lat,
                           double     *ext_lon,
                           double     *ext_p,
                           double     *T3)

{  
  char
    infile[FILE_STR],
    header[N_STR];
  int
    kk,ni,nj,nk,K,J,I;
  double
     aux_lon,
     aux_lat,
     aux_p,
     aux_T3,
     *ext_lat,
     *ext_lon,
     *ext_p,
     *T3;
  FILE
    *ext_heat;
  
  sprintf(infile,EPIC_PATH"/data/%s/ext_heat.%s",planet->name,planet->name);

/* Open external_heating file */
    ext_heat = fopen(infile,"r");
    if (!ext_heat) {
      fprintf(stderr,"Error: read_external_heating(): Cannot open file %s \n",infile);
      exit(1);
    }

/* Skip over 6-line header: */
  if (IAMNODE == 0) {
    for (kk = 0; kk < 6; kk++) {
      fgets(header,128,ext_heat);
    }
/* input number of data points*/ 
    fscanf(ext_heat,"%d,%d,%d",&ni,&nj,&nk);
  }
  
  ext_lon      = dvector(1, ni);
  ext_lat      = dvector(1, nj);
  ext_p        = dvector(1, nk);
 
  for (I = 1; I <= ni; I++) {
     for (J = 1; J <= nj; J++) {
        for (K = 1; K <= nk ; K++) {
           fscanf(ext_heat,"%lf,%lf,%lf,%lf",&aux_lon,&aux_lat,&aux_p,&aux_T3);
           ext_p[K]  = aux_p;
           T3[K,J,I] = aux_T3;            
        }
     ext_lat[J] = aux_lat;
     }
  ext_lon[I] = aux_lon;
  }

return;
}

/*======================= end of read_external_heating() ======================*/
 
/*======================= create_socket() =====================================*/

/* 
 * Create an internet domain tcp (stream) socket:
 */

void create_socket(int *sock,
                   int *socket_port)
{
  int
    slength;
  struct sockaddr_in 
    server;

  do {
    *sock = socket(AF_INET,SOCK_STREAM,0);
    if (*sock == -1) {
      perror("opening stream socket");
    }
  } while (*sock == -1);

  /* Name socket using wildcards */
  server.sin_family      = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port        = 0;
  if (bind(*sock,(struct sockaddr *)&server,sizeof(server)) < 0) {
    perror("binding stream socket");
    exit(1);
  }

  listen(*sock,8);

  slength = sizeof(server);
  if (getsockname(*sock,(struct sockaddr *)&server,&slength) < 0) {
    perror("getting socket name");
    exit(1);
  }

  /* Assign socket port */
  *socket_port = ntohs(server.sin_port);

  return;
}

/*======================= end of create_socket() ==============================*/

/*======================= accept_socket() =====================================*/

int accept_socket(int socket_port)
{
  int
    fd;

  do {
    fd = accept(socket_port,(struct sockaddr *)0,(size_t *)0);
    if (fd == -1) {
      perror("accepting socket connection");
    }  
  } while (fd == -1);   

  return fd;
}

/*======================= end of accept_socket() ==============================*/

/*======================= connect_socket() ====================================*/

int connect_socket(char *hostname,
                   int   socket_port)
{
  int
    fd,
    iret;
  static char
    old_hostname[80]="none";
  static struct hostent 
    *hp;
  struct sockaddr_in
    server;
  struct hostent
    *gethostbyname();

  if (strcmp(old_hostname,hostname) != 0) {
    if ((hp = gethostbyname(hostname)) == 0) {
      fprintf(stderr,"unknown host: %s\n",hostname);
      exit(2);
    }
    strcpy(old_hostname,hostname);
  }

  do {
    fd = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
    if (fd == -1) {
      perror("opening socket");
    }
  } while (fd == -1);

  server.sin_family = AF_INET;
  bcopy((char *)hp->h_addr,(char *)&server.sin_addr,hp->h_length);

  server.sin_port = htons((unsigned short)socket_port);
  do {
    iret = connect(fd,(struct sockaddr *)&server,sizeof(server));
    if (iret == -1) {
      perror("connecting socket");
    }
  } while (iret == -1);

  return fd;
}

/*======================= end of connect_socket() =============================*/

/*======================= input_double() ======================================*/

/* 
 * Read in double, or set to default if input is a return ('\n').
 * C.Santori, T.Dowling 
 */

double input_double(char prompt[N_STR], double def) 
{
  char  
    c,
    buffer[N_STR];
  int   
    len;
  double 
    ans;

  if (IAMNODE == 0) {
    fprintf(stderr,"%s[%g]: ",prompt,def);
    for (len = 0; (c = getchar()) != '\n' && len < N_STR; len++) {
      buffer[len]=c;
    }
    buffer[len] = '\0';
    if (len == 0) {
      ans = def;
    }
    else {
      sscanf(buffer,"%lf",&ans);
    }
  }

#if defined(EPIC_MPI)
  MPI_Bcast(&ans,1,MPI_DOUBLE,NODE0,para.comm);
#endif

  return ans;
}

/*====================== end input_double() ===================================*/

/*======================= input_float() =======================================*/

/* 
 * Read in float, or set to default if input is a return ('\n').
 * C.Santori, T.Dowling 
 */

float input_float(char prompt[N_STR], float def) 
{
  char  
    c,
    buffer[N_STR];
  int   
    len;
  float 
    ans;

  if (IAMNODE == 0) {
    fprintf(stderr,"%s[%g]: ",prompt,def);
    for (len = 0; (c = getchar()) != '\n' && len < N_STR; len++) {
      buffer[len]=c;
    }
    buffer[len] = '\0';
    if (len == 0) {
      ans = def;
    }
    else {
      sscanf(buffer,"%f",&ans);
    }
  }

#if defined(EPIC_MPI)
  MPI_Bcast(&ans,1,MPI_FLOAT,NODE0,para.comm);
#endif

  return ans;
}

/*====================== end input_float() =====================================*/

/*====================== input_int() ========================================*/

/* 
 * Read in int, or set to default if input is a return ('\n').
 * C.Santori, T.Dowling 
 */

int input_int(char prompt[N_STR], int def) 
{
  char  
    c,
    buffer[N_STR];
  int 
    ans,
    len;

  if (IAMNODE == 0) {
    fprintf(stderr,"%s[%d]: ",prompt,def);
    for (len = 0; (c = getchar()) != '\n' && len < N_STR; len++) {
      buffer[len]=c;
    }
    buffer[len] = '\0';
    if (len == 0) {
      ans = def;
    }
    else {
      sscanf(buffer,"%d",&ans);
    }
  }

#if defined(EPIC_MPI)
  MPI_Bcast(&ans,1,MPI_INT,NODE0,para.comm);
#endif

  return ans;
}

/*====================== end input_int() ====================================*/

/*====================== input_string() =====================================*/

/* 
 * Read in a string, or set to default if input is a return ('\n').
 *
 * C.Santori, T.Dowling 
 */

void input_string(char prompt[N_STR], char def[N_STR], char *ans) 
{
  char 
    c,
    buffer[N_STR];
  int  
    len;

  if (IAMNODE == 0) {
    fprintf(stderr,"%s[%s]: ",prompt,def);
    for (len = 0; (c = getchar()) != '\n' && len < N_STR; len++) {
      buffer[len]=c;
    }
    buffer[len] = '\0';
    if (len == 0) {
      strcpy(ans,def);
    }
    else {
      strcpy(ans,buffer);
      strcpy(def,buffer);
    }
  }

#if defined(EPIC_MPI)
  MPI_Bcast(ans,N_STR,MPI_CHAR,NODE0,para.comm);
#endif

}

/*====================== end input_string() ==================================*/

/*====================== input_sw_arrays() ===================================*/

/*
 * The shallow-water case is set up to read its initial data via stdin.
 * This is done to facilitate connection to Glenn Flierl's Matlab interface
 * for class use.
 */
void input_sw_arrays(void)
{
  int
    K,J,I;
  float
    ftmp;

  /* NOTE: assumed initial running on single processor here */
  /* Read gh array into p */
  fprintf(stderr,"Input gh array J: %d,%d   I: %d,%d \n",JLO,JHI,ILO,IHI);
  K = 1;
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      scanf("%f",&ftmp);
      P(K,J,I) = (double)ftmp;
    }
  }
  BC2D(&(P(KLO,JLO,ILO)),P_INDEX,K);

  /* Read u */
  fprintf(stderr,"Input u array J: %d,%d   I: %d,%d \n",JLO,JHI,ILO,IHI);
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      scanf("%f",&ftmp);
      U(K,J,I) = (double)ftmp;
    }
  }
  BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);

  /* Read v */
  fprintf(stderr,"Input v array J: %d,%d   I: %d,%d \n",JLO,JHI,ILO,IHI);
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      scanf("%f",&ftmp);
      V(K,J,I) = (double)ftmp;
    }
  }
  BC2D(&(V(KLO,JLO,ILO)),V_INDEX,K);

  /* Read u2 */
  fprintf(stderr,"Input u2 array J: %d,%d   I: %d,%d \n",JLO,JHI,ILO,IHI);
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      scanf("%f",&ftmp);
      U(K+1,J,I) = (double)ftmp;
    }
  }
  BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K+1);

  return;
}

/*====================== end of input_sw_arrays() ============================*/

/*====================== print_zonal_info() ==================================*/

void print_zonal_info(planetspec *planet,
                      double     *q)
{
  int
    J,I;
  double
    zf,zfy,fy,
    zeta;
  FILE
    *u_dat;

  if (IAMNODE == 0) {
    /* NOTE: not set up for MPI. */
    potential_vorticity(planet,q,grid.nk,0);

    u_dat = fopen(EPIC_PATH"/tmp/zonal_wind.dat","w");
    fprintf(u_dat,"Cloud-top zonal wind data \n");
    fprintf(u_dat," %s,  nk,nj,ni = %2d %2d %2d \n",
                   planet->name,grid.nk,grid.nj,grid.ni);
    fprintf(u_dat," zeta units: 1.e-4 s^-1; beta units: 1.e-12 m^-1 s^-1 \n\n");
    fprintf(u_dat," lat(deg)  u(m/s)      f       zeta"
                  "         zf       df/dy     d(zeta+f)/dy \n");
    I = ILO;
    for (J = JLO; J <= JHI; J++) {
      zeta = .5*(Q(J,I)-grid.f[2*J]+Q(J+1,I)-grid.f[2*(J+1)]);
      zf   = .5*(Q(J,I)+Q(J+1,I));
      zfy  = grid.n[2*J+1]*(Q(J+1,I)-Q(J,I));
      fy   = grid.n[2*J+1]*(grid.f[2*J+2]-grid.f[2*J]);
      /* change units on zeta terms */
      /* change units on beta=df/dy terms */
      zeta *= 1.e+4;
      zf   *= 1.e+4;
      zfy  *= 1.e+12;
      fy   *= 1.e+12;
      fprintf(u_dat," %5.1f   %7.2f  %10.3e %10.3e %10.3e %10.3e %10.3e \n",
                    grid.lat[2*J+1],U(grid.nk,J,I),
                    grid.f[2*J+1]*1.e+4,zeta,zf,fy,zfy);
    }

    fclose(u_dat);
  }

  return;
}

/*====================== end of print_zonal_info() ===========================*/

/*====================== print_vertical_column() =============================*/

void print_vertical_column(planetspec *planet,
                           int         J,
                           int         I)
{
  int
    K;
  double
    pressure,
    fpara,
    theta,
    temperature,
    mu,
    density,
    brunt,
    z_comp,
    *h_tmp;
  FILE
    *vert_dat;

  if (IAMNODE != 0 || strcmp(planet->name,"sw") == 0) {
    return;
  }

  /* Allocate memory for h_tmp[K]: */
  h_tmp = dvector(1,grid.nk);

  vert_dat = fopen(EPIC_PATH"/tmp/vertical.dat","w");
  fprintf(stderr,"\n Column data at lat = %6.2f, lon = %7.2f: \n",
                 grid.lat[2*J+1],grid.lon[2*I+1]);
  fprintf(stderr,"\n     k     press[mbar]      theta[K]   N[1/s]  \n");
  fprintf(vert_dat,"  Vertical profile for south-west corner of domain \n");
  fprintf(vert_dat,"   k   press[mbar]    temp[K]  theta[K]    N[1/s]   Z=P/(rhoRT) \n");
  for (K = 1; K <= grid.nk; K++) {
    theta = grid.theta[2*K];
    if (K <= KLAST_ACTIVE) {
      if (K == 1) {
        h_tmp[K] = (P(K,J,I)-0.)/(planet->g*
                   (grid.theta[2*K  ]-grid.theta[2*K+1]));
      }
      else {
        h_tmp[K] = (P(K,J,I)-P(K-1,J,I))/(planet->g*
                   (grid.theta[2*K-1]-grid.theta[2*K+1]));
      }
      if (h_tmp[K] <= 0.) {
        fprintf(stderr,"Error: print_vertical_column(): H(%d,%d,%d) = %f \n",K,J,I,h_tmp[K]);
        exit(1);
      }
      /* inside-the-layer pressure: */
      pressure    = P1(K,J,I);
      temperature = T(K,J,I);
      density     = RHO(K,J,I);
      brunt       = sqrt(get_brunt2(planet,2*K,J,I));

      /* 
       * Determine compressibility index, z_comp: 
       */
      if (strcmp(grid.eos,"ideal") == 0) {
        z_comp = 1.;
      }
      else if (strcmp(grid.eos,"virial") == 0) {
        z_comp = 1.+sum_xx(planet,b_vir,temperature)*pressure;
      }
      else {
        z_comp = 1.;
      }
    }
    else {
      brunt = 0.;
    }
    fprintf(stderr,"   %4.1f                   %9.1f  %8.4f \n", 
            (double)K,grid.theta[2*K],brunt);
    if (K <= KLAST_ACTIVE) {
      fprintf(vert_dat," %4.1f  %11.5e  %9.1f %9.1f  %8.4f     %5.3f \n",
                       (double)K,pressure/100.,temperature,grid.theta[2*K],brunt,z_comp);
      fprintf(stderr,"-- %4.1f -- %11.5f -- %9.1f ----------- \n",
             (double)K+0.5,P(K,JLO,ILO)/100.,(grid.theta)[2*K+1]);
    }
  }

  fclose(vert_dat);
  free_dvector(h_tmp,1,grid.nk);

  return;
}



/*====================== end of print_vertical_column() ======================*/

/*====================== print_humidity_column() =============================*/

void print_humidity_column(planetspec *planet,
                           char       *outfile_str,
                           int         index,
                           char       *humidity_str,
                           double     *humidity,
                           double     *density,
                           int        J,
                           int        I)
{
  int
    K;
  FILE
    *outfile;

  outfile = fopen(outfile_str,"w");
  if (!outfile) {
    perror("print_humidity_column: fopen w");
  }

  fprintf(outfile,"system: %s \n",planet->name);
  if (strcmp(humidity_str,"total cloud density") == 0) {
    fprintf(outfile,"Total cloud density [kg/m^3] data at J,I = %d,%d \n\n",
                     J,I);
    fprintf(outfile,"    p[mbar]     cld_dens   dens[kg/m^3] \n");
  }
  else {
    fprintf(outfile,"%s %s [0,1] column data at J,I = %d,%d \n\n",
                     var.chem_name[index],humidity_str,J,I);
    fprintf(outfile,"    p[mbar]     %8s   dens[kg/m^3] \n",var.chem_name[index]);
  }

  for (K = 1; K <= grid.nk; K++) {
    fprintf(outfile," %11.3e %11.3e %11.3e\n",
                     P1(K,J,I)*1.e-2,humidity[K],density[K]);
  }

  fclose(outfile);

  return;
}

/*====================== end of print_humidity_column ========================*/

/*====================== scdswap() ===========================================*/

/*
 *  Byte swapping for 8, 4 and 2 byte quantities.  
 */
#include <stdio.h>

void scdswap(char *arr, 
             int   nbytes, 
             int   cnt)
{
  char 
    buf[4];
  register int 
    nb=nbytes;
  register char 
   *parr, *pend;

  pend = arr+nb*cnt;

  switch (nb) {
    default: 
      fprintf(stderr," Bad length to scdswap()\n");
      exit(99);

    case 1:
      break;

    case 2:  
      for (parr=arr; parr<pend; parr+=nb) {
        buf[0]  = parr[0];
        parr[0] = parr[1];
        parr[1] = buf[0];
      }
      break;

    case 4:  
      for (parr=arr; parr<pend; parr+=nb) {
        buf[0]  = parr[0];
        buf[1]  = parr[1];
        parr[0] = parr[3];
        parr[1] = parr[2];
        parr[2] = buf[1];
        parr[3] = buf[0];
      }
      break;

    case 8:  
      for (parr=arr; parr<pend; parr+=nb) {
        buf[0]  = parr[0];
        buf[1]  = parr[1];
        buf[2]  = parr[2];
        buf[3]  = parr[3];
        parr[0] = parr[7];
        parr[1] = parr[6];
        parr[2] = parr[5];
        parr[3] = parr[4];
        parr[4] = buf[3];
        parr[5] = buf[2];
        parr[6] = buf[1];
        parr[7] = buf[0];
      }
      break;
  } 

  return;
} 

/*======================= end of scdswap() ===================================*/

/*======================= declare_copyright() ================================*/

void declare_copyright(void)
{
  static int
    initialized=0;

  if (!initialized) {
    fprintf(stderr,"\n");
    fprintf(stderr," EPIC Model, Copyright (C) 1998 Timothy E. Dowling             \n");                                                                                         
    fprintf(stderr," This program is free software; you can redistribute it and/or \n");  
    fprintf(stderr," modify it under the terms of the GNU General Public License.  \n");    
    fprintf(stderr," This program is distributed WITHOUT ANY WARRANTY.             \n"); 
    fprintf(stderr,"\n");

    initialized = 1;
  } 
                                                                 
  return;
}

/*======================= end of declare_copyright() =========================*/

/*============================ methane_mass() =================================*/
void methane_mass(planetspec *planet,
                  char       *outfile)

{
  double 
    g, ginv, mlayer_ch4, mlayer_planet, p2, p1, dxdy;
  int
    K,J,I,index,kk;
 FILE
    *output;

  g      = planet->g;
  ginv   = 1./g;
  output = fopen(outfile,"a");
  fprintf(output,"Time: %05d,",
          var.time[0] / 86400 + var.time[1]*365);
  for (K = KLO; K<=KHI-1; K++) {
    kk = 2*K;
    mlayer_planet = 0;
    mlayer_ch4    = 0;
    for (J = JLO; J <= JHI; J++) {
      dxdy = 1/grid.mn[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        p2    = get_p(planet,P_INDEX,kk+1,J,I);
        p1    = get_p(planet,P_INDEX,kk-1,J,I);
	if (p2 <=p1) {
	   fprintf(output,"ERROR, p2 <= p1 in methane_mass()");
	}
	else {
           mlayer_ch4 += (p2-p1)*VAR(CH_4_INDEX,K,J,I)*dxdy;
	   mlayer_planet += (p2-p1)*dxdy;
	}
      }
    }
    mlayer_ch4    *= ginv;
    mlayer_planet *= ginv;
    
    fprintf(output,"%f, %f ",mlayer_ch4,mlayer_planet);
  }
  fprintf(output," \n");  
  return;
}

/*======================== end of methane_mass() ==============================*/


/* * * * * * * * * * * * * end of epic_funcs_io.c  * * * * * * * * * * * * * * */









