/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * *  epic_funcs_util.c  * * * * * * * * * * * * * * * * * * * * 
 *                                                                           *
 *       Timothy E. Dowling                                                  *
 *                                                                           *
 *       Functions for math, memory, and utility operations.                 *
 *       This file includes the following:                                   *
 *                                                                           *
 *           set_name_lists()                                                *
 *           make_arrays()                                                   *
 *           free_arrays()                                                   *
 *           extract_scalar()                                                *
 *           insert_scalar()                                                 *
 *           grid_to_domain()                                                *
 *           dvector()                                                       *
 *           spline(),splint()                                               *
 *           linint()                                                        *
 *           sech2()                                                         *
 *           find_root()                                                     *
 *           find_place_in_table()                                           *
 *           cnum(),cmult(),cadd(),csub()                                    *
 *           cexp(),cabsolute(),creal(),cimag                                *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

/* 
 * NOTE: EPIC_VIEW code needs to pass domain and planet as pointers, since they
 * originate as AVS input ports. Otherwise, their pointers are defined globally
 * in epic.h.
 */

/*======================= set_name_lists() ==================================*/

void set_name_lists(void) 
{
  int
    ii;

  /* Core prognostic variables: */
  strcpy(var.chem_name[U_INDEX],      "u");
  strcpy(var.chem_name[V_INDEX],      "v");
  strcpy(var.chem_name[P_INDEX],      "p");

  /* Para hydrogen fraction: */
  strcpy(var.chem_name[FPARA_INDEX],  "fpara");

  /* Optional humidities: */
  strcpy(var.chem_name[H_2O_INDEX],   "H_2O");
  strcpy(var.chem_name[NH_3_INDEX],   "NH_3");
  strcpy(var.chem_name[H_2S_INDEX],   "H_2S");
  strcpy(var.chem_name[CH_4_INDEX],   "CH_4");
  strcpy(var.chem_name[CO_2_INDEX],   "CO_2");
  strcpy(var.chem_name[DUST_INDEX],   "dust");

  /* Optional chemical-product humidities: */
  strcpy(var.chem_name[NH_4SH_INDEX], "NH_4SH");
  strcpy(var.chem_name[O_3_INDEX],    "O_3");
  strcpy(var.chem_name[AEROSOL_INDEX],"aerosol");

  /* Dry air: */
  strcpy(var.chem_name[DRY_AIR_INDEX],"dry_air");

  /*
   * Fill in variable units list:
   */
  strcpy(var.chem_units[U_INDEX],      "m/s");
  strcpy(var.chem_units[V_INDEX],      "m/s");
  strcpy(var.chem_units[P_INDEX],      "Pa");
  strcpy(var.chem_units[FPARA_INDEX],  "mole/mole");
  for (ii = FIRST_HUMIDITY; ii <= DRY_AIR_INDEX; ii++) {
    strcpy(var.chem_units[ii],"kg/kg");
  }

  /*
   * Fill in tendency name list and units list:
   */
  for (ii = 0; ii <= MAX_NVARS; ii++) {
    if (ii == U_INDEX || ii == V_INDEX) {
      sprintf(var.tend_name[ ii],"d%sdt",var.chem_name[ ii]);
      sprintf(var.tend_units[ii],"%s/s", var.chem_units[ii]);
    }
    else if (ii == P_INDEX) {
      sprintf(var.tend_name[ ii],"dhdt");
      sprintf(var.tend_units[ii],"kg/m^2/K/s");
    }
    else {
      sprintf(var.tend_name[ ii],"dh%sdt",       var.chem_name[ ii]);
      sprintf(var.tend_units[ii],"kg/m^2/K %s/s",var.chem_units[ii]);
    }
  }

  return;
}

/*======================= end of set_name_lists() ===========================*/

/*======================= make_arrays() =====================================*/

#if defined(EPIC_VIEW)
void make_arrays(planetspec *planet,
                 domainspec *domain)
#else 
void make_arrays(void)
#endif
/*
 * Allocate memory for variables,  
 * and fill data tables for theta, lat, lon, f, m, n, etc.
 *
 * NOTE: Usually call var_read() with portion = INIT_DATA
 *       before calling make_arrays(), to get size of model 
 *       and allocate memory for unchanging parameters.
 */
{
  char
    field_type[40];
  int    
    k,kk,j,jj,i,ii,it,
    nk,nj,ni,
    index,
    dims[3];       /*  Dimensions used when allocating field           */
  double  
    rln,rlt,       /*  radii of curvature                              */
    dlnr,dltr,     /*  dln, dlt in radians                             */
    dx,dy,         /*  f-plane grid spacings [meters]                  */
    lat,           /*  latitude                                        */
    re,rp,         /*  equatorial, polar radii                         */
    omega,         /*  planetary angular velocity  [1/s]               */
    wn,            /*  2 Pi/wavelength, used in low-pass filter        */
    m_45,n_45,     /*  used to set low-pass filter                     */
    kappap1,
    kappa_inv,
    tmp;
  int
    idbms=0;
  char
    dbmsname[]="make_arrays";

  /* 
   * Set the prognostic-variable name and index arrays and count the
   * number of variables.
   */
  set_name_lists();
  var.nvars = 0;
  for (index = 0; index < MAX_NVARS; index++) {
    if (var.chem_on[index]) {
      var.index[index] = var.nvars++;
    }
    else {
      var.index[index] = -1;
    }
  }

#if defined(EPIC_VIEW)
  /*
   *  Copy all domain information into the 
   *  appropriate grid and var locations.  The -1 argument
   *  in grid_to_domain() indicates a domain to grid-and-var copy.
   */
  grid_to_domain(domain,-1);
#endif
  
  nk = grid.nk;
  nj = grid.nj;
  ni = grid.ni;

#if defined(EPIC_MPI)
  /* Initialize the parallel-bookkeeping structure */
  mpispec_init();

  if (strcmp(grid.geometry,"globe") == 0) {
    if(para.nproc > grid.nj) {
      if (IAMNODE == 0) {
        fprintf(stderr,"This model must be run on <= %d processors\n",grid.nj);
      }
      exit(0);
    }
  }

#else
  grid.we_num_nodes = 1;
  /* Indicate whether the processor has poles in its range: */
  if (strcmp(grid.geometry,"globe") == 0) {
    if (grid.globe_latbot == -90.) {
      grid.is_spole = 1;
    }
    else {
      grid.is_spole = 0;
    }
    if (grid.globe_lattop == 90.) {
      grid.is_npole = 1;
    }
    else {
      grid.is_npole = 0;
    }
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    grid.is_spole = 0;
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      grid.is_npole = 0;
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      grid.is_npole = 1;
    }
  }
  else {
    grid.is_spole = 0;
    grid.is_npole = 0;
  }
#endif

  /* Set the shift integers used in the multidimensional array macros: */
  Ishift  = ILO-IPAD;
  Jshift  = JLO-JPAD;
  Kshift  = KLO-KPAD;
  Iadim   = IADIM;
  Jadim   = JADIM;
  Kadim   = KADIM;
  Nelem2d = NELEM2D;
  Nelem3d = NELEM3D;
  Shift2d = Ishift+Iadim*Jshift;
  Shift3d = Ishift+Iadim*Jshift+Nelem2d*Kshift;
  Shiftkj = Jshift+Jadim*Kshift;

  /*
   *  In order to handle half-integers we use the notation 
   *  theta[kk] = theta[2*k], lat[jj] = lat[2*j], etc.
   *
   *  Following Hsu and Arakawa (1990) we use   
   *  the geometric mean of each layer's upper and 
   *  lower theta for the middle theta.
   *
   */
  grid.lon   = dvector(0, 2*(ni+1));
  grid.lat   = dvector(0, 2*(nj+1));   
  grid.f     = dvector(0, 2*(nj+1));
  grid.m     = dvector(0, 2*(nj+1));
  grid.n     = dvector(0, 2*(nj+1));
  grid.mn    = dvector(0, 2*(nj+1));

  /*
   *  Compute lat:
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    if (grid.globe_latbot == -90.) {
      (grid.lat)[0] = -90.+(grid.dlt)*sqrt(.5);
    }
    else {
      (grid.lat)[0] = grid.globe_latbot;
    }
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      (grid.lat)[0] = -180.-(grid.dlt);
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      (grid.lat)[0] = 0.-(grid.dlt);
    }
  }
  else {
    fprintf(stderr,"Error: make_arrays(): Unrecognized geometry \"%s\" \n",grid.geometry);
    exit(1);
  }
  for (jj = 1; jj <= 2*(nj+1); jj++)  {      
    (grid.lat)[jj] = (grid.lat)[jj-1]+(grid.dlt)*.5;
  }
  if (strcmp(grid.geometry,"globe") == 0) {
    if (grid.globe_latbot == -90.) {
      /* poles are offset by extra dlt*sqrt(.5) */
      (grid.lat)[       0] = -90.;   
    }
    if (grid.globe_lattop == 90.) {  
      (grid.lat)[2*(nj+1)] =  90.; 
    }
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    if (strcmp(grid.f_plane_map,"polar") == 0) {
      (grid.lat)[2*(nj+1)] = 90.;
    }
  }

  /*
   *  Compute lon:
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    (grid.lon)[0] = grid.globe_lonbot-(grid.dln);
  }
  else {
    (grid.lon)[0] = -180.-(grid.dln);
  }
  for (ii = 1; ii <= 2*(ni+1); ii++) {
    (grid.lon)[ii] = (grid.lon)[ii-1]+(grid.dln)*.5;
  }

  /*
   *  Compute f, m, n:
   */
  omega = planet->omega;
  if (strcmp(grid.geometry,"globe") == 0) {
    dlnr  = grid.dln*DEG;
    dltr  = grid.dlt*DEG;
    re    = planet->re;
    rp    = planet->rp;
    for (jj = 1; jj < 2*(nj+1); jj++) {
      lat  = DEG*(grid.lat)[jj];
      rln  = re/sqrt( 1.+pow(rp/re*tan(lat),2.) );
      rlt  = rln/( cos(lat)*( pow(sin(lat),2.)+
             pow(re/rp*cos(lat),(double)2.)));
    
      (grid.f )[jj] = 2.*omega*sin(lat);
      (grid.m )[jj] = 1./(rln*dlnr);
      (grid.n )[jj] = 1./(rlt*dltr);
      (grid.mn)[jj] = (grid.m)[jj]*(grid.n)[jj]; 
    }
    /*
     *  The Arakawa and Lamb (1981) scheme calls for a 
     *  special (3/2)*dlt spacing for n next to the poles,
     *  as illustrated by their Fig. A2; their equation
     *  (A40) specifies mn at the poles.
     *
     *  NOTE: We find that the special spacing next to the poles is more accurately
     *  given by (1.+sqrt(.5))*dlt.  
     */
    if (grid.globe_latbot == -90.) {
      /* south pole */
      lat = DEG*(grid.lat)[0];
      (grid.f )[2*0  ]  = 2.*omega*sin(lat);
      (grid.n )[2*0+1] /= 1.+sqrt(.5);  
      /* wedge shaped area: */
      (grid.mn)[2*0+1]  = grid.n[2*0+1]*grid.n[2*0+1]/(.5*dlnr);
      (grid.mn)[2*0  ]  = 2.*(grid.mn)[2*0+1];
    }
    else {
      jj = 0;
      lat  = DEG*(grid.lat)[jj];
      rln  = re/sqrt( 1.+ pow(rp/re*tan(lat),(double)2.) );
      rlt  = rln/( cos(lat)*( pow(sin(lat),(double)2.)+
             pow(re/rp*cos(lat),(double)2.)));
    
      (grid.f )[jj] = 2.*omega*sin(lat);
      (grid.m )[jj] = 1./(rln*dlnr);
      (grid.n )[jj] = 1./(rlt*dltr);
      (grid.mn)[jj] = (grid.m)[jj]*(grid.n)[jj]; 
    }
    if (grid.globe_lattop == 90.) {
      /* north pole */
      lat = DEG*(grid.lat)[2*(nj+1)];
      (grid.f )[2*(nj+1)  ]  = 2.*omega*sin(lat);
      (grid.n )[2*(nj+1)-1] /= 1.+sqrt(.5);
      /* wedge shaped area: */
      (grid.mn)[2*(nj+1)-1]  = grid.n[2*(nj+1)-1]*grid.n[2*(nj+1)-1]/(.5*dlnr);
      (grid.mn)[2*(nj+1)  ]  = 2.*(grid.mn)[2*(nj+1)-1];
    }
    else {
      jj = 2*(nj+1);
      lat  = DEG*(grid.lat)[jj];
      rln  = re/sqrt( 1.+ pow(rp/re*tan(lat),(double)2.) );
      rlt  = rln/( cos(lat)*( pow(sin(lat),(double)2.)+
             pow(re/rp*cos(lat),(double)2.)));
    
      (grid.f )[jj] = 2.*omega*sin(lat);
      (grid.m )[jj] = 1./(rln*dlnr);
      (grid.n )[jj] = 1./(rlt*dltr);
      (grid.mn)[jj] = grid.m[jj]*grid.n[jj]; 
    }
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      lat = DEG*(grid.f_plane_lat0);
      dx  = 2.*(grid.f_plane_half_width)/ni;
      dy  = dx;
      for (jj = 0; jj <= 2*(nj+1); jj++) {
        (grid.f )[jj] = 2.*omega*sin(lat);
        (grid.m )[jj] = 1./dx;
        (grid.n )[jj] = 1./dy;
        (grid.mn)[jj] = 1./(dx*dy);
      }
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      lat  = DEG*(grid.f_plane_lat0);
      dlnr = grid.dln*DEG;
      dy   = grid.f_plane_half_width*(grid.dlt/90.);
      rln  = grid.f_plane_half_width+dy;
      for (jj = 0; jj < 2*(nj+1); jj++) {
        dx = rln*dlnr;
        (grid.f )[jj] = 2.*omega*sin(lat);
        (grid.m )[jj] = 1./dx;
        (grid.n )[jj] = 1./dy;
        (grid.mn)[jj] = grid.m[jj]*grid.n[jj];
        rln -= dy/2.;
      }
      /* pole */
      (grid.f )[2*(nj+1)  ]  = 2.*omega*sin(lat);
      (grid.n )[2*(nj+1)-1] /= 1.+sqrt(.5);
      /* wedge shaped area: */
      (grid.mn)[2*(nj+1)-1]  = grid.n[2*(nj+1)-1]*grid.n[2*(nj+1)-1]/(.5*dlnr);
      (grid.mn)[2*(nj+1)  ]  = 2.*(grid.mn)[2*(nj+1)-1];
    }
  }
  
  /*
   *  Sponge-layer constants.
   */
  if (grid.k_sponge > 0) {
    double
      t0_inv;

    t0_inv = 1./(5.*(double)grid.dt);
    grid.t_sponge_inv = dvector(1,grid.k_sponge);
    for (k = 1; k <= grid.k_sponge; k++) {
      tmp = (double)(grid.k_sponge+1-k)/(grid.k_sponge);
      grid.t_sponge_inv[k] = t0_inv*.5*(1.-cos(M_PI*tmp));
    }
  }

  /*
   *  Allocate memory for the prognostic variables, and diagnostic
   *  variables that are calculated in timestep().
   *
   *  NOTE: EPIC_VIEW indicates the AVS subroutine epic_view, in which 
   *        case the variable memory is already declared upstream.
   */

#if !defined(EPIC_VIEW)
#if !defined(EPIC_AVS) 
  /*
   *  Allocate vector memory for the prognostic variables:
   */
  for (index = 0; index < MAX_NVARS; index++) {
    if (var.chem_on[index]) {
      var.vector[index] = dvector(0,NELEM3D-1);
    }
  }

  /* Three KJ planes: */
  diag.vector[0] = dvector(0,Kadim*Jadim-1);
  diag.vector[1] = dvector(0,Kadim*Jadim-1);
  diag.vector[2] = dvector(0,Kadim*Jadim-1);
#else
  /* 
   *  Allocate AVSfield memory for the prognostic variables: 
   */
  dims[0] = IADIM, dims[1] = JADIM, dims[2] = KADIM;
  sprintf(field_type,"field 3D %d-vector double uniform",var.nvars);
  var.field = (AVSfield_double *)AVSdata_alloc(field_type,dims);

  /*
   *  Allocate AVSfield memory for KJ-plane diagnostic variables:
   */
  dims[0] = 1, dims[1] = JADIM, dims[2] = KADIM;
  sprintf(field_type,"field 3D 3-vector double uniform");
  diag.field  = (AVSfield_double *)AVSdata_alloc(field_type,dims);
#endif

  if (strcmp(planet->class,"terrestrial") == 0) {
    /*
     * Allocate memory for surface topography, multiplied by g:
     */
    var.surface_gz = dvector(0,NELEM2D-1);
  }
  /*
   *  Spinup zonal-wind profile:
   */
  var.u_spinup = dvector(0,NELEM3D-1);

#if !defined(EPIC_RECEIVE)
  /*
   *  Three tendency (d/dt) time planes are used in the 3rd Order 
   *  Adams-Bashforth timestep.
   */
  for (index = 0; index < MAX_NVARS; index++) {
    if (var.chem_on[index]) {
      var.vector[MAX_NVARS+index] = dvector(0,3*NELEM3D-1);
    }
  }
#endif

  /* 
   * Pointers to the time planes 
   */
  IT_MINUS2 = 0;
  IT_MINUS1 = 1;
  IT_ZERO   = 2;
#endif

  /*
   * Layer values of pressure:
   */
  var.p1 = dvector(0,NELEM3D-1);
  /*
   * Temporarily set using grid.p_avg[K] values, to aid bootstrapping p initialization
   * in epic_initial.c:
   */

  for (k = 1; k <= KLAST_ACTIVE; k++) {
    kappap1   = planet->kappa+1.;
    kappa_inv = 1./(kappap1-1.);
    for (j = JLO; j <= JHI; j++) {
      for (i = ILO; i <= IHI; i++) {
        if (k == 1) {
          /* top of top layer has p = 0 */
          P1(k,j,i) = grid.p_avg[k]*pow(kappap1,-kappa_inv);
        }
        else {
          P1(k,j,i) = pow((pow(grid.p_avg[k],kappap1)-pow(grid.p_avg[k-1],kappap1))/
                         (kappap1*(grid.p_avg[k]-grid.p_avg[k-1])),kappa_inv);
        }
      }
    }
  }
  if (strcmp(planet->class,"gas-giant") == 0) {
    /* Set interior layer pressure to atmosphere-interior interface pressure. */
    k = grid.nk;
    for (j = JLO; j <= JHI; j++) {
      for (i = ILO; i <= IHI; i++) {
        P1(k,j,i) = grid.p_avg[k-1];
      }
    }
  }

  /*
   * Heavily used diagnostics: temperature and density:
   */
  var.t    = dvector(0,NELEM3D-1);
  var.t2   = dvector(0,NELEM3D-1);
  var.rho  = dvector(0,NELEM3D-1);
  var.rho2 = dvector(0,NELEM3D-1);

  return;
}

/*====================== end of make_arrays() ===============================*/

/*====================== free_arrays() ======================================*/

void free_arrays(planetspec *planet)
    /*
      *  Free memory allocated by make_arrays():
      */
{
  int 
    it,
    nk,nj,ni,
    index;
  int
    idbms=0;
  char
    dbmsname[]="free_arrays";

  nk = grid.nk;
  nj = grid.nj;
  ni = grid.ni;

  free_dvector(var.rho2,0,NELEM3D-1);
  free_dvector(var.rho, 0,NELEM3D-1);
  free_dvector(var.t2,  0,NELEM3D-1);
  free_dvector(var.t,   0,NELEM3D-1);
  free_dvector(var.p1,  0,NELEM3D-1);
  
#if !defined(EPIC_VIEW)
#if !defined(EPIC_RECEIVE)
  for (index = MAX_NVARS-1; index >= 0; index--) {
    if (var.chem_on[index]) {
      free_dvector(var.vector[MAX_NVARS+index],0,3*NELEM3D-1);
    }
  }
#endif

#if !defined(EPIC_AVS)
  free_dvector(diag.vector[2],0,Kadim*Jadim-1);
  free_dvector(diag.vector[1],0,Kadim*Jadim-1);
  free_dvector(diag.vector[0],0,Kadim*Jadim-1);
  for (index = MAX_NVARS-1; index >= 0; index--) {
    if (var.chem_on[index]) {
      free_dvector(var.vector[index],0,NELEM3D-1);
    }
  }
#else
  AVSdata_free("field",(char *)var.field);
  AVSdata_free("field",(char *)diag.field);
#endif
  free_dvector(var.u_spinup,0,NELEM3D-1);
  if (strcmp(planet->class,"terrestrial") == 0) {
    free_dvector(var.surface_gz,0,NELEM2D-1);
  }
#endif

  if (grid.k_sponge > 0) {
    free_dvector(grid.t_sponge_inv,1,grid.k_sponge);
  }

  free_dvector(grid.mn,  0, 2*(nj+1));
  free_dvector(grid.n,   0, 2*(nj+1));
  free_dvector(grid.m,   0, 2*(nj+1));
  free_dvector(grid.f,   0, 2*(nj+1));
  free_dvector(grid.lat, 0, 2*(nj+1));   
  free_dvector(grid.lon, 0, 2*(ni+1));

  return;
}

/*======================= end of free_arrays() ==============================*/

/*======================= extract_scalar() ==================================*/

/*
 *  Extracts 2D plane from AVS field, or 3D memory on the ncube2.
 */

double *extract_scalar(int    index,
                       int    k,
                       double *buffji) 
{
  int
    j,i;

  if (index < MAX_NVARS) {
    for (j = JLO-JPAD; j <= JHI+JPAD; j++) {
      for (i = ILO-IPAD; i <= IHI+IPAD; i++) {
#if !defined(EPIC_AVS)
        BUFFJI(j,i) 
          = (var.vector[index])[i+(j)*Iadim+(k)*Nelem2d-Shift3d];
#else
        BUFFJI(j,i) 
          = FIELD(var.field,var.index[index],k,j,i);
#endif
      }
    }
  }
  else {
    for (j = JLO-JPAD; j <= JHI+JPAD; j++) {
      for (i = ILO-IPAD; i <= IHI+IPAD; i++) {
        BUFFJI(j,i) 
          = (var.vector[index])[i+(j)*Iadim+(k)*Nelem2d-Shift3d];
      }
    }
  }

  return buffji;
}

/*======================= end of extract_scalar() ===========================*/

/*======================= insert_scalar() ===================================*/

/*
 *  Inserts 2D plane into AVS field, or 3D memory for ncube2.
 */

void insert_scalar(int    index,
                   int    k,
                   double *buffji)
{
  int
    j,i;

  if (index < MAX_NVARS) {
    for (j = JLO-JPAD; j <= JHI+JPAD; j++) {
      for (i = ILO-IPAD; i <= IHI+IPAD; i++) {
#if !defined(EPIC_AVS)
        (var.vector[index])[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
          = BUFFJI(j,i);
#else
        FIELD(var.field,var.index[index],k,j,i) 
          = BUFFJI(j,i);
#endif
      }
    }
  }
  else {
    for (j = JLO-JPAD; j <= JHI+JPAD; j++) {
      for (i = ILO-IPAD; i <= IHI+IPAD; i++) {
        (var.vector[index])[i+(j)*Iadim+(k)*Nelem2d-Shift3d]
          = BUFFJI(j,i);
      }
    }
  }

  return;
}

/*======================= end of insert_scalar() ============================*/

#if defined(EPIC_AVS)
/*======================= grid_to_domain() ==================================*/

void grid_to_domain(domainspec *domain,
                    int         direction) 
{
  /*
   *  The domain structure holds information passed through an AVS port.
   *
   *  If direction ==  1, copy grid and var information to domain->
   *  If direction == -1, copy domain information to grid and var.
   */
  int 
    i,nk,nbytes;
  static int
    initialized=0;

  if (direction == 1) {
  /*
   *  grid information:
   */
    nk = grid.nk;
    domain->data_version       = grid.data_version;
    strncpy(domain->geometry,    grid.geometry,       GEOM_STR);
    strncpy(domain->view_host,   grid.view_host,            80);
    strncpy(domain->eos,         grid.eos,                   8);
    domain->globe_lonbot       = grid.globe_lonbot;
    domain->globe_lontop       = grid.globe_lontop;
    domain->globe_latbot       = grid.globe_latbot;
    domain->globe_lattop       = grid.globe_lattop;
    strncpy(domain->f_plane_map, grid.f_plane_map, GEOM_STR   );
    domain->f_plane_lat0       = grid.f_plane_lat0;
    domain->f_plane_half_width = grid.f_plane_half_width;
    domain->dt                 = grid.dt;
    domain->cfl_dt             = grid.cfl_dt;
    domain->nk                 = grid.nk;
    domain->nj                 = grid.nj;
    domain->ni                 = grid.ni;
    domain->dln                = grid.dln;
    domain->dlt                = grid.dlt;
    domain->theta_bot          = grid.theta_bot;
    domain->theta_top          = grid.theta_top;
    domain->press0             = grid.press0;
    domain->mont0              = grid.mont0;
    domain->hasten             = grid.hasten;
    domain->k_sponge           = grid.k_sponge;
    domain->init_type          = grid.init_type;
    domain->newt_cool_on       = grid.newt_cool_on;
    domain->thermal_conduct_on = grid.thermal_conduct_on;
    for (i = 0; i < TOPDIM; i++) {
      domain->wrap[i]          = grid.wrap[i];
      domain->pad[i]           = grid.pad[i];
    }
    domain->jlo                = grid.jlo;
    domain->jfirst             = grid.jfirst;
    domain->klast_active       = grid.klast_active;
    domain->we_num_nodes       = grid.we_num_nodes;
    domain->they_num_nodes     = grid.they_num_nodes;
    domain->itback             = grid.itback;
    domain->itout              = grid.itout;
    domain->itsave             = grid.itsave;
    domain->aux_a              = grid.aux_a;
    domain->aux_b              = grid.aux_b;
    domain->aux_c              = grid.aux_c;
    domain->aux_fa             = grid.aux_fa;
    domain->aux_fb             = grid.aux_fb;
    domain->aux_fc             = grid.aux_fc;

    domain->prandtl            = grid.prandtl;
    nbytes = (MAX_NU_ORDER+1)*sizeof(double);
    memcpy(&(domain->nu[0]),&(grid.nu[0]),nbytes);

    nbytes = (2*nk)*sizeof(double);
    memcpy(&(domain->theta[2]),&(grid.theta[2]),nbytes);

    nbytes = nk*sizeof(double);
    memcpy(&(domain->rgas[ 1]),&(grid.rgas[ 1]),nbytes);
    memcpy(&(domain->kappa[1]),&(grid.kappa[1]),nbytes);

    nbytes = (nk)*sizeof(double);
    memcpy(&(domain->p_avg[1]),&(grid.p_avg[1]),nbytes);

    nbytes = (nk)*sizeof(double);
    memcpy(&(domain->qmin[1]),&(grid.qmin[1]),nbytes);
    memcpy(&(domain->qmax[1]),&(grid.qmax[1]),nbytes);
    memcpy(&(domain->tmin[1]),&(grid.tmin[1]),nbytes);
    memcpy(&(domain->tmax[1]),&(grid.tmax[1]),nbytes);

  /*
   *  var information:
   */
    domain->time_fp_bar = var.time_fp_bar;
    domain->nvars       = var.nvars;
    for (i = U_INDEX; i <= DRY_AIR_INDEX; i++) {
      domain->index[  i] = var.index[  i];
      domain->chem_on[i] = var.chem_on[i];
      strcpy(domain->chem_name[ i],var.chem_name[ i]);
      strcpy(domain->tend_name[ i],var.tend_name[ i]);
      strcpy(domain->chem_units[i],var.chem_units[i]);
      strcpy(domain->tend_units[i],var.tend_units[i]);
    }
  }
  else if (direction == -1) {
    nk = domain->nk;

    /* grid information: */
    grid.data_version       = domain->data_version;
    strncpy(grid.geometry,    domain->geometry,       GEOM_STR);
    strncpy(grid.view_host,   domain->view_host,            80);
    strncpy(grid.eos,         domain->eos,                   8);
    grid.globe_lonbot       = domain->globe_lonbot;
    grid.globe_lontop       = domain->globe_lontop;
    grid.globe_latbot       = domain->globe_latbot;
    grid.globe_lattop       = domain->globe_lattop;
    strncpy(grid.f_plane_map, domain->f_plane_map, GEOM_STR   );
    grid.f_plane_lat0       = domain->f_plane_lat0;
    grid.f_plane_half_width = domain->f_plane_half_width;
    grid.dt                 = domain->dt;
    grid.cfl_dt             = domain->cfl_dt;
    grid.nk                 = domain->nk;
    grid.nj                 = domain->nj;
    grid.ni                 = domain->ni;
    grid.dln                = domain->dln;
    grid.dlt                = domain->dlt;
    grid.theta_bot          = domain->theta_bot;
    grid.theta_top          = domain->theta_top;
    grid.press0             = domain->press0;
    grid.mont0              = domain->mont0;
    grid.hasten             = domain->hasten;
    grid.k_sponge           = domain->k_sponge;
    grid.init_type          = domain->init_type;
    grid.newt_cool_on       = domain->newt_cool_on;
    grid.thermal_conduct_on = domain->thermal_conduct_on;
    for (i = 0; i < TOPDIM; i++) {
      grid.wrap[i]          = domain->wrap[i];
      grid.pad[i]           = domain->pad[i];
    }
    grid.jlo                = domain->jlo;
    grid.jfirst             = domain->jfirst;
    grid.klast_active       = domain->klast_active;
    grid.we_num_nodes       = domain->we_num_nodes;
    grid.they_num_nodes     = domain->they_num_nodes;
    grid.itback             = domain->itback;
    grid.itout              = domain->itout;
    grid.itsave             = domain->itsave;
    grid.aux_a              = domain->aux_a;
    grid.aux_b              = domain->aux_b;
    grid.aux_c              = domain->aux_c;
    grid.aux_fa             = domain->aux_fa;
    grid.aux_fb             = domain->aux_fb;
    grid.aux_fc             = domain->aux_fc;
    
    grid.prandtl            = domain->prandtl;
    nbytes = (MAX_NU_ORDER+1)*sizeof(double);
    memcpy(&(grid.nu[0]),&(domain->nu[0]),nbytes);

    nbytes = (2*nk)*sizeof(double);
    memcpy(&(grid.theta[2]),&(domain->theta[2]),nbytes);

    nbytes = nk*sizeof(double);
    memcpy(&(grid.rgas[ 1]),&(domain->rgas[ 1]),nbytes);
    memcpy(&(grid.kappa[1]),&(domain->kappa[1]),nbytes);

    nbytes = (nk)*sizeof(double);
    memcpy(&(grid.p_avg[1]),&(domain->p_avg[1]),nbytes);

    nbytes = (nk)*sizeof(double);
    memcpy(&(grid.qmin[1]),&(domain->qmin[1]),nbytes);
    memcpy(&(grid.qmax[1]),&(domain->qmax[1]),nbytes);
    memcpy(&(grid.tmin[1]),&(domain->tmin[1]),nbytes);
    memcpy(&(grid.tmax[1]),&(domain->tmax[1]),nbytes);

  /*
   *  var information:
   */
    var.time_fp_bar = domain->time_fp_bar;
    var.nvars       = domain->nvars;
    for (i = U_INDEX; i <= DRY_AIR_INDEX; i++) {
      var.index[  i] = domain->index[  i];
      var.chem_on[i] = domain->chem_on[i];
      strcpy(var.chem_name[ i],domain->chem_name[ i]);
      strcpy(var.tend_name[ i],domain->tend_name[ i]);
      strcpy(var.chem_units[i],domain->chem_units[i]);
      strcpy(var.tend_units[i],domain->tend_units[i]);
    }
  }
  else {
    fprintf(stderr,"direction != 1 or -1 in grid_to_domain() \n");
    exit(1);
  }

  return;
}

/*======================= end of grid_to_domain() ==============================*/
#endif

/*======================= dvector() ============================================*/

double *dvector(int nl, int nh)
      /*
       *  Allocates memory for a 1D double array 
       *  with range [nl..nh].
       */
{
  unsigned int  
    len_safe;
  int           
    nl_safe, nh_safe;
  double         
    *m;

  if (nh < nl) {
    fprintf(stderr,"called dvector(%d,%d) \n",nl,nh);
    exit(1);
  }

  nl_safe  = (nl < 0) ? nl : 0;
  nh_safe  = (nh > 0) ? nh : 0;
  len_safe = (unsigned)(nh_safe - nl_safe + 1);

  m = (double *)calloc(len_safe, sizeof(double));
  if (!m) {
    fprintf(stderr, "calloc error in dvector \n");
    exit(1);
  }
  m -= nl_safe;
  return m;
}

/*======================= end of dvector() ====================================*/

/*======================= free_dvector() ======================================*/

void  free_dvector(double *m, int nl, int nh)
      /*
       *  Frees memory allocated by dvector().
       */
{
  int  
    nl_safe;

  nl_safe = (nl < 0) ? nl : 0;
  m += nl_safe;
  free(m);
}

/*======================= end of free_dvector() ===============================*/

/*======================= spline() ============================================*/

int spline(double *x,  double *y,  int n, 
           double yp0, double ypn, double *y2)
     /* 
      *  Cubic spline routine (zero-offset arrays). 
      *  Adapted from Numerical Recipes in C, p. 96.
      */
{
  int     
    i, 
    k;
  double   
    p, 
    qn, 
    sig, 
    un, 
    *u;

  /*
   * Screen for dx=0 error:
   */
  for (i = 1; i <= n-2; i++) {
    if (x[i+1] == x[i-1]) {
      fprintf(stderr,"Error: spline(): x[%d] = x[%d] = %e \n",
                      i+1,i-1,x[i+1]);
      exit(1);
    }
    else if (x[i+1] == x[i]) {
      fprintf(stderr,"Error: spline(): x[%d] = x[%d] = %e \n",
                      i+1,i,x[i+1]);
      exit(1);
    }
    else if (x[i] == x[i-1]) {
      fprintf(stderr,"Error: spline(): x[%d] = x[%d] = %e \n",
                      i,i-1,x[i]);
      exit(1);
    }
  }
  
  u = dvector(0,n-2);
  
  if (yp0 > 0.99e+30) {
    y2[0] = 0.;
    u[ 0] = 0.;
  } 
  else {
    y2[0] = -0.5; 
    u[ 0] = (3./(x[1]-x[0]))*((y[1]-y[0])/(x[1]-x[0])-yp0);
  }

  for (i = 1; i <=  n-2; i++) {
    sig   = (x[i]-x[i-1])/(x[i+1]-x[i-1]);
    p     = sig*y2[i-1]+2.;
    y2[i] = (sig - 1.)/p;
    u[ i] = (y[i+1]-y[i  ])/(x[i+1]-x[i  ])
           -(y[i  ]-y[i-1])/(x[i  ]-x[i-1]);

    u[ i] = (6.*u[i]/(x[i+1]-x[i-1])-sig*u[i-1])/p;

  }

  if (ypn > 0.99e+30) {
    qn = 0.;
    un = 0.;
  }
  else {
    qn = 0.5;
    un = (3./(x[n-1]-x[n-2]))*(ypn-(y[n-1]-y[n-2])/(x[n-1]-x[n-2]));
  }
  
  y2[n-1] = (un-qn*u[n-2])/(qn*y2[n-2]+1.);
  
  for (k = n-2; k >= 0; k--) {
    y2[k] = y2[k]*y2[k+1]+u[k];
  }
  
  free_dvector(u,0,n-2);
  
  return 0;
}

/*======================= end of spline() ===================================*/

/*======================= splint() ==========================================*/

/*  
 *  Evaluates cubic-spline interpolations.
 *  This version assumes you have already found the correct position
 *  in the tables, unlike the Numerical Recipes version. 
 *  The function find_place_in_table() is used for this purpose.
 */

double splint(double xx, double *x, double *y, double *y2, double dx)
{
  double  
    a, 
    b,
    answer;
  
  a = ( *(x+1) -  xx  )/dx;
  b = (   xx   - *(x) )/dx;

  answer = a * *(y) + b * *(y+1) + ( (a*a*a-a) * *(y2  )
                                   + (b*b*b-b) * *(y2+1) )*dx*dx/6.;
  
  return answer;
}

/*======================= end of splint() ===================================*/

/*======================= linint() ==========================================*/

/*
 * Evaluates linear interpolation using same arguments as splint().
 */

double linint(double xx, double *x, double *y, double *y2, double dx)
{
  double
    answer;
  
  answer = *(y) + (*(y+1) - *(y))*(xx - *(x))/dx;

  return answer;
}

/*======================= end of linint() ===================================*/

/*======================= sech2() ===========================================*/

double sech2(double xx)
     /*
      *  Evaluates the square of the hyperbolic secant
      */
{
  double 
    a,
    result;

  a = exp(xx) + exp(-xx);
  result = 4.0/(a*a);

  return result;
}

/*======================= end of sech2() =====================================*/

/*======================= find_root() ========================================*/

/*
 * Adapted from zridder(), Numerical Recipes in C, 2nd ed., p.358.
 */

#define MAXIT 60

double find_root(double (*func)(double),
                 double x1,
                 double x2,
                 double xacc)
{
  int
    j;
  double
    ans,fh,fl,fm,fnew,
    s,xh,xl,xm,xnew;

  fl = (*func)(x1);
  fh = (*func)(x2);
  if ((fl > 0. && fh < 0.) || (fl < 0. && fh > 0.)) {
    xl  = x1;
    xh  = x2;
    ans = DBL_MAX;
    for (j = 0; j < MAXIT; j++) {
      xm = 0.5*(xl+xh);
      fm = func(xm);
      s  = sqrt(fm*fm-fl*fh);
      if (s == 0.) {
        return ans;
      }
      xnew = xm+(xm-xl)*((fl > fh ? 1. : -1.)*fm/s);
      if (fabs(xnew-ans) <= xacc) {
        return ans;
      }
      ans  = xnew;
      fnew = func(ans);
      if (fnew == 0.) {
        return ans;
      }
      if ((fnew > 0. ? fabs(fm) : -fabs(fm)) != fm) {
        xl = xm;
        fl = fm;
        xh = ans;
        fh = fnew;
      }
      else if ((fnew > 0. ? fabs(fl) : -fabs(fl)) != fl) {
        xh = ans;
        fh = fnew;
      }
      else if ((fnew > 0. ? fabs(fh) : -fabs(fh)) != fh) {
        xl = ans;
        fl = fnew;
      }
      else {
        fprintf(stderr,"Warning: find_root(): should never get here. \n");
      }
      if (fabs(xh-xl) <= xacc) {
        return ans;
      }
    }
    fprintf(stderr,"Error: find_root(): exceeded MAXIT = %d \n",MAXIT);
    exit(1);
  }
  else {
    if (fl == 0.) {
      return x1;
    }
    if (fh == 0.) {
      return x2;
    }
    fprintf(stderr,"Error: find_root(): not bracketed: fl,fh = %f %f \n",fl,fh);
    exit(1);
  }

  return 0.;  /* Should never get here. */
}

/*======================= end of find_root() =================================*/

/*======================= find_place_in_table() ==============================*/

/*
 * Find place in table. Useful for applying cubic splines, etc.
 */
int find_place_in_table(int     n,
                        double *table,
                        double *x,
                        double *x_d)
{
  int
    i;

  i = n-1;
  while (*x < table[i] && i > 0) {  
    i--;
    if (i == 0) {
      *x = MAX(*x,table[0]);
      break;
    }
  }
  *x_d = table[i+1]-table[i];

  return i;
}

/*======================= end of find_place_in_table() =======================*/

/*======================= cnum() =============================================*/

complex cnum(double x, double y)
{
  complex
    ans;

  ans.x = x;
  ans.y = y;

  return ans;
}

/*======================= end of cnum() ======================================*/

/*======================= cmult() ============================================*/

complex cmult(complex z1,complex z2)
{
  complex
    ans;

  ans.x = (z1.x)*(z2.x)-(z1.y)*(z2.y);
  ans.y = (z1.x)*(z2.y)+(z1.y)*(z2.x);

  return ans;
}
/*======================= end of cmult() =====================================*/

/*======================= cadd() =============================================*/

complex cadd(complex z1,complex z2)
{
  complex
    ans;

  ans.x = (z1.x)+(z2.x);
  ans.y = (z1.y)+(z2.y);

  return ans;
}

/*======================= end of cadd() ======================================*/

/*======================= csub() =============================================*/

complex csub(complex z1,complex z2)
{
  complex
    ans;

  ans.x = (z1.x)-(z2.x);
  ans.y = (z1.y)-(z2.y);

  return ans;
}

/*======================= end of csub() ======================================*/

/*======================= cexp() =============================================*/

complex cexp(complex z)
{
  complex 
    ans;

  ans.x = exp(z.x)*cos(z.y);
  ans.y = exp(z.x)*sin(z.y);

  return ans;
}

/*======================= end of cexp() ======================================*/

/*======================= cabsolute() ========================================*/

/*
 * NOTE: For LINUX with -D_BSD_SOURCE, cabs() is defined, such that a 
 * type-mismatch error occurs if we call this function cabs().
 */
 
double cabsolute(complex z)
{

  return sqrt((z.x)*(z.x)+(z.y)*(z.y));
}

/*======================= end of cabsolute() =================================*/

/*======================= creal() ============================================*/

double creal(complex z) 
{
  return (z.x);
}

/*======================= end of creal() =====================================*/

/*======================= cimag() ============================================*/

double cimag(complex z)
{
  return (z.y);
}

/*======================= end of cimag() =====================================*/

/* * * * * * * * * * *  end of epic_funcs_util.c  * * * * * * * * * * * * * * */









