/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*======================= heating() =========================================*/

/*
 * Return heat per unit mass, Q, calculated at the 
 * bottom interface of layer K.
 */

#include <epic.h>

/* Amplitude of initial heating perturbation: */
#define HEAT_SEED 0.0
/* 
 * Set ADJUST_TEQ_MAX > 0. to adjust Teq such that int(wh)dxdy = 0,
 * subject to the restriction Teq_adj <= ADJUST_TEQ_MAX. 
 */
#define ADJUST_TEQ_MAX 100.

/*
 * Function prototypes:
 */
double heat_flux_top(planetspec *planet,
                     int         J,
                     int         I);
double heat_flux_bot(planetspec *planet,
                     int         J,
                     int         I);
void test_temp_eq_ura(planetspec *planet);

void rain(planetspec *planet,
                    double     *heat,
                    int         K);

void external_heating(planetspec *planet,
                      double     *heat,
		      int         K);

/*
 * Shift macros:
 */
#define FLUX_TOP(j,i) flux_top[i+(j)*Iadim-Shift2d]
#define DFLUX(j,i)    dflux[   i+(j)*Iadim-Shift2d]

/*
 * Main subroutine:
 */
void heating(planetspec *planet,
             double     *heat,
             int         K)
/*
 * Sets heating variable, Q. 
 *  
 * If called from EPIC_VIEW, also writes heating profiles to a file.
 *
 */
{
  int 
    kk,
    J,I,
    flag,nkm1,nkm1ni,shift;
  long
    offset;
  static int
    initialized=0,
    ntp;
  static double
    *pdat,
    *tdat,
    *t_y2dat,
    *t,
    *lapt,
    *BuffD[2];
  double
    t_eq,t_eq0,p_d,neglogp,
    t_cool,cp_over_time,cprh2,
    theta,dtheta,
    fpara,dfpdt,fgibb,mu,
    theta_o,theta_p,
    fpe,uoup,time_fp_inv,
    temp_top,temp_bot,temp_mid,
    alpha,pressure,flux_bot,
    add_heat,
    tmp;
  double
    area,
    t_eq_adj,
    numerator,
    denominator;
  double
    h,*flux_top,*dflux;
  char   
    header[N_STR],
    infile[N_STR];
  FILE    
    *heat_bin;
  /* 
   * The following are part of DEBUG_MILESTONE statements: 
   */
  int
    idbms=0;
  char
    dbmsname[]="epic_heating";

  if (!initialized) {
    /* Allocate memory: */
    t        = dvector(0,Nelem2d-1);
    BuffD[0] = dvector(0,Nelem2d-1);
    if (grid.thermal_conduct_on == 1) {
      lapt     = dvector(0,Nelem2d-1);
      BuffD[1] = dvector(0,Nelem2d-1);
    }

    if (grid.newt_cool_on == TRUE &&
        strcmp(planet->name,"uranus") != 0) {
      /* 
       * Need radiative equilibrium T(p) profile if temp_eq() not implemented.
       * Use nominal T(p), adjusted at each time so that int(wh)dxdy = 0.
       *
       * Input t_vs_p  data:
       */
      ntp = read_t_vs_p(planet,0.,0.,NULL,NULL,NULL,NULL,INIT_DATA);

      /* Allocate memory: */
      pdat    = dvector(0,ntp-1);
      tdat    = dvector(0,ntp-1);
      t_y2dat = dvector(0,ntp-1);

      read_t_vs_p(planet,0.,0.,NULL,NULL,pdat,tdat,NONINIT_DATA);

      /* In order of increasing p: */
      for (kk = 0; kk < ntp; kk++) {
        /* spline on -log p */
        pdat[kk] = -log(pdat[kk]);  
      }
      spline(pdat,tdat,ntp,1.e+30,1.e+30,t_y2dat);
    }
    initialized = 1;
  }
  /* End initialization */

#if defined(EPIC_VIEW) 
  nkm1   = KLAST_ACTIVE;
  nkm1ni = nkm1*grid.ni;
  shift  = 1+1*nkm1+grid.jlo*nkm1ni;
  /* 
   * Open file to output random-accessed heating-profile data. 
   *
   * NOTE: For gcc, opening with "a" always writes to the end of file,
   *       so this mode is not used here.
   *
   * Create an empty heat.bin if none exists, so that "r+" works
   * the first time.
   */
  heat_bin = fopen(EPIC_PATH"/tmp/heat.bin","r+b");
  if (!heat_bin) {
    heat_bin = fopen(EPIC_PATH"/tmp/heat.bin","wb");
    if (!heat_bin) {
      perror("epic_heat: fopen heat.bin wb");
    }
  }
#endif

  /* Zero heat array */
  memset(heat,0,Nelem2d*sizeof(double));

  /* Heat = 0 on top interface: */
  if(K <= 0) {
    return;
  }

  /* Check for out-of-bounds K: */
  if (K > KLAST_ACTIVE) {
    fprintf(stderr,"Error in epic_heating, K = %d > KLAST_ACTIVE = %d \n",
                   K,KLAST_ACTIVE);
    exit(1);
  }

  if (grid.thermal_conduct_on == 1) {
    /* 
     * Calculate horizontal Laplacian of T, including 
     * multiplication by conductivity coefficient, kcond(planet,T): 
     */
    laplacian(planet,K,t,NULL,lapt,NULL,BuffD[0],BuffD[1],kcond);

    /* Divide by density: */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        /* alpha = 1/density */
        alpha      = 1./RHO2(K,J,I);
        LAPT(J,I) *= alpha;
      }
    }

    /* 
     * Add vertical term: 
     */
    /* Calculate heat flux above interface: */
    flux_top = BuffD[0];
    if (K == 1) {
      /* Heat flux is a boundary condition at top of model: */
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          FLUX_TOP(J,I) = heat_flux_top(planet,J,I);
        }
      }
    }
    else {
      dtheta = grid.theta[2*K-1]-grid.theta[2*K+1];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          temp_top      = T2(K-1,J,I);
          temp_mid      = T( K,  J,I);
          temp_bot      = T2(K  ,J,I); 
          alpha         = 1./RHO(K,J,I);
          h             = get_h(planet,2*K,J,I);
          FLUX_TOP(J,I) = kcond(planet,temp_mid)*(temp_top-temp_bot)/
                          (alpha*dtheta*h);
        }
      }
    }

    /* Calculate heat flux below interface, and take difference: */
    dflux = flux_top;
    if (K == KLAST_ACTIVE) {
      /* Heat flux is a boundary condition at bottom of model: */
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          DFLUX(J,I) -= heat_flux_bot(planet,J,I);
        }
      }
    }
    else {
      dtheta = grid.theta[2*K+1]-grid.theta[2*(K+1)+1];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          temp_top    = T2(K,  J,I);
          temp_mid    = T( K+1,J,I);
          temp_bot    = T2(K+1,J,I);
          alpha       = 1./RHO(K+1,J,I);
          h           = get_h(planet,2*(K+1),J,I);
          DFLUX(J,I) -= kcond(planet,temp_mid)*(temp_top-temp_bot)/
                        (alpha*dtheta*h);
        }
      }
    }

    /* 
     * Divide by (dtheta*H) instead of (alpha*dtheta*H), 
     * to take into account final division by density.
     */
    dtheta = grid.theta[2*K]-grid.theta[2*(K+1)];
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        h = get_h(planet,2*K+1,J,I);
        LAPT(J,I) += DFLUX(J,I)/(h*dtheta);
      }
    }
    /* No need to update edges on lapt. */
  }

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      if (grid.thermal_conduct_on == 1) {
        /* 
         * Add thermal conduction: 
         */
        add_heat   = LAPT(J,I);
        HEAT(J,I) += add_heat;
      }
      else {
        add_heat = 0.;
      }

#if defined(EPIC_VIEW) 
      /* 
       * Record conductive heating: 
       */
      offset = (long)(((K+I*nkm1+J*nkm1ni-shift)*5+0)*sizeof(double));
      fseek(heat_bin,offset,SEEK_SET);
      fwrite(&add_heat,sizeof(double),1,heat_bin);
#endif

    }
  }

  cprh2 = 2.5*planet->rgas;

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      if (var.chem_on[FPARA_INDEX] == CHEM_ACTIVE) {
        /*
         * Add ortho-para hydrogen latent heating:
         * NOTE: use interface_value for fpara to get flag.
         */
        fpara = interface_value(planet,FPARA_INDEX,K,J,I,&flag);
        return_enthalpy(planet,fpara,P(K,J,I),T2(K,J,I),&fgibb,&fpe,&uoup);
        time_fp_inv = P(K,J,I)/var.time_fp_bar;

        time_fp_inv *= grid.hasten;

        if (flag == 1) {
          /* fpara = fpe */
          dfpdt = 0.;
        }
        else {
          dfpdt = (fpe-fpara)*time_fp_inv;
        }
        /* 
         * Call return_theta() to get theta_o,theta_p. 
         * These only depend on p,T.
         */
        return_theta(planet,fpara,P(K,J,I),T2(K,J,I),&theta_o,&theta_p);
        add_heat   = (planet->x_h2)*(uoup+cprh2*T2(K,J,I)*log(theta_p/theta_o))*dfpdt;
        HEAT(J,I) += add_heat;  
      }
      else {
        add_heat = 0.;
      }

#if defined(EPIC_VIEW) 
      /* 
       * Record fpara heating: 
       */
      offset = (long)(((K+I*nkm1+J*nkm1ni-shift)*5+1)*sizeof(double));
      fseek(heat_bin,offset,SEEK_SET);
      fwrite(&add_heat,sizeof(double),1,heat_bin);
#endif

    }
  }

  for (J = JLO; J <= JHI; J++) {
    if (grid.newt_cool_on == TRUE &&
        strcmp(planet->name,"uranus") != 0) {
      /* 
       * Specify latitude dependence of radiative cooling profile: 
       */
      if (strcmp(planet->name,"venus") == 0) {
        /* Equilibrium temperature:  Venus (U. Arizona), Tomasko, pp 604. */
        t_eq0 = pow(cos(grid.lat[2*J+1]*DEG),1.4/4.);
      }
      else {
        t_eq0 = 1.;
      }
    }
    for (I = ILO; I <= IHI; I++) {
      if (grid.newt_cool_on == TRUE) {
        t_cool       = t_rad(planet,P(K,J,I))/grid.hasten;
        cp_over_time = (grid.rgas[K]/grid.kappa[K])/t_cool;

        /*
         * Determine T_eq:
         */
        if (strcmp(planet->name,"uranus") == 0) {
          t_eq = temp_eq(planet,grid.lat[2*J+1],P(K,J,I),TIME*grid.hasten);

          /*** Test function for temp_eq 
          test_temp_eq_ura(planet);
          ***/
        }
        else {
          neglogp = -log(P(K,J,I));
          if (neglogp >= pdat[ntp-1]) {
            /* below range of data */
            t_eq = t_eq0*tdat[ntp-1];
          }
          else if (neglogp <= pdat[0]) {
            /* above range of data */
            t_eq = t_eq0*tdat[0];
          }
          else {
            /* find place in table */
            kk   = find_place_in_table(ntp,pdat,&neglogp,&p_d);
            t_eq = t_eq0*splint(neglogp,pdat+kk,tdat+kk,t_y2dat+kk,p_d);
          }
        }
        add_heat   = cp_over_time*(t_eq-T2(K,J,I));
        HEAT(J,I) += add_heat;
        /* 
         *Additional heating source to be used in simple
         *hot jupiter like planets 24 March 2004 (Danie&Raul)
         */
        if (strcmp(planet->name,"hotjupiter") == 0){
           if (-90. <= grid.lon[2*I+1] && grid.lon[2*I+1] <= 90.) {
              if (0. <= grid.lat[2*J+1]) {
                  add_heat = cp_over_time*50.*cos(grid.lat[2*J+1]*DEG)*cos(grid.lon[2*I+1]*DEG);
              }
              else {
                  add_heat = cp_over_time*150.*cos(grid.lat[2*J+1]*DEG)*cos(grid.lon[2*I+1]*DEG);
              }
              HEAT(J,I) += add_heat;
           }
        } 
      }
      else {
        add_heat = 0.;
      } 

#if defined(EPIC_VIEW)
      /* 
       * Record Newtonian cooling: 
       */
      offset = (long)(((K+I*nkm1+J*nkm1ni-shift)*5+2)*sizeof(double));
      fseek(heat_bin,offset,SEEK_SET);
      fwrite(&add_heat,sizeof(double),1,heat_bin);
#endif

    }
  }

  if (TIME == 0 && HEAT_SEED != 0. ) {
    /* 
     * Add initial heating perturbation: 
     */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        HEAT(J,I) += HEAT_SEED*
                     cos(.5*2.*M_PI*((double)J)/grid.nj)*
                     sin(6.*2.*M_PI*((double)I)/grid.ni);
      }
    }
    if (IAMNODE == 0) {
      fprintf(stderr,"epic_heating: HEAT_SEED = %e, K = %2d \n",
              HEAT_SEED,K);
    }
  }

  if (ADJUST_TEQ_MAX > 0. &&
      grid.newt_cool_on == TRUE) {
    /* 
     * Adjust T_eq such that the area average of hw = h*dtheta/dt = 0.
     */
    numerator   = 0.;
    denominator = 0.;
    for (J = JLO; J <= JHI; J++) {
      area = 1./grid.mn[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        /* Calculate h for bottom of layer: */
        h            = get_h(planet,2*K+1,J,I);
        t_cool       = t_rad(planet,P(K,J,I))/grid.hasten;
        cp_over_time = (grid.rgas[K]/grid.kappa[K])/t_cool;
        tmp          = h/T2(K,J,I)*area;
        numerator   -= tmp*HEAT(J,I);
        denominator += tmp*cp_over_time;
      }
    }

#if defined(EPIC_MPI)
    /* Global sum */ 
    tmp = numerator;
    MPI_Allreduce(&tmp,&numerator,1,MPI_DOUBLE,MPI_SUM,para.comm);
    tmp = denominator;
    MPI_Allreduce(&tmp,&denominator,1,MPI_DOUBLE,MPI_SUM,para.comm);
#endif

    t_eq_adj = numerator/denominator;

    /* Restrict fabs(t_eq_adj) <= ADJUST_TEQ_MAX: */
    if (fabs(t_eq_adj) > ADJUST_TEQ_MAX) {
      if (IAMNODE == NODE0) {
        fprintf(stderr,"Warning: heating(): t_eq_adj(K=%2d): %.1f > %.1f \n",
                        K,fabs(t_eq_adj),ADJUST_TEQ_MAX);
      }
    }
    if (t_eq_adj < 0.) {
      t_eq_adj = MAX(t_eq_adj,-ADJUST_TEQ_MAX);
    }
    else {
      t_eq_adj = MIN(t_eq_adj, ADJUST_TEQ_MAX);
    }
  }

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      if (ADJUST_TEQ_MAX > 0. &&
          grid.newt_cool_on == TRUE) {

        t_cool       = t_rad(planet,P(K,J,I))/grid.hasten;
        cp_over_time = (grid.rgas[K]/grid.kappa[K])/t_cool;
        /* 
         * This adjustment of t_eq ensures that
         * the area average of h*dtheta/dt = 0. 
         * (assuming fabs(t_eq_adj) <= ADJUST_TEQ_MAX).
         */
        add_heat   = cp_over_time*t_eq_adj;
        HEAT(J,I) += add_heat; 
      }
      else {
        add_heat = 0.;
      }

#if defined(EPIC_VIEW)
      /* 
       * Record adjustment heating: 
       */
      offset = (long)(((K+I*nkm1+J*nkm1ni-shift)*5+3)*sizeof(double));
      fseek(heat_bin,offset,SEEK_SET);
      fwrite(&add_heat,sizeof(double),1,heat_bin);
      /* 
       * Record total heating: 
       */
      fwrite(&(HEAT(J,I)),sizeof(double),1,heat_bin);
#endif

    }
  }

#if defined(EPIC_VIEW) 
  fclose(heat_bin);
#endif
  return;
}

/*======================= end of heating() ==================================*/

/*======================= t_rad() ===========================================*/

/*
 * Return radiative cooling time [s], given pressure [mks].
 */

double t_rad(planetspec *planet,
             double      pressure)
{
  int
    kk;
  static int
    initialized=0,
    n_t_cool;
  static double
    *p_t_cool_dat,     
    *t_cool_dat,         
    *t_cool_y2dat;
  double
    t_cool,
    p_t_cool_d,   
    neglogp,
    t_spinup;
  char   
    header[N_STR],
    infile[N_STR];
  FILE
    *t_cool_vs_p,
    *relax_times;  

  if (grid.newt_cool_on != TRUE) {
    fprintf(stderr,"Error: epic_heating.c, t_rad() called with newt_cool_on == %d \n",
            grid.newt_cool_on);
    exit(1);
  }

  if (!initialized) {
    initialized = 1;

    /*
     * Input t_cool_vs_p:
     */
    if (IAMNODE == 0) {
      sprintf(infile,EPIC_PATH"/data/%s/t_cool_vs_p.%s",planet->name,planet->name);
      t_cool_vs_p = fopen(infile,"r");
      for (kk = 0; kk < 6; kk++) {
        fgets(header,100,t_cool_vs_p); 
      }
      /* input number of data points */
      fscanf(t_cool_vs_p, "%d", &n_t_cool); 
    }

#if defined(EPIC_MPI)
    MPI_Bcast(&n_t_cool,1,MPI_INT,NODE0,para.comm);
#endif

    /* allocate memory */
    p_t_cool_dat = dvector(0,n_t_cool-1);
    t_cool_dat   = dvector(0,n_t_cool-1);
    t_cool_y2dat = dvector(0,n_t_cool-1);

    if (IAMNODE == 0) {
      relax_times = fopen(EPIC_PATH"/tmp/relax_times.dat","w");
      fprintf(relax_times," %s, grid.hasten = %3.0f \n",planet->name,grid.hasten);
      fprintf(relax_times,"   p[mbar]        trad[s]        tfp[s]      tdrag[s] \n");

      /* stored in order of decreasing p (increasing -log p) */
      for (kk = n_t_cool-1; kk >= 0;  kk--) {  
        fscanf(t_cool_vs_p, "%lf %*lf %lf", (p_t_cool_dat+kk), (t_cool_dat+kk));
        /* convert from mbar to mks */
        p_t_cool_dat[kk] *= 1.e+2; 

        /* Output table of relaxation times vs pressure: */
        if (grid.prandtl == 0.) {
          t_spinup = 1.e+20;
        }
        else {
          t_spinup = t_cool_dat[kk]/(grid.prandtl*grid.hasten);
        }
        fprintf(relax_times," %e  %e  %e  %e \n",
                p_t_cool_dat[kk]/100.,
                t_cool_dat[kk]/grid.hasten,
                var.time_fp_bar/p_t_cool_dat[kk]/grid.hasten,
                t_spinup);
      }
      fclose(t_cool_vs_p);
      fclose(relax_times);
    }

#if defined(EPIC_MPI)
    MPI_Bcast(p_t_cool_dat,n_t_cool,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(t_cool_dat,  n_t_cool,MPI_DOUBLE,NODE0,para.comm);
#endif

    for (kk = 0; kk < n_t_cool; kk++) {
      /* spline on -log p  */
      p_t_cool_dat[kk] = -log(p_t_cool_dat[kk]);  
    }
    spline(p_t_cool_dat,t_cool_dat,n_t_cool,1.e+30,1.e+30,t_cool_y2dat);
  }
  /* End of initialization. */

  /*
   *  Interpolate to get Newtonian cooling time:
   */
  neglogp = -log(pressure);
  if (neglogp >= p_t_cool_dat[n_t_cool-1]) {
    /* past range of t_cool data */
    t_cool = t_cool_dat[n_t_cool-1];
  }
  else if (neglogp <= p_t_cool_dat[0]) {
    /* past range of t_cool data */
    t_cool = t_cool_dat[0];
  }
  else {
    kk     = find_place_in_table(n_t_cool,p_t_cool_dat,&neglogp,&p_t_cool_d);
    t_cool = splint(neglogp,p_t_cool_dat+kk,t_cool_dat+kk,t_cool_y2dat+kk,p_t_cool_d);
  }

  return t_cool;
}

/*======================= end of t_rad() ====================================*/

/*======================= temp_eq() =========================================*/

/*
 * Time 0 is north spring equinox.
 * NOTE: We use dynamic-latitude in the EPIC model (for Uranus, this is opposite
 *       in sign to conventional latitude).
 */

#define CT(k,j,i) (ct[i+(ni+1)*((j-1)+nj*(k-1))])

double temp_eq(planetspec *planet,
               double      latitude,
               double      pressure,
               double      time)
{
  int
    k,j,i,
    kk,jj;
  static int
    initialized=0,
    nk,nj,ni;
  char
    infile[N_STR],
    checkstr[N_STR];
  double
    ans,
    dsin_y,dlogp,yy,pp,
    temp_equil[2][2],
    logp_k,logpress,
    lat,lattop,latbot,
    sin_y_j,sin_y,
    freqtime,
    tmp;
  static double
    freq,
    logpbot,logptop,
    sin_ybot,sin_ytop;
  static complex
    *ct;
  complex
    ccomp,
    carg,
    ctemp;
  FILE
   *t_cool_vs_yp;

  if (grid.newt_cool_on != TRUE) {
    fprintf(stderr,"Error: epic_heating.c, temp_eq() called with newt_cool_on == %d \n",
            grid.newt_cool_on);
    exit(1);
  }

  if (!initialized) {
    initialized  = 1;

    freq         = 2.*M_PI/(planet->orbit_period*365.*24.*60.*60.);

    if (IAMNODE == 0) {
      /* Input data: */
      sprintf(infile,EPIC_PATH"/data/%s/t_cool_vs_yp.%s",
                     planet->name,planet->name);
      t_cool_vs_yp = fopen(infile,"r");

      fscanf(t_cool_vs_yp,"%d %d %d %lf %lf %lf %lf",
             &ni,&nj,&nk,&logptop,&logpbot,&sin_ybot,&sin_ytop);
    }

#if defined(EPIC_MPI)
    MPI_Bcast(&ni,      1,MPI_INT,   NODE0,para.comm);
    MPI_Bcast(&nj,      1,MPI_INT,   NODE0,para.comm);
    MPI_Bcast(&nk,      1,MPI_INT,   NODE0,para.comm);
    MPI_Bcast(&logptop, 1,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(&logpbot, 1,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(&sin_ybot,1,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(&sin_ytop,1,MPI_DOUBLE,NODE0,para.comm);
#endif

    /* convert pressures from cgs to log(mks): */
    logptop = log(logptop/10.);
    logpbot = log(logpbot/10.);
    if (IAMNODE == 0) {
      fscanf(t_cool_vs_yp,"%lf %lf %lf %lf",
                           &tmp,&tmp,&tmp,&tmp);
      fscanf(t_cool_vs_yp,"%lf %lf",
                           &tmp,&tmp);
    }

    /* Allocate memory: */
    ct = (complex *)calloc(nk*nj*(ni+1),sizeof(complex));

    if (IAMNODE == 0) {
      for (k = 1; k <= nk; k++) {
        for (j = 1; j <= nj; j++) {
          for (i = 0; i <= ni; i++) {
            fscanf(t_cool_vs_yp,"%lf %lf",
                  &(CT(k,j,i).x),&(CT(k,j,i).y));
          }
        }
      }

      fscanf(t_cool_vs_yp,"%33c",checkstr);
      *(checkstr+33) = '\0';
      if (strcmp(checkstr,"\n Done with Fourier coefficients.") != 0) {
        fprintf(stderr,"Error in t_cool_vs_yp: \n");
        fprintf(stderr,"checkstr = %s \n",checkstr);
        exit(1);
      }
      fclose(t_cool_vs_yp);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(ct,nk*nj*(ni+1),EPIC_MPI_COMPLEX,NODE0,para.comm);
#endif
  }
  /* end of initialization */


  /*
   * Check for out-of-bounds:
   */
  logpress = log(pressure);
  if (logpress > logpbot) {
    logpress = logpbot;
    fprintf(stderr,"Warning: heating: temp_eq: p=%e > pbot=%e \n",
                    pressure/100.,exp(logpbot)/100.);
  }
  else if (logpress < logptop) {
    logpress = logptop;
    fprintf(stderr,"Warning: heating: temp_eq: p=%e < ptop=%e \n",
                    pressure/100.,exp(logptop)/100.);
  }
  lattop   = asin(sin_ytop)/DEG;
  latbot   = asin(sin_ybot)/DEG;
  lat      = MIN(lattop,latitude);
  lat      = MAX(latbot,lat);
  sin_y    = sin(lat*DEG);
  sin_y    = MIN(sin_ytop,sin_y);
  sin_y    = MAX(sin_ybot,sin_y);
  
  /*
   * Determine position in table:
   */
  k      = 1;
  logp_k = logpbot;
  dlogp  = (logptop-logpbot)/(nk-1);
  while (logp_k >= logpress) {
    k++;
    logp_k = logpbot+dlogp*(double)(k-1);
  }
  pp = (logp_k-logpress)/dlogp;

  j       = 1;
  sin_y_j = sin_ybot;
  dsin_y  = (sin_ytop-sin_ybot)/(nj-1);
  while (sin_y_j <= sin_y) {
    j++;
    sin_y_j = sin_ybot+dsin_y*(double)(j-1);
  }
  yy = (sin_y_j-sin_y)/dsin_y;

  /* 
   * k,j now refers to the grid point just north and just above
   * the input position (or equal to it).
   *
   * Calculate Teq for the four corners surrounding the input (lat,p):
   */

  freqtime = freq*time;
  for (kk = 0; kk <= 1; kk++) {
    for (jj = 0; jj <= 1; jj++) {
      if (k-kk > 0 && j-jj > 0) {
        ctemp = cnum(0.,0.);
        for (i = 0; i <= ni; i++) {
          /* Sum over complex Fourier components: */
          carg  = cnum(0.,freqtime*(double)i);
          ccomp = cmult(CT(k-kk,j-jj,i),cexp(carg));
          ctemp = cadd(ctemp,ccomp);
        }
        /* Take real part for answer: */
        temp_equil[kk][jj] = creal(ctemp);
      }
      else {
        temp_equil[kk][jj] = 0.;
      }
    }
  }

  /*
   * The time dependence has now been resolved.
   * Interpolate in (sin lat, log p) to get answer:
   */
  ans = (   pp)*(   yy)*temp_equil[1][1]+
        (   pp)*(1.-yy)*temp_equil[1][0]+
        (1.-pp)*(   yy)*temp_equil[0][1]+
        (1.-pp)*(1.-yy)*temp_equil[0][0];
       
  return ans;
}

/*======================= end of temp_eq() ==================================*/

/*==================== heat_flux_top() ======================================*/

double heat_flux_top(planetspec *planet,
                     int         J,
                     int         I)
{
  double
    flux;

  if (strcmp(planet->name,"triton") == 0) {
    /* Ionospheric heating value. See Yelle, Lunine, and Hunten (1991). */
    flux = -1.15e-6;
  }
  else {
    fprintf(stderr,"Error: heating: heat_flux_top not defined for %s \n",
                   planet->name);
    exit(1);
  }

  return flux;
}

/*==================== end of heat_flux_top() ===============================*/


/*==================== heat_flux_bot() ======================================*/

double heat_flux_bot(planetspec *planet,
                     int         J, 
                     int         I)
{
  double 
    flux;

  /****** Need to put in real physics here ******/
  if (strcmp(planet->name,"triton") == 0) {
    flux = -1.15e-6;
  }
  else {
    fprintf(stderr,"Error: heating: heat_flux_bot not defined for %s \n",
                   planet->name);
    exit(1);
  }

  return flux;
}

/*==================== end of heat_flux_bot() ===============================*/

/*==================== test_temp_eq_ura() ===================================*/

void test_temp_eq_ura(planetspec *planet)
{
  static double 
    time,
    latitude,
    y,
    pressure,
    logp,ptop,pbot,
    eq_temp;
  char
    outfile[100];
  FILE
    *out;

  ptop = .1;
  pbot = 2000.; 
  time = input_double("Input time [fraction of Uranian year]:\n",time);

  sprintf(outfile,"teq_%1d.dat",(int)(time*100.));
  out = fopen(outfile,"w");

  fprintf(out," 21   21  -90. 90. %f %f \n",ptop,pbot);
  fprintf(out," lat[deg] press[mbar]  eq_temp[K] \n");

  for (y = -1.; y <= 1.; y+= 2./20.) {
    latitude = asin(y)/DEG;
    for (logp = log(ptop); logp < log(pbot+.1); logp += (log(pbot)-log(ptop))/20.) {
      pressure = exp(logp);
      eq_temp  = temp_eq(planet,latitude,pressure*100.,
                         time*planet->orbit_period*365.*24.*60.*60.);
      fprintf(out," %5.1f     %6.1f     %5.1f \n",
                  latitude,pressure,eq_temp);
    }
  }
  fclose(out);

  return;
}

/*======================= end of test_temp_eq_ura() =========================*/

/*======================== rain()==================================*/


/* P. Stratman
 * 
 *  Checks for relative humidity > 1.0, if greated than 1, updates specific
 *  humidity value to coincide with a RH = 1.0.
 */
 
void rain(planetspec *planet,
          double     *heat,
          int         K)
{
  int 
     J,I,index,index2,kk;
  double
     mu,mu_i,temperature,fp,pressure,theta, 
     density_1,density_2,sh_1,sh_2,coeff,p_sat;
  static int
     initialized = FALSE;
  static double
     *buffji;
     
  if(initialized == FALSE){
  
   /** allocating memory **/
    buffji = dvector(0,Nelem2d-1);
      
    initialized = TRUE;
  }
  
  kk = 2*K;
  theta = grid.theta[kk];
  
  for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++){
    
    if (var.chem_on[index]==CHEM_PASSIVE){
      
      relative_humidity(planet, index, buffji, K);   
      
      for (J = JLO; J<=JHI; J++){
        
	for (I = ILO; I<= IHI; I++){
	
	   if (VAR(index,K,J,I) > 0.1) {
	      fprintf(stderr,"WARNING: SH > 0.1 before modified by rain() at (K,J,I) = "
	       "(%d,%d,%d) \n",K,J,I);
	   }
   
          if (BUFFJI(J,I) > 1.){ 
            
	    mu_i        = molar_mass(planet,var.chem_name[index]);
            pressure    = get_p(planet, P_INDEX, kk, J, I);
            fp          = get_chem(planet, FPARA_INDEX, kk, J, I);
            temperature = return_temp(planet, fp, pressure, theta);
            mu          = avg_molar_mass(planet, kk, J, I);
            sh_1        = VAR(index,K,J,I);
            p_sat       = return_sat_vapor_p(var.chem_name[index], temperature);
            density_1   = return_dens(planet, fp, pressure, theta, mu, 
	                       PASSING_THETA);
            coeff       = mu_i*p_sat/(R_GAS*temperature);
            density_2   = density_1*(1.-sh_1)+coeff;
            sh_2        = coeff/density_2;
          
            for (index2 = FIRST_HUMIDITY; index2 <= LAST_HUMIDITY; index2++){
	      if(index2 == index){
                VAR(index2,K,J,I) = sh_2;
              }
              else if(var.chem_on[index2]==CHEM_PASSIVE){
                VAR(index2,K,J,I) *= density_1/density_2;
	      }
            }
          } 
        }
      }
    }
  }
  return;
}
/*======================== end of rain()===========================*/

/*========================= external_heating() ====================*/

/*
 * Danie Mao-Chang Liang
 */
void external_heating(planetspec *planet,
                      double     *heat,
		      int         K)
{
/*
  register int
    J,I;
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI/2; I++) {
    }
  }
*/
  /* Need to call B2D here. */
 /* BC2D(&(HEAT(JLO,ILO),NO_INDEX,1);*/
  return;
}

/*========================= end of external_heating() =============*/

/************************ end of epic_heating.c ******************************/
