/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Charles Santori, Feb 1997                                       *
 *                                                                 *
 * This function finds the smallest eigenvalue in the equation:    *
 *                                                                 *
 *     z  d    -z    1    d(phi)               2                   *
 *    e * --( e  * -----  ------)  =  -(lambda)  (phi)             *
 *        dz       L_d^2   dz                  n                   *
 *                                                                 *
 * and expresses the answer in terms of c = f_o / lambda.  As the  *
 * resolution is increased, the values of the smallest eigenvalues *
 * (the ones of interest) should converge to fixed values; the     *
 * new eigenvalues are the largest ones.                           *
 *                                                                 *
 * Gas-giant planets:                                              *
 * The bottom boundary condition is phi = 0, and the top boundary  *
 * condition is d(phi)/dz = 0.                                     *
 *                                                                 *
 * Terrestrial planets:                                            *
 * Not yet implemented.                                            *
 *                                                                 *
 * In the computation below, all quantities are stored in arrays   *
 * with indices that increase by two every layer, even for         *
 * quantities that aren't actually calculated both inside the      *
 * layers and at the boundaries.                                   *
 *                                                                 *
 * Note: Unconventionally for EPIC code, here z increases as the   *
 *       array index increases.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

/* 
 * Function prototypes: 
 */
void hqr(double **a,
         int      n,
         double  *wr,
         double  *wi);
double **dmatrix(long nrl, 
                 long nrh, 
                 long ncl, 
                 long nch);
void free_dmatrix(double **m, 
                  long     nrl, 
                  long     nrh, 
                  long     ncl, 
                  long     nch);
void quicksort(double *mag, 
               int     left, 
               int     right);

/*===================== low_mode() ==================================*/

double low_mode(int J)
{
  int 
    I,K,
    kk,nn,nlen,
    count,elem,
    row,col;
  double 
    press,temperature,density,fpara,
    theta,mu,
    brunt,hscale,f0,rad_def,
    g_inv,d_th1_inv,h_tmp,clow,
    elem1,elem2,elem3,mult,
    *ld,*z,*dz,*fact,*trid,
    *wr,*wi,**a,
    c;
  FILE
    *outfile;

  I    = ILO;
  nlen = grid.nk-3;

  /* Allocate memory: */
  z    = dvector(0,2*nlen+1);
  dz   = dvector(0,2*nlen+1);
  ld   = dvector(0,2*nlen+1);
  fact = dvector(0,2*nlen+1);
  trid = dvector(1,3*nlen-2);
  wr   = dvector(1,nlen);
  wi   = dvector(1,nlen);
  a    = dmatrix(1,nlen,1,nlen);

  /* Empty matrix a: */
  for (row = 1; row <= nlen; row++) {
    for (col = 1; col <= nlen; col++) {
      a[row][col] = 0.;
    }
  }

  /* Set f0, g_inv: */
  f0    = (grid.f)[2*J+1];
  g_inv = 1./planet->g;


  /* Calculate z: */
  kk = 2*grid.nk-1;
  for (nn = 0; nn <= 2*nlen+1; nn++) {
    z[nn] = 0.;
    press = get_p(planet,P_INDEX,kk,J,I);
    z[nn] = -log(press/grid.press0);
    if (nn >= 2) {
      dz[nn-1] = z[nn]-z[nn-2];
    }
    kk--;
  }

  /* Calculate L_d: */
  K = grid.nk-1;
  for (nn = 1; nn <= 2*nlen-1; nn+=2) {
    ld[nn] = 0.;
    fpara       = 0.25;
    press       = get_p(planet,P_INDEX,2*K,J,I);
    theta       = grid.theta[2*K];
    temperature = return_temp(planet,fpara,press,theta);
    mu          = avg_molar_mass(planet,2*K,J,I);
    density     = return_dens(planet,fpara,press,temperature,mu,PASSING_T);
    d_th1_inv   = g_inv/(grid.theta[2*K-1]-grid.theta[2*K+1]);
    h_tmp       = (P(K,J,I)-P(K-1,J,I))*d_th1_inv;
    brunt       = sqrt(get_brunt2(planet,2*K,J,I));
    hscale      = (planet->rgas)*temperature*g_inv;
    ld[nn]      = brunt*hscale/f0;
    K--;
  } 

  /* Calculate e^-z/(L_d^2*dz): */
  for (nn = 1; nn <= 2*nlen-1; nn+=2) {
    fact[nn] = exp(-z[nn])/(ld[nn]*ld[nn]*dz[nn]);
  }

  /* Form matrix: */

  count = 1;
  mult  = exp(z[2*1])/dz[2*1];
  elem2 = -(fact[1]+fact[3])*mult;
  elem3 = fact[3]*mult;

  a[1][1]       = elem2;
  a[2][1]       = elem3;
  trid[count++] = elem2;
  trid[count++] = elem3;
  
  for (row = 2; row <= nlen-1; row++) {
    mult  = exp(z[2*row])/dz[2*row];
    elem1 = fact[2*row-1]*mult;
    elem2 = -(fact[2*row-1]+fact[2*row+1])*mult;
    elem3 = fact[2*row+1]*mult;

    a[row-1][row] = elem1;
    a[row  ][row] = elem2;
    a[row+1][row] = elem3;
    trid[count++] = elem1;
    trid[count++] = elem2;
    trid[count++] = elem3;
  }

  mult  =  exp(z[2*nlen])/dz[2*nlen];
  elem1 =  fact[2*nlen-1]*mult;
  elem2 = -fact[2*nlen-1]*mult;

  a[nlen-1][nlen] = elem1;
  a[nlen  ][nlen] = elem2;
  trid[count++]   = elem1;
  trid[count++]   = elem2;

  /* Find the eigenvalues: */
  hqr(a,nlen,wr,wi);

  /* Sort in increasing order: */
  quicksort(wr,1,nlen);

  /* Write eigenvalues to a file: */
  outfile = fopen(EPIC_PATH"/tmp/eigenvalues.dat","w");
  fprintf(outfile," System: %s, J = %d, lat = %4.2f, f0 = %9.3e \n",
                    planet->name,J,grid.lat[2*J+1],fabs(f0));
  fprintf(outfile,"  n   c[m/s]   L_d[km]\n");
  for (row = 1; row <= nlen; row++) {
    if (wr[row] != 0.) {
      rad_def = 1./sqrt(-wr[row]);
      c       = fabs(f0)*rad_def;
    }
    else{
      c = -1.;
    }
    fprintf(outfile," %2d %8.2f %8.1f\n",
                    row,c,rad_def*1.e-3);
    /* Save gravest eigenvalue: */
    if (row == 1) {
      clow = c;
    }
  }

  fclose(outfile);

  /* Free allocated memory: */
  free_dvector(z,   0,2*nlen+1);
  free_dvector(dz,  0,2*nlen+1);
  free_dvector(ld,  0,2*nlen+1);
  free_dvector(fact,0,2*nlen+1);
  free_dvector(trid,1,3*nlen-2);
  free_dvector(wr,  1,nlen);
  free_dvector(wi,  1,nlen);
  free_dmatrix(a,1,nlen,1,nlen);

  return clow;
}

/*===================== end of low_mode() ===========================*/

/*===================== hqr() =======================================*/

/* 
 * From Numerical Recipes. 
 * Finds eigenvalues of a real Hessenberg matrix. 
 */

#define HQRSIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))

void hqr(double **a, int n, double *wr, double *wi)
{
  int nn,m,l,k,j,its,i,mmin;
  double z,y,x,w,v,u,t,s,r,q,p,anorm;
  
  anorm=fabs(a[1][1]);
  for (i=2;i<=n;i++)
    for (j=(i-1);j<=n;j++)
      anorm += fabs(a[i][j]);
  nn=n;
  t=0.0;
  while (nn >= 1) {
    its=0;
    do {
      for (l=nn;l>=2;l--) {
	s=fabs(a[l-1][l-1])+fabs(a[l][l]);
	if (s == 0.0) s=anorm;
	if ((fabs(a[l][l-1]) + s) == s) break;
      }
      x=a[nn][nn];
      if (l == nn) {
	wr[nn]=x+t;
	wi[nn--]=0.0;
      } else {
	y=a[nn-1][nn-1];
	w=a[nn][nn-1]*a[nn-1][nn];
	if (l == (nn-1)) {
	  p=0.5*(y-x);
	  q=p*p+w;
	  z=sqrt(fabs(q));
	  x += t;
	  if (q >= 0.0) {
	    z=p+HQRSIGN(z,p);
	    wr[nn-1]=wr[nn]=x+z;
	    if (z) wr[nn]=x-w/z;
	    wi[nn-1]=wi[nn]=0.0;
	  } else {
	    wr[nn-1]=wr[nn]=x+p;
	    wi[nn-1]= -(wi[nn]=z);
	  }
	  nn -= 2;
	} else {
	  if (its == 30){
	    fprintf(stderr, "Warning: hqr(): too many iterations\n");
	    return;
	  }
	  if (its == 10 || its == 20) {
	    t += x;
	    for (i=1;i<=nn;i++) a[i][i] -= x;
	    s=fabs(a[nn][nn-1])+fabs(a[nn-1][nn-2]);
	    y=x=0.75*s;
	    w = -0.4375*s*s;
	  }
	  ++its;
	  for (m=(nn-2);m>=l;m--) {
	    z=a[m][m];
	    r=x-z;
	    s=y-z;
	    p=(r*s-w)/a[m+1][m]+a[m][m+1];
	    q=a[m+1][m+1]-z-r-s;
	    r=a[m+2][m+1];
	    s=fabs(p)+fabs(q)+fabs(r);
	    p /= s;
	    q /= s;
	    r /= s;
	    if (m == l) break;
	    u=fabs(a[m][m-1])*(fabs(q)+fabs(r));
	    v=fabs(p)*(fabs(a[m-1][m-1])+fabs(z)+fabs(a[m+1][m+1]));
	    if ((u+v) == v) break;
	  }
	  for (i=m+2;i<=nn;i++) {
	    a[i][i-2]=0.0;
	    if (i != (m+2)) a[i][i-3]=0.0;
	  }
	  for (k=m;k<=nn-1;k++) {
	    if (k != m) {
	      p=a[k][k-1];
	      q=a[k+1][k-1];
	      r=0.0;
	      if (k != (nn-1)) r=a[k+2][k-1];
	      if ((x=fabs(p)+fabs(q)+fabs(r)) != 0.0) {
		p /= x;
		q /= x;
		r /= x;
	      }
	    }
	    if ((s=HQRSIGN(sqrt(p*p+q*q+r*r),p)) != 0.0) {
	      if (k == m) {
		if (l != m)
		  a[k][k-1] = -a[k][k-1];
	      } else
		a[k][k-1] = -s*x;
	      p += s;
	      x=p/s;
	      y=q/s;
	      z=r/s;
	      q /= p;
	      r /= p;
	      for (j=k;j<=nn;j++) {
		p=a[k][j]+q*a[k+1][j];
		if (k != (nn-1)) {
		  p += r*a[k+2][j];
		  a[k+2][j] -= p*z;
		}
		a[k+1][j] -= p*y;
		a[k][j] -= p*x;
	      }
	      mmin = nn<k+3 ? nn : k+3;
	      for (i=l;i<=mmin;i++) {
		p=x*a[i][k]+y*a[i][k+1];
		if (k != (nn-1)) {
		  p += z*a[i][k+2];
		  a[i][k+2] -= p*r;
		}
		a[i][k+1] -= p*q;
		a[i][k] -= p;
	      }
	    }
	  }
	}
      }
    } while (l < nn-1);
  }
}

/*===================== end of hqr() ================================*/

/*===================== dmatrix() ===================================*/

#define NR_END 1
#define FREE_ARG char*
/* 
 * These matrix functions are from Numerical Recipes. 
 */

double **dmatrix(long nrl,long nrh,long ncl,long nch)
/* 
 * Allocate a double matrix with subscript range m[nrl..nrh][ncl..nch]. 
 */
{
  long 
    i, 
    nrow=nrh-nrl+1,
    ncol=nch-ncl+1;
  double 
    **m;

  /* Allocate pointers to rows: */
  m=(double **) malloc((size_t)((nrow+NR_END)*sizeof(double*)));
  if (!m){
    fprintf(stderr, "Allocation failure 1 in matrix()");
    return;
  }
  m += NR_END;
  m -= nrl;

  /* Allocate rows and set pointers to them */
  m[nrl]=(double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(double)));
  if (!m[nrl]){
    fprintf(stderr, "allocation failure 2 in matrix()");
    return;
  }
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for (i = nrl+1; i <= nrh; i++) {
    m[i]=m[i-1]+ncol;
  }

  /* Return pointer to array of pointers to rows: */
  return m;
}

/*===================== end of dmatrix() ============================*/

/*===================== free_dmatrix() ==============================*/

void free_dmatrix(double **m, long nrl, long nrh, long ncl, long nch)
/* free a double matrix allocated by dmatrix() */
{
  free((FREE_ARG) (m[nrl]+ncl-NR_END));
  free((FREE_ARG) (m+nrl-NR_END));
}

/*===================== end of free_dmatrix() =======================*/

/*===================== quicksort() =================================*/
/* 
 * From K & R 2nd ed., p. 87: 
 */
void quicksort(double *mag, 
               int     left, 
               int     right)
{
  int 
    i,last;
  void 
    swap(double *mag,int i,int j);

  if (left >= right) {
    return;
  }

  swap(mag,left,(left+right)/2);
  last = left;
  for (i = left+1; i <= right; i++) {
    if (mag[i] > mag[left]) {
      swap(mag, ++last, i);
    }
  }
  swap(mag,left,last);

  quicksort(mag,left,  last-1);
  quicksort(mag,last+1,right);

  return;
}

/*===================== end of quicksort() ==========================*/

/*===================== swap() ======================================*/

void swap(double *mag,int i,int j)
{
  double 
    temp;

  temp   = mag[i];
  mag[i] = mag[j];
  mag[j] = temp;

  return;
}

/*===================== end of swap() ===============================*/

/* * * * * * * * * * * * end of epic_init_eigen.c  * * * * * * * * * */
