/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * epic_init_funcs.c * * * * * * * * * * * * * * * * * * * * * 
 *                                                                           *
 *       T. Dowling, C. Santori                                              *
 *                                                                           *
 *       This file includes the following functions:                         *
 *                                                                           *
 *           set_p_balanced()                                                *
 *           set_u_balanced()                                                *
 *           set_uamp()                                                      *
 *           set_u_spinup()                                                  *
 *           init_with_t()                                                   *
 *           init_with_u()                                                   *
 *           init_with_p_avg()                                               *
 *           init_with_qm()                                                  *
 *           init_fpara_as_fpe()                                             *
 *           init_v_by_continuity()                                          *
 *           init_viscosity()                                                *
 *           init_humidity()                                                 *
 *           t_yp(),fp_yp()                                                  *
 *           mont_from_q()                                                   *
 *           mont_sor(),mont_fill()                                          *
 *           mq_init(),mqerr()                                               *
 *           outm(),outp(),outh(),outu()                                     *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

typedef struct {
  char
    npolar;
  int
    maxit,
    checkit,
    kbot_in,
    nuzero,
    ztyp,
    ptyp,
    j_eq,
    j_pol,
    j_eq2,
    j_inc,
    j_sorlo,
    j_sorhi;  
  double
    c_q,
    pfact,
    rrmanual,
    x,
    utop;
} mqspec;

/*
 * Function prototypes:
 */
int mont_from_q(double *mont3d, 
                mqspec  params);
int mont_sor(double *mont3d,
	     double *q0,
             double *p_shrunk,
             mqspec params);
int mont_fill(double *mont3d,
	      double *q0,
	      double *p_shrunk,
	      mqspec params);
int mq_init(double *dlt, double *dlti, double *dltlti, double *cci,
	    double *kapi, double *dc, double **av, double **bv,
	    double **dth, double **dthi, double **dththi, mqspec params);
int mqerr(int J, int J0, int K, double dltlti, double cci, double kapi,
	  double dc, double *av, double *bv, double *dthi, double *mont3d,
	  double *q0, mqspec params);

void outm(int j0,int j1,int j12,int k0,int k1,double *mont3d);
void outp(int j0,int j1,int j12,int k0,int k1);
void outh(int j0,int j1,int j12,int k0,int k1);
void outu(int j0,int j1,int j12,int k0,int k1);


/*====================== set_p_balanced() ====================================*/

#define MAXIT_SET_P_BALANCED 60

void set_p_balanced(planetspec *planet,
                    double     *mont3d) 
{
  int
    K,J,I,
    kk,it;
  double
    exner,
    temperature,
    old_temp,
    fp,
    theta,
    d_th0_inv,
    dmontdth,
    cpr_inv,
    fgibb,
    fpe,uoup;

  cpr_inv = 1./planet->cp;

  for (K = 1; K < grid.nk; K++) {
    kk    = 2*K+1;
    theta = grid.theta[kk];
    d_th0_inv = 1./(grid.theta[2*K]-grid.theta[2*K+2]);
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        fp          = get_chem(planet,FPARA_INDEX,kk,J,I);
        dmontdth    = (MONT3D(K,J,I)-MONT3D(K+1,J,I))*d_th0_inv;
        /*
         * Sanity check:
         */
        if (dmontdth <= 0. && d_th0_inv > 0.) {
          fprintf(stderr,"\nError: set_p_balanced(): MONT(%d,%d,%d)=%e > MONT(%d,%d,%d)=%e\n",
                          K+1,J,I,MONT3D(K+1,J,I),K,J,I,MONT3D(K,J,I));
          exit(1);
        }

        exner       = dmontdth;
        temperature = exner*theta*cpr_inv;
        if (var.chem_on[FPARA_INDEX]) {
          /* 
           * Add fpara term. Iterate to get temperature. 
           */
          for (it = 0; it < MAXIT_SET_P_BALANCED; it++) {
            old_temp = temperature;
            return_enthalpy(planet,fp,P(K,J,I),old_temp,&fgibb,&fpe,&uoup);
            exner       = dmontdth+fgibb*(FPARA(K,J,I)-FPARA(K+1,J,I))*d_th0_inv;
            temperature = exner*theta*cpr_inv;
            if (fabs(old_temp-temperature) < 1.e-9*temperature) {
              break;
            }
          }
          if (it >= MAXIT_SET_P_BALANCED) {
            if (IAMNODE == NODE0) {
              fprintf(stderr,"Warning: set_p_balanced(): exceeded MAXIT = %d ",
                             MAXIT_SET_P_BALANCED);
            }
          }
        }
        P(K,J,I) = return_press(planet,fp,temperature,theta);
      }
    }
    BC2D(&(P(KLO,JLO,ILO)),P_INDEX,K);
  }

  /* 
   * Bottom layer for terrestrial planets: 
   */
  if (strcmp(planet->class,"terrestrial") == 0) {
    fp    = .25;
    K     = grid.nk;
    kk    = 2*K+1;
    theta = grid.theta[kk];
    d_th0_inv = 1./grid.theta[2*K];
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        exner       = (MONT3D(K,J,I)-SURFACE_GZ(J,I))*d_th0_inv;
        temperature = exner*theta*cpr_inv;
        P(K,J,I)    = return_press(planet,fp,temperature,theta);
      }
    }
    BC2D(&(P(KLO,JLO,ILO)),P_INDEX,K);
  }

  return;
}

/*====================== end of set_p_balanced() =============================*/

/*====================== set_u_balanced() ====================================*/

/* 
 * Initialize u using the balance formula:
 * u*u*sin(lat)/r+f*u+(d/dy Mont+fgibb d/dy fpara) = 0.
 */

#define UDY_ON_V(j,i) udy_on_v[i+(j)*Iadim-Shift2d]

void set_u_balanced(double *mont,
                    double *ffgibb,
                    int     K) 
{
  int
    J,I,
    need_eq;
  double
    *udy_on_v,  
    n_2j,rln,
    cc,tmp;

  /*
   * Calculate geostrophic u/n's on v grid, udy_on_v.
   * Allocate memory: 
   */
  udy_on_v = dvector(0,Nelem2d-1);

  need_eq = 0;

  for (J = JFIRST; J <= JHI; J++) {
    n_2j = grid.n[2*J];
    rln  = 1./(grid.m[2*J]*grid.dln*DEG);
    for (I = ILO; I <= IHI; I++) {
      if (fabs(grid.lat[2*J]) >= grid.dlt) {
        cc = (MONT(J,I)-MONT(J-1,I));
        if (var.chem_on[FPARA_INDEX] && ffgibb != NULL) {
          cc += .5*(FFGIBB(J,I)+FFGIBB(J-1,I))*
                   (FPARA(K,J,I)-FPARA(K,J-1,I));
        }
        cc *= n_2j/(rln*planet->omega*.5*grid.f[2*J]);
        UDY_ON_V(J,I) = planet->omega*rln*(sqrt(1.-cc)-1.)/grid.n[2*J];
      }
      else {
        need_eq = 1;
      }
    }
  }

  if (grid.wrap[1] == 0) {
    if (JLO == grid.jlo) {
      /* Southern edge */
      for (I = ILO; I <= IHI; I++) {
        /* linear extrapolation of u/n */
        tmp = (UDY_ON_V(2,I)-UDY_ON_V(1,I))/
              (grid.lat[2*2]-grid.lat[2*1]);
        UDY_ON_V(0,I) = UDY_ON_V(1,I)+tmp*
                 (grid.lat[2*0]-grid.lat[2*1]);
      }
    }
    if (JHI == grid.nj) {
      /* Northern edge */
      for (I = ILO; I <= IHI; I++) {
        /* linear extrapolation */
        tmp = (UDY_ON_V(grid.nj,I)-UDY_ON_V(grid.nj-1,I))/
              (grid.lat[2*grid.nj]-grid.lat[2*(grid.nj-1)]);
        UDY_ON_V(grid.nj+1,I) = UDY_ON_V(grid.nj,I)+tmp*
                 (grid.lat[2*(grid.nj+1)]-grid.lat[2*grid.nj]);
      }
    }
  }
  BC2D(&(UDY_ON_V(JLO,ILO)),NO_INDEX,1);

  if (need_eq == 1) {
    int
      ismooth;
    double
      *tmpu;

    /* Set equatorial udy_on_v assuming a parabolic u/n: */
    for (J = JFIRST; J <= JHI; J++) {
      if (fabs(grid.lat[2*J]) < grid.dlt) {
        if (grid.lat[2*J] >= 0.) {
          if (J+2 > JHI+JPAD) {
            fprintf(stderr,"error: J+2 > JHI+JPAD in epic_initial \n");
            exit(1);
          }
          for (I = ILO; I <= IHI; I++) {
            UDY_ON_V(J,I) = (4.*UDY_ON_V(J+1,I)-UDY_ON_V(J+2,I))/3.;
          }
        }
        else {
          if (J-2 < JLO-JPAD) {
            fprintf(stderr,"error: J-2 < JLO-JPAD in epic_initial \n");
            exit(1);
          }
          for (I = ILO; I <= IHI; I++) {
            UDY_ON_V(J,I) = (4.*UDY_ON_V(J-1,I)-UDY_ON_V(J-2,I))/3.;
          }
        }
      }
    }
    /* Smooth u around equator */
    /* Allocate memory for tmpu */
    tmpu = dvector(JFIRST,JHI);
    for (I = ILO; I <= IHI; I++) {
      for (ismooth = 1; ismooth <= 1; ismooth++) {
        for (J = JFIRST; J <= JHI; J++) {
          tmpu[J] = UDY_ON_V(J,I);
        }
        for (J = JFIRST; J <= JHI; J++) {
          if (fabs(grid.lat[2*J]) <= 2.*grid.dlt) {
            /* Savitsky-Golay weighting */
            if (J-5 < JFIRST || J+5 > JHI) {
              fprintf(stderr,"initial: problem with Savitsky-Golay weighting \n");
              exit(1);
            }
            UDY_ON_V(J,I) = -0.084*(tmpu[J-5]+tmpu[J+5])
                            +0.021*(tmpu[J-4]+tmpu[J+4])
                            +0.103*(tmpu[J-3]+tmpu[J+3])
                            +0.161*(tmpu[J-2]+tmpu[J+2])
                            +0.196*(tmpu[J-1]+tmpu[J+1])
                            +0.207*(tmpu[J]);
          }
        }
      }
    }
    BC2D(&(UDY_ON_V(JLO,ILO)),NO_INDEX,1);
    free_dvector(tmpu,JFIRST,JHI);
  }

  /* Average udy_on_v onto u grid: */
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      U(K,J,I) = .25*((UDY_ON_V(J  ,I  )+
                       UDY_ON_V(J,  I-1))+
                      (UDY_ON_V(J+1,I  )+
                       UDY_ON_V(J+1,I-1)))*grid.n[2*J+1];
    }
  }
  BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);
  /* Free allocated memory */
  free_dvector(udy_on_v,0,Nelem2d-1);

  return;
}

/*====================== end set_u_balanced() ================================*/

/*====================== set_uamp() ==========================================*/

void set_uamp(int     uz_type,
              double *uamp,
              double (*uzonal)(double))
{
  int
    K,J;
  double
    pressure,
    tmp,
    aux1,
    c_wind,
    *factor;

  /*
   * Allocate memory:
   */
  factor = dvector(KLO,KHI);

  if (uz_type == UZ_GALILEO || uz_type == (UZ_GALILEO+10)) {
    /* Galileo Probe Doppler wind profile: */
    /* Normalize to 670 mbar: */
    pressure = 670.*100.;
    tmp      = 1./galileo_u(pressure);
    for (K = KLO; K < KHI; K++) {
      /* 
       * P(K,J,I) is not set yet. Set P(K,JLO,ILO) equal to grid.p_avg[K]
       * so that get_p() will work here.
       */
      P(K,JLO,ILO) = grid.p_avg[K];
    }
    for (K = KLO; K <= KHI; K++) {
      if (K == KHI && strcmp(planet->class,"gas-giant") == 0) {
        /* 
         * Abyssal layer. Use atmosphere-interior interface pressure: 
         */
        pressure = grid.p_avg[grid.nk-1];
      }
      else {
        pressure = get_p(planet,P_INDEX,2*K,JLO,ILO);
      }
      for (J = JLO; J <= JHI; J++) {
        UAMP(K,J) = galileo_u(pressure)*tmp;
      }
    }
  }
  else if (uz_type == UZ_CZS || uz_type == (UZ_CZS+10)) {

     for (K = KLO; K <= KHI; K++) {
       P(K,JLO,ILO) = grid.p_avg[K];
     }

     fprintf(stderr,"\n");

     for (K = KLO; K <= KHI; K++) {
       if (K == KHI && strcmp(planet->class,"gas-giant") == 0) {
         pressure = grid.p_avg[grid.nk-1];
       }
       else {
         pressure = get_p(planet,P_INDEX,2*K,JLO,ILO);
       }
       pressure *= 0.01;

       if (pressure >= 680.) {
         /*
          * Change in amplitude and therefore vertical shear, but
          * not change in zonal shear
          */
          aux1 = (5./2.7)*log(pressure/680.);
          for (J = JLO; J <= JHI; J++) {
             UAMP(K,J) = 1.0+aux1/uzonal((grid.lat)[2*J+1]);
          }
       }
       else {
         /*
          *above the clouds decay according to Gierasch criteria
          */
         aux1 = 1. + (1./2.4) * log(pressure/680.);
         if (aux1 <= 0.0) {
           aux1 = 0.0;
         }
         for (J = JLO; J <= JHI; J++) {
            UAMP(K,J) = aux1;
         }
       }
    }
  }

  else if (uz_type == UZ_GIERASCH || uz_type == (UZ_GIERASCH+10)) {
    /* 
     * Wind vertical structure according to Gierasch paper
     * Icarus 67,456-483 (1986)
     */
     c_wind = 1000.;
     c_wind = input_float("Value of C_wind? [1000. = inf]\n",c_wind);

     for (K = KLO; K <= KHI; K++) {
       P(K,JLO,ILO) = grid.p_avg[K];
     }

     fprintf(stderr,"\n");

     for (K = KLO; K <= KHI; K++) {
       if (K == KHI && strcmp(planet->class,"gas-giant") == 0) {
         pressure = grid.p_avg[grid.nk-1];
       }
       else {
         pressure = get_p(planet,P_INDEX,2*K,JLO,ILO);
       }
       pressure *= 0.01; 
       
       for (J = JLO; J <= JHI; J++){
       if (pressure >= 680.) {
          if (c_wind < 1000. && uzonal((grid.lat)[2*J+1]) > 0.0){
             aux1 = 1. + (1./c_wind) * log(pressure/680.);
          }
          else {
             aux1 = 1;
          }
       }
       else {
         aux1 = 1. + (1./2.4) * log(pressure/680.);
         if (aux1 <= 0.0) {
           aux1 = 0.0;
         }
       }
       factor[K] = aux1;
       UAMP(K,J) = aux1;
       }
      fprintf(stderr,"Layer[%d]: P -> %f ...uamp -> %f \n",K,pressure,factor[K]);
    }
  } 
  else if (uz_type == UZ_CONSTANT || uz_type == (UZ_CONSTANT+10)) {
    /* Winds constant with height: */
    for (K = KLO; K <= KHI; K++) {
      for (J = JLO; J <= JHI; J++) {
        UAMP(K,J) = 1.0;
      }
    }
  }
  else if (uz_type == UZ_LINEAR || uz_type == (UZ_LINEAR+10)) {
    /* Winds decreasing with height linearly with respect to k: */
    for (K = KLO; K <= KHI; K++) {
      for (J = JLO; J <= JHI; J++) {
        UAMP(K,J) = (double)(K-1)/(grid.nk-1);
      }
    }
  }
  else {
    for (K = KLO; K <= KHI; K++) {
      if (IAMNODE == 0) {
        fprintf(stderr,"Layer %d.   ", K);
      }
      UAMP(K,JLO) = input_double("Input uamp: ",1.0);
      for (J = JLO+1; J <= JHI; J++) {
        UAMP(K,J) = UAMP(K,JLO);
      }
    }
  }
  
  return;
}

/*====================== end of set_uamp() ===================================*/

/*====================== set_u_spinup() ======================================*/

void set_u_spinup(int     uz_type,
                  double  *uamp,
                  double (*uzonal)(double))
{
  int
    K,J,I;
  double
    tmp;

  if (uamp && uzonal) {
    if (uz_type >= 10) {
      /* Spinup to original U for sponge and k > KLAST_ACTIVE */
      for (K = KLO; K <= KHI; K++) {
        if (K <= grid.k_sponge || K > KLAST_ACTIVE) {
          for (J = JLO; J <= JHI; J++) {
            for (I = ILO; I <= IHI; I++) {
              U_SPINUP(K,J,I) = UAMP(K,J)*uzonal((grid.lat)[2*J+1]);
            }   
          }
        }
      }
      /* Spinup to specified constant in main layers (often 0.) */
      tmp = input_double("Input constant u_spinup: \n",0.);
      for (K = grid.k_sponge+1; K <= KLAST_ACTIVE; K++) {
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            U_SPINUP(K,J,I) = tmp;
          }
        }
      }
    }
    else {
      for (K = KLO; K <= KHI; K++) {
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            U_SPINUP(K,J,I) = UAMP(K,J)*uzonal((grid.lat)[2*J+1]);
          }   
        }
      }
    }
  }
  else {
    for (K = KLO; K <= grid.k_sponge; K++) {
      if (K <= grid.k_sponge || K > KLAST_ACTIVE) {
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            U_SPINUP(K,J,I) = U(K,J,I);
          }
        }
      }
    }
    if (uz_type < 10) {
      /* Main active layers */
      for (K = grid.k_sponge+1; K <= KLAST_ACTIVE; K++) {
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            U_SPINUP(K,J,I) = U(K,J,I);
          }
        }
      }
    }
    /* Case uz_type >= 10 assumed to already have constant U_SPINUP set. */
  }

  /* 
   * NOTE: BC2D has not been applied to U_SPINUP.
   */

  return;
}

/*====================== end of set_u_spinup() ===============================*/

/*====================== init_with_t() =======================================*/

/* 
 *  Initialize with T(lat,p) data:
 *  1. Calculate mont in bottom layer, k = nk.
 *  2. Find pressure given temperature,theta,fpara (requires iteration).
 *  3. Use hydrostatic balance to get mont in next layer up.
 *  4. Use gradient balance to get u from mont.
 */

void init_with_t(planetspec *planet,
                 double      ntp,
                 double     *pdat,
                 double     *tdat,
                 double     *t_y2,
                 double     *uamp,
                 double     (*uzonal)(double))
{
  int
    K,J,I,
    kk,mode;
  double
    dp,pl,p1,p2,prt,swap,latr,
    fgibb,fpe,uoup,temperature,
    theta_ortho,theta_para,exner,
    theta_errorl,theta_error,
    d_th0,theta_target,
    fp_guess,t_guess,theta_guess,
    neg_log_p,p_d;
  double
    *mont,
    *ffgibb;

  mode = input_int("T,fp from data or synthetic [0 or 1]: \n", 0);

  /*
   * Allocate memory:
   */
  mont   = dvector(0,Nelem2d-1);
  ffgibb = dvector(0,Nelem2d-1);
  /*
   * Calculate mont for k = nk.
   */
  if (strcmp(planet->class,"gas-giant") == 0) {
    /* Use gradient balance with u_nk. */
    K = grid.nk;
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        U(K,J,I) = UAMP(K,J)*uzonal((grid.lat)[2*J+1]);
      }
    }
    BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);
  }
  mont_nk(planet,mont);

  /* 
   * We are given T(p,lat) and fp(p,lat),  
   * and need to iterate to find p(theta,lat). 
   * Use secant method (Numerical Recipes in C, p. 357).
   *
   * Form the Montgomery streamfunction and use it to geostrophically 
   * initialize u for all the other layers.
   */
  for (K = grid.nk-1; K >= 1; K--) {
    if (K > 1) {
      d_th0 = grid.theta[2*K  ]-grid.theta[2*K+2];
    }
    else {
      /* special delta-theta in top layer: */
      d_th0 = grid.theta[2*K  ]-grid.theta[2*K+2];
    }
    theta_target = grid.theta[2*K+1];
    if (mode == 1) {
      /* synthetic t_yp needs t_p values */
      neg_log_p   = -log(grid.p_avg[K]);
      kk          = find_place_in_table(ntp,pdat,&neg_log_p,&p_d);
      temperature = splint(neg_log_p,pdat+kk,tdat+kk,t_y2+kk,p_d);
    }
    for (J = JLO; J <= JHI; J++) {
      latr = grid.lat[2*J+1]*DEG;
      /* Pick bounds on p:  */
      p1   = grid.p_avg[K]*.9;
      p2   = grid.p_avg[K]*1.1;
      for (I = ILO; I <= IHI; I++) {
        /* Calculate theta(p) */
        if (mode == 0) {
          t_guess = t_yp(p1,latr,mode,NULL);
        }
        else {
          /* synthetic t_yp returns variation only */
          t_guess = temperature+t_yp(p1,latr,mode,NULL);
        }
        fp_guess     = fp_yp(p1,latr,mode);
        theta_guess  = return_theta(planet,fp_guess,p1,t_guess,&theta_ortho,&theta_para);
        theta_errorl = theta_guess-theta_target;
        /* Calculate theta(p) */
        if (mode == 0) {
          t_guess = t_yp(p2,latr,mode,NULL);
        }
        else {
          t_guess = temperature+t_yp(p2,latr,mode,NULL);
        }
        fp_guess     = fp_yp(p2,latr,mode);
        theta_guess  = return_theta(planet,fp_guess,p2,t_guess,&theta_ortho,&theta_para);
        theta_error  = theta_guess-theta_target;
        if (fabs(theta_errorl) < fabs(theta_error)) {
          prt = p1;
          pl  = p2;
          swap         = theta_errorl;
          theta_errorl = theta_error;
          theta_error  = swap;
        }
        else {
          pl  = p1;
          prt = p2;
        }
        dp = DBL_MAX;
        while (fabs(dp) > 1.) {
          dp = (pl-prt)*theta_error/(theta_error-theta_errorl);
          pl = prt;
          theta_errorl = theta_error;
          prt += dp;
          /* Calculate theta(p) */
          if (mode == 0) {
            t_guess = t_yp(prt,latr,mode,NULL);
          }
          else {
            t_guess = temperature+t_yp(prt,latr,mode,NULL);
          }
          fp_guess    = fp_yp(prt,latr,mode);
          theta_guess = return_theta(planet,fp_guess,prt,t_guess,&theta_ortho,&theta_para);
          theta_error = theta_guess-theta_target;
        }
        if (var.chem_on[FPARA_INDEX]) {
          FPARA(K,J,I) = fp_guess;
          if (K == grid.nk-1) {
            /* set fpara in bottom layer to value in layer above */
            FPARA(K+1,J,I) = fp_guess;
          }
        }
        P(K,J,I)   = prt;
        exner      = planet->cp*t_guess/theta_target;
        MONT(J,I) += exner*d_th0;
        return_enthalpy(planet,fp_guess,P(K,J,I),t_guess,&fgibb,&fpe,&uoup);
        /* store fgibb for later use */
        FFGIBB(J,I) = fgibb;
        if (var.chem_on[FPARA_INDEX]) {
          MONT(J,I)  -= fgibb*(FPARA(K,J,I)-FPARA(K+1,J,I));
        }
      }
    }
    BC2D(&(P(KLO,JLO,ILO)),P_INDEX,K);
    if (var.chem_on[FPARA_INDEX]) {
      BC2D(&(FPARA(KLO,JLO,ILO)),FPARA_INDEX,K);
      if (K == grid.nk-1) {
        BC2D(&(FPARA(KLO,JLO,ILO)),FPARA_INDEX,K+1);
      }
    }
    BC2D(&(MONT(  JLO,ILO)),NO_INDEX,1);
    BC2D(&(FFGIBB(JLO,ILO)),NO_INDEX,1);

    /* 
     * Initialize u using the balance formula:
     * u*u*sin(lat)/r+f*u+(d/dy Mont+fgibb d/dy fpara) = 0.
     */
    set_u_balanced(mont,ffgibb,K);
  }
  /* Free allocated memory: */
  free_dvector(mont,  0,Nelem2d-1);
  free_dvector(ffgibb,0,Nelem2d-1);

  return;
}

/*====================== end of init_with_t() ================================*/

/*====================== init_with_u() =======================================*/

/* 
 *  1. Specify u(y,theta).
 *  2. Calculate mont from u using geostrophic balance.
 *  3. Calculate p from mont using hydrostatic balance.
 *
 * For simplicity, use the standard hydrostatic and gradient 
 * relations without the extra terms from hydrogen (ortho-para) 
 * thermodynamics.
 */

void init_with_u(planetspec *planet,
                 double     *uamp,
                 double     (*uzonal)(double))
{
  int
    K,J,I,
    kk,
    jj,itmp;
  double
    exner,d_th0,
    temperature,fpara,pressure,theta,
    mont_max,
    tmp,tmp2;
  double
    *mont3d,
    *mont_k,
    *sendbuf;

  for (K = 1; K <= grid.nk; K++) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        U(K,J,I) = UAMP(K,J)*uzonal((grid.lat)[2*J+1]);
      }
    }
    BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);
  }

  /*
   * Allocate memory:
   */
  mont3d  = dvector(0,Nelem3d-1);
  mont_k  = dvector(1,grid.nk);
  sendbuf = dvector(1,grid.nk);
  /*
   * Calculate mont3d from the zonal-wind profile assuming gradient balance:
   *
   * First, store horizontal variation:
   */
  for (K = 1; K <= grid.nk; K++) {
    mont_geostrophic(planet,mont3d+(K-Kshift)*Nelem2d,grid.mont0,K);
    /* Determine minimum mont for layer */
    mont_k[K] = DBL_MAX;
    I         = ILO;
    for (J = JLO; J <= JHI; J++) {
      mont_k[K] = MIN(mont_k[K],MONT3D(K,J,I));
    } 
  }
#if defined (EPIC_MPI)
  memcpy(&sendbuf[1],&mont_k[1],grid.nk*sizeof(double));
  MPI_Allreduce(&sendbuf[1],&mont_k[1],grid.nk,MPI_DOUBLE,MPI_MIN,para.comm);
#endif
  /* Find j index of maximum mont */
  mont_max = -DBL_MAX;
  I = ILO;
  for (K = 1; K <= grid.nk; K++) {
    for (J = JLO; J <= JHI; J++) {
      tmp = fabs(MONT3D(K,J,I)-mont_k[K]);
      if (tmp > mont_max) {
        jj       = J;
        mont_max = tmp;
      }
    }
  }
#if defined(EPIC_MPI)
  /* Compare notes on different processors */
  tmp  = mont_max;
  tmp2 = tmp;
  MPI_Allreduce(&tmp2,&tmp,1,MPI_DOUBLE,MPI_MAX,para.comm);
  if (mont_max < tmp) {
    jj = INT_MAX;
  }
  itmp = jj;
  MPI_Allreduce(&itmp,&jj,1,MPI_INT,MPI_MIN,para.comm);
#endif
  if (IAMNODE == 0) {
    fprintf(stderr,"\nApplied p_avg[k] at lat = %5.1f.\n\n", grid.lat[2*jj+1]);
  }
  /* Correct mont planes using p_avg at J = jj: */
  if (jj >= JLO && jj <= JHI) {
    tmp = MONT3D(grid.nk,jj,ILO);
    for (K = grid.nk-1; K >= 1; K--) {
      d_th0       = grid.theta[2*K]-grid.theta[2*K+2];
      kk          = 2*K+1;
      fpara       = get_chem(planet,FPARA_INDEX,kk,J,I);
      pressure    = grid.p_avg[K];
      theta       = grid.theta[kk];
      temperature = return_temp(planet,fpara,pressure,theta);
      exner       = planet->cp*temperature/theta;   
      /* tmp holds mont */
      tmp        += exner*d_th0;
      /* mont_k holds correction term */
      mont_k[K] = MONT3D(K,jj,ILO)-tmp;
    }
  }
  else {
    memset(&(mont_k[1]),0,grid.nk*sizeof(double));
  }
#if defined(EPIC_MPI)
  memcpy(&sendbuf[1],&mont_k[1],grid.nk*sizeof(double));
  MPI_Allreduce(&sendbuf[1],&mont_k[1],grid.nk,MPI_DOUBLE,MPI_SUM,para.comm);
#endif
  for (K = grid.nk-1; K >= 1; K--) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        MONT3D(K,J,I) -= mont_k[K];
      }
    }
    /* No need to apply horizontal boundary conditions to mont3d. */
  }

  /* 
   * Calculate p from mont using hydrostatic approximation:
   */
  set_p_balanced(planet,mont3d);

  /* Free allocated memory */
  free_dvector(sendbuf,1,grid.nk);
  free_dvector(mont_k, 1,grid.nk);
  free_dvector(mont3d, 0,Nelem3d-1);

  return;
}

/*====================== end of init_with_u() ================================*/

/*====================== init_with_p_avg() ===================================*/
    
/* 
 * Initialize pressure with p_avg. 
 */

void init_with_p_avg(void)
{
  int
    K,J,I;

  for (K = 1; K <= KLAST_ACTIVE; K++) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        P(K,J,I) = grid.p_avg[K];
      }
    }
    BC2D(&(P(KLO,JLO,ILO)),P_INDEX,K);
  }

  return;
}

/*====================== end of init_with_p_avg() ============================*/

/*====================== init_with_qm() ======================================*/

                   /**************************/
                   /* M(q) Initialization    */
                   /* Installed Spring, 1997 */
                   /* Charles Santori        */
                   /**************************/

void init_with_qm(init_defaultspec *def,
                  double           (*uzonal)(double))
{
  int 
    K,J,I,
    manual,j0,j1,j12,
    k0,k1,qmrepeat,ptyp;
  double
    *mont3d;
  mqspec 
    params;

  /* Allocate memory: */
  mont3d = dvector(0,Nelem3d-1);

  while (1){
    def->mq_c_q = params.c_q = 
      input_double("Input c_q parameter:\n", def->mq_c_q);
    def->mq_pfact = params.pfact =
      input_double("Input pressure multiplier, used for the shrink-and-stretch procedure:\n",
		     def->mq_pfact);
    def->mq_ztyp = params.ztyp =
      input_int("For q_0(theta), use (0) zeta = 0 or (1) zeta = const?\n", def->mq_ztyp);
    fprintf(stderr, "\n\nSOR selections:\n");
    fprintf(stderr, "\n(%d) Set pressure constant at top, use u vs. lat at bottom,\n", SORPTOP);
    fprintf(stderr,    "    use T(p) on equatorward side, and set u = 0 at pole.\n");
    fprintf(stderr, "\n(%d) Set u = 0 at top, use u vs. lat at bottom,\n", SORUTOP);
    fprintf(stderr,    "    use T(p) on equatorward side, and set u = 0 at pole.\n");
    fprintf(stderr, "\n(%d) Set u = 0 at top, set pressure constant at bottom,\n", SORPANDU);
    fprintf(stderr,    "    use T(p) on equatorward side, and set u = 0 at pole.\n");
    fprintf(stderr, "\nSimple fill selections:\n");
    fprintf(stderr, "\n(%d) Set pressure constant at bottom, use u vs. lat at bottom,\n", FILUP);
    fprintf(stderr,    "    use T(p) on equatorward side, and set u = 0 at pole.\n");
    fprintf(stderr, "\n(%d) Set pressure constant at top, set u = 0 (in some frame) at top,\n", FILDWN);
    fprintf(stderr,    "    use T(p) on equatorward side, and set u = 0 at pole.\n");
    fprintf(stderr, "\n(%d) Set pressure constant at top, use u vs. lat at bottom,\n", FILLFT);
    fprintf(stderr,    "    use T(p) at pole, and set u = 0 at pole.\n");
    fprintf(stderr, "\nRecommended: %d or %d.\n\n", SORPTOP, SORUTOP);
    ptyp = def->mq_ptyp = 
      params.ptyp = input_int("Select boundary conditions:\n", def->mq_ptyp);

    if (strcmp(planet->class,"gas-giant") == 0) {
      /* Fill in U on the bottom layer for mont_nk() call in mont_from_q(). */
      K = grid.nk;
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          U(K,J,I) = uzonal((grid.lat)[2*J+1]);
        }
      }
      BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);
    }

    /* Calculate Montgomery Potential */
    if (ptyp == SORPANDU || ptyp == FILUP){
      fprintf(stderr, "nk = %d.  Enter bottom layer for application of M(q):", grid.nk);
      params.kbot_in = input_int("\n", grid.nk);
      if (ptyp == FILUP){
        fprintf(stderr, "Input x parameter: the ratio by which winds increase in each successive\n");
        fprintf(stderr, "(next-highest) layer below k = %d :", params.kbot_in);
        def->mq_x = params.x = 
          input_double("\n", def->mq_x);
      }
    }
    if (ptyp == FILDWN) {
      params.utop = input_double("Input u at top, equatorial boundary:\n", 0.);
    }
    if (ptyp == SORUTOP || ptyp == SORPANDU) {
      params.nuzero = input_int("Input number of layers on top with u set to zero:\n", 1);
    }
    if (ptyp == SORPTOP || ptyp == SORUTOP || ptyp == SORPANDU) {
      def->mq_manual = manual =
        input_int("Would you like manual control over the relaxation rate? (1 = yes)\n",def->mq_manual);
      if (manual == 1) {
        fprintf(stderr, "You will get to repeatedly specify a number of iterations, and a multiplier\n");
        fprintf(stderr, "to omega, the relaxation parameter.\n\n");
        def->mq_checkit = params.checkit =
          input_int("Input convergeance check interval:\n", def->mq_checkit);
        params.rrmanual = input_double("Input multiplier to omega:\n", 1.);
        if (params.rrmanual < 0.) {
          fprintf(stderr, "Negative value not allowed.\n\n");
	  continue;
        }
      }
      else{
        def->mq_maxit = params.maxit =
          input_int("Input maximum allowed number of iterations:\n", def->mq_maxit);
        params.rrmanual = -1.;
        params.checkit  = 50;
      }
    }
    if (mont_from_q(mont3d,params) == -1) continue;	
    /* 
     * Calculate u from mont using gradient balance:
     */
    for (K = 1; K <= grid.nk; K++) {
      set_u_balanced(mont3d+Nelem2d*(K-1),NULL,K);
    }
    /* 
     * Calculate p from mont using hydrostatic approximation:
     */
    set_p_balanced(planet,mont3d);

    while (1) {
      def->mq_qmrepeat = qmrepeat =
        input_int("(0) Continue, (1) Try new c, (2) View results, or (3) Measure c_1?\n",
                  def->mq_qmrepeat);
      if (qmrepeat < 2) break;
      if (qmrepeat == 2){
        j0  = input_int("Enter lowest J:\n", JLO);
        j1  = input_int("Enter highest J:\n", JHI);
        j12 = input_int("Enter increment in J:\n", 1);
        k1  = input_int("Enter highest K:\n", grid.nk);
        k0  = input_int("Enter lowest K:\n", 1);
        outm(j0,j1,j12,k0,k1,mont3d);
        outp(j0,j1,j12,k0,k1);
        outh(j0,j1,j12,k0,k1);
        outu(j0,j1,j12,k0,k1);
      }
      else {
        def->mq_jzero = J = 
          input_int("Enter J (-1 for all J) :\n", def->mq_jzero);
        if (J < -1 || J > grid.nj+1) {
          fprintf(stderr, "Out of bounds.\n");
        }
        else if (J == -1) {
          fprintf(stderr, "\nPaste these into your IDL procedure:\n\nc_1 = [");
          for (J = 0; J <= grid.nj; J++) {
            fprintf(stderr, "%f", low_mode(J));
            if (J != grid.nj) fprintf(stderr, ", ");
          }
          fprintf(stderr, "]\n\nlat = [");
          for (J = 0; J <= grid.nj; J++) {
            fprintf(stderr, "%f", grid.lat[2*J+1]);
            if (J != grid.nj) fprintf(stderr, ", ");
          }
          fprintf(stderr, "]\n\n");
        }
        else {
          fprintf(stderr, "\nc_1[%d] = %f\n\n", J, low_mode(J));
        }
      }
    }
    if (!qmrepeat) break;
  }

  /* Free allocated memory */
  free_dvector(mont3d,0,Nelem3d-1);

  return;
}

/*====================== end of init_with_qm() ===============================*/

/*====================== init_fpara_as_fpe() =================================*/

/*
 * Iterate to get fpara = fpe(temp(fpara,pressure,theta)):
 */

#define FPE_TOLERANCE 1.e-5

void init_fpara_as_fpe(planetspec *planet)
{
  int
    K,J,I;
  double
    fp_guess,t_guess,theta,
    fgibb,fpe,uoup,
    pressure;

  for (K = 1; K <= KLAST_ACTIVE; K++) {
    theta = grid.theta[2*K];
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        pressure = get_p(planet,P_INDEX,2*K,J,I);
        /* Iterate */
        do {
          fp_guess = FPARA(K,J,I);
          t_guess  = return_temp(planet,fp_guess,pressure,theta);
          return_enthalpy(planet,fp_guess,pressure,t_guess,&fgibb,&fpe,&uoup);
          FPARA(K,J,I) = fpe;
        } while (fabs(fp_guess-fpe) > FPE_TOLERANCE);
      }
    }
    BC2D(&(FPARA(KLO,JLO,ILO)),FPARA_INDEX,K);
  }

  /* Fill in fpara for k = nk, set to interface value */
  K = grid.nk;
  theta = grid.theta[2*K];
  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      pressure = P(K-1,J,I);
      /* Iterate */
      do {
        fp_guess = FPARA(K,J,I);
        t_guess  = return_temp(planet,fp_guess,pressure,theta);
        return_enthalpy(planet,fp_guess,pressure,t_guess,&fgibb,&fpe,&uoup);
        FPARA(K,J,I) = fpe;
      } while (fabs(fp_guess-fpe) > FPE_TOLERANCE);
    }
  }
  BC2D(&(FPARA(KLO,JLO,ILO)),FPARA_INDEX,K);

  return;
}

/*====================== end of init_fpara_as_fpe() ==========================*/

/*====================== init_v_by_continuity() ==============================*/

/*
 * Initialize v to satisfy the steady continuity equation:  
 * d(vh)/dy = -d(wh)/dtheta.
 * Assumes grid.ni = 1 (i.e., a meridional-plane model).
 *
 * The macros EP_FLUX_Z() and EP_FLUX_Y() use the diag structure, which is
 * defined globally in epic.h.
 */

void init_v_by_continuity(planetspec *planet)
{
  int
    K,J;
  double
    *heat,*wh0,*wh1,*vh1,*vh2,*ptmp,
    h,d_th1_inv,theta,exner,fpara,tmp;

  /* Allocate memory: */
  heat = dvector(0,Nelem2d-1);
  wh0  = dvector(0,Nelem2d-1);
  wh1  = dvector(0,Nelem2d-1);
  vh1  = dvector(0,grid.nj+1);
  vh2  = dvector(0,grid.nj+1);

  /* Calculate wh for top of layer nk (ie, bottom of layer nk-1): */
  K     = grid.nk-1;
  theta = grid.theta[2*K+1];
  heating(planet,heat,K);
  fprintf(stdout,"Applying heating to layer K = %d --> epic_init_funcs.c_1\n",K);  /*RAUL*/
  for (J = JLO; J <= JHI; J++) {
    fpara      = get_chem(planet,FPARA_INDEX,2*K+1,J,ILO);
    exner      = (planet->cp)*return_temp(planet,fpara,P(K,J,ILO),theta)/theta;
    h          = get_h(planet,2*K+1,J,ILO);
    WH0(J,ILO) = h*HEAT(J,ILO)/exner;

    /* Load ep_flux_z with dtheta/dt: */
    EP_FLUX_Z(K,J) = HEAT(J,ILO)/exner;
  }

  for (K = grid.nk-1; K >= 1; K--) {
    ptmp = wh1;
    wh1  = wh0;
    wh0  = ptmp;
 
    /* Clear wh0: */
    memset(wh0,0,Nelem2d*sizeof(double));

    if (K > 1) {
      /* Calculate h and heat for top of current layer */
      heating(planet,heat,K-1);
      fprintf(stdout,"Applying heating to layer K-1 = %d --> epic_init_funcs.c_2\n",K-1);  /*RAUL*/

      /* Start calculation of wh0: */
      for (J = JLO; J <= JHI; J++) {
        h          = get_h(planet,2*K-1,J,ILO);
        WH0(J,ILO) = h*HEAT(J,ILO);
      }
    }

    if (K > 1) {
      theta = grid.theta[2*K-1];
      for (J = JLO; J <= JHI; J++) {
        /* Calculate exner for the top of current layer: */
        fpara = get_chem(planet,FPARA_INDEX,2*K-1,J,ILO);
        exner = (planet->cp)*return_temp(planet,fpara,P(K-1,J,ILO),theta)/theta;

        /* Finish calculation of wh0: */
        WH0(J,ILO) /= exner;

        /* Load ep_flux_z with dtheta/dt: */
        EP_FLUX_Z(K-1,J) = HEAT(J,ILO)/exner;
      }
    }

    if (K > 1) {
      d_th1_inv = 1./(grid.theta[2*K-1]-grid.theta[2*K+1]);
    }
    else {
      d_th1_inv = 1./(grid.theta[2*K  ]-grid.theta[2*K+1]);
    }
    /* 
     * Do global integral on all processors. If both poles
     * are included, run from poles in both directions, take 
     * weighted average.
     */
    vh1[0] = 0.;
    for (J = 0; J < grid.nj; J++) {
      vh1[J+1] = vh1[J]+(WH1(J,ILO)-WH0(J,ILO))*
                        d_th1_inv/grid.mn[2*J+1];
    }
    vh2[grid.nj+1] = 0.;
    for (J = grid.nj; J > 0; J--) {
      vh2[J] = vh2[J+1]-(WH1(J,ILO)-WH0(J,ILO))*
                        d_th1_inv/grid.mn[2*J+1];
    }
    /* Assign v */
    for (J = JFIRST; J <= JHI; J++) {
      if (grid.lat[2*0] == -90. && grid.lat[2*(grid.nj+1)] == 90.) {
        /* both poles present, use weighted average */
        tmp = (double)J/(grid.nj+1);
      }
      else if (fabs(grid.lat[2*0]) >= fabs(grid.lat[2*(grid.nj+1)])) {
        /* southern edge closest to pole */
        tmp = 0.;
      }
      else {
        /* northern edge closest to pole */
        tmp = 1.;
      } 
      V(K,J,ILO) = (vh1[J]*(1.-tmp)+vh2[J]*tmp)*
                   grid.m[2*J]/(.5*(get_h(planet,2*K,J,  ILO)
                                   +get_h(planet,2*K,J-1,ILO)));

      /* Load ep_flux_y with v */
      EP_FLUX_Y(K,J) = V(K,J,ILO);
    }
  }

  /* Free allocated memory: */
  free_dvector(vh2, 0,grid.nj+1);
  free_dvector(vh1, 0,grid.nj+1);
  free_dvector(wh1, 0,Nelem2d-1);
  free_dvector(wh0, 0,Nelem2d-1);
  free_dvector(heat,0,Nelem2d-1);

  return;
}

/*====================== end of init_v_by_continuity() =======================*/

/*====================== init_viscosity() ====================================*/

/* 
 * Initialize viscosity coefficients. 
 * The factor A in nu2 = A*dx*dx/dt should be less than about (3./44.)
 * for computational stability of viscosity with the 3rd order 
 * Adams-Bashforth timestep (1./30. is known to work). 
 * See Dowling et al (1998) for details.
 * We usually run with nu2 = 0.
 *
 * The factor A in nu4 = A*dx*dx*dx*dx/dt should be less than about (1./240.)
 * for computational stability of hyperviscosity.  We use this term
 * to reduce gridscale noise.
 */

void init_viscosity(planetspec       *planet,
                    init_defaultspec *def)
{
  double
    dx,dt,
    lat,rln,rlt,
    m_45,n_45,
    max_nu[MAX_NU_ORDER+1]; 

  /*
   * Calculate dx for CFL calculations:
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    lat  = 45.*DEG;
    rln  = planet->re/sqrt( 1.+ pow(planet->rp/planet->re*tan(lat),2.) );
    rlt  = rln/( cos(lat)*( pow(sin(lat),2.) +
                pow(planet->re/planet->rp*cos(lat),2.) ) );
    m_45 = 1./(rln*grid.dln*DEG);
    n_45 = 1./(rlt*grid.dlt*DEG);
    dx   = sqrt(1./(m_45*m_45+n_45*n_45));
  }
  else if (strcmp(grid.geometry,"f-plane")  == 0) {
    if (strcmp(grid.f_plane_map,"polar") == 0) {
      rln  = .5*grid.f_plane_half_width;
      m_45 = 1./(rln*grid.dln*DEG);
      n_45 = 1./(grid.f_plane_half_width/grid.nj);
      dx   = sqrt(1./(m_45*m_45+n_45*n_45));
    }
    else {
      dx = grid.f_plane_half_width/grid.ni;
    }
  }
  else {
    fprintf(stderr,"Error: init_viscosity(): Unrecognized grid.geometry %s\n",
                    grid.geometry);
    exit(1);
  }

  dt = (double)(grid.dt);

  /* R. LeBeau's estimates of max nu's (Dowling et al, 1998) */
  max_nu[0] = (1./3.  )/dt;
  max_nu[2] = (1./30. )*dx*dx/dt;
  max_nu[4] = (1./240.)*pow(dx,4.)/dt;
  max_nu[6] = (1./800.)*pow(dx,6.)/dt;

  def->nu[2] = input_double("nu[2], percent of max\n",def->nu[2]);
  def->nu[4] = input_double("nu[4], percent of max\n",def->nu[4]);
  def->nu[6] = input_double("nu[6], percent of max\n",def->nu[6]);
  grid.nu[2] = def->nu[2]*max_nu[2];
  grid.nu[4] = def->nu[4]*max_nu[4];
  grid.nu[6] = def->nu[6]*max_nu[6];

  return;
}

/*====================== end of init_viscosity() =============================*/

/*====================== init_humidity() =====================================*/

/*
 * Set up initial specific-humidity fields.
 */

void init_humidity(planetspec       *planet,
                   init_defaultspec *def)
{
  int
    K,J,I,
    kk,
    index;
  double
    solar,
    temperature,
    fpara,
    pressure,
    theta,
    sat_vapor_p,
    x,x_sat,
    x1,x2,xx,
    mu,avg_mu,dry_air,
    lnkp[MAX_NVARS],lnppbar,ppbar,
    mole_fraction[MAX_NVARS],
    mole_fraction0[MAX_NVARS];
  char
    min_element[MAX_NVARS][4],
    prompt[64];

  /*
   * Establish elemental mole fractions. These are stored in def->mole_fraction[],
   * indexed by the carried condensible whose least-solar-abundant element matches 
   * the element in question. The value of def->mole_fraction[] should equal the sum of 
   * the mole fractions of all the molecules that include that element.
   */
  for (index = FIRST_HUMIDITY; index < FIRST_CHEM_PRODUCT; index++) {
    if (var.chem_on[index]) {
      solar = solar_fraction(var.chem_name[index],BY_NUMBER,min_element[index]);
      fprintf(stderr,"Solar mole fraction of %s is %g \n",
                      min_element[index],solar);
      sprintf(prompt,"Input %s mole fraction:\n",min_element[index]);
      def->mole_fraction[index] = input_double(prompt,def->mole_fraction[index]);
    }
    else {
      def->mole_fraction[index] = 0.;
    }
  }

  for (J = JLO; J <= JHI; J++) {
    for (I = ILO; I <= IHI; I++) {
      /* 
       * Start at lowest layer with elemental mole fractions 
       * set to specified values. 
       */
      for (index = FIRST_HUMIDITY; index < FIRST_CHEM_PRODUCT; index++) {
        mole_fraction0[index] = def->mole_fraction[index];
      }
      for (K = KHI; K >= KLO; K--) {
        kk          = 2*K;
        theta       = grid.theta[kk];
        pressure    = get_p(planet,P_INDEX,kk,J,I);  
        fpara       = get_chem(planet,FPARA_INDEX,kk,J,I);
        temperature = return_temp(planet,fpara,pressure,theta);
        if (var.chem_on[NH_4SH_INDEX]) {
          /* 
           * Eq (12) of Killen and Flasar (1996, Icarus 119, 67-89): 
           */
          lnkp[NH_4SH_INDEX] = 34.151-10834./temperature;
          ppbar              = pressure*pressure*1.e-10;
        }
        for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
          if (var.chem_on[index]) {
            /*
             * Adjust for chemistry: 
             */
            if (index == NH_4SH_INDEX) {
              /* 
               * Determine if NH_4SH forms. 
               * Use eq (12) of Killen and Flasar (1996).
               * NOTE: Pressure in (12) is in bars.
               */
              /* Mole fractions of N and S: */
              x1 = mole_fraction0[NH_3_INDEX];
              x2 = mole_fraction0[H_2S_INDEX];
              xx = ppbar*x1*x2;
              if (xx > 0.) {
                lnppbar = log(xx);
              }
              else {
                lnppbar = -DBL_MAX;
              }
              if (lnppbar > lnkp[NH_4SH_INDEX]) {
                /* NH_4SH forms. */
                mole_fraction[NH_4SH_INDEX] = MIN(x1,x2);
              }
              else {
                /* NH_4SH doesn't form: */
                mole_fraction[NH_4SH_INDEX] = 0.;
              }
              /* Adjust NH_3 and H_2S: */
              mole_fraction[NH_3_INDEX] -= mole_fraction[NH_4SH_INDEX];
              mole_fraction[H_2S_INDEX] -= mole_fraction[NH_4SH_INDEX];
            }
            else {
              /* 
               * Set chemical mole fraction: 
               */
              mole_fraction[index] = mole_fraction0[index];
            }
          }
        }
        for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
          if (var.chem_on[index]) {
            /* 
             * Calculate saturation vapor pressure: 
             */
            sat_vapor_p = return_sat_vapor_p(var.chem_name[index],temperature);
            /* 
             * Calculate saturation mole fraction: 
             */
            x_sat  = sat_vapor_p/pressure;
            x_sat *= 1.0;
            /* 
             * Assume the mole fraction is bounded by the saturation curve.
             * Store temporarily in VAR, to be converted to specific humidity
             * after all chemicals are done.
             *
             */
            x                = mole_fraction[index];
            VAR(index,K,J,I) = MIN(x,x_sat);
          }
        }
        for (index = FIRST_HUMIDITY; index < FIRST_CHEM_PRODUCT; index++) {
          if (var.chem_on[index]) {
            /*
             * Assume that as we travel upwards, elemental mole fractions stay
             * reduced once they has been limited by saturation.
             */
            mole_fraction0[index] = VAR(index,K,J,I);
            /* 
             * Adjust for chemistry: 
             */ 
            if (index == NH_3_INDEX || index == H_2S_INDEX) {
              if (var.chem_on[NH_4SH_INDEX]) {
                mole_fraction0[index] += VAR(NH_4SH_INDEX,K,J,I);
              }
            }
          }
        }

        /* 
         * Convert VAR's from mole fraction to specific humidity: 
         */
        dry_air = 1.;
        avg_mu  = 0.;
        for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
          if (var.chem_on[index]) {
            x        = VAR(index,K,J,I);
            dry_air -= x;
            avg_mu  += x*molar_mass(planet,var.chem_name[index]);
          }
        }
        x       = dry_air;
        avg_mu += x*molar_mass(planet,"dry_air");
        for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
          if (var.chem_on[index]) {
            VAR(index,K,J,I) *= molar_mass(planet,var.chem_name[index])/avg_mu;
          }
        }
      }
    }
  }

  return;
}

/*====================== end of init_humidity() ==============================*/

/*======================== new_init_humidity() ==============================*/

void new_init_humidity(planetspec       *planet,
                   init_defaultspec *def)

{

int
  K,J,I,kk,index,index2;
double
  mu, mu_i, pressure, fp, temperature, coeff,
  sh_1, sh_2, p_sat, density_1, density_2, theta;
static int
  initialized = FALSE;
static double
  *buffji;

  if(initialized == FALSE) {
  /** allocating memory **/
    buffji = dvector(0,Nelem2d-1);
    initialized = TRUE;
  }


  /*
   * Creates a layer of 100% relative humidity for a denoted pressure range.
   */

  for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {

    /* Input pressure range for cloud */

    if (var.chem_on[index]==CHEM_PASSIVE) {
    
      grid.aux_fa = input_double("Input upper pressure range for cloud [mbar] \n",500.);
      grid.aux_fb = input_double("Input lower pressure range for cloud [mbar] \n",1000.);

      /* Convert pressures from mbar to mks: */
    
      grid.aux_fa *= 100.;
      grid.aux_fb *= 100.;

      for (K=KLO; K<=KHI; K++) {
        kk = 2*K;
        theta = grid.theta[kk];
        for (J=JLO; J<=JHI; J++) {
          for (I=ILO; I<=IHI; I++) {

            if (P(K,J,I)>=grid.aux_fa && P(K,J,I)<=grid.aux_fb) {
              mu_i = molar_mass(planet,var.chem_name[index]);
              pressure = get_p(planet, P_INDEX, kk, J, I);
              fp = get_chem(planet, FPARA_INDEX, kk, J, I);
              temperature = return_temp(planet, fp, pressure, theta);
              mu = avg_molar_mass(planet, kk, J, I);
              sh_1 = VAR(index,K,J,I);
              p_sat = return_sat_vapor_p(var.chem_name[index], temperature);
              density_1 = return_dens(planet, fp, pressure, theta, mu, 
	                       PASSING_THETA);
              coeff = mu_i*p_sat/(R_GAS*temperature);
              density_2 = density_1*(1-sh_1)+coeff;
              sh_2 = coeff/density_2;
          
              for (index2 = FIRST_HUMIDITY; index2 <= LAST_HUMIDITY; index2++){
            
                if(index2 == index){
                  VAR(index2,K,J,I) = sh_2;
                }                
                else if(var.chem_on[index2]==CHEM_PASSIVE){
                  VAR(index2,K,J,I) *= density_1/density_2;
                }                
              }
            }
          
            else {
              VAR(index,K,J,I) = 0.00001;
            }
          }
        }   
  	}
    }
  }
  return;
}

/*======================== end of new_init_humidity() ========================*/

/*====================== t_yp() ==============================================*/

/*
 * Interpolates T(y,p) data table using a bicubic spline.
 */

#define T_DATA(k,j)    t_data[   j+(k)*nlat]
#define T_DATA_Y2(k,j) t_data_y2[j+(k)*nlat]

double t_yp(double            p, 
            double            latr, 
            int               mode,
            init_defaultspec *def)
{
  char
    string[N_STR];
  int
    k,j,kk;
  static int
    initialized=0,
    nlat,
    npress;
  double 
    z,temp,lat_d,
    log_p,neg_log_p,p_d,
    a4 = 2.;
  static double
    *lat,
    *press,
    *t_data,
    *t_data_y2,
    *t_k,
    *t_k_y2;

  if (!initialized) {
    char
      header[80];
    FILE
      *infile;

    sprintf(string,EPIC_PATH"/data/%s/iris/temperature.dat",planet->name);
    infile = fopen(string,"r");
    if (!infile) {
      fprintf(stderr,"Cannot find input file %s \n",string);
      exit(1);
    }
    fscanf(infile,"%d %d",&nlat,&npress);
    fscanf(infile,"%s",header);
    /* Allocate memory: */
    lat       = dvector(0,  nlat-1);
    press     = dvector(0,npress-1);
    t_k       = dvector(0,npress-1);
    t_k_y2    = dvector(0,npress-1);
    t_data    = dvector(0,nlat*npress-1);
    t_data_y2 = dvector(0,nlat*npress-1);

    for (k = 0; k < npress; k++) {
      fscanf(infile,"%lf",press+k);
      /* convert press to mks */
      press[k] *= 100.;
      /* spline on log p */
      press[k] = log(press[k]);
    }
    fscanf(infile,"%s",header);
    for (j = 0; j < nlat; j++) {
      fscanf(infile,"%lf",lat+j);
      /* convert lat to radians */
      lat[j] *= DEG;
      for (k = 0; k < npress; k++) {
        fscanf(infile,"%lf",&(T_DATA(k,j)));
      }
    }
    fclose(infile);
    /* Compute spline information for each row: */
    for (k = 0; k < npress; k++) {
      spline(lat,&(T_DATA(k,0)),nlat,1.e+30,1.e+30,&(T_DATA_Y2(k,0)));
    }
    if (IAMNODE == 0) {
      /* Echo range limits: */
      fprintf(stderr,"Range limits: lat[%2.1f,%2.1f]  p[%2.1f,%2.1f] \n",
                     lat[0]/DEG,lat[nlat-1]/DEG,
                     exp(press[npress-1])/100.,exp(press[0])/100.);
    }

    if (def) {
      /* Set default ranges */
      /* latbot, lattop in degrees */
      def->globe_latbot = lat[0     ]/DEG;
      def->globe_lattop = lat[nlat-1]/DEG;
      /* ptop, pbot in mbar */
      def->ptop         = exp(press[0       ])/100.;
      def->pbot         = exp(press[npress-1])/100.;
    }

    initialized = 1;
  }
  if (p == 0) return;

  if (mode == 1) {
    /* Synthetic data: */
    z    = -log(p/grid.press0);
    temp = a4*z*cos(4.*latr);
  }
  else if (mode == 0) {
    /* Data */
    if (latr < lat[0]) {
      fprintf(stderr,"error: t_yp(), lat = %f below data range = %f \n",
              latr/DEG,lat[0]/DEG);
      exit(1);
    }
    else if (latr > lat[nlat-1]) {
      fprintf(stderr,"error: t_yp(), lat = %f above data range = %f \n",
              latr/DEG,lat[nlat-1]/DEG);
      exit(1);
    }
    j = find_place_in_table(nlat,lat,&latr,&lat_d);
    /* Make column at latr */
    for (k = 0; k < npress; k++) {
      t_k[k] = splint(latr,lat+j,&(T_DATA(k,j)),&(T_DATA_Y2(k,j)),lat_d);
    }
    spline(press,t_k,npress,1.e+30,1.e+30,t_k_y2);

    log_p = log(p);
    if (log_p >= press[0] && log_p <= press[npress-1]) {
      k    = find_place_in_table(npress,press,&log_p,&p_d);
      temp = splint(log_p,press+k,t_k+k,t_k_y2+k,p_d);
    }
    else if (log_p > press[0]) {
      fprintf(stderr,"Error: t_yp(): p(%5.1f) = %e out of bounds (increase floor_tp) \n",
                      latr/DEG,p/100.);
      exit(1);
    }
    else {
      fprintf(stderr,"Error: t_yp(): p(%5.1f) = %e out of bounds (decrease ceiling_tp) \n",
                      latr/DEG,p/100.);
      exit(1);
    }
  }
  else {
    fprintf(stderr,"Error: t_yp(), unrecognized mode. \n");
    exit(1);
  }

  return temp;
}

/*====================== end of t_yp() ========================================*/

/*====================== fp_yp() ==============================================*/

/*
 * Interpolates fp data table using a bicubic spline.
 * Uses data-boundary value when pressure out of range.
 */

#define FP_DATA(k,j)    fp_data[   j+(k)*nlat]
#define FP_DATA_Y2(k,j) fp_data_y2[j+(k)*nlat]

double fp_yp(double p, 
             double latr, 
             int    mode)
{
  char
    string[N_STR];
  int
    k,j;
  static int
    initialized=0,
    nlat,
    npress;
  double 
    z,fp,lat_d,
    log_p,p_d,
    a1 =  0.5,
    a2 =  0.1,
    a3 = -0.05;
  static double
    *lat,
    *press,
    *fp_data,
    *fp_data_y2,
    *fp_k,
    *fp_k_y2;

  if (!initialized) {
    char
      header[80];
    FILE
      *infile;

    sprintf(string,EPIC_PATH"/data/%s/iris/fpara.dat",planet->name);
    infile = fopen(string,"r");
    fscanf(infile,"%d %d",&nlat,&npress);
    fscanf(infile,"%s",header);
    /* Allocate memory */
    lat        = dvector(0,  nlat-1);
    press      = dvector(0,npress-1);
    fp_k       = dvector(0,npress-1);
    fp_k_y2    = dvector(0,npress-1);
    fp_data    = dvector(0,nlat*npress-1);
    fp_data_y2 = dvector(0,nlat*npress-1);

    for (k = 0; k < npress; k++) {
      fscanf(infile,"%lf",press+k);
      /* convert press to mks */
      press[k] *= 100.;
      /* spline on log p */
      press[k] = log(press[k]);
    }
    fscanf(infile,"%s",header);
    for (j = 0; j < nlat; j++) {
      fscanf(infile,"%lf",lat+j);
      /* convert lat to radians */
      lat[j] *= DEG;
      for (k = 0; k < npress; k++) {
        fscanf(infile,"%lf",&(FP_DATA(k,j)));
      }
    }
    fclose(infile);
    /* Compute spline information for each row: */
    for (k = 0; k < npress; k++) {
      spline(lat,&(FP_DATA(k,0)),nlat,1.e+30,1.e+30,&(FP_DATA_Y2(k,0)));
    }
    initialized = 1;
  }

  if (mode == 1) {
    /* Synthetic data */
    z  = -log(p/grid.press0);
    fp = a1+a2*z+a3*(1.-cos(4.*latr));
  }
  else if (mode == 0) {
    /* Data */
    if (latr < lat[0]) {
      fprintf(stderr,"error: fp_yp(), lat = %f below data range = %f \n",
              latr/DEG,lat[0]/DEG);
      exit(1);
    }
    else if (latr > lat[nlat-1]) {
      fprintf(stderr,"error: fp_yp(), lat = %f above data range = %f \n",
              latr/DEG,lat[nlat-1]/DEG);
      exit(1);
    }
    j = find_place_in_table(nlat,lat,&latr,&lat_d);
    /* Make column at latr */
    for (k = 0; k < npress; k++) {
      fp_k[k] = splint(latr,lat+j,&(FP_DATA(k,j)),&(FP_DATA_Y2(k,j)),lat_d);
    }
    spline(press,fp_k,npress,1.e+30,1.e+30,fp_k_y2);

    log_p = log(p);
    if (log_p < press[0]) {
      log_p = press[0];
    }
    else if (log_p > press[npress-1]) {
      log_p = press[npress-1];
    }
    k  = find_place_in_table(npress,press,&log_p,&p_d);
    fp = splint(log_p,press+k,fp_k+k,fp_k_y2+k,p_d);
  }
  else {
    fprintf(stderr,"error: epic_initial, unrecognized mode in fp_yp() \n");
    exit(1);
  }

  return MIN(fp,1.);

}

/*====================== end of fp_yp() ======================================*/

/*====================== mont_from_q() =======================================*/

int mont_from_q(double *mont3d,
	        mqspec params){

  int I, J, K, j0, j1, jj0, jj1, jj2, j_tp, sret, ptyp = params.ptyp;
  double g_inv, d_th1_inv;
  double h, *q0, z0bot;
  double tmp, d_th0, exner;
  double lat, delat, latavg, deltamont;
  double p_shrunk[201];

  /* Initialize J limits */
  latavg = (grid.globe_latbot + grid.globe_lattop)*.5;
  if (latavg > 5.){
    if (grid.globe_lattop != 90.){
      fprintf(stderr, "WARNING: North pole not included.\n");
    }
    params.npolar = 1;
    params.j_eq = JLO;
    params.j_pol = JHI;
    params.j_eq2 = JLO+1;
    params.j_inc = 1;
    params.j_sorlo = JLO+1;
    params.j_sorhi = JHI;
  }
  else if (latavg < -5.){
    if (grid.globe_latbot != -90.){
      fprintf(stderr, "WARNING: South pole not included.\n");
    }
    params.npolar = 0;
    params.j_eq = JHI;
    params.j_pol = JLO;
    params.j_eq2 = JHI-1;
    params.j_inc = -1;
    params.j_sorlo = JLO;
    params.j_sorhi = JHI-1;
  }
  else{
    fprintf(stderr, "Error in epic_init_funcs: not clear which hemisphere to do.\nExitting.\n\n");
    exit(1);
  }

  /* Create a new p_shrunk[K] array which will be used by the SOR or
     fill routine. */
  for (K = 1; K <= KLAST_ACTIVE; K++){
    p_shrunk[K] = grid.p_avg[K] * params.pfact;
  }

  /* Clear out M to be safe. */
  I = ILO;
  for (K = 1; K <= grid.nk; K++){
    for (J = JLO; J <= JHI; J++){
      MONT3D(K,J,I) = 0.;
    }
  }

  /* 
   * Fill in M for bottom layer. 
   * For gas-giant planets, this assumes that u_nk has been properly set.
   */
  mont_nk(planet,mont3d+(K-Kshift)*Nelem2d);

  /* Fill in M along the equatorward side by integrating P. */
  /* This is mostly copied from the geostrophic initialization. */
  j_tp = (params.ptyp == FILLFT) ? params.j_pol : params.j_eq;
  tmp = MONT3D(grid.nk,j_tp,I);
  for (K = grid.nk-1; K >= 1; K--) {
    d_th0       = grid.theta[2*K]-grid.theta[2*K+2];
    exner       = planet->cp*pow(p_shrunk[K]/grid.press0,planet->kappa);
    /* tmp holds mont */
    tmp        += exner*d_th0;
    MONT3D(K,j_tp,I) = tmp;
  }

  /* Find q0 along equatorward (or polar) side, setting zeta = 0  */
  /* or zeta = constant, according to specification.              */
  /* H calculation taken from epic functions.                     */
  /* This is not efficient code, but it is not a real time drain. */
  q0 = (double *)calloc(grid.nk, sizeof(double));

  if (ptyp != FILLFT){
    j0 = (params.npolar) ? JLO : JHI-1;
    j1 = (params.npolar) ? JLO+1 : JHI;
    jj0 = 2*params.j_eq;
    jj1 = jj0+1;
    jj2 = jj0+2;
    z0bot = grid.m[jj1]*grid.n[jj1]*
      (U(grid.nk,j0,I) / grid.m[jj0] - U(grid.nk,(j1),I) / grid.m[jj2]);
  }
  else{
    jj1 = (params.npolar) ? 2*(grid.nj+1) : 0;
    jj2 = (params.npolar) ? 2*grid.nj + 1 : 1;
    /* Need polar form of zeta (with correct sign!) */
    z0bot = params.j_inc*U(grid.nk,params.j_pol,I)*(grid.mn)[jj1]/((grid.m)[jj2]);
  }
  g_inv = 1./planet->g;
  for (K = 2; K <= grid.nk-1; K++){
    d_th1_inv = g_inv/(grid.theta[2*K-1]-grid.theta[2*K+1]);
    h = (p_shrunk[K] - p_shrunk[K-1])*d_th1_inv;
    q0[K] = grid.f[jj1];
    if (params.ztyp) q0[K] += z0bot;
    q0[K] /= h;
  }

  /* Call the solver of choice */

  if (ptyp == SORPTOP || ptyp == SORUTOP || ptyp == SORPANDU){
    sret = mont_sor(mont3d, q0, (double *)p_shrunk, params);
  }
  else if (ptyp == FILUP || ptyp == FILDWN || ptyp == FILLFT){
    sret = mont_fill(mont3d, q0, (double *)p_shrunk, params);
  }
  else{
    fprintf(stderr, "Error in ptyp in epic_init_funcs.c\n\n");
    sret = -1;
  }

  if (sret != -1){
    /* Adjust Montgomery potential to match the pressures along the
       side.  This calculation could be simplified, but I want
       to leave it easy to put in a more compicated M from p function. */
    tmp = MONT3D(grid.nk,j_tp,I);
    for (K = grid.nk - 1; K >= 1; K--){
      d_th0       = grid.theta[2*K]-grid.theta[2*K+2];
      exner       = planet->cp*pow(grid.p_avg[K]/grid.press0,planet->kappa);
      tmp        += exner*d_th0;
      deltamont = tmp - MONT3D(K,j_tp,I);
      for (J = JLO; J <= JHI; J++){
	MONT3D(K,J,I) += deltamont;
      }
    }
    /* Fill in MONT3D over all I grid points */
    for (K = 1; K <= grid.nk; K++){
      for (J = JLO; J <= JHI; J++){
	for (I = ILO+1; I <= IHI; I++){
	  MONT3D(K,J,I) = MONT3D(K,J,ILO);
	}
      }
      BC2D(&(MONT3D(KLO,JLO,ILO)),NO_INDEX,K);
    }
  }

  free(q0);
  return sret;
}

/*====================== end of mont_from_q() ================================*/

/*====================== mont_sor() ==========================================*/

#define TOL 1.e-10

int mont_sor(double *mont3d,
	     double *q0,
             double *p_shrunk,
             mqspec params){

/*
   This routine can solve three cases:

   1) Constant pressure top boundary,
   2) u = 0 top boundary (applied to optional # of top layers),
   3) u = 0 top (applied to top # layers) and constant pressure bottom.

   The input kbot_in only matters for case 3.  The other 6 inputs must
   always have appropriate values.

   Note the extra effort to mirror iterations applied to the two poles,
   to enable exact comparison.
*/

  char tripped;
  int I, J, K, p, p2, it, jj, kk, nj, nk;
  int ktop, kbot, kchecktop, kcheckbot, ptyp;
  int maxit, checkit = params.checkit;
  int jstart, jstop, dj, dj2;
  double cci;
  double dlt, dlti, dltlti;
  double dmr, dml, ptop, pbot;
  double rr, w, cheby, emax, oldemax, e, coef;
  double kapi, dmrtop, dmlbot;
  double dc, qfact, qterm, qcoef, wfact;
  double latterm, latcoef;
  double dmrsgn, dmlsgn;
  double *av, *bv, *dth, *dthi, *dththi;

  I = ILO;
  nj = JHI - JLO + 1;
  nk = grid.nk;
  ptyp = params.ptyp;
  dj = params.j_inc;
  dj2 = 2*dj;


  if (ptyp == SORPTOP){
    ktop = 2;
    kbot = nk-1;
    kchecktop = ktop+1;
    kcheckbot = kbot;
    ptop = p_shrunk[1];
    dmrtop = (planet->cp) * pow(ptop/grid.press0, (planet->kappa));
  }
  else if (ptyp == SORUTOP){
    ktop = 1+params.nuzero;
    kbot = nk-1;
    kchecktop = ktop;
    kcheckbot = kbot;
  }
  else if (ptyp == SORPANDU){
    ktop = 1+params.nuzero;
    kbot = params.kbot_in - 1;
    kchecktop = ktop;
    kcheckbot = kbot - 1;
    if (kbot <= ktop){
      fprintf(stderr, "Bad kbot_in at mont_sor.\n\n");
      return -1;
    }
    pbot = p_shrunk[kbot];
    dmlbot = (planet->cp) * pow(pbot/grid.press0, (planet->kappa));
  }
  else{
    fprintf(stderr, "Invalid ptyp in mont_sor.\n\n");
    return -1;
  }

  /* Initialize grid variables */
  if (mq_init(&dlt, &dlti, &dltlti, &cci, &kapi, &dc, &av, &bv,
	      &dth, &dthi, &dththi, params) == -1) return -1;

  /* Initialize Grid */
  /* Set u to zero on top layer(s) if called for */
  if (ptyp == SORUTOP || ptyp == SORPANDU){
    for (K = 1; K <= params.nuzero; K++){
      for (J = params.j_sorlo; J <= params.j_sorhi; J++){
	MONT3D(K,J,I) = MONT3D(K,params.j_eq,I);
      }
    }
  }
  /* Interpolate guess for interior: */
  for (K = ktop; K <= kbot; K++){
    for (J = params.j_sorlo; J <= params.j_sorhi; J++){
      if (ptyp == SORUTOP || ptyp == SORPANDU){
	MONT3D(K,J,I) = (MONT3D(K,params.j_eq,I)-MONT3D(nk,params.j_eq,I)) /
	  (MONT3D(ktop-1,params.j_eq,I)-MONT3D(nk,params.j_eq,I))
	    * (MONT3D(ktop-1,J,I)-MONT3D(nk,J,I)) + MONT3D(nk,J,I);
      }
      else{
	MONT3D(K,J,I) = (MONT3D(K,params.j_eq,I)-MONT3D(nk,params.j_eq,I)) /
	  (MONT3D(2,params.j_eq,I)-MONT3D(nk,params.j_eq,I))
	    * (MONT3D(2,params.j_eq,I)-MONT3D(nk,J,I)) + MONT3D(nk,J,I);
      }
    }
  }

  /* Initialize SOR parameters */

  w = 1.;
  /* Square of the Jacobi Radius: */
  rr = pow((cos(M_PI/(double)nj) + cos(M_PI/(double)(kbot))) / 2., 2.);
  /* Multiply by manual control factor */
  if (params.rrmanual < 0.){
    wfact = 1.;
    maxit = params.maxit;
  }
  else{
    wfact = params.rrmanual;
    maxit = checkit+1;
  }

  /* Start Iterating */
  cheby = 0.5;
  for (it = 0; it < maxit; it++){
    tripped = 0;
    emax = 0.;
    for (p = 0; p < 2; p++){                   /* p is parity */
      p2 = p;
      for (K = ktop; K <= kbot; K++){          /* K is potential temperature */
	kk = 2*K;
	jstart = params.j_eq2 + p2*params.j_inc;
	jstop = params.j_pol + dj2 - (params.j_pol - jstart) % 2;
	for (J = jstart; J != jstop; J += dj2){ 
	  jj = 2*J+1;                          /* J is latitude */
	  /* Check for negative theta derivatives */
	  if (K >= kchecktop){
	    dmr = dthi[kk-1]*(MONT3D((K-1),J,I) - MONT3D(K,J,I));
	    dmrsgn = 1.;
	    if (dmr < 0.){
	      dmr *= -1.;
	      tripped = 1;
	      dmrsgn = -1.;
	    }
	  }
	  else{
	    dmr = dmrtop;
	    dmrsgn = 0.;
	  }
	  if (K <= kcheckbot){
	    dml = dthi[kk+1]*(MONT3D(K,J,I) - MONT3D((K+1),J,I));
	    dmlsgn = 1.;
	    if (dml < 0.){
	      dml *= -1.;
	      tripped = 1;
	      dmlsgn = -1.;
	    }
	  }
	  else{
	    dml = dmlbot;
	    dmlsgn = 0.;
	  }

	  qfact = dc*q0[K]*exp((MONT3D(K,params.j_eq,I)-MONT3D(K,J,I))*cci);
	  qterm = qfact*dthi[kk]*(pow(dmr,kapi) - pow(dml,kapi)); 
	  qcoef = -qfact*kapi*dthi[kk]*
	    (dmrsgn*pow(dmr,kapi-1.)*dthi[kk-1]+dmlsgn*pow(dml,kapi-1.)*dthi[kk+1])
	    - qterm*cci;

	  if (J == params.j_pol){
	    /* Apply u=0 polar boundary condition */
	    if (params.npolar){
	      latterm = av[jj]*dltlti*bv[jj-1]*(-MONT3D(K,J,I) + MONT3D(K,(J-1),I));
	      latcoef = -av[jj]*dltlti*bv[jj-1];
	    }
	    else{
	      latterm = av[jj]*dltlti*bv[jj+1]*( MONT3D(K,(J+1),I) - MONT3D(K,J,I));
	      latcoef = -av[jj]*dltlti*bv[jj+1];
	    }
	  }
	  else{
	    latterm = av[jj]*dltlti*
	      (bv[jj+1]*MONT3D(K,(J+1),I) - (bv[jj+1]+bv[jj-1])*MONT3D(K,J,I) +
	       bv[jj-1]*MONT3D(K,(J-1),I));
	    latcoef = -av[jj]*dltlti*(bv[jj+1]+bv[jj-1]);
	  }

	  /* Calculate error */

	  e = latterm + qterm + grid.f[jj];

	  /* Calculate the derivative of e[] with respect to
	     m[shift].  This will be used in something resembling
	     Newton's method, called a Newton Gauss-Seidel
	     iteration.  Note the bandaids that take care of the
	     negative theta derivatives. */

	  coef = latcoef + qcoef;

	  /* Finally, make the adjustment */

	  MONT3D(K,J,I) -= wfact*w*e / coef;

          emax = (emax >= fabs(e)) ? emax : fabs(e);
	}
	p2 = !p2;
      }
      w = 1./(1. - cheby*rr*w);
      cheby = 0.25;
    }
    if (fabs(emax) < TOL) break;
    if (!(it%10)){
      fprintf(stderr, "%d\tMax. Error: %e\n", it, emax);
    }
    if (it > 0 && (!(emax > 1.e-20 && emax < 1.e20))){
      fprintf(stderr, "SOR iteration out of bounds.  Quitting.\n");
      return -1;
    }
    if (!(it%checkit)){
      if (params.rrmanual < 0.){
	if (emax > oldemax && it != 0){
	  fprintf(stderr, "SOR iteration is not converging.  Quitting.\n");
	  return -1;
	}
	oldemax = emax;
      }
      else if (it != 0){
	checkit = input_int("Input next convergeance check interval:\n", checkit);
	wfact = input_double("Input new multiplier to omega:\n", wfact);
	it = 0;
	maxit = checkit+1;
      }
    }
  }

  fprintf(stderr, "\nFinished.\n");
  fprintf(stderr, "Iterations: %d\n", it);
  fprintf(stderr, "Max error: %e", emax);
  fprintf(stderr, "\n\n");

  if (tripped) fprintf(stderr, "\n\nNEGATIVE THETA DERIVATIVES PRESENT.\n\n");

  /* Clean up boundaries as appropriate. */

  if (ptyp == SORPTOP){
    /* Fill in top M row using the constant pressure condition. */
    K = 1;
    for (J = params.j_sorlo; J <= params.j_sorhi; J++){
      MONT3D(K,J,I) = MONT3D((K+1),J,I) + dmrtop*(grid.theta[2] - grid.theta[4]);
    }
  }
  if (ptyp == SORPANDU){
    /* Fill in bottom M row using the constant pressure condition. */
    K = kbot+1;
    for (J = params.j_sorlo; J <= params.j_sorhi; J++){
      MONT3D(K,J,I) = MONT3D((K-1),J,I) - dmlbot*(grid.theta[2*K-2] - grid.theta[2*K]);
    }
  }
  /* Fill in remaining rows if necessary using u constant with height. */
  if  (ptyp == SORPANDU){
    for (K = kbot+2; K <= nk; K++){
      for (J = params.j_eq2; J != params.j_pol+dj; J += dj){
	MONT3D(K,J,I) = MONT3D(K,(J-dj),I) + MONT3D((K-1),J,I) - MONT3D((K-1),(J-dj),I);
      }
    }
  }

  /* Free Arrays */

  free(av);
  free(bv);
  free(dth);
  free(dthi);
  free(dththi);

  return 0;
}

/*====================== end of mont_sor() ===================================*/

/*====================== mont_fill() =========================================*/

int mont_fill(double *mont3d,
	      double *q0,
	      double *p_shrunk,
	      mqspec params){

/*
   This routine can solve three cases:

   1) Constant pressure and u(lambda) bottom bc's,
   2) Constant pressure and u=0 top bc's,
   3) u=0 and T(p) polar bc's
*/

  char tripped = 0;
  int I, J, K, jj, kk, nj, nk, dj, kbot, numfails;
  double dlt, dlti, dltlti;
  double dmr, dmrcomp, dml, dmlcomp, dmrtop, ptop;
  double kappa, kapi, cci;
  double dc, qfact, latterm;
  double *av, *bv, *dth, *dthi, *dththi;
  double qterm, e, g_inv, d_th1_inv, h, deltamont;
  double deltam, deltambot, r0;

  I = ILO;
  nj = JHI - JLO + 1;
  nk = grid.nk;
  kbot = params.kbot_in;
  dj = params.j_inc;

  /* Initialize grid variables */
  if (mq_init(&dlt, &dlti, &dltlti, &cci, &kapi, &dc, &av, &bv,
	      &dth, &dthi, &dththi, params) == -1) return -1;
  kappa = planet->kappa;

  /* Initialize grid */

  if (params.ptyp == FILUP){
    /* Set up for upward fill */
    kbot = params.kbot_in;
    if (kbot < nk){
      /* Fill in kbot layer with winds from nk layer */
      K = kbot;
      deltamont = MONT3D(K,params.j_eq,I) - MONT3D(nk,params.j_eq,I);
      for (J = params.j_sorlo; J <= params.j_sorhi; J++){
	MONT3D(K,J,I) = MONT3D(nk,J,I) + deltamont;
      }
      /* Fill in everything below kbot with winds according to
	 the x parameter (in the geostrophic approximation). */
      for (K = kbot+1; K <= nk; K++){
	for (J = params.j_eq2; J != params.j_pol+dj; J += dj){
	  MONT3D(K,J,I) = MONT3D(K,(J-dj),I) +
	    (MONT3D((K-1),J,I) - MONT3D((K-1),(J-dj),I))/params.x;
	}
      }
    }
    /* The layer immediately above the bottom layer must be
       filled in using the bottom derivative condition.         */
    K = kbot-1;
    for (J = params.j_eq2; J != params.j_pol+dj; J += dj){
      MONT3D(K,J,I) = MONT3D(K,(J-dj),I) +
	(MONT3D(kbot,J,I) - MONT3D(kbot,(J-dj),I))*params.x;
    }
    /* Fill in rest of interior with error markers */
    for (K = kbot-2; K >= 1; K--){
      for (J = params.j_sorlo; J <= params.j_sorhi; J++){
	MONT3D(K,J,I) = 1.e+20;
      }
    }
  }
  else if (params.ptyp == FILDWN){
    /* Set up for downward fill */
    /* First, the top layer must be filled in using the u=utop condition.*/
    K = 1;
    jj = (params.npolar) ? 2*JLO+1 : 2*(JHI+1)-1; 
    r0 = (planet->re)/
      sqrt( 1.+pow((planet->rp)/(planet->re)*tan(grid.lat[jj]*DEG),2.) );
    for (J = params.j_eq2; J != params.j_pol+dj; J += dj){
      jj = 2*J+1-dj;
      MONT3D(K,J,I) = MONT3D(K,(J-dj),I) - dj*dlt*grid.f[jj]*params.utop/(av[jj]*r0);
    }
    /* Next, the layer immediately below the top layer must be
       filled in using the top derivative condition.   */
    K = 2;
    for (J = params.j_eq2; J != params.j_pol+dj; J += dj){
      MONT3D(K,J,I) = MONT3D(K,(J-dj),I) + MONT3D((K-1),J,I) -
	MONT3D((K-1),(J-dj),I);
    }
  }
  else if (params.ptyp == FILLFT){
    /* Set up for fill to left */
    ptop = p_shrunk[1];
    dmrtop = (planet->cp) * pow(ptop/grid.press0, (planet->kappa));
  }
  else{
    fprintf(stderr, "Invalid ptyp in mont_fill.\n\n");
    return -1;
  }

  /* In the following loops (a different one for each case), the current J and K
     correpond to the center of the elliptic operator.  A point in the Montgomery
     potential is changed somewhere off center, depending on the case. */

  if (params.ptyp == FILUP){
    for (K = kbot-1; K >= 2; K--){             /* K is potential temperature */
      kk = 2*K;
      numfails = 0;
      for (J = params.j_sorlo; J <= params.j_sorhi; J++){   /* J is latitude */
	jj = 2*J+1;
	if (J == params.j_pol){
	  /* Apply u=0 polar boundary condition */
	  if (params.npolar){
	    latterm = av[jj]*dltlti*bv[jj-1]*(-MONT3D(K,J,I) + MONT3D(K,(J-1),I));
	  }
	  else{
	    latterm = av[jj]*dltlti*bv[jj+1]*( MONT3D(K,(J+1),I) - MONT3D(K,J,I));
	  }
	}
	else{
	  latterm = av[jj]*dltlti*
	    (bv[jj+1]*MONT3D(K,(J+1),I) - (bv[jj+1]+bv[jj-1])*MONT3D(K,J,I) +
	     bv[jj-1]*MONT3D(K,(J-1),I));
	}
	dml = dthi[kk+1]*(MONT3D(K,J,I) - MONT3D((K+1),J,I));
	qfact = dc*q0[K]*exp((MONT3D(K,params.j_eq,I)-MONT3D(K,J,I))*cci);
	dmrcomp = pow(dml,kapi) - dth[kk]*(latterm + grid.f[jj])/qfact;
	if (dmrcomp <= 0.){
	  tripped = 1;
	  numfails++;
	  MONT3D((K-1),J,I) = 1.e20;
	}
	dmr = pow(dmrcomp, kappa);
	MONT3D((K-1),J,I) = MONT3D(K,J,I) + dth[kk-1]*dmr;
	/* Check for success */
	if (!tripped) mqerr(J, params.j_eq, K, dltlti, cci, kapi,
			    dc, av, bv, dthi, mont3d, q0, params);
      }
      if (numfails >= JHI-JLO) break;
    }
  }
  else if (params.ptyp == FILDWN){
    for (K = 2; K <= nk-1; K++){             /* K is potential temperature */
      kk = 2*K;
      numfails = 0;
      for (J = params.j_sorlo; J <= params.j_sorhi; J++){ /* J is latitude */
	jj = 2*J+1;
	if (J == params.j_pol){
	  /* Apply u=0 polar boundary condition */
	  if (params.npolar){
	    latterm = av[jj]*dltlti*bv[jj-1]*(-MONT3D(K,J,I) + MONT3D(K,(J-1),I));
	  }
	  else{
	    latterm = av[jj]*dltlti*bv[jj+1]*( MONT3D(K,(J+1),I) - MONT3D(K,J,I));
	  }
	}
	else{
	  latterm = av[jj]*dltlti*
	    (bv[jj+1]*MONT3D(K,(J+1),I) - (bv[jj+1]+bv[jj-1])*MONT3D(K,J,I) +
	     bv[jj-1]*MONT3D(K,(J-1),I));
	}
	dmr = dthi[kk-1]*(MONT3D((K-1),J,I) - MONT3D(K,J,I));
	qfact = dc*q0[K]*exp((MONT3D(K,params.j_eq,I)-MONT3D(K,J,I))*cci);
	dmlcomp = pow(dmr,kapi) + dth[kk]*(latterm + grid.f[jj])/qfact;
	if (dmlcomp <= 0.){
	  tripped = 1;
	  numfails++;
	  MONT3D((K+1),J,I) = 1.e20;
	}
	dml = pow(dmlcomp, kappa);
	MONT3D((K+1),J,I) = MONT3D(K,J,I) - dth[kk+1]*dml;
	/* Check for success */
	if (!tripped) mqerr(J, params.j_eq, K, dltlti, cci, kapi,
			    dc, av, bv, dthi, mont3d, q0, params);
      }
      if (numfails >= JHI-JLO) break;
    }
  }
  else if (params.ptyp == FILLFT){
    for (J = params.j_pol; J != params.j_eq; J -= dj){
      jj = 2*J+1;
      numfails = 0;
      for (K = 2; K < nk; K++){
	kk = 2*K;
	if (K > 2){
	  /* Check for negative theta derivatives */
	  dmr = dthi[kk-1]*(MONT3D((K-1),J,I) - MONT3D(K,J,I));
	  if (dmr < 0.){
	    tripped = 1;
	    numfails++;
	    MONT3D(K,(J-dj),I) = 1.e20;
	  }
	}
	else{
	  dmr = dmrtop;
	  if (J != params.j_pol) MONT3D((K-1),J,I) = MONT3D(K,J,I) +
	    MONT3D((K-1),params.j_pol,I) - MONT3D(K,params.j_pol,I);
	}
	dml = dthi[kk+1]*(MONT3D(K,J,I) - MONT3D((K+1),J,I));
	if (dml < 0.){
	  tripped = 1;
	  numfails++;
	  MONT3D(K,(J-dj),I) = 1.e20;
	}
	qfact = dc*q0[K]*exp((MONT3D(K,params.j_pol,I)-MONT3D(K,J,I))*cci);
	qterm = qfact*dthi[kk]*(pow(dmr,kapi) - pow(dml,kapi)); 

	if (J == params.j_pol){
	  latterm = MONT3D(K,J,I);
	}
	else{
	  latterm = ((bv[jj+1] + bv[jj-1])*MONT3D(K,J,I) - bv[jj+dj]*MONT3D(K,(J+dj),I)) / bv[jj-dj];
	}
	MONT3D(K,(J-dj),I) = latterm - (dlt*dlt / (av[jj]*bv[jj-dj])) * (qterm + grid.f[jj]);
	/* Check for success */
	if (!tripped) mqerr(J, params.j_pol, K, dltlti, cci, kapi,
			    dc, av, bv, dthi, mont3d, q0, params); 
      }
      if (numfails >= nk-3) break;
    }
  }
  else{
    fprintf(stderr, "Invalid ptyp in mont_fill.\n\n");
    return -1;
  }

  if (!tripped){
    fprintf(stderr, "\nAll layers finished.\n");
  }
  else{
    fprintf(stderr, "\nNegative theta derivative occured.\n");
  }
  /* Free Arrays */  
  free(av);
  free(bv);
  free(dth);
  free(dthi);
  free(dththi);

  return 0;
}

/*====================== end of mont_fill() ==================================*/

/*====================== mq_init() ===========================================*/

int mq_init(double *dlt, double *dlti, double *dltlti, double *cci,
	    double *kapi, double *dc, double **av, double **bv,
	    double **dth, double **dthi, double **dththi, mqspec params){
  
  int jj, kk, nk = grid.nk;
  double re, rp, rv, Rv, lat;

  /* Initialize constants */
  *dlt = grid.dlt*DEG;
  *dlti = 1./(*dlt);
  *dltlti = (*dlti)*(*dlti);
  *cci = 1. / (params.c_q * params.c_q);
  if ((*cci) < 0){
    fprintf(stderr, "Bad cci at mq_init.\n\n");
    return -1;
  }
  *kapi = 1./(planet->kappa);
  *dc = (grid.press0)*pow((planet->cp),-(*kapi))/(planet->g);  

  /* Initialize vectors */

  *av = (double *)malloc((2*(JHI+2))*sizeof(double));
  *bv = (double *)malloc((2*(JHI+2))*sizeof(double));
  *dth = (double *)calloc((2*(nk+1)), sizeof(double));
  *dthi = (double *)calloc((2*(nk+1)), sizeof(double));
  *dththi = (double *)calloc((2*(nk+1)), sizeof(double));

  /* Compute whole-step differentials: */
  for (kk = 2*nk-1; kk >= 3; kk--){
    (*dth)[kk] = grid.theta[kk-1] - grid.theta[kk+1];
    (*dthi)[kk] = 1./((*dth)[kk]);
    (*dththi)[kk] = ((*dthi)[kk]) * ((*dthi)[kk]);
  }

  /* Do the latitude vectors. */
  re = planet->re;
  rp = planet->rp;
  for (jj = 2*JLO+1; jj <= 2*JHI+1; jj++){
    lat = grid.lat[jj]*DEG;
    /* These next three formulas from epic_functions.c: */
    rv = re/sqrt( 1.+pow(rp/re*tan(lat),2.) );
    Rv = rv/( cos(lat)*( pow(sin(lat),2.)+
			pow(re/rp*cos(lat),(double)2.)));
    (*av)[jj] = 1./(rv*Rv);
    (*bv)[jj] = rv/(grid.f[jj]*Rv);
  }
  return 0;
}

/*====================== end of mq_init() ====================================*/

/*====================== mqerr() =============================================*/

int mqerr(int J, int J0, int K, double dltlti, double cci, double kapi,
	  double dc, double *av, double *bv, double *dthi, double *mont3d,
	  double *q0, mqspec params){

  /* This function is for debugging purposes.  Call it for a particular interior
     (J,K) and see if the M(q) equation has been satisfied there.  Uses same
     tolerance defined in mont_sor.
     J0 Specifies the origin for q0.  */

  int I=ILO, jj = 2*J+1, kk = 2*K;
  double dmr, dml, qfact, qterm, latterm, e;

  if (J >= JHI || J <= JLO || K <= 1 || K >= grid.nk) return 0;
  dmr = dthi[kk-1]*(MONT3D((K-1),J,I) - MONT3D(K,J,I));
  dml = dthi[kk+1]*(MONT3D(K,J,I) - MONT3D((K+1),J,I));
  qfact = dc*q0[K]*exp((MONT3D(K,J0,I)-MONT3D(K,J,I))*cci);
  qterm = qfact*dthi[kk]*(pow(dmr,kapi) - pow(dml,kapi)); 
  latterm = av[jj]*dltlti*
    (bv[jj+1]*MONT3D(K,(J+1),I) - (bv[jj+1]+bv[jj-1])*MONT3D(K,J,I) +
     bv[jj-1]*MONT3D(K,(J-1),I));
  /* Calculate error */    
  e = latterm + qterm + grid.f[jj];
  if (!(fabs(e) < TOL)){
    fprintf(stderr, "Error above tolerance at J,K = %d, %d: %e\n", J, K, e);
  }
}

/*====================== end of mqerr() ======================================*/

/*====================== outm() ==============================================*/

void outm(int j0,int j1,int j12,int k0,int k1,double *mont3d) {
  /* 
   * Display M rectangle - assumes M has been calculated. 
   */
  int 
    K,J,I,
    swap,l;

  I   = ILO;
  j12 = abs(j12);
  if (j1 < j0){
    swap = j1;
    j1   = j0;
    j0   = swap;
  }
  if (k1 < k0){
    swap = k1;
    k1   = k0;
    k0   = swap;
  }
  fprintf(stderr, "\nFinal M field returned to initial:\n\n");
  fprintf(stderr, "\t");
  for (J = j0; J <= j1; J += j12){
    fprintf(stderr, "%.2f\t", grid.lat[2*J+1]);
  }
  fprintf(stderr, "\n\n");
  for (K = k0; K <= k1; K++){
    fprintf(stderr, "%.2f\t", grid.theta[2*K]);
    l=0;
    for (J = j0; J <= j1; J += j12){
      fprintf(stderr, "%.0f\t", MONT3D(K,J,I));
      l++;
    }
    fprintf(stderr, "\n");
    if (l>20) fprintf(stderr, "\n");
  }
}

/*====================== end of outm() =======================================*/

/*====================== outp() ==============================================*/

void outp(int j0,int j1,int j12,int k0,int k1) {
  /* 
   * Display P rectangle - assumes P has been calculated. 
   */
  int 
    K,J,I,
    swap,l;

  I   = ILO;
  j12 = abs(j12);
  if (j1 < j0){
    swap = j1;
    j1   = j0;
    j0   = swap;
  }
  if (k1 < k0){
    swap = k1;
    k1   = k0;
    k0   = swap;
  }
  k1--;

  fprintf(stderr, "\nFinal P field:\n\n");
  fprintf(stderr, "\t");
  for (J = j0; J <= j1; J += j12){
    fprintf(stderr, "%.2f\t", grid.lat[2*J+1]);
  }
  fprintf(stderr, "\n\n");
  for (K = k0; K <= k1; K++){
    fprintf(stderr, "%.2f\t", grid.theta[2*K+1]);
    l = 0;
    for (J = j0; J <= j1; J += j12){
      fprintf(stderr, "%.0f\t", P(K,J,I));
      l++;
    }
    fprintf(stderr, "\n");
    if (l>20) fprintf(stderr, "\n");
  }
}

/*====================== end of outp() =======================================*/

/*====================== outh() ==============================================*/

void outh(int j0,int j1,int j12,int k0,int k1) {
  /* 
   * Display H rectangle - assumes P (not H) has been calculated. 
   */
  int 
    K,J,I,
    swap,l;
  double 
    h;

  I   = ILO;
  j12 = abs(j12);
  if (j1 < j0){
    swap = j1;
    j1   = j0;
    j0   = swap;
  }
  if (k1 < k0){
    swap = k1;
    k1   = k0;
    k0   = swap;
  }
  k1--;
  fprintf(stderr, "\nFinal H field inside layers:\n\n");
  fprintf(stderr, "\t");
  for (J = j0; J <= j1; J += j12){
    fprintf(stderr, "%.2f\t", grid.lat[2*J+1]);
  }
  fprintf(stderr, "\n\n");
  for (K = k0; K <= k1; K++){
    fprintf(stderr, "%.2f\t", grid.theta[2*K+1]);
    l=0;
    for (J = j0; J <= j1; J += j12){
      h = get_h(planet,2*K,J,I);
      fprintf(stderr, "%.1f\t",h);
      l++;
    }
    fprintf(stderr, "\n");
    if (l>20) fprintf(stderr, "\n");
  }
}

/*====================== end of outh() =======================================*/

/*====================== outu() ==============================================*/

void outu(int j0,int j1,int j12,int k0,int k1) {
  /* 
   * Display U rectangle - assumes U has been calculated. 
   */
  int 
    K,J,I,
    swap,l;

  I   = ILO;
  j12 = abs(j12);
  if (j1 < j0){
    swap = j1;
    j1   = j0;
    j0   = swap;
  }
  if (k1 < k0){
    swap = k1;
    k1   = k0;
    k0   = swap;
  }
  fprintf(stderr, "\nFinal u field:\n\n");
  fprintf(stderr, "\t");
  for (J = j0; J <= j1; J += j12){
    fprintf(stderr, "%.2f\t", grid.lat[2*J+1]);
  }
  fprintf(stderr, "\n\n");
  for (K = k0; K <= k1; K++){
    fprintf(stderr, "%.2f\t", grid.theta[2*K]);
    l=0;
    for (J = j0; J <= j1; J += j12){
      fprintf(stderr, "%.2f\t", U(K,J,I));
      l++;
    }
    fprintf(stderr, "\n");
    if (l>20) fprintf(stderr, "\n");
  }
}

/*====================== end of outu() =======================================*/

/* * * * * * * * * * * * end of epic_init_funcs.c * * * * * * * * * * * * * * */









