/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * epic_initial.c  * * * * * * * * * * * * * * * * * * * * * 
 *                                                                           *
 *  Timothy E. Dowling                                                       *
 *                                                                           *
 *  Creates initial epic.nc input file for the EPIC atmospheric model.       *
 *                                                                           *
 *  The Hsu and Arakawa (1990) horizontal differencing scheme                *
 *  uses a staggered grid called the C grid.                                 *
 *                                                                           *
 *  For the globe geometry, the numbering is:                                *
 *                                                                           *
 *                            -180 deg             +180 deg                  *
 *                               |                   |                       *
 *       j = nj+1   -- q----0----q----0----q----0----q -- north pole         *
 *                     |    :    |    :    |    :    |                       *
 *                     |    :    |    :    |    :    |                       *
 *       j = nj+1/2 -- u....p....u....p....u....p....u                       *
 *                     |    :    |    :    |    :    |                       *
 *       j = nj     -- q----v----q----v----q----v----q                       *
 *                     |    :    |    :    |    :    |                       *
 *                     u....p....u11..p11..u....p....u -- equator            *
 *                     |    :    |    :    |    :    |                       *
 *       j = 1      -- q----v----q11--v11--q----v----q                       *
 *                     |    :    |    :    |    :    |                       *
 *       j = 1/2    -- u....p....u....p....u....p....u                       *
 *                     |    :    |    :    |    :    |                       *
 *                     |    :    |    :    |    :    |                       *
 *       j = 0      -- q----0----q----0----q----0----q -- south pole         *
 *                     |         |         |         |                       *
 *                   i = 0     i = 1     i = ni    i = ni+1                  *
 *                                                                           *
 *   The Hsu and Arakawa (1990) vertical differencing scheme                 *
 *   uses potential temperature, theta, for the vertical coordinate.         *
 *   The horizontal velocities, u and v, and the pressure, press,            *
 *   are staggered in the vertical.  The layer thickness, h, is              *
 *   -dp/d_theta. For the bottom layer (k = nk) of a gas-giant planet        *
 *   the horizontal velocities and the Montgomery potential, mont,           *
 *   are specified:                                                          *
 *                                                                           *
 *                     (outer space)                                         *
 *                                                                           *
 *     k = 1/2    ........ p = 0 ...... theta[1] = infinity                  *
 *                                                                           *
 *     k = 1              u, v, h       theta[2] = (1+kappa)*theta_top       *
 *                                                                           *
 *     k = 3/2    ......... p ......... theta[3] = theta_top                 *
 *                                                                           *
 *     k = nk-1           u, v, h       theta[2*nk-2] = geometric mean       *
 *                                                                           *
 *     k = nk-1/2 ......... p ......... theta[2*nk-1] = theta_bot            *
 *                                                                           *
 *     k = nk           u, v, mont      theta[2*nk  ] = theta_bot            *
 *                      specified                                            *
 *                                                                           *
 *            (interior of gas-giant planet)                                 *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

extern chem_element
  *Element;

/*
 *  Data from the books "Venus," "Mars," "Saturn," "Uranus," 
 *  University of Arizona Press; Lindal et al (1983, Icarus) for Titan. 
 *
 *  NOTE: planet->cp and planet->kappa are reassigned according to the value of cpr 
 *  returned from the initialization routine thermo_setup(), in order to maintain
 *  self-consistency with the thermodynamic functions.
 *
 *  {name,class,
 *   re,rp,omega, 
 *   cp,rgas,kappa=rgas/cp,g,
 *   x_he,x_h2,x_3,
 *   a[AU],e,orbit_period[yrs]}
 */
 planetspec 
   /* Venus numbers are relevant to stratosphere */
   venus   = {"venus","terrestrial",
               6052.e+3,6052.e+3,2.99e-7,
               860.,191.,0.222,8.80,
               0.,0.,1.,
               0.72333,0.00675,.61521},
   earth   = {"earth","terrestrial",
               6378.e+3,6357.e+3,7.294e-5,
               1000.,287.1,0.287,9.81,
               0.,0.,1.,
               1.000,0.0167,1.00004},
   mars    = {"mars","terrestrial",
               3394.e+3,3378.e+3,7.088e-5,
               860.,192.,0.223,3.72,
               0.,0.,1.,
               1.524,0.093,1.88089},
   jupiter = {"jupiter","gas-giant",
               71492.e+3,66852.e+3,1.759e-4, 
               11290.,3500.,.31,22.88,
               0.136,0.864,0.,
               5.20,0.0481,11.8623},
   hotjupiter = {"hotjupiter","gas-giant",
               1e8,1e8,2.1e-5, 
               11290.,3500.,.31,9.5,
               0.136,0.864,0.,
               0.05,0.0,0.01023},
   saturn  = {"saturn","gas-giant",
               60330.e+3,54180.e+3,1.638e-4, 
               8667.,3900.,.45,10.50,
               0.04,0.96,0.,
               9.573,0.052,29.458},
   titan   = {"titan","terrestrial",
               2575.e+3,2575.e+3,4.56e-6,
               1069.,297.,.278,1.35,
               0.,0.,1.,
               9.573,0.052,29.458},
   uranus  = {"uranus","gas-giant",
               25560.e+3,24972.e+3,1.013e-4, 
               9280.,3480.,.375,7.77,
               0.15,0.85,0.,
               19.32,0.046,84.01},
   neptune = {"neptune","gas-giant",
               24764.e+3,24343.e+3,1.083e-4, 
               9280.,3480.,.375,11.00,
               0.19,0.81,0.,
               30.28,0.008,164.79},
   triton  = {"triton","terrestrial",
               1352.6e+3,1352.6e+3,1.237e-5,
               1344.,297.,.221,.78,
               0.,0.,1.,
               30.28,0.008,164.79},
   shwater = {"sw","gas-giant",
               1.,1.,1.,
               1.,1.,1.,1.,
               0.,0.,1.,
               1.,1.,1.};

init_defaultspec 
  defaults;

/*
 * Function prototypes:
 */
void read_defaults(init_defaultspec *def);

void write_defaults(init_defaultspec *def);

void init_with_t(planetspec *planet,
                 double      ntp,
                 double     *pdat,
                 double     *tdat,
                 double     *t_y2,
                 double     *uamp,
                 double     (*uzonal)(double));

void init_with_qm(init_defaultspec *def,
                  double           (*uzonal)(double));

void init_with_u(planetspec *planet,
                 double     *uamp,
                 double     (*uzonal)(double));

void init_with_p_avg(void);

void init_fpara_as_fpe(planetspec *planet);

void init_v_by_continuity(planetspec *planet);

void init_viscosity(planetspec       *planet,
                    init_defaultspec *def);

void new_init_humidity(planetspec       *planet,
                   init_defaultspec *def);         

void set_uamp(int     uz_type,
              double *uamp,
              double (*uzonal)(double));

void set_u_spinup(int     uz_type,
                  double  *uamp,
                  double  (*uzonal)(double));  

/*======================= main() ============================================*/

int main(int   argc,
         char *argv[])
{
  char   
    header[N_STR],       /*  header line for reading t_vs_p             */
    system_id[N_STR],    /*  Name of planet or system to study          */
    outfile[FILE_STR],   /*  used to output v,w data                    */
    sflag[16];           /*  string to hold command-line flags          */
  int    
    spacing_type,        /*  distance between layers choice             */
    geostr_init,         /*  geostrophic initialization                 */
    uz_type,             /*  vertical wind profile choice               */
    kk,jj,ii,index,      /*  utility indices                            */
    K,J,I,               /*  counters                                   */
    ntp,                 /*  full length of t_vs_p data file            */ 
    floor_tp,            /*  bottom of shortened t_vs_p data            */
    ceiling_tp,          /*  top+1 of shortened t_vs_p data             */
    flag,itmp,count,
    chem_on,
    sw=0;
  double  
    (*uzonal)(double),   /*  Pointer for zonal-wind function            */
    u_ven(double),       /*  Venus wind profile                         */
    u_jup(double),       /*  Jupiter cloud-top zonal-wind profile       */
    u_sat(double),       /*  Saturn      "                              */
    u_ura(double),       /*  Uranus      "                              */
    u_nep(double),       /*  Neptune     "                              */
    u_tri(double),       /*  Triton dummy u(y)                          */
    u_nul(double);       /*  returns 0.                                 */
  double
    *mont3d,             /*  montgomery potential (k,j,i)               */
    *q,                  /*  potential vorticity                        */
    d_th1,               /*  delta theta, used to compute h from p      */
    tmp,tmp2,            /*  temporary storage                          */
    *uamp,               /*  amplitude for u_spinup                     */
    max_temp=-DBL_MAX,   /*  used for plotting temperatures             */
    min_temp= DBL_MAX,   /*  used for plotting temperatures             */
    neg_log_p;           /*  used to estimate qmin, qmax                */
  double  
    *pdat,*tdat,         /*  t_vs_p data                                */
    *thetadat,           /*  theta corresponding to t_vs_p data         */ 
    *p_y2,th_d,          /*  used in cubic-spline of p(theta)           */
    *th_y2,p_d,          /*  used in cubic-spline of theta(p)           */
    *t_y2;               /*  used in cubic-spline of kappa              */
  double
    fpara,pressure,temperature,fgibb,fpe,uoup,cpr,
    theta,theta_ortho,theta_para,
    kappap1,kappa_inv;
  FILE   
    *t_vs_p;             /*  pointer to t_vs_p data file                */
  /* 
   * The following are part of DEBUG_MILESTONE statements: 
   */
  int
    idbms=0;
  char
    dbmsname[]="epic_initial";


#if defined(EPIC_MPI)
  MPI_Init(&argc,&argv);
  para.comm = MPI_COMM_WORLD;
  MPI_Errhandler_set(para.comm,MPI_ERRORS_RETURN);
  MPI_Comm_rank(para.comm,&para.iamnode);
  MPI_Comm_size(para.comm,&para.nproc);
  para.ndim = NINT(log((double)para.nproc)/log(2.));;
#endif
  
  /* Model version number: */
  grid.data_version = 3.85;
  /* Declare copyright: */
  declare_copyright();

  /* 
   * Interpret command-line arguments: 
   */
  if (argc > 1) {
    for (count = 1; count < argc; count++) {
      sscanf(argv[count],"%s",sflag);
      if (strcmp(sflag,"-help") == 0 ||
          strcmp(sflag,"-h")    == 0) {
        /* Print help, exit: */
        system("more "EPIC_PATH"/help/epic_initial.help");
        exit(1);
      }
      else {
        fprintf(stderr,"Unrecognized epic_initial command-line flag: %s \n",sflag);
        exit(1);
      }
    }
  }

  /*
   * Fill in variable, tendency, and units name lists:
   */
  set_name_lists();

  /* 
   * Use 1 bar for the reference pressure.  
   * NOTE: This value is assumed in fortran thermodynamics subroutines.
   */
  grid.press0 = 1.e+5;  /* 1 bar in mks */

  /*
   *  Read default parameter settings:
   */
  read_defaults(&defaults);

  if (IAMNODE == NODE0) {
    /*  Print welcome statement: */
    fprintf(stderr,"\n");    
    fprintf(stderr,"        ______/      __       __   ___/        __     \n");
    fprintf(stderr,"       /            /     /       /          /   __/ \n");  
    fprintf(stderr,"      ______/      ______/       /          /       \n");
    fprintf(stderr,"     /            /             /          (       \n");    
    fprintf(stderr," _________/   ___/        _________/    \\_______/\n");
    fprintf(stderr," \n\n             WELCOME TO THE EPIC MODEL   \n\n");
    fprintf(stderr,"           Version: %4.2f\n",grid.data_version);
    fprintf(stderr,"Working geometries: globe, f-plane \n");
    fprintf(stderr,"   Working planets: Jupiter, Saturn, Uranus, Neptune \n");
    fprintf(stderr,"                    Venus,Triton, hotjupiter \n");
    fprintf(stderr,"Shallow-water case: sw \n");
    fprintf(stderr,"Under construction: Earth, Mars, Titan \n\n");
  }
  
  /*
   * Choose system:
   */
  ii = 0;
  while (ii == 0) {
    input_string("Choose system to initialize \n",defaults.system_id,system_id);
    /* Make first letter lower case: */
    system_id[0] = (char)tolower((int)system_id[0]);

    /* 
     * Inquire if starting with observed zonal winds or zero:
     */
    defaults.u_init = input_int("Use zero or cloud-top zonal winds? [0 or 1]\n",
                      defaults.u_init); 

    /* Scan for implemented systems: */
    ii = 1;
    if      (strcmp(system_id,"venus")   == 0) {
      planet                  = &venus;
      uzonal                  = u_ven;
      grid.newt_cool_on       = TRUE;
      grid.thermal_conduct_on = FALSE;
    }
    else if (strcmp(system_id,"jupiter") == 0) {
      planet                  = &jupiter;
      uzonal                  = u_jup;
      grid.newt_cool_on       = TRUE;
      grid.thermal_conduct_on = FALSE;
    }
    else if (strcmp(system_id,"hotjupiter") == 0) {
      planet                  = &hotjupiter;
      uzonal                  = u_jup;
      grid.newt_cool_on       = TRUE;
      grid.thermal_conduct_on = FALSE;
    }
    else if (strcmp(system_id,"saturn")  == 0) {
      planet                  = &saturn;
      uzonal                  = u_sat;
      grid.newt_cool_on       = TRUE;
      grid.thermal_conduct_on = FALSE;
    }
    else if (strcmp(system_id,"uranus")  == 0) {
      planet                  = &uranus;
      uzonal                  = u_ura;   
      grid.newt_cool_on       = TRUE;
      grid.thermal_conduct_on = FALSE;
      if (defaults.u_init) {
        grid.aux_a = input_int("Use even or odd zonal-wind profile? [0 or 1]\n",0);
      }
    }
    else if (strcmp(system_id,"neptune") == 0) {
      planet                  = &neptune;
      uzonal                  = u_nep;
      grid.newt_cool_on       = TRUE;
      grid.thermal_conduct_on = FALSE;
    }
    else if (strcmp(system_id,"triton") == 0) {
      planet                  = &triton;
      uzonal                  = u_tri;
      /* Thermal conduction dominates: */
      grid.newt_cool_on       = FALSE; 
      grid.thermal_conduct_on = TRUE;
    }
    else if (strcmp(system_id,"sw")      == 0) {
      /* 
       * Special shallow-water case.  
       * Choose parameters such that p output corresponds to gh in a 
       * 1-1/2 layer (nk = 2) shallow-water model.
       */
      sw                      = 1;
      planet                  = &shwater;
      planet->g               = 1.;
      grid.press0             = 1.;
      planet->kappa           = 1.;
      planet->cp              = planet->g*grid.press0;
      planet->rgas            = planet->cp;
      grid.newt_cool_on       = FALSE;
      grid.thermal_conduct_on = FALSE;
    }
    else {
      /* unimplemented system */
      ii = 0;
      strcpy(defaults.system_id,"Jupiter");
      system_id[0] = (char)toupper((int)system_id[0]);
      if (IAMNODE == 0) {
        fprintf(stderr,"\n\"%s\" is not defined.\n\n", system_id);
      }
    }
    /* Use zero zonal winds if requested above */
    if (!defaults.u_init) {
      uzonal = u_nul;
    }
  }
  /*
   * Choose equation of state:
   */
  input_string("Choose equation of state [ideal,virial]\n",defaults.eos,grid.eos);

  if (!sw) {
    /*
     * Inquire about initializing with T(y,p), T(p), or q(M):
     */
    sprintf(header,"Initialize with T(p),T(lat,p),q(M)? [%1d=T(p),%1d=T(lat,p),%1d=q(M)]\n",
                    INIT_TP,INIT_T,INIT_QM);
    defaults.init_type = grid.init_type = input_int(header,defaults.init_type);
    if (grid.init_type == INIT_T) {
      /* 
       * Call t_yp() to echo range limits on lat, p, 
       * and set defaults.pbot,defaults.ptop: 
       */
      t_yp(0,0,0,&defaults);
    }
    else {
      /* 
       * Input ptop,pbot (external units are mbars, internal are mks):
       */
      defaults.ptop = 100.*input_double("Target at-rest pressure at k = 1 1/2 (model's top) [mbar]\n",
                                        defaults.ptop/100.);
      defaults.pbot = 100.*input_double("Target at-rest pressure at k = nk-1/2 (model's bottom) [mbar]\n",
                                        defaults.pbot/100.);
    }
  }

  /*
   * Choose geometry:
   */
  input_string("Choose geometry \n",defaults.geometry,grid.geometry);
  if (strcmp(grid.geometry,"f-plane") == 0) {
    grid.wrap[2] = 0;
    grid.wrap[0] = 1;
    grid.pad[2]  = 0;
    grid.pad[0]  = 1;
    if (sw) {
      planet->omega = .5*(double)input_float("Input f [1/s]\n",0.);
      grid.f_plane_lat0=defaults.f_plane_lat0=90.;
    }
    else {
      grid.f_plane_lat0=defaults.f_plane_lat0
        = input_double("Latitude of f-plane [deg] \n",defaults.f_plane_lat0);
    }
    input_string("Choose mapping: cartesian or polar \n",
                 defaults.f_plane_map,grid.f_plane_map);
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      grid.wrap[1] = 1;
      grid.pad[1]  = 1;
      grid.jlo     = 1;
      grid.jfirst  = 1;
      grid.f_plane_half_width=defaults.f_plane_half_width
        = input_double("Half-width [km] \n",defaults.f_plane_half_width);
      /* convert km to m */
      grid.f_plane_half_width *= 1000.;
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      grid.wrap[1] = 0;
      grid.pad[1]  = 1;
      grid.jlo     = 0;
      grid.jfirst  = 1;
      if (sw) {
        grid.f_plane_half_width=defaults.f_plane_half_width
          = .5*(double)input_float("Radius \n",1000.);
      }
      else {
        grid.f_plane_half_width=defaults.f_plane_half_width
          = input_double("Radius [km] \n",defaults.f_plane_half_width);
        /* convert km to m */
        grid.f_plane_half_width *= 1000.;
      }
    }
    else {
      fprintf(stderr,"Unrecognized f-plane mapping. \n");
      exit(1);
    }
  }
  else if (strcmp(grid.geometry,"globe") == 0) {
    grid.wrap[2] = 0;
    grid.wrap[1] = 0;
    grid.wrap[0] = 1;
    grid.pad[2]  = 0;
    grid.pad[1]  = 1;
    grid.pad[0]  = 1;
    grid.jlo     = 0;
    grid.jfirst  = 1;

    if (sw) {
      planet->re = planet->rp = (double)input_float("Radius\n",1000.);
      planet->omega           = (double)input_float("Omega\n",1.);
    }

    grid.globe_lonbot=defaults.globe_lonbot
      = input_double("Lowest longitude [deg] \n",defaults.globe_lonbot);
    if (grid.globe_lonbot < -180.) {
      fprintf(stderr,"lonbot must be >= -180.  Setting lonbot to -180. \n");
      grid.globe_lonbot = defaults.globe_lonbot = -180.;
    }
    grid.globe_lontop=defaults.globe_lontop
      = input_double("Highest longitude [deg] \n",defaults.globe_lontop);
    if (grid.globe_lontop > 180.) {
      fprintf(stderr,"lontop must be <= 180. Setting lontop to 180. \n");
      grid.globe_lontop = defaults.globe_lontop = 180.;
    }

    grid.globe_latbot=defaults.globe_latbot
      = input_double("Lowest latitude [deg] \n",defaults.globe_latbot);
    if (grid.globe_latbot < -90.) {
      fprintf(stderr,"latbot must be >= -90.  Setting latbot to -90. \n");
      grid.globe_latbot = defaults.globe_latbot = -90.;
    }
    grid.globe_lattop=defaults.globe_lattop
      = input_double("Highest latitude [deg] \n",defaults.globe_lattop);
    if (grid.globe_lattop > 90.) {
      fprintf(stderr,"lattop must be <= 90. Setting lattop to 90. \n");
      grid.globe_lattop = defaults.globe_lattop = 90.;
    }
  }
  else {
    fprintf(stderr,"initial: unrecognized geometry: %s \n",grid.geometry);
    exit(1);
  }

  if (!sw) {
    /*
     * Set up thermodynamics subroutines:
     */
    thermo_setup(planet,&cpr);
    /* Assign thermodynamics function's reference cpr to planet->cp */
    planet->cp    = cpr*planet->rgas;
    planet->kappa = 1./cpr;
  }

  /*
   *  Inquire about prognostic variables:
   */
  fprintf(stderr,"\n Core prognostic variables: \n");
  for (index = U_INDEX; index <= P_INDEX; index++) {
    fprintf(stderr,"  %2d. %-s \n",index,var.chem_name[index]);
  }
  /* 
   * Turn on u,v,p and "dry_air": 
   */
  var.chem_on[U_INDEX]       = CHEM_ACTIVE;
  var.chem_on[V_INDEX]       = CHEM_ACTIVE;
  var.chem_on[P_INDEX]       = CHEM_ACTIVE;
  var.chem_on[DRY_AIR_INDEX] = CHEM_ACTIVE;

  fprintf(stderr," Optional prognostic variables: \n");
  for (index = P_INDEX+1; index < FIRST_CHEM_PRODUCT; index++) {
    fprintf(stderr,"  %2d. %-s \n",index,var.chem_name[index]);
  }
  fprintf(stderr,"\n");

  defaults.add_optional_var  
    = input_int("Add optional variable(s)? [1=yes, 0=no]\n",defaults.add_optional_var);
  if (defaults.add_optional_var == TRUE) {
    index = P_INDEX;
  }
  else {
    index = -1;
  }
  while (index >= 0) {
    /* Wrap around to the beginning of the list after end: */
    index = (index < FIRST_CHEM_PRODUCT) ? index+1 : P_INDEX+1;
    index = input_int("Choose variable to add [-1 to quit loop]\n",index);
    if (index < 0) break;
    if (index <= P_INDEX || index >= FIRST_CHEM_PRODUCT) {
      fprintf(stderr,"Index %d out of valid range [%d,%d]\n",
                      index,P_INDEX+1,MAX_NVARS-1);
      index = P_INDEX;
    }
    else {
      sprintf(header,"Thermodynamically active, passive, or don't add [%1d,%1d,%1d]?\n",
                      CHEM_ACTIVE,CHEM_PASSIVE,CHEM_OFF);
      chem_on = input_int(header,defaults.chem_on[index]);
      if (chem_on == CHEM_ACTIVE || chem_on == CHEM_PASSIVE || chem_on == CHEM_OFF) {
        defaults.chem_on[index] = var.chem_on[index] = chem_on;
      }
      else {
        fprintf(stderr,"Valid choices are %1d,%1d,%1d \n",
                       CHEM_ACTIVE,CHEM_PASSIVE,CHEM_OFF);
        index--;
      }
    }
  }
  /*
   * Inquire about chemical products:
   */
  if (defaults.add_optional_var == TRUE) {
    fprintf(stderr,"\n Optional chemical products: \n");
    for (index = FIRST_CHEM_PRODUCT; index <= LAST_HUMIDITY; index++) {
      if (index == NH_4SH_INDEX) {
        if (!var.chem_on[NH_3_INDEX] || !var.chem_on[H_2S_INDEX]) {
          /* Can't form NH_4SH. */
          defaults.chem_on[index] = var.chem_on[index] = CHEM_OFF;
          continue;
        }
      }
      fprintf(stderr,"  %2d. %-s \n",index,var.chem_name[index]);
    }
    fprintf(stderr,"\n");

    index = FIRST_CHEM_PRODUCT-1;
    while (index >= 0) {
      /* Wrap around to the beginning of the list after end: */
      index = (index < LAST_HUMIDITY) ? index+1 : FIRST_CHEM_PRODUCT;
      index = input_int("Choose chemical product to add [-1 to quit loop]\n",index);
      if (index < 0) break;
      if (index < FIRST_CHEM_PRODUCT || index > LAST_HUMIDITY) {
        fprintf(stderr,"Index %d out of valid range [%d,%d]\n",
                        index,FIRST_CHEM_PRODUCT,LAST_HUMIDITY);
        index = FIRST_CHEM_PRODUCT;
      }
      else {
        sprintf(header,"Thermodynamically active, passive, or don't add [%1d,%1d,%1d]?\n",
                        CHEM_ACTIVE,CHEM_PASSIVE,CHEM_OFF);
        chem_on = input_int(header,defaults.chem_on[index]);
        if (chem_on == CHEM_ACTIVE || chem_on == CHEM_PASSIVE || chem_on == CHEM_OFF) {
          defaults.chem_on[index] = var.chem_on[index] = chem_on;
        }
        else {
          fprintf(stderr,"Valid choices are %1d,%1d,%1d \n",
                         CHEM_ACTIVE,CHEM_PASSIVE,CHEM_OFF);
          index--;
        }
      }
    }
    fprintf(stderr,"\n");
  }

  /* 
   * Special-case parameters: 
   */
  if (var.chem_on[FPARA_INDEX] == CHEM_ACTIVE) {
    defaults.time_fp = 
      input_double("Ortho-para conversion time at 1 bar [s, nominal = 3.e+8]?\n",defaults.time_fp);
    var.time_fp_bar = defaults.time_fp*1000.*100.;
  }


  if (sw) {
    grid.nk = 2;
  }
  else {
    grid.nk = 201;
    while (grid.nk > 200) {
      defaults.nk = grid.nk = 
        input_int("Input the number of vertical layers, nk\n", defaults.nk);
      if (grid.nk > 200) {
        if (IAMNODE == 0) {
          fprintf(stderr,"WARNING: nk must be <= 200. The restriction is in avs_epic.h \n");
        }
      }
    }
  }

  /*
   * Set grid.klast_active (a.k.a. KLAST_ACTIVE):
   */
  if (strcmp(planet->class,"gas-giant") == 0) {
    grid.klast_active = grid.nk-1;
  }
  else if (strcmp(planet->class,"terrestrial") == 0) {
    grid.klast_active = grid.nk;
  }
  else {
    fprintf(stderr,"Error in epic_initial, planet->class = %s unrecognized.\n",
                    planet->class);
  }

  /*
   * Inquire about vertical spacing between layers:
   */
  if (sw) {
    spacing_type = defaults.spacing_type = SPACING_SW;
  }
  else {
    if (IAMNODE == 0) {
      fprintf(stderr,"\n");
      sprintf(header,"Layer intervals even in: %1d => theta \n"
                     "                         %1d => log theta \n"
                     "                         %1d => log pressure \n",
                     SPACING_THETA,SPACING_LOGTHETA,SPACING_LOGP);
    }
    defaults.spacing_type = spacing_type = input_int(header,defaults.spacing_type);
  }

  /*
   * Inquire about number of longitude gridpoints:
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    if (IAMNODE == 0) {
      fprintf(stderr,"\nInput the number of longitude gridpoints, \n");
    }
    defaults.ni = grid.ni = 
      input_int("which must be an integer power of two\n",defaults.ni);
    grid.dln = (grid.globe_lontop-grid.globe_lonbot)/(grid.ni);
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      defaults.ni = grid.ni =
        input_int("\nInput the number of gridpoints on a side\n",defaults.ni);
      grid.dln = 360./(grid.ni);
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      if (IAMNODE == 0) {
        fprintf(stderr,"\nInput the number of longitude gridpoints, \n");
      }
      defaults.ni = grid.ni = 
        input_int("which must be an integer power of two\n",defaults.ni);
      grid.dln = 360./(grid.ni);
    }
  }

  /*
   * Inquire about number of latitude grid points:
   */
  if (strcmp(grid.geometry,"globe") == 0) {
    defaults.nj = grid.nj = 
      input_int("\nInput the number of latitude gridpoints\n",defaults.nj);
    if (grid.globe_latbot == -90. &&
        grid.globe_lattop ==  90.) {
      /* 
       * Poles are offset by extra sqrt(.5)*dlt.
       * NOTE: This is more accurate than the .5*dlt prescribed by Arakawa and Lamb.
       */
      grid.dlt = 180./((grid.nj+1)+sqrt(.5)+sqrt(.5));
    }
    else if (grid.globe_latbot == -90.) {
      grid.dlt = (grid.globe_lattop+90.)/((grid.nj+1)+sqrt(.5));
    }  
    else if (grid.globe_lattop ==  90.) {
      grid.dlt = (90.-grid.globe_latbot)/((grid.nj+1)+sqrt(.5));
    }
    else {
      grid.dlt = (grid.globe_lattop-grid.globe_latbot)/(grid.nj+1);
    }
  }
  else if (strcmp(grid.geometry,"f-plane") == 0) {
    if (strcmp(grid.f_plane_map,"cartesian") == 0) {
      defaults.nj = grid.nj = grid.ni;
      grid.dlt   = grid.dln;
    }
    else if (strcmp(grid.f_plane_map,"polar") == 0) {
      /* 
       * "latitude" runs from 0 at the edge (j = 1) 
       *  to 90 in the center (j = nj+1) */
      defaults.nj = grid.nj =
        input_int("\nInput the number of radial gridpoints\n",defaults.nj);
      grid.dlt = 90./((grid.nj+1-1)+sqrt(.5));
    }
  }

  if (!sw) {
    /* 
     * Input temperature sounding data, t_vs_p: 
     */
    ntp = read_t_vs_p(planet,0.,0.,NULL,NULL,NULL,NULL,INIT_DATA);
    /* Allocate memory: */
    pdat     = dvector(0,ntp-1);
    tdat     = dvector(0,ntp-1);
    thetadat = dvector(0,ntp-1);
    p_y2     = dvector(0,ntp-1);
    th_y2    = dvector(0,ntp-1);
    t_y2     = dvector(0,ntp-1);
    read_t_vs_p(planet,defaults.ptop,defaults.pbot,&ceiling_tp,&floor_tp,
                pdat,tdat,NONINIT_DATA);

    for (kk = 0; kk < ntp; kk++) {
      if (var.chem_on[FPARA_INDEX]) {
        /* fpe only depends on temp: */
        fpara = .25;
        return_enthalpy(planet,fpara,0,tdat[kk],&fgibb,&fpe,&uoup);
        fpara = fpe;
      }
      else {
        fpara = .25;
      }
      thetadat[kk] = return_theta(planet,fpara,pdat[kk],tdat[kk],&theta_ortho,&theta_para);

      /*  spline on -log p */
      pdat[kk] = -log(pdat[kk]);  
    }
    /* 
     * Prepare for data table interpolations.
     */
    spline(thetadat,pdat,    ntp,1.e+30,1.e+30,p_y2);
    spline(pdat,    thetadat,ntp,1.e+30,1.e+30,th_y2);
    spline(pdat,    tdat,    ntp,1.e+30,1.e+30,t_y2);
  }
  
  /*
   * Set k_sponge:
   */
  if (grid.nk >= 5) {
    grid.k_sponge = NINT((double)grid.nk/5.);
    grid.k_sponge = (grid.k_sponge > 1) ? grid.k_sponge : 1;
  }
  else {
    /* no sponge for nk <= 4 */
    grid.k_sponge = 0;
  }
  if (!sw) {
    grid.k_sponge = input_int("Input k_sponge \n",grid.k_sponge);
  }

  /*
   *  Allocate memory for initial arrays:
   */
  make_arrays();
  
  /*
   * Inquire about Newtonian cooling and Rayleigh drag:
   */
  if (grid.newt_cool_on == TRUE) {
    defaults.prandtl = grid.prandtl = 
      input_double("Input Prandtl number [t_rad/t_drag, 0. => no drag]\n",defaults.prandtl);
  }
  else {
    /* Use prandtl to hold 1./t_drag: */
    if (defaults.prandtl == 0.) {
      defaults.prandtl = 1.e+20;
    }
    defaults.prandtl = grid.prandtl = 
      input_double("Rayleigh drag timescale [1.e+20 => infinity]\n",defaults.prandtl);
    if (grid.prandtl >= 1.e+20) {
      grid.prandtl = 0.;
    }
    else if (grid.prandtl == 0.) {
      fprintf(stderr,"Error: t_drag = 0. not defined. \n");
      exit(1);
    }
    else {
      grid.prandtl = 1./grid.prandtl;
    }
  }

  /*
   * Inquire about geostrophic initialization:
   */
  if (!sw && (grid.init_type != INIT_QM)) {
    defaults.geostr_init = geostr_init = 
      input_int("Balance pressures with zonal wind?"
                " (1 => yes, 0 => no)\n",defaults.geostr_init);
  }

  uz_type = -1;
  /* 
   * Allocate memory: 
   */
  uamp = dvector(0,Kadim*Jadim-1);
  /*
   * Inquire about vertical variation of zonal wind:
   */
  if (grid.init_type == INIT_T) {
    /* 
     * When initializing with T(y,p), u equals the standard profile in the bottom layer
     * and is set using the thermal-wind equation everywhere above.
     */
    defaults.uz_type = uz_type = UZ_CONSTANT;
  }
  else if ((grid.prandtl > 0. || geostr_init == TRUE) && (grid.init_type != INIT_QM)){
    if (IAMNODE == 0) {
      fprintf(stderr,"Vertical wind profile: \n");
      fprintf(stderr,"                       %1d => Galileo Probe [above data, u/u1=(p/p1)^ln(10/9)]\n",
                      UZ_GALILEO);
      fprintf(stderr,"                       %1d => Constant Zonal Shear \n",
                      UZ_CZS);
      fprintf(stderr,"                       %1d => Gierasch Scale Height criteria \n",
                      UZ_GIERASCH);
      fprintf(stderr,"                       %1d => Constant \n",
                      UZ_CONSTANT);
      fprintf(stderr,"                       %1d => Linear in k \n",
                      UZ_LINEAR);
      fprintf(stderr,"                       %1d => Input uamp by layer \n",
                      UZ_USER);
    }
    defaults.uz_type = uz_type = 
      input_int("(add 10 to set u_spinup == constant)\n",defaults.uz_type);
  }

  /*
   *  Initialize time:
   */
  var.time[0] = 0; 
  var.time[1] = 0; 

  /* 
   * Set theta,p_avg,rgas,kappa profiles:
   */

  /* Bottom interface values: */
  if (sw) {
    grid.theta_bot           = 1./planet->g;
    grid.p_avg[KLAST_ACTIVE] = grid.press0;
    grid.mont0               = 0.;
  }
  else {
    grid.theta_bot           = thetadat[floor_tp];
    grid.p_avg[KLAST_ACTIVE] = exp(-pdat[floor_tp]);
    fpara                    = .25;
    grid.mont0               = return_enthalpy(planet,fpara,pdat[floor_tp],
                                               tdat[floor_tp],&fgibb,&fpe,&uoup);
  }

  /* p_avg[0] used to set spinup winds below */
  grid.p_avg[0]  = 0.;   

  if (sw) {
    grid.theta_top = grid.theta_bot;
    grid.p_avg[1]  = grid.press0;
  }
  else {
    if (spacing_type == SPACING_THETA) {
      grid.theta_top = thetadat[ceiling_tp-1]+
                       (thetadat[floor_tp]-thetadat[ceiling_tp-1])
                      *(1.5-1.)/(((double)(KLAST_ACTIVE+1)-.5)-1.);
      /* set p_avg[1] */

      kk = find_place_in_table(ntp,thetadat,&(grid.theta_top),&th_d);
      /* pdat is -log p */
      grid.p_avg[1] = exp(-splint(grid.theta_top,thetadat+kk, 
                                   pdat+kk,p_y2+kk,th_d));
    } 
    else if (spacing_type == SPACING_LOGTHETA) {
      grid.theta_top = log(thetadat[ceiling_tp-1])+
                       (log(thetadat[floor_tp])-log(thetadat[ceiling_tp-1]))
                      *(1.5-1.)/(((double)(KLAST_ACTIVE+1)-.5)-1.);
      grid.theta_top = exp(grid.theta_top);
      /* set p_avg[1] */

      kk = find_place_in_table(ntp,thetadat,&(grid.theta_top),&th_d);
      /* pdat is -log p */
      grid.p_avg[1] = exp(-splint(grid.theta_top,thetadat+kk, 
                                   pdat+kk,p_y2+kk,th_d));
    } 
    else if (spacing_type == SPACING_LOGP) {
      neg_log_p = pdat[ceiling_tp-1]+
                 (pdat[floor_tp]-pdat[ceiling_tp-1])
                 *(1.5-1.)/(((double)(KLAST_ACTIVE+1)-.5)-1.);

      kk             = find_place_in_table(ntp,pdat,&neg_log_p,&p_d);
      grid.theta_top = splint(neg_log_p,pdat+kk,thetadat+kk,th_y2+kk,p_d);

      /* set p_avg[1] */
      grid.p_avg[1]  = exp(-neg_log_p);
    }
  }

  grid.theta[2] = (1.+(planet->kappa))*grid.theta_top;
  grid.theta[3] = grid.theta_top;

  if (strcmp(planet->class,"gas-giant") == 0) {
    grid.theta[2*(grid.nk-1)+1] = grid.theta_bot;
    grid.theta[2*grid.nk      ] = grid.theta_bot;
  }
  else if (strcmp(planet->class,"terrestrial") == 0) {
    grid.theta[2*grid.nk+1]     = grid.theta_bot;
  }
  else {
    fprintf(stderr,"Error in epic_initial, planet->class = %s unrecognized.\n",
                   planet->class);
    exit(1);
  }

  /* 
   * Set rgas, kappa for top layer:
   */
  K = 1;
  grid.rgas[ K] = planet->rgas;
  grid.kappa[K] = planet->kappa;
  
  /* 
   * Continue in K loop over rest of layers 
   */
  for (K = 2; K <= KLAST_ACTIVE; K++) {
    if (spacing_type == SPACING_THETA) {
      grid.theta[2*K+1] = grid.theta_top-(double)(K-1)/(double)(KLAST_ACTIVE-1)
                                 *(grid.theta_top-grid.theta_bot);
      kk = find_place_in_table(ntp,thetadat,&(grid.theta[2*K+1]),&th_d);
      /* 
       * A cubic spline is too stiff in low-N regions.
       * Instead, use linear interpolation.
       */
      /*****
      grid.p_avg[K] = splint(grid.theta[2*K+1],thetadat+kk,pdat+kk,p_y2+kk,th_d);
      *****/
      grid.p_avg[K] = linint(grid.theta[2*K+1],thetadat+kk,pdat+kk,p_y2+kk,th_d);

      /* pdat is -log p */
      grid.p_avg[K] = exp(-grid.p_avg[K]);
    } 
    else if (spacing_type == SPACING_LOGTHETA) {
      grid.theta[2*K+1] = exp( log(grid.theta_top)-(double)(K-1)/(double)(KLAST_ACTIVE-1)
                                 *(log(grid.theta_top)-log(grid.theta_bot)) );
      kk = find_place_in_table(ntp,thetadat,&(grid.theta[2*K+1]),&th_d);
      /* 
       * A cubic spline is too stiff in low-N regions.
       * Instead, use linear interpolation.
       */
      /*****
      grid.p_avg[K] = splint(grid.theta[2*K+1],thetadat+kk,pdat+kk,p_y2+kk,th_d);
       *****/
      grid.p_avg[K] = linint(grid.theta[2*K+1],thetadat+kk,pdat+kk,p_y2+kk,th_d);
      /* pdat is -log p */
      grid.p_avg[K] = exp(-grid.p_avg[K]);
    }
    else if (spacing_type == SPACING_LOGP) {
      neg_log_p = (double)(K-1)/(double)(KLAST_ACTIVE-1)
                     *( log((grid.p_avg)[           1])
                       -log((grid.p_avg)[KLAST_ACTIVE]) )
                  -log((grid.p_avg)[1]);
      grid.p_avg[K] = exp(-neg_log_p);

      kk = find_place_in_table(ntp,pdat,&neg_log_p,&p_d);
      /* 
       * A cubic spline is too stiff in low-N regions and 
       * yields nonmonotonic theta.  Instead, use linear interpolation.
       */
      /*****
      grid.theta[2*K+1] = splint(neg_log_p,pdat+kk,thetadat+kk,th_y2+kk,p_d);
       *****/
      grid.theta[2*K+1] = linint(neg_log_p,pdat+kk,thetadat+kk,th_y2+kk,p_d);
    }
    grid.theta[2*K]   = sqrt(grid.theta[2*K-1]*grid.theta[2*K+1]);

    /* 
     * Set rgas, kappa for layer:
     */
    grid.rgas[ K] = planet->rgas;
    grid.kappa[K] = planet->kappa;
  }

  if (strcmp(planet->class,"gas-giant") == 0) {
    grid.rgas[ grid.nk] = grid.rgas[ grid.nk-1];
    grid.kappa[grid.nk] = grid.kappa[grid.nk-1];
  }

  /* Check for monotonic theta */
  flag = 0;
  for (kk = 2+1; kk <= 2*KLAST_ACTIVE+1; kk+=2) {
    if ((kk > 3) && (grid.theta[kk] >= grid.theta[kk-2])) {
      flag = kk;
    }
  }
  if (flag > 0) {
    if (IAMNODE == 0) {
      fprintf(stderr,"Error: theta not monotonic at k = %4.1f: \n",
                     (double)(flag)/2.);
      fprintf(stderr,"  k      theta[K]  \n");
      for (kk = 2+1; kk <= 2*KLAST_ACTIVE+1; kk+=2) {
        fprintf(stderr," %4.1f  %10.1f \n",(double)(kk)/2.,grid.theta[kk]);
      }
    }
    exit(0);
  }

  if (var.chem_on[FPARA_INDEX]) {
    /*
     * Preliminary initialization of para-hydrogen fraction, fpara.
     * Set to fpe(T), for representative T for layer.
     */
    for (K = KLO; K <= KHI; K++) {
      if (K <= KLAST_ACTIVE) {
        kappap1   = grid.kappa[K]+1.;
        kappa_inv = 1./(kappap1-1.);
        if (K > 1) {
          pressure  =  pow((pow(grid.p_avg[K],kappap1)-pow(grid.p_avg[K-1],kappap1))/
                           (kappap1*(grid.p_avg[K]-grid.p_avg[K-1])),kappa_inv);
        }
        else {
          /* take pressure at k=1 layer bottom to stay in range of pdat */
          pressure  = grid.p_avg[K];
        }
        neg_log_p   = -log(pressure);
        kk          = find_place_in_table(ntp,pdat,&neg_log_p,&p_d);
        temperature = splint(neg_log_p,pdat+kk,tdat+kk,t_y2+kk,p_d);
      }
      else {
        /* use bottom temperature */
        temperature = tdat[floor_tp];
      }
      /* the choice of fpara doesn't matter for fpe */
      fpara = .25;
      return_enthalpy(planet,fpara,0,temperature,&fgibb,&fpe,&uoup);
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          FPARA(K,J,I) = fpe;
        }
      }
      BC2D(&(FPARA(KLO,JLO,ILO)),FPARA_INDEX,K);
    }
  }

  /*
   *  Set spinup zonal-wind profile:
   */
  if (grid.prandtl > 0. || geostr_init == TRUE) {
    set_uamp(uz_type,uamp,uzonal);
    set_u_spinup(uz_type,uamp,uzonal);
  }

  /*
   *  Initialize p and u.
   */
  if (sw) {
    input_sw_arrays();
  }
  else if (grid.init_type == INIT_T) {
    /*
     * Initialize with T(lat,p) data:
     */
    init_with_t(planet,ntp,pdat,tdat,t_y2,uamp,uzonal);
  }
  else if (grid.init_type == INIT_QM) {
    /* 
     * Initialize by inverting q(M):
     */
    init_with_qm(&defaults,uzonal);
  }
  else if (geostr_init == FALSE) {
    /*
     * Set P(K,J,I) = P_AVG(K) and leave U(K,J,I) = 0.;
     */
    init_with_p_avg();
  }
  else {
    init_with_u(planet,uamp,uzonal); 
  }

  /*
   * Store heavily used diagnostic variables for subsequent calculations:
   */
  if (IAMNODE == NODE0) {
    fprintf(stderr,"Calculating and storing heavily used diagnostic variables...\n");
  }
  store_temp_dens_p(planet,JLO,JHI,ILO,IHI);

  /*
   * Update U_SPINUP(K,J,I):
   */
  set_u_spinup(uz_type,NULL,NULL);

  /*
   * Initialize fpara as fpe:
   */
  if (var.chem_on[FPARA_INDEX] != CHEM_OFF & grid.init_type == INIT_TP) {
    init_fpara_as_fpe(planet);
  }

  if (grid.ni == 1 && !sw) {
    /*
     * Initialize v to satisfy the steady continuity equation:  
     * d(vh)/dy = -d(wh)/dtheta.
     */
    init_v_by_continuity(planet);
  }

#if !defined(EPIC_MPI)
  /* 
   * Write out diabatic circulation data to a file in tmp directory: 
   */
  write_ep(planet);
#endif

  /*
   * Initialize specific humidities:
   */
  init_humidity(planet,&defaults); 


  /* 
   * Input timestep: 
   */
  defaults.dt = grid.dt = 
    input_int("Input timestep, dt [s, int to prevent roundoff]\n", defaults.dt);

  /* 
   * Initialize viscosity coefficients: 
   */
  init_viscosity(planet,&defaults);

  if (sw) {
    defaults.hasten = grid.hasten = 1.;
  }
  else {
    defaults.hasten = grid.hasten =
      input_double("Input artificial hasten factor (1 = normal)\n",defaults.hasten);
  }

  /*
   *  Set qmin, qmax (used in plotting).
   */
  /* Allocate memory: */
  q = dvector(0,Nelem2d-1);
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    (grid.qmin)[K] = DBL_MAX;
    (grid.qmax)[K] =-DBL_MAX;
    potential_vorticity(planet,q,K,1);
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        (grid.qmin)[K] = MIN((grid.qmin)[K],fabs(Q(J,I)));
        (grid.qmax)[K] = MAX((grid.qmax)[K],fabs(Q(J,I)));
      }
    }
#if defined(EPIC_MPI)
    /* global min, max: */
    tmp = grid.qmin[K];
    MPI_Allreduce(&tmp,grid.qmin+K,1,MPI_DOUBLE,MPI_MIN,para.comm);
    tmp = grid.qmax[K];
    MPI_Allreduce(&tmp,grid.qmax+K,1,MPI_DOUBLE,MPI_MAX,para.comm);
#endif
  }

  /*  
   *  Set tmin, tmax (used in plotting).
   */
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        temperature = T2(K,J,I);
        /* find max, min temperatures */
        max_temp    = MAX(max_temp,temperature);
        min_temp    = MIN(min_temp,temperature);
      }
    }
    (grid.tmin)[K] = min_temp;
    (grid.tmax)[K] = max_temp;
  }

  /*
   * Flag DUDT,DVDT,DHDT with DBL_MAX to indicate this is the initial 
   * timestep. This information is used in timestep() to specify the 
   * appropriate Adams-Bashforth coefficients.
   */
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        DUDT(IT_MINUS2,K,J,I) = DBL_MAX;
        DHDT(IT_MINUS2,K,J,I) = DBL_MAX;
        DUDT(IT_MINUS1,K,J,I) = DBL_MAX;
        DHDT(IT_MINUS1,K,J,I) = DBL_MAX;
      }
    }
    for (J = JFIRST; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        DVDT(IT_MINUS2,K,J,I) = DBL_MAX;
        DVDT(IT_MINUS1,K,J,I) = DBL_MAX;
      }
    }
  }

  /*
   * Write defaults file:
   */
  write_defaults(&defaults);

  /* 
   * Print out zonal-wind information: 
   */
  print_zonal_info(planet,q);

  /* 
   * Print out vertical information: 
   */
  print_vertical_column(planet,JLO,ILO);

  /*
   * Call low_mode() to write vertical eigenvalues to $EPIC_PATH/tmp.
   * NOTE: Algorithm requires nk >= 5.
   */
  if (grid.nk >= 5) {
    low_mode(grid.nj/2);
  }

  /* 
   * Output epic.nc file (Network Common Data Form, netCDF):
   */
  var_write(planet,"epic.nc",0,VIA_FILE,ALL_DATA);

  /* 
   * Free allocated memory: 
   */
  free_arrays(planet); 
  free_dvector(uamp,0,Kadim*Jadim-1);
  free_dvector(q,   0,    Nelem2d-1);


  if (!sw) {
    free_dvector(t_y2,    0,ntp-1);
    free_dvector(th_y2,   0,ntp-1);
    free_dvector(thetadat,0,ntp-1);
    free_dvector(tdat,    0,ntp-1);
    free_dvector(p_y2,    0,ntp-1);
    free_dvector(pdat,    0,ntp-1);
  }
 
  return 0;
}

/*======================= end of main() =====================================*/

/*======================= u_ven() ===========================================*/

double u_ven(double lat)
    /* 
     * Calculate u(lat) for Venus model, where lat is in degrees.
     * NOTE: For Venus, the UAMP(K,J) profile contains the m/s dimension.
     */
{
  char   
    header[N_STR];
  int
    kk;
  static int
    nup,       
    initialized=0;
  static double
    *u_up_dat,   
    *p_up_dat,       
    *up_y2; 
  double
    uu;
  char
    infile[64];
  FILE
    *u_vs_p;          

  if (!initialized) {
    /* Read in u vs p data */
    if (IAMNODE == 0) {
      sprintf(infile,EPIC_PATH"/data/venus/u_vs_p.venus");
      u_vs_p = fopen(infile,"r");
      if (!u_vs_p) {
        fprintf(stderr,"Error: u_ven(): Cannot open %s \n",infile);
        exit(1);
      }
      for (kk = 0; kk < 6; kk++) {
        fgets(header,128,u_vs_p);  
      }
      /* input number of data points */
      fscanf(u_vs_p,"%d",&nup);  
    }
#if defined(EPIC_MPI)
    MPI_Bcast(&nup,1,MPI_INT,NODE0,para.comm);
#endif
    p_up_dat  = dvector(0, nup-1);
    u_up_dat  = dvector(0, nup-1);
    up_y2     = dvector(0, nup-1);
    if (IAMNODE == 0) {
      /* in order of increasing theta */
      for (kk = nup-1; kk >= 0; kk--) {  
        fscanf(u_vs_p,"%lf %lf",(p_up_dat+kk),(u_up_dat+kk));
        /* convert from mbar to mks */
        p_up_dat[kk] *= 1.e+2; 
        /*  spline on -log p */
        p_up_dat[kk] = -log(p_up_dat[kk]);  
      }
      fclose(u_vs_p);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(p_up_dat,nup,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(u_up_dat,nup,MPI_DOUBLE,NODE0,para.comm);
#endif
    spline(p_up_dat,u_up_dat,nup,1.e+30,1.e+30,up_y2);

    initialized = 1;
  }
  /* End initialization */

  uu = cos(lat*DEG);

  return uu;
}

/*======================= end of u_ven() ====================================*/

/*======================= u_jup() ===========================================*/

double u_jup(double lat)
     /*  
      *  Calculate u(lat) for Jupiter, where   
      *  (lat) is in degrees.
      */
{
  int 
    j;
  static int
    ndat,
    initialized=0;
  double 
    uu,
    d;
  static double 
    *ltdat, 
    *udat, 
    *y2;
  FILE
    *u_dat;           

  if (!initialized) {
    initialized = 1;

    if (IAMNODE == 0) {
      u_dat = fopen(EPIC_PATH"/data/jupiter/u_vs_lat.jupiter","r");
      fscanf(u_dat, "%d", &ndat);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(&ndat,1,MPI_INT,NODE0,para.comm);
#endif
    ltdat = dvector(0, ndat-1);
    udat  = dvector(0, ndat-1);
    y2    = dvector(0, ndat-1);
    if (IAMNODE == 0) {
      for (j = 0; j < ndat; j++) {
        fscanf(u_dat, "%lf %lf", (ltdat+j), (udat+j));
      }
      fclose(u_dat);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(ltdat,ndat,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(udat, ndat,MPI_DOUBLE,NODE0,para.comm);
#endif
    spline(ltdat, udat, ndat, 1.e+30, 1.e+30, y2);
  }
  /* End initialization */
  
  j = ndat-1;
  while (lat < ltdat[j] && j > 0) {
    j--;
  }
  
  d   = *(ltdat+j+1) - *(ltdat+j);
  uu  = splint(lat, ltdat+j, udat+j, y2+j, d);
  return uu;
}

/*======================= end of u_jup() ====================================*/

/*======================= u_sat() ===========================================*/

double u_sat(double lat)
{
  int 
    j;
  static int
    ndat,
    initialized=0;
  double 
    uu, 
    d;
  static double 
    *ltdat, 
    *udat, 
    *y2;
  FILE
    *u_dat;           

  if (!initialized) {
    initialized = 1;
    if (IAMNODE == 0) {
      u_dat  = fopen(EPIC_PATH"/data/saturn/u_vs_lat.saturn", "r");
      fscanf(u_dat, "%d", &ndat);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(&ndat,1,MPI_INT,NODE0,para.comm);
#endif
    ltdat = dvector(0, ndat-1);
    udat  = dvector(0, ndat-1);
    y2    = dvector(0, ndat-1);
    if (IAMNODE == 0) {
      for (j = 0; j < ndat; j++) {
        fscanf(u_dat, "%lf %lf", (ltdat+j), (udat+j));
      }
      fclose(u_dat);
    }
#if defined(EPIC_MPI)
    MPI_Bcast(ltdat,ndat,MPI_DOUBLE,NODE0,para.comm);
    MPI_Bcast(udat, ndat,MPI_DOUBLE,NODE0,para.comm);
#endif
    spline(ltdat, udat, ndat, 1.e+30, 1.e+30, y2);
  }
  /* End initialization */
  
  j = ndat-1;
  while (lat < ltdat[j] && j > 0) {
    j--;
  }
  
  d   = *(ltdat+j+1) - *(ltdat+j);
  uu  = splint(lat, ltdat+j, udat+j, y2+j, d);
  return uu;
}

/*======================= end of u_sat() ====================================*/

/*====================== u_ura() ============================================*/

double  u_ura(double lat)
     /*
      *  Calculate Uranus-like u(lat), where
      *  (lat) is in degrees.
      *
      * NOTE: Need to implement Karkoschka (1998, Science 280, 570-2):
      *        deg/day = (482.+sin(latr)*(-8.+sin(latr)*127.)) 
      */
{
  double   
    u0, 
    latn, 
    lats,
    latr,
    uu;

  latr = lat*DEG;

  if (grid.aux_a == 0) {
    u0   = -120.;
    lats =  -20.;
    latn =   20.;
    if (lat >= 0.) {
      uu = u0*cos(latr)*(1.-2.*(lat/latn)*(lat/latn)/(1+lat/latn));
      return uu;
    }
    else {
      uu = u0*cos(latr)*(1.-2.*(lat/lats)*(lat/lats)/(1+lat/lats));
      return uu;
    }
  }
  else if (grid.aux_a == 1) {
    uu = -4300.*cos(latr)*(lat-17.)/(lat-500.);
    return uu;
  }
  else {
    fprintf(stderr,"Unrecognized grid.aux_a in u_ura \n");
    exit(1);
  }
    
}
/*======================= end of u_ura() ====================================*/

/*====================== u_nep() ============================================*/

#define USE_SPHERICAL_FIT  0
#define USE_POLYNOMIAL_FIT 1

double  u_nep(double lat)
     /*
      *  Calculate Neptune u(lat), where
      *  (lat) is in degrees. From Ingersoll et al., Neptune, 
      *  University of Arizona Press.  Using Table 1, deg/hr -> m/s.
      */
{
  static int
    which_fit,
    initialized=0;
  double   
    uu,rln,
    sinlat;
  static double
    a1,a2,a3,
    b1,b2,b3,
    re,rp;

  if (!initialized) {
    /*
     * Choose between Sromovsky et al's (1993) 4th order even-polynomial fit
     * and LeBeau and Dowling's (1998) 4th order even-spherical-harmonic fit:
     */
    which_fit = USE_SPHERICAL_FIT;

    re = planet->re;
    rp = planet->rp;
    a1 = -3.19;
    a2 =  6.7e-4;
    a3 =  2.54e-7;
    /* sin(lat) coeffs, from Legendre fit */
    b1 = -2.906;
    b2 = -2.603;
    b3 = 14.31;

    initialized = 1;
  }
  /* End initialization */

  rln  = re/sqrt( 1.+ pow(rp/re*tan(lat*DEG),2.) );

  if (which_fit == USE_POLYNOMIAL_FIT) {
    uu = (  a1
          +(a2
           +a3*lat*lat)*lat*lat)*rln*DEG/3600.;
  }
  else if (which_fit == USE_SPHERICAL_FIT) {
    sinlat = sin(lat*DEG);
    uu     = (  b1
              +(b2
              +b3*sinlat*sinlat)*sinlat*sinlat)*rln*DEG/3600.;
  }
  else {
    fprintf(stderr,"Error: u_nep(): unknown which_fit = %d \n",which_fit);
    exit(1);
  }

  return uu;
}

/*======================= end of u_nep() ====================================*/

/*======================= u_tri() ===========================================*/

double u_tri(double lat)
{
  static int
    initialized=0;
  static double
    amp;
  double
    uu;

  if (!initialized) {
    amp = input_double("Amplitude for Triton zonal wind [m/s]\n",10.);
    initialized = 1;
  }

/****
  uu = amp*cos(lat*DEG);
*****/
  /* Midlatitude jets: */
  uu = amp*.5*(1.-cos(4.*lat*DEG));

  return uu;
}

/*======================= end of u_tri() ====================================*/

/*======================= u_nul() ===========================================*/

double u_nul(double lat)
{
  return 0.;
}

/*======================= end of u_nul() ====================================*/

/*======================= read_defaults() ===================================*/

/*
 * NOTE: To use the READ* macros in epic_io_macros.h, dummy
 *       io and fd variables are declared.
 */

void read_defaults(init_defaultspec *def) 
{
  int
    index,
    nc_id,nc_err,
    fd,
    io_type=VIA_FILE,
    node   =NODE0;
  double
    solar;
  char
    min_element[4];
  static char
    **gattname=NULL,
    **varname =NULL;
  static int
    ngatts=0,
    nvars =0;
  nc_type
    the_nc_type;
  FILE
    *io;

  lookup_netcdf(EPIC_PATH"/tmp/init_defaults.nc",
                &nc_id,&ngatts,&gattname,&nvars,&varname);

  strcpy(def->geometry,"globe");
    READC(NODE0,*&,def,->,geometry, GEOM_STR);
  strcpy(def->system_id,  "Jupiter");
    READC(NODE0,*&,def,->,system_id,8);
  strcpy(def->f_plane_map,"polar");
    READC(NODE0,*&,def,->,f_plane_map,GEOM_STR);
  strcpy(def->eos,        "ideal");
    READC(NODE0,*&,def,->,eos,8);

  def->nk                 =  10;
    READI(NODE0,&,def,->,nk,1);
  def->nj                 =  64;
    READI(NODE0,&,def,->,nj,1);
  def->ni                 =  128;
    READI(NODE0,&,def,->,ni,1);
  def->dt                 =  120;
    READI(NODE0,&,def,->,dt,1);
  def->init_type          =  INIT_TP;
    READI(NODE0,&,def,->,init_type,1);
  def->add_optional_var   =  FALSE;
    READI(NODE0,&,def,->,add_optional_var,1);
  def->chem_on[U_INDEX]   =  CHEM_ACTIVE;
  def->chem_on[V_INDEX]   =  CHEM_ACTIVE;
  def->chem_on[P_INDEX]   =  CHEM_ACTIVE;
  for (index = P_INDEX+1; index <= LAST_HUMIDITY; index++) {
    def->chem_on[index]   = CHEM_OFF;
  }
    READI(NODE0,*&,def,->,chem_on,MAX_NVARS);
  def->uz_type            =  UZ_GALILEO;
    READI(NODE0,&,def,->,uz_type,1);
  def->spacing_type       =  SPACING_LOGP;
    READI(NODE0,&,def,->,spacing_type,1);
  def->geostr_init        =  TRUE;
    READI(NODE0,&,def,->,geostr_init,1);
  def->u_init             =  1;
    READI(NODE0,&,def,->,u_init,1);
  def->mq_ztyp            =  0;
    READI(NODE0,&,def,->,mq_ztyp,1);
  def->mq_ptyp            =  1;
    READI(NODE0,&,def,->,mq_ptyp,1);
  def->mq_maxit           =  500;
    READI(NODE0,&,def,->,mq_maxit,1);
  def->mq_manual          =  0;
    READI(NODE0,&,def,->,mq_manual,1);
  def->mq_checkit         =  50;
    READI(NODE0,&,def,->,mq_checkit,1);
  def->mq_qmrepeat        =  0;
    READI(NODE0,&,def,->,mq_qmrepeat,1);
  def->mq_jzero           =  64;
    READI(NODE0,&,def,->,mq_jzero,1);

  def->globe_lonbot       = -180.;
    READD(NODE0,&,def,->,globe_lonbot,1);
  def->globe_lontop       =  180.;
    READD(NODE0,&,def,->,globe_lontop,1);
  def->globe_latbot       = -90.;
    READD(NODE0,&,def,->,globe_latbot,1);
  def->globe_lattop       =  90.;
    READD(NODE0,&,def,->,globe_lattop,1);
  def->f_plane_half_width =  90.;
    READD(NODE0,&,def,->,f_plane_half_width,1);
  def->ptop               =  1.*100.;
    READD(NODE0,&,def,->,ptop,1);
  def->pbot               =  1000.*100.;
    READD(NODE0,&,def,->,pbot,1);
  def->prandtl            =  0.;
    READD(NODE0,&,def,->,prandtl,1);
  def->hasten             =  1.;
    READD(NODE0,&,def,->,hasten,1);
  def->nu[2]              =  0.;
  def->nu[4]              =  0.;
  def->nu[6]              =  0.5;
    READD(NODE0,*&,def,->,nu,MAX_NU_ORDER+1);
  def->time_fp            =  3.e+8;
    READD(NODE0,&,def,->,time_fp,1);
  def->mq_c_q             =  700.;
    READD(NODE0,&,def,->,mq_c_q,1);
  def->mq_pfact           =  1.;
    READD(NODE0,&,def,->,mq_pfact,1);
  def->mq_x               =  1.;
    READD(NODE0,&,def,->,mq_x,1);
  for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
    solar = solar_fraction(var.chem_name[index],BY_NUMBER,min_element);
    (def->mole_fraction)[index] = solar;
  }
    READD(NODE0,*&,def,->,mole_fraction,MAX_NVARS);


  return;
}

/*======================= end of read_defaults() ============================*/

/*======================= write_defaults() ==================================*/

/*
 * NOTE: To use the WRITE* macros in epic_io_macros.h, dummy
 *       io and fd variables are declared.
 */

void write_defaults(init_defaultspec *def)
{
  int
    index,
    nc_id,nc_err,
    fd,
    io_type=VIA_FILE,
    node   =NODE0;
  nc_type
    the_nc_type;
  FILE
    *io;

  nc_err = nc_create(EPIC_PATH"/tmp/init_defaults.nc",NC_CLOBBER,&nc_id);

  WRITEC(NODE0,*&,def,->,geometry, GEOM_STR);
  WRITEC(NODE0,*&,def,->,system_id,8);
  WRITEC(NODE0,*&,def,->,f_plane_map,GEOM_STR);
  WRITEC(NODE0,*&,def,->,eos,8);

  WRITEI(NODE0,&,def,->,nk,1);
  WRITEI(NODE0,&,def,->,nj,1);
  WRITEI(NODE0,&,def,->,ni,1);
  WRITEI(NODE0,&,def,->,dt,1);
  WRITEI(NODE0,&,def,->,init_type,1);
  WRITEI(NODE0,&,def,->,add_optional_var,1);
  WRITEI(NODE0,*&,def,->,chem_on,MAX_NVARS);
  WRITEI(NODE0,&,def,->,uz_type,1);
  WRITEI(NODE0,&,def,->,spacing_type,1);
  WRITEI(NODE0,&,def,->,geostr_init,1);
  WRITEI(NODE0,&,def,->,u_init,1);
  WRITEI(NODE0,&,def,->,mq_ztyp,1);
  WRITEI(NODE0,&,def,->,mq_ptyp,1);
  WRITEI(NODE0,&,def,->,mq_maxit,1);
  WRITEI(NODE0,&,def,->,mq_manual,1);
  WRITEI(NODE0,&,def,->,mq_checkit,1);
  WRITEI(NODE0,&,def,->,mq_qmrepeat,1);
  WRITEI(NODE0,&,def,->,mq_jzero,1);

  WRITED(NODE0,&,def,->,globe_lonbot,1);
  WRITED(NODE0,&,def,->,globe_lontop,1);
  WRITED(NODE0,&,def,->,globe_latbot,1);
  WRITED(NODE0,&,def,->,globe_lattop,1);
  WRITED(NODE0,&,def,->,f_plane_half_width,1);
  WRITED(NODE0,&,def,->,ptop,1);
  WRITED(NODE0,&,def,->,pbot,1);
  WRITED(NODE0,&,def,->,prandtl,1);
  WRITED(NODE0,&,def,->,hasten,1);
  WRITED(NODE0,*&,def,->,nu,MAX_NU_ORDER+1);
  WRITED(NODE0,&,def,->,time_fp,1);
  WRITED(NODE0,&,def,->,mq_c_q,1);
  WRITED(NODE0,&,def,->,mq_pfact,1);
  WRITED(NODE0,&,def,->,mq_x,1);
  WRITED(NODE0,*&,def,->,mole_fraction,MAX_NVARS);

  nc_close(nc_id);

  return;
}

/*======================= end of write_defaults() ===========================*/

/* * * * * * * * * * * *  end of epic_initial.c  * * * * * * * * * * * * * * */







