/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * epic_main.c * * * * * * * * * * * * * * * * * 
 *                                                                           *
 *     Timothy E. Dowling                                                    *
 *                                                                           *
 *     Contributions from A. Fischer, E.E. Charrette, J. Harrington,         *
 *     R. Lebeau, C. Santori.                                                *
 *                                                                           *
 *     Explicit Planetary Isentropic-Coordinate (EPIC)                       *
 *     atmospheric model for Jupiter, Saturn, Uranus, and Neptune.           * 
 *                                                                           * 
 *     C version.                                                            * 
 *                                                                           * 
 *     The scheme follows Hsu and Arakawa (1990) "Numerical modeling         *
 *     of the atmosphere with an isentropic vertical coordinate,"            *
 *     Monthly Weather Review, 118: 1933-1959, and Arakawa and Lamb (1981)   *
 *     "A potential enstrophy and energy conserving scheme for the shallow   *
 *     water equations,"  Monthly Weather Review, 109: 18-36.                *
 *                                                                           *
 *     The model has nk vertical layers.  The vertical coordinate is         *
 *     potential temperature, theta. The top layer is denoted by k = 1.      * 
 *     Pressure is defined at the layer interfaces.                          *
 *     For gas-giant planets, variables in the bottom layer (k = nk) are     *
 *     specified externally.                                                 *
 *                                                                           *
 *     The horizontal velocity variables u and v are marched forward in      *
 *     time using the 3rd-order Adams-Bashforth method.  For a               *
 *     discussion of this timestep see Durran (1991): "The                   *
 *     third-order Adams-Bashforth method: an attractive alternative         * 
 *     to leapfrog time differencing," Monthly Weather Review, 119:          *
 *     702-720.                                                              *
 *                                                                           *
 *     The small longitudinal grid spacings near the poles violate the       *
 *     CFL numerical stability criterion, but the instability is removed     *
 *     by filtering the tendencies.                                          *
 *                                                                           *
 *     Time is in seconds; length is in meters.                              *
 *                                                                           *
 *     The input file epic.nc is generated by epic_initial.c.                *
 *     Output files are labeled by the time.                                 *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

extern chem_element
  *Element;

/*======================= main() ============================================*/

main (int   argc,
      char *argv[]) 
{ 
  /*
   * NOTE: structures planet, grid, var, and diag are
   *       declared globally in epic.h.
   */
  char  
    infile[80],            /*  input file                                */
    outfile[80],           /*  name of output data file, labeled by time */
    savdir[80],            /*  directory in which to save data           */
    sflag[16];             /*  string to hold command-line flag          */
  int
    io_type,
    socket_port,
    count;
  /* 
   * The following are part of DEBUG_MILESTONE statements: 
   */
  char
    dbmsname[]="epic_main";
  int
    idbms=0;

#ifdef EPIC_MPI
  MPI_Init(&argc,&argv);
  para.comm = MPI_COMM_WORLD;
  MPI_Errhandler_set(para.comm,MPI_ERRORS_RETURN);
  MPI_Comm_rank(para.comm,&para.iamnode);
  MPI_Comm_size(para.comm,&para.nproc);
#endif

  /* 
   * Interpret command-line arguments: 
   */
  /* Start with defaults: */
  grid.itback = 10000;
  grid.itout  = 1000;
  grid.itsave = 1000000;
  socket_port = 0;
  sprintf(grid.view_host,"none");
  if (argc == 1) {
    /* Print help, exit: */
    /* Declare copyright: */
    declare_copyright();
    system("more "EPIC_PATH"/help/epic_main.help");
    exit(1);
  }
  else {  
    /* Read flags: */
    for (count = 1; count < argc; count++) {
      sscanf(argv[count],"%s",sflag);
      if (strcmp(sflag,"-help") == 0 ||
          strcmp(sflag,"-h")    == 0) {
        /* Print help, exit: */
        /* Declare copyright: */
        declare_copyright();
        system("more "EPIC_PATH"/help/epic_main.help");
        exit(1);
      }
      else if (strcmp(sflag,"-itback") == 0) {
        sscanf(argv[++count],"%d",&(grid.itback));
      }
      else if (strcmp(sflag,"-itsave") == 0) {
        sscanf(argv[++count],"%d",&(grid.itsave));
      }
      else if (strcmp(sflag,"-itout") == 0) {
        sscanf(argv[++count],"%d",&(grid.itout));
      }
      else if (strcmp(sflag,"-socket") == 0) {
        sscanf(argv[++count],"%d",&socket_port);
      }
      else if (strcmp(sflag,"-view_host") == 0) {
        sscanf(argv[++count],"%s",grid.view_host);
      }
      else {
        if (count == argc-1) {
          /* The last command-line argument is the input filename: */
          sscanf(argv[argc-1],"%s",infile);
        }
        else {
          fprintf(stderr,"Unrecognized epic command-line flag: %s \n",sflag);
          exit(1);
        }
      }
    }
  }

  /* Allocate memory */
  if((planet=( planetspec *)malloc(sizeof( planetspec))) == 0) {
    fprintf(stderr,"Error: epic_main: allocating space for planet \n");
    exit(0);
  }

  if (strcmp(infile,"stdio") == 0) {
    io_type = VIA_STDIO;
  }
  else {
    io_type = VIA_FILE;
  }

  /* 
   * Read model size, allocate memory for arrays,
   * read in rest of data:
   */
  var_read(planet,infile,0,io_type,INIT_DATA);
  make_arrays();
  var_read(planet,infile,0,io_type,NONINIT_DATA);
 
  if (socket_port > 0) {
    /* Send initial data to EPIC_RECEIVE */
    var_write(planet,NULL,socket_port,VIA_SOCKET,INIT_DATA);
    var_write(planet,NULL,socket_port,VIA_SOCKET, UVP_DATA);
  }

  if (io_type == VIA_FILE) {
    char
      *ptr;

    /* cd to infile directory so saved files go there */
    strcpy(savdir,infile);
    ptr = strrchr(savdir,'/');
    if (ptr != NULL) {
      *ptr = '\0';
      if(chdir(savdir) == -1){
        fprintf(stderr,"Warning: epic_main: Couldn't cd to save directory. \n");
        fprintf(stderr,savdir);
      }
    }

#if defined (EPIC_MPI)
    /* Synchronize processors: */
    MPI_Barrier(para.comm);
#endif

  }

  grid.itime = 0;
  while (socket_port > 0 || grid.itime <= grid.itout) {
    if (io_type == VIA_FILE) {
      /* Save data when requested. */
      if ( (grid.itime)%(grid.itsave) == 0 && grid.itime > 0) {
        sprintf(outfile, "epic%05d-%02d:%02d:%02d.nc",
                var.time[0] / 86400 + var.time[1]*365, /* days    */
               (var.time[0] /(60*60))  % 24,           /* hours   */
               (var.time[0] / 60)      % 60,           /* minutes */
                var.time[0]            % 60);          /* seconds */
        /* zero indicates no expansion in zonal direction */
        var_write(planet,outfile,0,VIA_FILE,ALL_DATA);
      } 

      /* Write out backup frame periodically; always have a good one. */
      if (  (grid.itime)%(grid.itback) == 0 && grid.itime > 0) {
        sprintf(outfile, "epic_back1.nc");
        if (IAMNODE == 0) { 
          if (0 != rename(outfile, "epic_back2.nc") ) {
            if (errno != ENOENT){
              perror(outfile);
            } 
            else {
              errno = 0;
            }
          }
        }
        var_write(planet,outfile,0,VIA_FILE,ALL_DATA);

#ifdef EPIC_SINGLE
        /* 
         * Write out EP flux data to a file in tmp directory: 
         */
        write_ep(planet);
#endif

      }
    } 
    
#ifdef EPIC_SINGLE
    else if (io_type == VIA_STDIO) {
      if ( (grid.itime)%(grid.itsave) == 0) {
        /* Output u,v,p to stdout as ascii floats */
        uvp_out(planet);
      } 
      if ( (grid.itime)%(grid.itback) == 0 && grid.itime > 0) {
        /* Write end result to facilitate restarting: */
        var_write(planet,"last_epic.nc",0,VIA_FILE,ALL_DATA);
      }
    }
#endif

    if (socket_port > 0) {
      /* Send data to EPIC_RECEIVE periodically */
      if ((grid.itime)%(grid.itout) == 0 && grid.itime > 0) {
        var_write(planet,NULL,socket_port,VIA_SOCKET,UVP_DATA);
      }
    }

    if (socket_port > 0 || grid.itime < grid.itout) {
      /* Take a step: */
      timestep();

    }

    (grid.itime)++;
  }  

  free_arrays(planet); 
  free(planet);

#if defined(EPIC_MPI)
  MPI_Finalize();
#endif

  return;
}

/*======================= end of main()  ====================================*/

/* * * * * * * * * * * * end of epic_main.c  * * * * * * * * * * * * * * * * */
