/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * epic_timestep.c * * * * * * * * * * * * * * * * * * * * * 
 *                                                                           *
 *  Timothy E. Dowling                                                       *
 *                                                                           *
 *  Integrate the prognostic variables ahead one timestep.                   *
 *                                                                           *
 *  We use the 3rd-order Adams-Bashforth timestep for u and v.               *
 *  This timestep is discussed by D. Durran (1991, MWR 119, 702-720).        *
 *  It is appropriate for dissipative terms as well as for                   *
 *  conservative terms, does not suffer from the time-splitting              *
 *  numerical instability of the leap-frog timestep, and is more             *
 *  accurate than the leapfrog timestep.  Its main drawback is that          *
 *  it requires more memory because it uses two previous time derivatives.   *
 *                                                                           *
 *  We use Hsu and Arakawa's predictor-corrector timestep for horizontal     *
 *  advection of p and the thermodynamical variables.  This advection        *
 *  scheme is positive-definite and handles steep gradients.  The result     *
 *  of this horizontal advection is converted into a 3rd order Adams         *
 *  Bashforth tendency, and then combined with the diabatic terms.           *
 *                                                                           *
 *  We calculate the primitive, isentropic-coordinate Eliassen-Palm (EP)     *
 *  flux and EP flux divergence as a diagnostic. See Chap. 3 of              *
 *  "Middle Atmosphere Dynamics," Andrews et al, 1987, Academic.             *                                         *
 *                                                                           *
 *  The thermodynamical functions were written by P. Gierasch.               *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>
#include <epic_sw_schemes.h>

#define DT   ((double)grid.dt)

    /*
     *  Working 2D arrays:
     */
#define NUM_WORKING_BUFFERS 6
static double 
  *BuffD[NUM_WORKING_BUFFERS];

/* Arrays for calculation of EP flux */
#define U_AVG(j)   u_avg[j-JLO]
#define H_AVG(j)   h_avg[j-JLO]
#define VH_AVG(j) vh_avg[j-JLO]
#define WH_AVG(j) wh_avg[j-JLO]

/*======================= timestep() ========================================*/


void timestep(void)
{
  int    
    K,J,I,
    kk,
    index,
    itmp,itmp2,sign;
  static int 
    initialized=0,
    sw=0;
  double  
    al, be,       /*  Arakawa and Lamb (1981) eqn. (3.34)               */
    ga, de,       /*         "                       "                  */
    ep1,ep2,      /*         "                       "                  */
    ph1,ph2,      /*         "                       "                  */
    ab[3],        /*  Adams-Bashforth coefficients                      */
    m_2jp1,
    m_2j_inv, m_2jp2_inv,
    n_2j,n_2jp1_inv,mn_2jp1,
    mn_2j_inv, mn_2jp2_inv,
    dx,
    d_th0,d_th1,d_th2,d_th3,d_th1_inv,
    *u_pt,*v_pt,*h_pt,dt,
    uavg,vavg,
    *uh,*vh,*q,*kin,
    *mont,*wh0,*wh1,*exner,*heat,
    fgibb,fpe,uoup,
    theta,fpara,pressure,temperature,cpr, 
    time_fp_inv,dfpdt,
    *lap1,*lap2,*laph,*dh_ptdt,*ptmp,nu,nu0,
    h_old,h_new,
    tmp,aux;
  static double 
     dx0=-1.,
    *h,
    *u_avg,*h_avg,*vh_avg,*wh_avg;
  /* 
   * The following are part of DEBUG_MILESTONE statements: 
   */
  int
    idbms=0;
  char
    dbmsname[]="epic_timestep";

  if(!initialized) {
    /* 
     * Allocate space for working arrays: 
     */
    for (I = 0; I < NUM_WORKING_BUFFERS; I++) {
      BuffD[I]  = dvector(0,Nelem2d-1);
    }
    h = dvector(0,Nelem2d-1);

    if (strcmp(planet->name,"sw") == 0) {
      sw = 1;
    }

    /* Used in calculation of EP flux: */
    u_avg  = dvector(0,Jadim-1);
    h_avg  = dvector(0,Jadim-1);
    vh_avg = dvector(0,Jadim-1);
    wh_avg = dvector(0,Jadim-1);

    if (!sw) {
      /*
       *  Setup P. Gierasch's fortran thermodynamic functions:
       */
      thermo_setup(planet,&cpr);
    }
    initialized=1;
  }
  /* End of initialization */

  /* 
   * Store heavily used diagnostic variables. 
   */
  store_temp_dens_p(planet,JLO,JHI,ILO,IHI);
 
  if (grid.itime%10 == 0 || dx0 < 0.) {
    /*
     * Calculate CFL dt at the beginning and every 10 timesteps.
     * Use if smaller than current grid.dt.
     * The function cfl_dt() is defined in epic_funcs_diag.c.
     */
    grid.cfl_dt = cfl_dt(planet,&dx0);
    
    if (!sw) {
      grid.dt = MIN(grid.dt,grid.cfl_dt);
    }
    if (grid.dt == 0 && TIME > 0.) {
      /* Timestep has shrunk to zero, exit model. */
      fprintf(stderr,"EPIC: dt has shrunk to zero, exiting. \n");
      exit(1);
    }
  }

  /*
   *  Clear tendency timeframes for timeframe IT_ZERO.
   */
  for (index = 0; index < MAX_NVARS; index++) {
    if (var.chem_on[index]) {
      memset(var.vector[MAX_NVARS+index]+IT_ZERO*Nelem3d,0,Nelem3d*sizeof(double));
    }
  }

  /* 
   * Specify Adams-Bashforth coefficients. The DBL_MAX flags are set in
   * epic_initial.c. 
   */
  if (DUDT(IT_MINUS2,KLO,JLO,ILO) == DBL_MAX ||
      DVDT(IT_MINUS2,KLO,JLO,ILO) == DBL_MAX ||
      DHDT(IT_MINUS2,KLO,JLO,ILO) == DBL_MAX) {
    if (DUDT(IT_MINUS1,KLO,JLO,ILO) == DBL_MAX ||
        DVDT(IT_MINUS1,KLO,JLO,ILO) == DBL_MAX ||
        DHDT(IT_MINUS1,KLO,JLO,ILO) == DBL_MAX) {
      /* 
       * Use 1st-order Adams-Bashforth, aka forward difference,
       * for initial step.
       */
      ab[0] = 1.;
      ab[1] = 0.;
      ab[2] = 0.;
    }
    else {
      /*
       * Use 2nd-order Adams-Bashforth for second step.
       */
      ab[0] =  3./2.;
      ab[1] = -1./2.;
      ab[2] =     0.;
    }
  }
  else {
   /*
    * Use 3rd-order Adams-Bashforth in general.
    */
    ab[0] =  23./12.;
    ab[1] = -16./12.;
    ab[2] =   5./12.;
  }

 /*
  *  Calculate the horizontal advection terms.  
  */ 
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    /*
     *  Assign zero'd memory to uh,vh,q,kin:
     */ 
    memset(BuffD[0],0,Nelem2d*sizeof(double));
    memset(BuffD[1],0,Nelem2d*sizeof(double));
    memset(BuffD[2],0,Nelem2d*sizeof(double));
    memset(BuffD[3],0,Nelem2d*sizeof(double));
    uh  = BuffD[0];      
    vh  = BuffD[1];    
    q   = BuffD[2];  
    kin = BuffD[3];  

    /* Calculate h from p for this layer */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        H(J,I) = get_h(planet,2*K,J,I);
      }
    }
    BC2D(&(H(JLO,ILO)),NO_INDEX,1);

    for (J = JLO; J <= JHI; J++) {
      n_2jp1_inv = .5/(grid.n)[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        UH(J,I) = U(K,J,I)*(H(J,I)+H(J,I-1))*n_2jp1_inv;
      }
    }
    BC2D(&(UH(JLO,ILO)),NO_INDEX,1);

    for (J = JFIRST; J <= JHI; J++) {
      m_2j_inv  = .5/(grid.m)[2*J];
      VH_AVG(J) = 0.;
      U_AVG( J) = 0.;
      for (I = ILO; I <= IHI; I++) {
        VH(J,I)    = V(K,J,I)*(H(J,I)+H(J-1,I))*m_2j_inv;
        VH_AVG(J) += VH(J,I);
        U_AVG( J) += (U(K,J,I)+U(K,J,I+1)+U(K,J-1,I)+U(K,J-1,I+1));
      }
      /* NOTE: only works if i-direction is not decomposed. */
      VH_AVG(J) /= grid.ni;
      U_AVG( J) /= grid.ni;
      if (grid.ni > 1 && !sw) {
        /*
         * Calculate meridional component of EP flux:
         */
        EP_FLUX_Y(K,J) = 0.;
        for (I = ILO; I <= IHI; I++) {
          EP_FLUX_Y(K,J) += (VH(J,I)-VH_AVG(J))*
                            (U(K,J,I)+U(K,J,I+1)+U(K,J-1,I)+U(K,J-1,I+1)-U_AVG(J));
        }
        /* NOTE: Only works if i-direction not decomposed. */
        EP_FLUX_Y(K,J) /= -(double)(4*grid.ni)*grid.dln*DEG;
      }
      else {
        /* For ni = 1 case, store v in place of ep_flux_y: */
        EP_FLUX_Y(K,J) = V(K,J,ILO);
      }
    }
    BC2D(&(VH(JLO,ILO)),NO_INDEX,1);

    /*
     *  Calculate potential vorticity, q:
     */
    potential_vorticity(planet,q,K,1);

    /*
     *  Calculate kinetic energy per unit mass, kin:
     */
    for (J = JLO; J <= JHI; J++) {
      mn_2jp1     =    (grid.mn)[2*J+1];
      mn_2j_inv   = 1./(grid.mn)[2*J  ];
      mn_2jp2_inv = 1./(grid.mn)[2*J+2];
      for (I = ILO; I <= IHI; I++) {
        KIN(J,I) = (      U(K,J,  I+1)*U(K,J,  I+1)+
                          U(K,J,  I  )*U(K,J,  I  ) 
                +mn_2jp1*(V(K,J+1,I  )*V(K,J+1,I  )*mn_2jp2_inv 
                         +V(K,J,  I  )*V(K,J,  I  )*mn_2j_inv  ) )*.25;
      }
    }
    BC2D(&(KIN(JLO,ILO)),NO_INDEX,1);
  
    /*
     *  Add q*uh, q*vh and kin terms to dudt:
     */
    for (J = JLO; J <= JHI; J++) {
      m_2jp1 = (grid.m)[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        DUDT(IT_ZERO,K,J,I) += ((AL_U*VH(J+1,I  )+BE_U*VH(J+1,I-1)
                                +GA_U*VH(J  ,I-1)+DE_U*VH(J,  I  )
		                +EP2*UH( J  ,I-1)-EP1*UH( J,  I+1))*SW_COEF
                              +KIN(J,I-1)-KIN(J,I))*m_2jp1;  
 
      }
    }
    /*
     *  Add q*uh, q*vh and kin terms to dvdt:
     */
    for (J = JFIRST; J <= JHI; J++) {
      n_2j = (grid.n)[2*J];
      for (I = ILO; I <= IHI; I++) {
        DVDT(IT_ZERO,K,J,I) += ((-GA_V*UH(J  ,I+1)-DE_V*UH(J  ,I  )
		                 -AL_V*UH(J-1,I  )-BE_V*UH(J-1,I+1)
                                 +PH2*VH( J-1,I  )-PH1*VH( J+1,I  ))*SW_COEF
                              +KIN(J-1,I)-KIN(J,I))*n_2j;
      }
    }

    /*
     *  Add uh and vh terms to dhdt, and 
     *  flux divergence terms to thermodynamical tendencies.
     *
     *  NOTE: hsu_predict_correct() uses the working buffer memory, so call it only 
     *  after you are done with uh, vh, etc. Also, it uses the dhdt, etc, tendency memory 
     *  for temporary storage, so call it before adding the diabatic terms to the tendencies.
     */
    if (HSU_PREDICT_CORRECT == TRUE) {
      hsu_predict_correct(h,K,ab);
    }
    else {
      horizontal_divergence_terms(uh,vh,K);
    }
  }

  /*
   * Calculate the pressure-gradient and diabatic terms.
   *
   * Run through the vertical layers from bottom to top to  
   * integrate the hydrostatic equation and obtain the Montgomery potential. 
   *
   * For gas-giant planets, the wind in the deep layer (k=nk) is specified as
   * a boundary condition that sets the interior value of the Montgomery potential.
   * For terrestrial planets, the surface topography is used to calculate the
   * bottom Montgomery potential.
   *
   * The quantity wh refers to (Dtheta/Dt)*h;  wh0 is for the
   * top of the layer and wh1 is for the bottom of the layer.
   *
   * Assign memory to wh0,wh1,exner,mont,heat:
   */
  memset(BuffD[0],0,Nelem2d*sizeof(double));
  memset(BuffD[1],0,Nelem2d*sizeof(double));
  memset(BuffD[2],0,Nelem2d*sizeof(double));
  memset(BuffD[3],0,Nelem2d*sizeof(double));
  memset(BuffD[4],0,Nelem2d*sizeof(double));
  wh0   = BuffD[0];
  wh1   = BuffD[1];
  exner = BuffD[2];
  mont  = BuffD[3];
  heat  = BuffD[4];

 /*
  * Determine mont for k = nk (the deepest layer).
  *   gas-giant: via gradient balance with u_nk,
  * terrestrial: via surface_gz.
  */
  mont_nk(planet,mont); 

 /*
  *  Determine exner for bottom of layer KLAST_ACTIVE:
  */
  K = KLAST_ACTIVE;
  theta = grid.theta[2*K+1];
  for (J = JLO; J <= JHI; J++) {
    if (sw) {
      /* kappa = 1. */
      for (I = ILO; I <= IHI; I++) {
        EXNER(J,I) = planet->cp*P(K,J,I)/grid.press0;
      }
    }
    else {
      for (I = ILO; I <= IHI; I++) {
        EXNER(J,I) = (planet->cp)*T2(K,J,I)/theta;
      }
    }
  }
  /* No need to apply horizontal boundary conditions to exner */

  if (!sw) {
    /* Calculate wh for bottom of layer KLAST_ACTIVE: */
    heating(planet,heat,K);
    rain(planet,heat,K);
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        WH0(J,I) = get_h(planet,2*K+1,J,I)*HEAT(J,I)/EXNER(J,I);
      }
      if (grid.ni == 1) {
        /* Store w for case ni = 1 */
        EP_FLUX_Z(K,J) = HEAT(J,ILO)/EXNER(J,ILO);
      }
    }
    BC2D(&(WH0(JLO,ILO)),NO_INDEX,1);
  }
  for (K = KLAST_ACTIVE; K >= 1; K--) {
    if (!sw) {
      /* 
       * New wh1 is old wh0 (new bottom wh is old top wh): 
       */
      ptmp = wh1;
      wh1  = wh0;
      wh0  = ptmp;
    
      /* Clear wh0: */
      memset(wh0,0,Nelem2d*sizeof(double));
    }

    if (K > 1 && !sw) {
      /*
       *  Calculate 2D heating array for top of layer K.
       *  The function heating() resides in its own file,
       *  $EPIC_PATH/src/shared/epic_heating.c.
       */
      heating(planet,heat,K-1);
      rain(planet,heat,K-1);

      /* Start calculation of wh0: */
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          WH0(J,I) = get_h(planet,2*K-1,J,I)*HEAT(J,I);
        }
      }
      /* No need to apply horizontal boundary conditions to wh0 here */
    }

    /* Calculate h for current layer */
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        H(J,I) = get_h(planet,2*K,J,I);
      }
    }
    BC2D(&(H(JLO,ILO)),NO_INDEX,1);

    /* Calculate various delta-thetas: */
    if (K > 1) {
      d_th0     = grid.theta[2*K  ]-grid.theta[2*K+2];
      d_th1     = grid.theta[2*K-1]-grid.theta[2*K+1];
      d_th2     = grid.theta[2*K-1]-grid.theta[2*K  ];
      d_th3     = grid.theta[2*K  ]-grid.theta[2*K+1];
      d_th1_inv = 1./d_th1;
    }
    else {
      /* special delta-thetas in top layer: */
      d_th0     = grid.theta[2*K  ]-grid.theta[2*K+2];
      d_th1     = grid.theta[2*K  ]-grid.theta[2*K+1];
      d_th1_inv = 1./d_th1;
    }

    if (!sw) {
      /* 
       * Start calculation of z-component of EP flux.
       *
       * Because avg(mont_x) = 0., avg(p'*mont_x') = avg(p*mont_x).
       *
       * NOTE: For EP_FLUX_Z, k = nk case effectively uses p(nk+.5)*mont(nk) 
       * instead of p(nk+.5)*.5*[mont(nk+1)+mont(nk)].
       */
      if (K > 1) {
        if (grid.ni > 1) {
          for (J = JLO; J <= JHI; J++) {
            EP_FLUX_Z(K,J) = 0.;
            for (I = ILO; I <= IHI; I++) {
              EP_FLUX_Z(K,J) += P(K,J,I)/planet->g*
                                (MONT(J,I+1)-MONT(J,I-1))*grid.m[2*J+1]*.25;
            }
          }
        }
      }
    }

    /*
     *  Calculate mont, exner, wh's:
     */
    if (K < grid.nk) {
      /* k = nk case (terrestrial planets) has mont(nk) set prior to loop. */
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          MONT(J,I) += EXNER(J,I)*d_th0;
        }
      }
      if (var.chem_on[FPARA_INDEX]) {
        /* add extra term for fpara thermodynamics */
        kk    = 2*K+1;
        theta = grid.theta[kk];
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            fpara       = get_chem(planet,FPARA_INDEX,kk,J,I);
            temperature = T2(K,J,I);
            return_enthalpy(planet,fpara,P(K,J,I),temperature,&fgibb,&fpe,&uoup);
            MONT(J,I) -= fgibb*(FPARA(K,J,I)-FPARA(K+1,J,I));
          }
        }
      }
      BC2D(&(MONT(JLO,ILO)),NO_INDEX,1);
    }

    if (K > 1) {
      theta = grid.theta[2*K-1];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          if (sw) {
            /* kappa = 1. */
            EXNER(J,I) = (planet->cp)*P(K,J,I)/grid.press0;
          }
          else {
            /* Calculate exner for the top of current layer: */
            EXNER(J,I) = (planet->cp)*T2(K-1,J,I)/theta;

            /* Finish calculation of wh0: */
            WH0(J,I) /= EXNER(J,I); 
          }        
        }
      }
    }
    if (!sw) {
      BC2D(&(WH0(JLO,ILO)),NO_INDEX,1);
    }
    /* No need to apply horizontal boundary conditions to exner */

    /*
     * Finish calculation of vertical component of EP flux.
     */
    if (K > 1 && !sw) {
      for (J = JLO; J <= JHI; J++) {
        WH_AVG(J) = 0.;
        U_AVG( J) = 0.;
        for (I = ILO; I <= IHI; I++) {
          WH_AVG(J) += WH1(J,I);
          U_AVG( J) += U(K,J,I)+U(K,J,I+1)+U(K-1,J,I)+U(K-1,J,I+1);
        }
        /* NOTE: only works if i-direction not decomposed: */
        WH_AVG(J) /= grid.ni;
        U_AVG( J) /= grid.ni;
        if (grid.ni > 1) {
          for (I = ILO; I <= IHI; I++) {
            EP_FLUX_Z(K,J) += P(K,J,I)/planet->g*
                             (MONT(J,I+1)-MONT(J,I-1))*grid.m[2*J+1]*.25+
                             (WH1(J,I)-WH_AVG(J))*
                             (U(K,J,I)+U(K,J,I+1)+U(K-1,J,I)+U(K-1,J,I+1)-U_AVG(J))*.25;
          }
          /* NOTE: only works if i-direction not decomposed: */
          EP_FLUX_Z(K,J) /= -(double)grid.ni*grid.m[2*J+1]*grid.dln*DEG;
        }
        else {
          /* 
           * For ni = 1 case, store w in place of ep_flux_z. Note that
           * HEAT and EXNER refer to the top of the layer, but we'll make
           * w refer to the bottom of the layer, hence the K-1 index.
           */
          EP_FLUX_Z(K-1,J) = HEAT(J,ILO)/EXNER(J,ILO);
        }
      }
    }

    if (!sw) {
      /*
       *  Add wh terms to dhdt.
       *  Add vertical divergence to thermodynamical tendencies and 
       *  source-sink terms.
       */
      theta = grid.theta[2*K];

      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          DHDT(IT_ZERO,K,J,I) += (WH1(J,I)-WH0(J,I))*d_th1_inv;
        }
        for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
          /* 
           * Passive advection terms: 
           */
          if (var.chem_on[index]) {
            if (K > 1) {
              for (I = ILO; I <= IHI; I++) {
                DVARDT(index,IT_ZERO,K,J,I) += 
                  (WH1(J,I)*get_chem(planet,index,2*K+1,J,I)
                  -WH0(J,I)*get_chem(planet,index,2*K-1,J,I))*d_th1_inv;
              }
            }
            else {
              /* wh0 = 0. */
              for (I = ILO; I <= IHI; I++) {
                DVARDT(index,IT_ZERO,K,J,I) += 
                  (WH1(J,I)*get_chem(planet,index,2*K+1,J,I)
                  -0.)*d_th1_inv;
              }
            }
          }
        }
        /*
         * Specific source-sink terms:
         */
        if (var.chem_on[FPARA_INDEX] == CHEM_ACTIVE) {
          for (I = ILO; I <= IHI; I++) {
            fpara       = FPARA(K,J,I);
            pressure    = P1(   K,J,I);
            temperature = T(    K,J,I);
            return_enthalpy(planet,fpara,pressure,temperature,&fgibb,&fpe,&uoup);
            time_fp_inv = pressure/var.time_fp_bar;

            time_fp_inv *= grid.hasten;

            dfpdt = (fpe-fpara)*time_fp_inv;
            DHFPARADT(IT_ZERO,K,J,I) += H(J,I)*dfpdt;
          }
        }
      }
    }

    /*
     *  Add mont terms to dudt and dvdt:
     */
    for (J = JLO; J <= JHI; J++) {
      m_2jp1 = (grid.m)[2*J+1];
      for (I = ILO; I <= IHI; I++) {
        DUDT(IT_ZERO,K,J,I) += (MONT(J,I-1)-MONT(J,I))*m_2jp1;
      }
    }
    if (var.chem_on[FPARA_INDEX]) {
      /* add extra term for fpara thermodynamics */
      kk    = 2*K;
      theta = grid.theta[kk];
      for (J = JLO; J <= JHI; J++) {
        m_2jp1 = (grid.m)[2*J+1];
        for (I = ILO; I <= IHI; I++) {
          /* Need to average quantities onto u grid */
          fpara       = .5*(FPARA(K,J,I-1)+FPARA(K,J,I));
          pressure    = .5*(P1(   K,J,I-1)+P1(   K,J,I));
          temperature = return_temp(planet,fpara,pressure,theta);
          return_enthalpy(planet,fpara,pressure,temperature,&fgibb,&fpe,&uoup);
          DUDT(IT_ZERO,K,J,I) += fgibb*(FPARA(K,J,I-1)-FPARA(K,J,I))*m_2jp1;
        }
      }
    }

    for (J = JFIRST; J <= JHI; J++) {
      n_2j   = (grid.n)[2*J];
      for (I = ILO; I <= IHI; I++) {
        DVDT(IT_ZERO,K,J,I) += (MONT(J-1,I)-MONT(J,I))*n_2j;
      }
    }
    if (var.chem_on[FPARA_INDEX]) {
      /* add extra term for fpara thermodynamics */
      kk    = 2*K;
      theta = grid.theta[kk];
      for (J = JFIRST; J <= JHI; J++) {
        n_2j   = (grid.n)[2*J];
        for (I = ILO; I <= IHI; I++) {
          /* Need to average quantities onto v grid */
          fpara       = .5*(FPARA(K,J-1,I)+FPARA(K,J,I));
          pressure    = .5*(P1(   K,J-1,I)+P1(   K,J,I));
          temperature = return_temp(planet,fpara,pressure,theta);
          return_enthalpy(planet,fpara,pressure,temperature,&fgibb,&fpe,&uoup);
          DVDT(IT_ZERO,K,J,I) += fgibb*(FPARA(K,J-1,I)-FPARA(K,J,I))*n_2j;
        }
      }
    }

    if (!sw) {
      /*
       *  Add wh*u and wh*v terms to dudt and dvdt:
       */
      for (J = JLO; J <= JHI; J++) {
        if (K > 1) {
          for (I = ILO; I <= IHI; I++) {
            DUDT(IT_ZERO,K,J,I) += ( (WH0(J,I)+WH0(J,I-1))
	                             *(U(K,J,I)-U(K-1,J,I))
	                             +(WH1(J,I)+WH1(J,I-1))
	                             *(U(K+1,J,I)-U(K,J,I)) )
	                             /( (H(J,I)+H(J,I-1))*d_th1 )*.5;
          }
        }
        else {
          /* wh0 = 0 */
          for (I = ILO; I <= IHI; I++) {
            DUDT(IT_ZERO,K,J,I) += (  (WH1(J,I)+WH1(J,I-1))
	                             *(U(K+1,J,I)-U(K,J,I)) )
	                             /( (H(J,I)+H(J,I-1))*d_th1 )*.5;
          }
        }
      }
      for (J = JFIRST; J <= JHI; J++) {
        if (K > 1) {
          for (I = ILO; I <= IHI; I++) {
            DVDT(IT_ZERO,K,J,I) += ( (WH0(J,I)+WH0(J-1,I))
	                             *(V(K,J,I)-V(K-1,J,I))
	                             +(WH1(J,I)+WH1(J-1,I))
	                             *(V(K+1,J,I)-V(K,J,I)) ) 
	                             /( (H(J,I)+H(J-1,I))*d_th1 )*.5;
          }
        }
        else {
          /* wh0 = 0 */
          for (I = ILO; I <= IHI; I++) {
            DVDT(IT_ZERO,K,J,I) += (  (WH1(J,I)+WH1(J-1,I))
	                             *(V(K+1,J,I)-V(K,J,I)) ) 
	                             /( (H(J,I)+H(J-1,I))*d_th1 )*.5;
          }
        }
      }
    }
  }

#if defined(EPIC_MPI)
  /* Update borders on ep_flux_y, ep_flux_z: */
  MPG_Cart_edgeexch(para.comm_kj,2, 
                    &(para.dimlen[1]),&(para.npad[1]),MPI_DOUBLE, 
                    &(EP_FLUX_Y(Kshift,Jshift)));
  MPG_Cart_edgeexch(para.comm_kj,2,
                    &(para.dimlen[1]),&(para.npad[1]),MPI_DOUBLE,
                    &(EP_FLUX_Z(Kshift,Jshift)));
#endif

  if (grid.ni > 1 && !sw) {
    /* 
     * Compute divergence of EP flux:
     */
    for (K = 1; K <= KLAST_ACTIVE; K++) {
      for (J = JLO; J <= JHI; J++) {
        H_AVG(J) = 0.;
        for (I = ILO; I <= IHI; I++) {
          H_AVG(J) += get_h(planet,2*K,J,I);
        }
        /* NOTE: only works if i-direction not decomposed: */
        H_AVG(J) /= grid.ni;
      }

      if (K > 1) {
        d_th1_inv = 1./(grid.theta[2*K-1]-grid.theta[2*K+1]);
      }
      else {
        /* special delta-thetas in top layer: */
        d_th1_inv = 1./(grid.theta[2*K  ]-grid.theta[2*K+1]);
      }
      for (J = JLO; J <= JHI; J++) {
        if (K > 1) {
          EP_FLUX_DIV(K,J) = (EP_FLUX_Z(K-1,J)-EP_FLUX_Z(K,J))*d_th1_inv;
        }
        else {
          EP_FLUX_DIV(K,J) = (0.-EP_FLUX_Z(K,J))*d_th1_inv;
        }
        m_2jp1     = (grid.m)[2*J+1];
        mn_2jp1    = (grid.mn)[2*J+1];
        if (J == grid.jlo && IS_SPOLE) {
          m_2j_inv = 0.;
        }
        else {
          m_2j_inv   = 1./(grid.m)[2*J  ];
        }
        if (J == grid.nj && IS_NPOLE) {
          m_2jp2_inv = 0.;
        }
        else {
          m_2jp2_inv = 1./(grid.m)[2*J+2];
        }
        EP_FLUX_DIV(K,J) += mn_2jp1*((EP_FLUX_Y(K,J+1)*m_2jp2_inv
                                     -EP_FLUX_Y(K,J  )*m_2j_inv  ) );

        /* NOTE: Normalize ep_flux_div to have units of acceleration: */
        EP_FLUX_DIV(K,J) *= grid.m[2*J+1]*grid.dln*DEG/H_AVG(J);
      }
    }
#if defined(EPIC_MPI)
    /* Update borders on ep_flux_div: */
    MPG_Cart_edgeexch(para.comm_kj,2, 
                      &(para.dimlen[1]),&(para.npad[1]),MPI_DOUBLE, 
                      &(EP_FLUX_DIV(Kshift,Jshift)));
#endif
  }
  if (grid.prandtl > 0.) {
    /* 
     * Add Rayleigh friction: 
     */
    for (K = KLO; K <= KHI; K++) {
      for (J = JLO; J <= JHI; J++) {
        /* compute zonal average of u */
        uavg = 0.;
        for (I = ILO; I <= IHI; I++) {
          /* NOTE: Only valid if I direction isn't cut */
          uavg += U(K,J,I);
        }
        uavg /= (grid.ni);
        if (grid.newt_cool_on == TRUE) {
          for (I = ILO; I <= IHI; I++) {
            pressure = .5*(P1(K,J,I)+P1(K,J,I-1));
            nu0      = grid.hasten*grid.prandtl/t_rad(planet,pressure);
            DUDT(IT_ZERO,K,J,I) -= nu0*(uavg-U_SPINUP(K,J,I));
          }
        }
        else {
          /* grid.prandtl holds 1/t_drag */
          for (I = ILO; I <= IHI; I++) {
            nu0      = grid.hasten*grid.prandtl;
            DUDT(IT_ZERO,K,J,I) -= nu0*(uavg-U_SPINUP(K,J,I));
          }
        }
      }
      for (J = JFIRST; J <= JHI; J++) {
        /* compute zonal average of v */
        vavg = 0.;
        for (I = ILO; I <= IHI; I++) {
          vavg += V(K,J,I);
        }
        vavg /= grid.ni;
        if (grid.newt_cool_on == TRUE) {
          for (I = ILO; I <= IHI; I++) {
            pressure = .5*(P1(K,J,I)+P1(K,J-1,I));
            nu0 = grid.hasten*grid.prandtl/t_rad(planet,pressure);
            DVDT(IT_ZERO,K,J,I) -= nu0*vavg;
          }
        }
        else {
          /* grid.prandtl holds 1/t_drag */
          for (I = ILO; I <= IHI; I++) {
            nu0 = grid.hasten*grid.prandtl;
            DVDT(IT_ZERO,K,J,I) -= nu0*vavg;
          }
        }
      }
    }
  }
  /*
   *  Add hyperviscosity to control numerical instability.
   */
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    /* 
     * Apply viscosity and/or hyperviscosity to (u,v).
     *
     * Determine if all viscosity coefficients are zero: 
     */
    tmp = 0.;
    for (itmp2 = 2; itmp2 <= MAX_NU_ORDER; itmp2+=2) {
      tmp += grid.nu[itmp2];
    }
    if (tmp > 0.) {
      memset(BuffD[0],0,Nelem2d*sizeof(double));
      memset(BuffD[1],0,Nelem2d*sizeof(double));
      memset(BuffD[2],0,Nelem2d*sizeof(double));
      memset(BuffD[3],0,Nelem2d*sizeof(double));
      u_pt = extract_scalar(0,K,BuffD[0]);
      v_pt = extract_scalar(1,K,BuffD[1]);
      lap1 = BuffD[2];
      lap2 = BuffD[3];
      sign = 1;
      itmp = 2;
      while (tmp > 0.) {
        laplacian(planet,K,u_pt,v_pt,lap1,lap2,BuffD[4],BuffD[5],NULL);
        if (grid.nu[itmp] > 0.) {
          /* Apply viscosity */
          for (J = JLO; J <= JHI; J++) {
            /* Scale viscosity coefficient to avoid CFL violation */
            dx = sqrt(1./(grid.m[2*J+1]*grid.m[2*J+1]
                         +grid.n[2*J+1]*grid.n[2*J+1]));
            nu = (double)sign*(grid.nu[itmp])*pow(MIN(1.,dx/dx0),(double)itmp);
            for (I = ILO; I <= IHI; I++) {
              DUDT(IT_ZERO,K,J,I) += nu*LAP1(J,I);
            }
          }
          for (J = JFIRST; J <= JHI; J++) {
            /* scale viscosity coefficient to avoid CFL violation */
            dx = sqrt(1./(grid.m[2*J]*grid.m[2*J]
                         +grid.n[2*J]*grid.n[2*J]));
            nu = (double)sign*(grid.nu[itmp])*pow(MIN(1.,dx/dx0),(double)itmp);
            for (I = ILO; I <= IHI; I++) {
              DVDT(IT_ZERO,K,J,I) += nu*LAP2(J,I);
            }
          }
        }
        sign *= -1;
        itmp +=  2;
        ptmp  = u_pt;
        u_pt  = lap1;
        lap1  = ptmp;
        ptmp  = v_pt;
        v_pt  = lap2;
        lap2  = ptmp;
        /* determine if all remaining viscosity coefficients are zero */
        tmp = 0.;
        for (itmp2 = itmp; itmp2 <= MAX_NU_ORDER; itmp2+=2) {
          tmp += grid.nu[itmp2];
        }
      }
    }

    /* 
     * Apply hyperviscosity to p and the thermodynamical variables.
     *
     * Determine if all hyperviscosity coefficients are zero: 
     */
    tmp = 0.;
    for (itmp2 = 4; itmp2 <= MAX_NU_ORDER; itmp2+=2) {
      tmp += grid.nu[itmp2];
    }
    if (tmp > 0.) {
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          H(J,I) = get_h(planet,2*K,J,I);
        }
      }
      /* No need to apply BC2D to H here */

      /*
       * Cycle through mass variables:
       */
      for (index = P_INDEX; index <= LAST_HUMIDITY; index++) {
        if (var.index[index] > -1) {
          /* Prepare variable */
          memset(BuffD[0],0,Nelem2d*sizeof(double));
          memset(BuffD[1],0,Nelem2d*sizeof(double));
          h_pt    = BuffD[0];
          laph    = BuffD[1];
          dh_ptdt = var.vector[MAX_NVARS+index]+
                    (K-KLO+KPAD)*Nelem2d+IT_ZERO*Nelem3d;
          if (index == 2) {
            /* variable is h */
            for (J = JLO; J <= JHI; J++) {
              for (I = ILO; I <= IHI; I++) {
                H_PT(J,I) = H(J,I);
              }
            }
            BC2D(&(H_PT(JLO,ILO)),NO_INDEX,1);
          }
          else {
            h_pt = extract_scalar(index,K,h_pt);
            /* Weight thermodynamical variable by h */
            for (J = JLO; J <= JHI; J++) {
              for (I = ILO; I <= IHI; I++) {
                H_PT(J,I) *= H(J,I);
              }
            }
            BC2D(&(H_PT(JLO,ILO)),NO_INDEX,1);
          }
          sign = 1;
          itmp = 2;
          tmp  = 1.;
          while (tmp > 0.) {
            laplacian(planet,K,h_pt,NULL,laph,NULL,BuffD[4],BuffD[5],NULL);
            if (grid.nu[itmp] > 0. && itmp >= 4) {
              /* Apply hyperviscosity */
              for (J = JLO; J <= JHI; J++) {
                /* Scale viscosity coefficient to avoid CFL violation */
                dx = sqrt(1./(grid.m[2*J+1]*grid.m[2*J+1]
                             +grid.n[2*J+1]*grid.n[2*J+1]));
                nu = (double)sign*(grid.nu[itmp])*pow(MIN(1.,dx/dx0),(double)itmp);
                for (I = ILO; I <= IHI; I++) {
                  DH_PTDT(J,I) += nu*LAPH(J,I);
                }
              }
            }
            sign *= -1;
            itmp +=  2;
            ptmp  = h_pt;
            h_pt  = laph;
            laph  = ptmp;
            /* determine if all remaining viscosity coefficients are zero */
            tmp = 0.;
            for (itmp2 = itmp; itmp2 <= MAX_NU_ORDER; itmp2+=2) {
              tmp += grid.nu[itmp2];
            }
          }
        }
      }
    }
  }

  /*
   *  Add Rayleigh friction term to damp gravity-wave reflections
   *  at the model's top.  This is the "sponge."
   */
  for (K = 1; K <= grid.k_sponge; K++) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        DUDT(IT_ZERO,K,J,I) -= (grid.t_sponge_inv[K])*(U(K,J,I)-U_SPINUP(K,J,I));
      }
    }
    for (J = JFIRST; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        DVDT(IT_ZERO,K,J,I) -= (grid.t_sponge_inv[K])*V(K,J,I);
      }
    }
  }  
  /*
   *  Filter tendencies to remove CFL violation at poles:
   */
  if (strcmp(grid.geometry,"globe")    == 0 ||
     (strcmp(grid.geometry,"f-plane")  == 0 &&
      strcmp(grid.f_plane_map,"polar") == 0))  {
    filter();
  }

  if (grid.ni == 1) {
    /* 
     * For ni = 1 case, store -dudt in place of ep_flux_div:
     */
    for (K = 1; K <= KLAST_ACTIVE; K++) {
      for (J = JLO; J <= JHI; J++) {
        EP_FLUX_DIV(K,J) = -DUDT(IT_ZERO,K,J,ILO);
      }
    }
#if defined(EPIC_MPI)
    /* Update borders on ep_flux_div: */
    MPG_Cart_edgeexch(para.comm_kj,2, 
                      &(para.dimlen[1]),&(para.npad[1]),MPI_DOUBLE, 
                      &(EP_FLUX_DIV(Kshift,Jshift)));
#endif
  }

  /*
   * Advance variables.
   *
   * Start by converting variables into form used in tendencies.
   *
   * Convert p to h. This is done in place by starting at K = KLAST_ACTIVE. 
   * Convert thermodynamical variables to h-weighted.
   */
  for (K = KLAST_ACTIVE; K >= 1; K--) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        H(J,I) = get_h(planet,2*K,J,I);
      }
    }
    /* No need to apply BC2D to H here */

    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        /* Use p to store h */
        P(K,J,I) = H(J,I);
      }
      for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
        if (var.chem_on[index]) {
          for (I = ILO; I <= IHI; I++) {
            VAR(index,K,J,I) *= H(J,I);
          }
        }
      }
    }
  }
  /* No need to apply horizontal boundary conditions here */

  /* 
   * Start at k = 1 to facilitate integration of h to get p 
   */
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        U(K,J,I) += DT*( ab[0]*DUDT(IT_ZERO,  K,J,I)
                        +ab[1]*DUDT(IT_MINUS1,K,J,I)
                        +ab[2]*DUDT(IT_MINUS2,K,J,I) );
      }
    }
    for (J = JFIRST; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        V(K,J,I) += DT*( ab[0]*DVDT(IT_ZERO,  K,J,I)
                        +ab[1]*DVDT(IT_MINUS1,K,J,I)
                        +ab[2]*DVDT(IT_MINUS2,K,J,I) );
      }
    }
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        /* p is holding h */
        h_old = P(K,J,I);
        h_new = P(K,J,I)+DT*( ab[0]*DHDT(IT_ZERO,  K,J,I)
                             +ab[1]*DHDT(IT_MINUS1,K,J,I)
                             +ab[2]*DHDT(IT_MINUS2,K,J,I) );

        if (h_new > H_TINY) {
          P(K,J,I) = h_new;
        }
        else {
          /* 
           * This is a "massless" region.
           * Apply H_TINY threshold, and set winds equal to 
           * winds in layer above: 
           */
          P(K,J,I) = H_TINY;
          if (K > 1) {
            U(K,J,I) = U(K-1,J,I);
            V(K,J,I) = V(K-1,J,I);
          }
        }
      }
      for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
        if (var.chem_on[index]) {
          for (I= ILO; I <= IHI; I++) {
            /* var is h weighted */
            VAR(index,K,J,I) += DT*( ab[0]*DVARDT(index,IT_ZERO,  K,J,I)
                                    +ab[1]*DVARDT(index,IT_MINUS1,K,J,I)
                                    +ab[2]*DVARDT(index,IT_MINUS2,K,J,I) );
            /* restore var to not h weighted */
            VAR(index,K,J,I) /= P(K,J,I);
          }
        }
      }
      /* 
       * Restore p by integrating h:
       */
      if (K > 1) {
        d_th1 = planet->g*(grid.theta[2*K-1]-grid.theta[2*K+1]);
        for (I = ILO; I <= IHI; I++) {
          /* p(k) is holding h(k) = -1/g dp/dtheta */
          P(K,J,I) = P(K-1,J,I)+P(K,J,I)*d_th1;
        }
      }
      else {
        /* p = 0 at top of top layer; special d_th1 at top layer */
        d_th1 = planet->g*(grid.theta[2*K]-grid.theta[2*K+1]);
        for (I = ILO; I <= IHI; I++) {
          /* p(k) is holding h(k) */
          P(K,J,I) = 0.+P(K,J,I)*d_th1;
        }
      }
    }
    /* Apply boundary conditions: */
    for (index = 0; index < MAX_NVARS; index++) {
      if (var.chem_on[index]) {
        BC2D(&(VAR(index,KLO,JLO,ILO)),index,K);
      }
    }
  }

  if (strcmp(planet->class,"gas-giant") == 0) {
    /*
     *  Advance gas-giant deep-layer zonal wind:
     */
    K = KHI;
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
          U(K,J,I) += DT*( ab[0]*DUDT(IT_ZERO,  K,J,I)
                          +ab[1]*DUDT(IT_MINUS1,K,J,I)
                          +ab[2]*DUDT(IT_MINUS2,K,J,I) );
      }
    }
    BC2D(&(U(KLO,JLO,ILO)),U_INDEX,K);
  }

  /* 
   * The variables are now all advanced and restored to their usual forms. 
   */

  /*
   *  Advance time:
   */
  var.time[0] += grid.dt;
  if (var.time[0] >= YEAR) {
    var.time[1]++;
    var.time[0] -= YEAR;
  }

  /*
   *  Cycle time index backwards:
   */
  itmp      = IT_MINUS2;
  IT_MINUS2 = IT_MINUS1;
  IT_MINUS1 = IT_ZERO;
  IT_ZERO   = itmp;

  return;
}

/*======================= end of timestep() =====================================*/

/*======================= hsu_predict_correct() =================================*/

/* 
 * Used to convert answer into the form of a 3rd-Order Adams-Bashforth tendency: 
 */
#define DADTM1(j,i) dadtm1[i+(j)*Iadim-Shift2d]
#define DADTM2(j,i) dadtm2[i+(j)*Iadim-Shift2d]

/*
 *  Hsu and Arakawa's horizontal predictor-corrector timestep for the
 *  continuity equations. This is a positive-definite advection scheme 
 *  that handles steep gradients.
 */
void hsu_predict_correct(double *h,
                         int     K,
                         double *ab) {
  int   
    J,I,
    index;
  double
    *a,*a_diff1,*a_diff2,*a_old,
    *um,*up,*u2d,*vm,*vp,*v2d,
    uhatp,uhatm,vhatp,vhatm,mu,
    al,gap,gam,bep,bem,behatp,behatm,g,
    mn_2jp1,mn_2j,m_2j_inv,n_2jp1_inv,tmp,
    *ff,*a_pred,*aaa,
    *dadtm1,*dadtm2;

  /* 
   * Zonal step:
   */
  memset(BuffD[0],0,Nelem2d*sizeof(double));
  memset(BuffD[1],0,Nelem2d*sizeof(double));
  memset(BuffD[2],0,Nelem2d*sizeof(double));
  um  = BuffD[0];
  up  = BuffD[1];
  u2d = extract_scalar(0,K,BuffD[2]);
      
  for (J = JLO; J <= JHI; J++) {
    n_2jp1_inv = 1./(grid.n)[2*J+1];
    for (I = ILO; I <= IHI; I++) {
      UM(J,I) = MIN(0.,U2D(J,I))*n_2jp1_inv;
      UP(J,I) = MAX(0.,U2D(J,I))*n_2jp1_inv;
    }
  }
  BC2D(&(UM(JLO,ILO)),NO_INDEX,1);
  BC2D(&(UP(JLO,ILO)),NO_INDEX,1);
  /* done with u2d memory */

  /*  
   *  Cycle through mass variables:
   */
  for (index = P_INDEX; index <= LAST_HUMIDITY; index++) {
    if (var.chem_on[index]) {
      /* use tendency memory for working space */
      a = var.vector[MAX_NVARS+index]+(K-KLO+KPAD)*Nelem2d+IT_ZERO*Nelem3d;
      if (index == P_INDEX) {
        /* variable is h */
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            A(J,I) = H(J,I);
          }
        }
      }
      else {
        /* weight by h */
        a = extract_scalar(index,K,a);
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            A(J,I) *= H(J,I);
          }
        }
      }
      BC2D(&(A(JLO,ILO)),NO_INDEX,1);

      /* 
       * Predictor: 
       */
      memset(BuffD[4],0,Nelem2d*sizeof(double));
      ff = BuffD[4];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          FF(J,I) = UP(J,I)*A(J,I-1)+UM(J,I)*A(J,I);
        }
      }
      BC2D(&(FF(JLO,ILO)),NO_INDEX,1);
      memset(BuffD[5],0,Nelem2d*sizeof(double));
      a_pred = BuffD[5];
      for (J = JLO; J <= JHI; J++) {
        mn_2jp1 = (grid.mn)[2*J+1];
        for (I = ILO; I <= IHI; I++) {
          A_PRED(J,I) = A(J,I)+DT*(FF(J,I)-FF(J,I+1))*mn_2jp1;
        }
      }
      BC2D(&(A_PRED(JLO,ILO)),NO_INDEX,1);
      /* 
       * Corrector: 
       */
      memset(BuffD[2],0,Nelem2d*sizeof(double));
      memset(BuffD[3],0,Nelem2d*sizeof(double));
      a_diff1 = BuffD[2];
      a_diff2 = BuffD[3];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          FF(J,I) = .5*(UP(J,I)*(A_PRED(J,I  )+A(J,I-1))
                       +UM(J,I)*(A_PRED(J,I-1)+A(J,I  )));
          A_DIFF1(J,I) = A_PRED(J,I)-A(J,I-1);
          A_DIFF2(J,I) = A_PRED(J,I)-A(J,I+1);
        }
      }
      BC2D(&(A_DIFF1(JLO,ILO)),NO_INDEX,1);
      BC2D(&(A_DIFF2(JLO,ILO)),NO_INDEX,1);
      /* finished with a_pred memory */
      memset(BuffD[5],0,Nelem2d*sizeof(double));
      aaa = BuffD[5];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          AAA(J,I)  = (A(J,I-1)-2.*A(J,I)+A(J,I+1))*
                      (A(J,I-1)-2.*A(J,I)+A(J,I+1));
        }
      }
      BC2D(&(AAA(JLO,ILO)),NO_INDEX,1);
      for (J = JLO; J <= JHI; J++) {
        mn_2jp1 = (grid.mn)[2*J+1];
        for (I = ILO; I <= IHI; I++) {
          uhatp  =  sqrt(UP(J,I)*UP(J,I-1));
          uhatm  = -sqrt(UM(J,I)*UM(J,I+1));
          /* Courant number, mu */
          mu     = (UP(J,I)-UM(J,I))*mn_2jp1*DT;
          mu     = MIN(mu,.5);
          al     = (1.+mu)/6.;
          gap    = AAA(J,I-1)/(AAA(J,I-1)+A(J,I-1)*A(J,I));
          gap    = gap*gap;
          gam    = AAA(J,I  )/(AAA(J,I  )+A(J,I-1)*A(J,I));
          gam    = gam*gam;
          bep    = 1.+(1./(2.*al)-1.)*gap;
          bem    = 1.+(1./(2.*al)-1.)*gam;
          behatp = 1.-gap;
          behatm = 1.-gam;
          g = -al*( UP(J,I)*bep*A_DIFF1(J,I  )
                  -uhatp*behatp*A_DIFF1(J,I-1)
                   +UM(J,I)*bem*A_DIFF2(J,I-1)
                  -uhatm*behatm*A_DIFF2(J,I  ));
          FF(J,I) += g;
        }  
      }
      BC2D(&(FF(JLO,ILO)),NO_INDEX,1);
      for (J = JLO; J <= JHI; J++) {
        mn_2jp1 = (grid.mn)[2*J+1];
        for (I = ILO; I <= IHI; I++) {
          A(J,I) += DT*(FF(J,I)-FF(J,I+1))*mn_2jp1;
        }
      }
      BC2D(&(A(JLO,ILO)),NO_INDEX,1);
    }
  }

  /* 
   * Meridional step:
   */
  memset(BuffD[0],0,Nelem2d*sizeof(double));
  memset(BuffD[1],0,Nelem2d*sizeof(double));
  memset(BuffD[2],0,Nelem2d*sizeof(double));
  vm = BuffD[0]; 
  vp = BuffD[1];
  v2d = extract_scalar(1,K,BuffD[2]);

  for (J = JFIRST; J <= JHI; J++) {
    m_2j_inv = 1./(grid.m)[2*J];
    for (I = ILO; I <= IHI; I++) {
      VM(J,I) = MIN(0.,V2D(J,I))*m_2j_inv;
      VP(J,I) = MAX(0.,V2D(J,I))*m_2j_inv;
    }
  }
  BC2D(&(VM(JLO,ILO)),NO_INDEX,1);
  BC2D(&(VP(JLO,ILO)),NO_INDEX,1);
  /* finished with v2d memory */

  /*  
   *  Cycle through mass variables:
   */
  for (index = P_INDEX; index <= LAST_HUMIDITY; index++) {
    if (var.chem_on[index]) {
      a = var.vector[MAX_NVARS+index]+(K-KLO+KPAD)*Nelem2d+IT_ZERO*Nelem3d;
      /* 
       * Predictor: 
       */
      memset(BuffD[4],0,Nelem2d*sizeof(double));
      ff = BuffD[4];
      for (J = JFIRST; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          FF(J,I) = VP(J,I)*A(J-1,I)+VM(J,I)*A(J,I);
        }
      }
      BC2D(&(FF(JLO,ILO)),NO_INDEX,1);
      memset(BuffD[5],0,Nelem2d*sizeof(double));
      a_pred = BuffD[5];
      for (J = JLO; J <= JHI; J++) {
        mn_2jp1 = (grid.mn)[2*J+1];
        for (I = ILO; I <= IHI; I++) {
          A_PRED(J,I) = A(J,I)+DT*(FF(J,I)-FF(J+1,I))*mn_2jp1;
        }
      }
      BC2D(&(A_PRED(JLO,ILO)),NO_INDEX,1);

      /*
       * Corrector:
       */
      for (J = JFIRST; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          FF(J,I) = .5*(VP(J,I)*(A_PRED(J,I  )+A(J-1,I))
                       +VM(J,I)*(A_PRED(J-1,I)+A(J,I  )));
        }
      }
      memset(BuffD[2],0,Nelem2d*sizeof(double));
      memset(BuffD[3],0,Nelem2d*sizeof(double));
      a_diff1 = BuffD[2];
      a_diff2 = BuffD[3];
      if (strcmp(grid.geometry,"f-plane") == 0 
          && strcmp(grid.f_plane_map,"cartesian") == 0) {
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            A_DIFF1(J,I) = A_PRED(J,I)-A(J-1,I);
            A_DIFF2(J,I) = A_PRED(J,I)-A(J+1,I);
          }
        }
      }
      else if (strcmp(grid.geometry,"globe") == 0 ||
              (strcmp(grid.geometry,"f-plane") == 0 
               && strcmp(grid.f_plane_map,"polar") == 0))  {
        /* NOTE: do interior points, ie J = JFIRST; J < JHI */
        for (J = JFIRST; J < JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            A_DIFF1(J,I) = A_PRED(J,I)-A(J-1,I);
            A_DIFF2(J,I) = A_PRED(J,I)-A(J+1,I);
          }
        }
        /* special a_diff's at ends */
        J = JLO;
        if (JLO == grid.jlo) {
          for (I = ILO; I <= IHI; I++) {
            A_DIFF1(J,I) = 0.;
            A_DIFF2(J,I) = A_PRED(J,I)-A(J+1,I);
          }
        }
        J = JHI;
        if (JHI == grid.nj) {
          /* edge */
          for (I = ILO; I <= IHI; I++) {
            A_DIFF1(J,I) = A_PRED(J,I)-A(J-1,I);
            A_DIFF2(J,I) = 0.;
          }
        }
        else {
          /* interior */
          for (I = ILO; I <= IHI; I++) {
            A_DIFF1(J,I) = A_PRED(J,I)-A(J-1,I);
            A_DIFF2(J,I) = A_PRED(J,I)-A(J+1,I);
          }
        }
      } 
      else {
        fprintf(stderr,"Unrecognized geometry in epic_timestep \n");
        exit(1);
      }
      BC2D(&(A_DIFF1(JLO,ILO)),NO_INDEX,1);
      BC2D(&(A_DIFF2(JLO,ILO)),NO_INDEX,1);

      /* finished with a_pred memory */
      memset(BuffD[5],0,Nelem2d*sizeof(double));
      aaa = BuffD[5];
      for (J = JLO; J <= JHI; J++) {
        for (I = ILO; I <= IHI; I++) {
          AAA(J,I)  = (A(J-1,I)-2.*A(J,I)+A(J+1,I))*
                      (A(J-1,I)-2.*A(J,I)+A(J+1,I));
        }
      }
      BC2D(&(AAA(JLO,ILO)),NO_INDEX,1);
      for (J = JFIRST; J <= JHI; J++) {
        mn_2j = (grid.mn)[2*J];
        for (I = ILO; I <= IHI; I++) {
          vhatp  =  sqrt(VP(J,I)*VP(J-1,I));
          vhatm  = -sqrt(VM(J,I)*VM(J+1,I));
          /* Courant number, mu */
          mu     = (VP(J,I)-VM(J,I))*mn_2j*DT;
          mu     = MIN(mu,.5);
          al     = (1.+mu)/6.;
          gap    = AAA(J-1,I)/(AAA(J-1,I)+A(J-1,I)*A(J,I));
          gap    = gap*gap;
          gam    = AAA(J,I  )/(AAA(J,I  )+A(J-1,I)*A(J,I));
          gam    = gam*gam;
          bep    = 1.+(1./(2.*al)-1.)*gap;
          bem    = 1.+(1./(2.*al)-1.)*gam;
          behatp = 1.-gap;
          behatm = 1.-gam;
          g = -al*( VP(J,I)*bep*A_DIFF1(J,  I)
                  -vhatp*behatp*A_DIFF1(J-1,I)
                   +VM(J,I)*bem*A_DIFF2(J-1,I)
                  -vhatm*behatm*A_DIFF2(J,  I));
          FF(J,I) += g;
        }  
      }
      BC2D(&(FF(JLO,ILO)),NO_INDEX,1);

      /* 
       * Store answer as 3rd order Adams-Bashforth tendency:
       */
      memset(BuffD[2],0,Nelem2d*sizeof(double));
      a_old = BuffD[2];
      if (index == P_INDEX) {
        /* variable is h */
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            A_OLD(J,I) = H(J,I);
          }
        }
      }
      else {
        a_old = extract_scalar(index,K,a_old);
        /* weight thermodynamical variable by h */
        for (J = JLO; J <= JHI; J++) {
          for (I = ILO; I <= IHI; I++) {
            A_OLD(J,I) *= H(J,I);
          }
        }
      }
      /* No need to apply horizontal boundary conditions to a_old */

      dadtm1 = var.vector[MAX_NVARS+index]+(K-KLO+KPAD)*Nelem2d+IT_MINUS1*Nelem3d;
      dadtm2 = var.vector[MAX_NVARS+index]+(K-KLO+KPAD)*Nelem2d+IT_MINUS2*Nelem3d;
      for (J = JLO; J <= JHI; J++) {
        mn_2jp1 = (grid.mn)[2*J+1];
        for (I = ILO; I <= IHI; I++) {
          A(J,I) += DT*(FF(J,I)-FF(J+1,I))*mn_2jp1;
          /* Convert to the form of an Adams-Bashforth tendency: */
          A(J,I) = ((A(J,I)-A_OLD(J,I))/DT-ab[1]*DADTM1(J,I)
                                          -ab[2]*DADTM2(J,I))/ab[0];
        }
      }
    }
  }

  return;
}

/*======================= end of hsu_predict_correct() ==========================*/

/*======================= horizontal_divergence_terms() =========================*/

void horizontal_divergence_terms(double  *uh,
                                 double  *vh,
                                 int      K) 
{
    /*
     * Simple horizontal divergence code for calculating tendency terms.
     */
  int
    J,I,
    index;
  double
    n_2jp1_inv,
    m_2j_inv,
    m_2jp2_inv,
    mn_2jp1;

  for (J = JLO; J <= JHI; J++) {
    mn_2jp1 = (grid.mn)[2*J+1];
    for (I = ILO; I <= IHI; I++) {
      DHDT(IT_ZERO,K,J,I) += (UH(J,I)-UH(J,I+1)
                             +VH(J,I)-VH(J+1,I))*mn_2jp1;
    }
    for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
      if (var.chem_on[index]) {
        for (I = ILO; I <= IHI; I++) {
          DVARDT(index,IT_ZERO,K,J,I) += 
             (UH(J,  I  )*(VAR(index,K,J,  I  )+VAR(index,K,J,  I-1))
             -UH(J,  I+1)*(VAR(index,K,J,  I+1)+VAR(index,K,J,  I  ))
             +VH(J,  I  )*(VAR(index,K,J,  I  )+VAR(index,K,J-1,I  ))
             -VH(J+1,I  )*(VAR(index,K,J+1,I  )+VAR(index,K,J,  I  )))*mn_2jp1*.5;
        }
      }
    }
  }

  return;
}

/*======================= end of horizontal_divergence_terms() ==================*/

/*======================= filter() ==============================================*/

void filter(void)
/* 
 * Filter to remove the CFL violation at the poles.  
 *
 * NOTE: We experimented with the filter scheme described by 
 * Kar, Turco, Mechoso, Arakawa (1994) MWR 122, 205-222, but it did not work.
 * We are using a filter that works in practice, even though it may suffer
 * minor problems like spurious vorticity creation as described by 
 * Kar et al.
 *
 * NOTE: ni must be an integer power of 2, in order to use realft().
 */
{
  int 
    K,J,I,
    index;
  double
    *a;
  static int 
    initialized=0;
  static double 
    *data, 
    *lowpass_h,
    *lowpass_v;

  /* no need to filter if ni = 1 */
  if (grid.ni == 1) {
    return;
  }

  if(!initialized) {
    double 
      m0,n0,rln,rlt,lat0,re,rp,
      r,tmp0,tmp1;

    initialized=1;
    data      = dvector(1,grid.ni);
    lowpass_h = dvector(0,Nelem2d-1);
    lowpass_v = dvector(0,Nelem2d-1);

    if (strcmp(grid.geometry,"globe") == 0) {
      re   = planet->re;
      rp   = planet->rp;
      /* Filter is applied poleward of LAT0 */
      lat0 = LAT0*DEG;  
      rln  = re/sqrt( 1.+ pow(rp/re*tan(lat0),2.) );
      rlt  = rln/( cos(lat0)*( pow(sin(lat0),2.) +
                   pow(re/rp*cos(lat0),2.) ) );
      m0 = 1./(rln*grid.dln*DEG);
      n0 = 1./(rlt*grid.dlt*DEG);
    }
    else if (strcmp(grid.geometry,"f-plane")  == 0 &&
             strcmp(grid.f_plane_map,"polar") == 0) {
      /* nj = 2*(nj+1-1)/2 */
      m0 = grid.m[grid.nj];
      n0 = grid.n[grid.nj];
    }
    /* h, u grid */
    for (J = JLO; J <= JHI; J++) {
      if (fabs( (grid.lat)[2*J+1] ) >= (double)LAT0) {
        LOWPASS_H(J,1) = 1.;       
        /* r is roughly 1.5 at a pole, 1 otherwise: */  
        r    = grid.mn[2*J+1]/(grid.m[2*J+1]*grid.n[2*J+1]);
        tmp0 = ((grid.n)[2*J+1]/n0)/((grid.m)[2*J+1]/m0);
        for (I = 2; I <= grid.ni/2+1; I++) {
          tmp1 = 1./sin((I-1)*(grid.dln)*DEG/2.);
          /* 
           * Increase filter strength by taking square root. 
           * This was found to be needed for Neptune runs.
           */
          tmp1 = sqrt(tmp1);
          tmp1 *= r*tmp0;
          tmp1 = (tmp1 < 1.) ? tmp1 : 1.;
          LOWPASS_H(J,I) = tmp1;
        }
      }
    }
    /* v grid */
    for (J = JFIRST; J <= JHI; J++) {
      if (fabs( (grid.lat)[2*J] ) >= (double)LAT0) {
        LOWPASS_V(J,1) = 1.;       
        /* r is roughly 1.5 at a pole, 1 otherwise: */  
        r    = grid.mn[2*J]/(grid.m[2*J]*grid.n[2*J]);
        tmp0 = ((grid.n)[2*J]/n0)/((grid.m)[2*J]/m0);
        for (I = 2; I <= grid.ni/2+1; I++) {
          tmp1 = 1./sin((I-1)*(grid.dln)*DEG/2.);
          /* 
           * Increase filter strength by taking square root. 
           * This was found to be needed for Neptune runs.
           */
          tmp1 = sqrt(tmp1);
          tmp1 *= r*tmp0;
          tmp1 = (tmp1 < 1.) ? tmp1 : 1.;
          LOWPASS_V(J,I) = tmp1;
        }
      }
    }
  } 
  /* end of initialization */

  for (K = KLO; K <= KLAST_ACTIVE; K++) {
    /* 
     * Filter dhdt and the thermodynamical tendencies: 
     */
    for (index = P_INDEX; index <= LAST_HUMIDITY; index++) {
      if (var.chem_on[index]) {
        a = var.vector[MAX_NVARS+index]+
            (K-KLO+KPAD)*Nelem2d+IT_ZERO*Nelem3d;
        for (J = JLO; J <= JHI; J++) {
          if (fabs( (grid.lat)[2*J+1] ) >= (double)LAT0) {
            for (I = 1; I <= grid.ni; I++) {
              data[I] = A(J,I);
            }
            realft(data,grid.ni,1);
            data[1] *= LOWPASS_H(J,1);
            for (I = 2; I <= (grid.ni)/2; I++) {
              data[2*I-1] *= LOWPASS_H(J,I);
              data[2*I  ] *= LOWPASS_H(J,I);
            }
            data[2] *= LOWPASS_H(J,(grid.ni)/2+1);
            realft(data,grid.ni,-1);
            for (I = 1; I <= grid.ni; I++) {
              A(J,I) = data[I]*(2./(grid.ni));
            }
          }
        }
      }
    }
    /*
     * Filter dudt (use lowpass_h since u and h have same grid in latitude):
     */
    for (J = JLO; J <= JHI; J++) {
      if (fabs( (grid.lat)[2*J+1] ) >= (double)LAT0) {
        for (I = 1; I <= grid.ni; I++) {
          data[I] = DUDT(IT_ZERO,K,J,I);
        }
        realft(data,grid.ni,1);
        data[1] *= LOWPASS_H(J,1);
        for (I = 2; I <= (grid.ni)/2; I++) {
          data[2*I-1] *= LOWPASS_H(J,I);
          data[2*I  ] *= LOWPASS_H(J,I);
        }
        data[2] *= LOWPASS_H(J,(grid.ni)/2+1);
        realft(data,grid.ni,-1);
        for (I = 1; I <= grid.ni; I++) {
          DUDT(IT_ZERO,K,J,I) = data[I]*(2./(grid.ni));
        }
      }
    }
    /* 
     * Filter dvdt: 
     */
    for (J = JFIRST; J <= JHI; J++) {
      if (fabs( (grid.lat)[2*J] ) >= (double)LAT0) {
        for (I = 1; I <= grid.ni; I++) {
          data[I] = DVDT(IT_ZERO,K,J,I);
        }
        realft(data,grid.ni,1);
        data[1] *= LOWPASS_V(J,1);
        for (I = 2; I <= (grid.ni)/2; I++) {
          data[2*I-1] *= LOWPASS_V(J,I);
          data[2*I  ] *= LOWPASS_V(J,I);
        }
        data[2] *= LOWPASS_V(J,(grid.ni)/2+1);
        realft(data,grid.ni,-1);
        for (I = 1; I <= grid.ni; I++) {
          DVDT(IT_ZERO,K,J,I) = data[I]*(2./(grid.ni));
        }
      }
    }
  }

  return;
}

/*======================= end of filter() ===================================*/

/*======================= four1() ===========================================*/

      /*
       *  FFT routine.
       *  Numerical Recipes in C, 2nd ed, p. 507.
       *  Assumes data length is a power of two.
       */

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr

void four1(double data[], unsigned long nn, int isign)
{
  unsigned long 
    n, mmax, m, j, istep, i;
  double 
    tempr, tempi;
  double 
    wtemp, wr, wpr, wpi, wi, theta;

  n = nn << 1;
  j = 1;
  for (i = 1; i < n; i+= 2) {
    if (j > i) {
       SWAP(data[j  ],data[i  ]);
       SWAP(data[j+1],data[i+1]);
    }
    m = n >> 1;
    while (m >= 2 && j > m) {
      j -= m;
      m >>= 1;
    }
    j += m;
  }
  mmax = 2;
  while (n > mmax) {
    istep = mmax << 1;
    theta = isign*(2*M_PI/mmax);
    wtemp = sin(0.5*theta);
    wpr = -2.0*wtemp*wtemp;
    wpi = sin(theta);
    wr = 1.0;
    wi = 0.0;
    for (m = 1; m < mmax; m += 2) {
      for (i = m; i <= n; i += istep) {
        j = i + mmax;
        tempr = wr*data[j]-wi*data[j+1];
        tempi = wr*data[j+1]+wi*data[j];
        data[j  ] = data[i  ]-tempr;
        data[j+1] = data[i+1] - tempi;
        data[i  ] += tempr;
        data[i+1] += tempi;
      }
      wr = (wtemp = wr)*wpr - wi*wpi + wr;
      wi = wi*wpr + wtemp*wpi + wi;
    }
    mmax = istep;
  }
}
      
/*======================= end of four1() ====================================*/

/*======================= realft() ==========================================*/

      /*
       *  Real FFT routine.
       *  Numerical Recipes in C, 2nd ed, p. 513.
       *  Assumes data length is a power of 2.
       *  Result of inverse must be multiplied by 2/n.
       */
void realft(double data[], unsigned long n, int isign)
{
  void 
    four1(double data[], unsigned long nn, int isign);
  unsigned long 
    i, i1, i2, i3, i4, np3;
  double 
    c1=0.5,c2,h1r,h1i,h2r,h2i;
  double 
    wr,wi,wpr,wpi,wtemp,theta;

  theta = M_PI/(double) (n>>1);
  if (isign == 1) {
    c2 = -0.5;
    four1(data, n>>1, 1);
  } 
  else {
    c2 = 0.5;
    theta = -theta;
  }
  wtemp = sin(0.5*theta);
  wpr   = -2.0*wtemp*wtemp;
  wpi   = sin(theta);
  wr    = 1.0+wpr;
  wi    = wpi;
  np3   = n+3;
  for (i = 2; i <= (n>>2); i++) {
    i4=1+(i3=np3-(i2=1+(i1=i+i-1)));
    h1r=c1*(data[i1]+data[i3]);
    h1i=c1*(data[i2]-data[i4]);
    h2r= -c2*(data[i2]+data[i4]);
    h2i = c2*(data[i1]-data[i3]);
    data[i1]=h1r+wr*h2r-wi*h2i;
    data[i2]=h1i+wr*h2i+wi*h2r;
    data[i3]=h1r-wr*h2r+wi*h2i;
    data[i4]= -h1i+wr*h2i+wi*h2r;
    wr=(wtemp=wr)*wpr-wi*wpi+wr;
    wi=wi*wpr+wtemp*wpi+wi;
  }
  if (isign == 1) {
    data[1] = (h1r=data[1])+data[2];
    data[2] = h1r - data[2];
  }
  else {
    data[1] = c1*((h1r=data[1])+data[2]);
    data[2] = c1*(h1r-data[2]);
    four1(data, n>>1, -1);
  }
}

/*======================= end of realft() ====================================*/

/* * * * * * * * * * * * end of epic_timestep.c * * * * * * * * * * * * * * * */
