**** Compile with: f77 -O -fpe2 -o radiate radiate.f

      program radiate
c
c RADIATE4 is one of a series of programs which solves for the stream-
c function on a given planet.  The general method is to find matrices
c which represent different fields in the atmosphere.  The dimensions of 
c most of the matrices are three: the matrix itself has different sines 
c of latitude in different rows and different pressures in different 
c columns.  The matrices themselves are complex and a series of four of 
c of them represent a Fourier series up to the specified order. Te is formatted 
c in this manner.  
c RADIATE4 is intended to be used along with DIFFEQ4/DE4 and EVALALL4.  
c The function of this program is to create the "forcing" matrix and other 
c arrays which are used to solve for the streamfunction.  The inputs used 
c in this program can be found in data statements (most of which are in 
c the subroutines), "insola.inp," "psi_bound.dat," "limits.dat," 
c "t_rad_params_new.dat," and files containing Bo, dimensions, and other 
c constants.  Those parameters which the user may desire to alter before 
c execution are in the separate files, generally excluding "insola.inp".  
c Also note that different bands (nu, A, gamma, etc.) should be used if 
c the user intends to run this for some planet other than Uranus.  
c In general, these programs which solve for the streamfunction on a 
c given planet correspond to summary papers by B. Conrath, P. Gierasch, and 
c S. Leroy.  
c RADIATE4 is the version of INSOLA which incorporates the formulas 
c devised by Barney Conrath (Goddard).  It is otherwise exactly the same as 
c INSOLA4. (1/12/87)
c				- Stephen Leroy 
c 
c Here follows a list of orbital data read in and calculated during 
c execution. . .  
c
c	r = radial distance from sun, in AU, 
c	phi = orbital longitude measured from equinox (radians), 
c	deltam = maximum solar declination (radians), 
c	delta = solar declination at a given time of year (t), 
c	e = orbital eccentricity, c	theta = latitude (radians), 
c	zeta = hour angle (radians) measured from local noon, and 
c	mu0 = cos(delta)*cos(theta)*cos(zeta) + sin(delta)*sin(theta).  
c 
c The following is a list of heating/forcing-term variables used in 
c this program. . .  
c
c	te:		the actual forcing matrix in Kelvins,
c	te_annual:	average value of te over year at a given p,
c	mubar:		the average molecular weight,
c	nu:		the frequencies of heating/cooling in 1/cm,
c	y:		sine of latitude,
c	p:		pressure normalized by p_base,
c	y_limits:	the boundary values of y,
c	p_limits:	the boundary values of p,
c	p_base:		pressure at base of atmosphere,
c	dQdT:		change in thermal heating with respect totemperature, 
c	dQdT_ave:	average dQdT over year at a given y and p, 
c	dQdT_annual:	average dQdT over year at a given p (dQdT_ave averaged c				over y),
c	grav:		gravity at base of atmosphere,
c	Rgas:		ideal gas law constant,
c	cp:		thermodynamic value for heating (?),
c	d,s(,A,gamma):	parameters in empirical heating formula,
c	 
c Departures from the equilibrium temperature (higher order Fourier terms) 
c are now determined by dividing the heating by dQdT. See the last two 
c pages of the notes of B. Conrath.  (3-June-87) 
c A correction as to the determination of delta has been made.  Rather 
c than delta = deltam * sin(phi), it has been correctly found that 
c sin(delta) = sin(deltam) * sin(phi).  (4-June-87) 
c A major change has been made to the general algorithm of the program.  
c To establish an annual average temperature, the program used to balance 
c the daily heating and cooling, and then average those equilibrium 
c temperatures over the year.  Now the program balances the annual heating 
c and cooling to find an annual average temperature.  (8-June-87) 
c The mixing ratios of the three gases are now made a function of pressure.
c Along with the new table of mixing ratios, some of the mixing ratio
c dependent quantities are now functions of pressure as well.  Toeffect these 
c modifications, all the mixing ratio dependent quantities are redimensioned 
c to handle 101 times as many values. Because of these changes, the 
c program is renamed RADIATE5. (28-September-87) 
c Some severe changes had to be made to the units in several spots.  For the 
c most part, units are cgs except for the Curtis-Godson pressure, and other 
c quantities whose units are specified when they are not cgs.  This should 
c complete RADIATE5. (7-November-87) 
c No longer is the Brunt frequency itself written to the output file, but 
c rather its value squared.  This was done to avoid the problem of square- 
c roots of negative values of N**2, which is physically possible.  Note that 
c this change must be made in DIFFEQ4 and DE4 as well.  (16-December-87) 
c Because we are taking the real part of a Fourier transform which involves 
c only nonnegative orders, the inverse Fourier transform will give only one 
c half the complex coefficient for the nonzero orders (to make the tranform 
c itself a real value, we only take its real part).  (7-January-88)

	parameter (pi=3.141592654,nm=101,n_its = 48, n_fourier=6)
	implicit double precision (a-h, o-z)
	real*8 mubar, nu, inclin, long0 
	logical first, out
	complex*16 te
	character*40 file
	character*79 label
	dimension te(0:n_fourier,nm,nm), y_limits(2), p_limits(2)
	dimension te_annual(nm), rn2(nm), 
     1			dQdT_ave(nm,nm), dQdT_annual(nm)
	dimension nu(6), d(6), S(6), F(3), s1(6)
c  May 88: new arrays for exchange integral formulation.
	dimension p_array(nm), pmb(nm), T_array(nm), heating_array(nm),
     1  cooling_array(nm), dfdp_array(nm), sline2_array(nm),
     1  T_equil_array(nm), dQdT_ave_array(nm)

	common /qs_arrays/nu, d, s, F, s1, fdnu1, fdnu2, c1, c2, pe, FI
	common /t_rad_params/p_base, radius, rotation_rate, 
     1		sigma, Rgas, grav, cp, mubar, h2_fraction
	common /orbital_params/a, e, phi0, deltam, phi_night_start,
     1             phi_night_end
	common /limits/p_limits, y_limits, nrows, ncols

c  Many of the variables in the t_rad_params common block are now rendered
c  useless because they were initially intended for use in INSOLA4.  RADIATE4
c  operates a very different formula.

	data epsilon/23.441/

c  Get planet specific (orbital) information.

	open( unit=2, name='insola.inp', type='old', readonly)
	read(2,104) label
104	format(a)
	read(2,*) alpha1, delta1, inclin, omega, pitwid, a, e, long0
	close(unit=2)

c  Convert and calculate appropriate values.

	deg=180/pi		!Convert these values to radians
	delta1 = delta1/deg
	alpha1 = alpha1/deg
	epsilon = epsilon/deg
	inclin = inclin/deg
	long0 = long0/deg
	omega = omega/deg
	pitwid = pitwid/deg

c  Calculate orbital period. It is not used in this program, but is 
c  written into the output file for future reference.

	P_orb = 3.1557E7 * a**1.5	! a in AU, P_orb in seconds.

c  We need to convert the above astronomical numbers to numbers relevant
c  to the following calculations.

	call angles(alpha2,delta2,epsilon,inclin,omega,pitwid,alpha1,delta1)
	deltam = pi/2 - abs(delta2)
	phi0 = 3*pi/2. - alpha2
	if(phi0.lt.0.) phi0 = phi0 + 2*pi

c  Get inputs...

	write(*,191)
191	format(' Do you want updating output? (1=yes, 2=no) ',$)
	read( *, *) ianswer
	out = (ianswer .eq. 1)			!"out" is TRUE when ianswer
						!is 1, FALSE otherwise.

	write(*,101)				!This is the only time the
101	format(' File with dimensions: ',$)	!the user directs the program
	read(*,102)file				!to retrieve the dimensions.
102	format(a40)
	open(unit=2,name=file,type='old',readonly)
	read(2,*)nrows,ncols
	close(unit=2)

	write(*,110)
110	format(' What is the minimum you wish 
     1 to assign (1 - RT/p/cp dT/dp)'
     1   ,/,' where it occurs in forming N^2? 
     1 Choice will not affect T(p).',
     1   /,/,' Stability_minimum = ',$)
	read(*,*)  stability_minimum

						!The same goes for the limits
	open(unit=2,file='limits',type='old',readonly)
	read(2,*)p_limits,y_limits
	close(unit=2)
	p_base = p_limits(2)

	write(*,103)				!Get output filename
103	format(' Output Te file: ',$)
	read(*,102)file

	call t_rad_init_new		!Retrieve planet specific information

c  May 88: we need two pressure arrays. pmb counts from the top down.
	do ip = 1, ncols
	     call evaluate_pressure(ip, p )
	     p_array(ip) = p
	end do
	do ip = 1, ncols
	     pmb(ip) = p_array(ncols + 1 - ip) * p_base / 1000.
	end do

c  Do calculations...

	iy_last = (nrows+1) / 2
	do iy=1,nrows			!iy is an index for y, sin(latitude)

	    y=(nrows-iy)/(nrows-1.)*y_limits(1)+(iy-1)/(nrows-1.)*y_limits(2)
	    theta=asin(y)		!theta is latitude

c  Find orbital longitude of polar night edge.

	if( (deltam .eq. 0.) .or. (theta .eq. 0.)) then
		phi_night_start = 2*pi
		phi_night_end   = 0.
	else
	test_sine = cos(theta) / sin(deltam)
		if( test_sine .ge. 1. ) then
			phi_night_start = 2*pi
			phi_night_end   = 0.
		else
			phi_half_night = acos(test_sine)
			if( theta .lt. 0.) then
				phi_night_start = pi/2. - phi_half_night
				phi_night_end   = pi/2. + phi_half_night
			else
				phi_night_start = 3*pi/2. - phi_half_night
				phi_night_end   = 3*pi/2. + phi_half_night
			endif
		endif
	endif

	    do ip=1,ncols				!ip is an index for p

c  May 88: Compute sline2 as array
	call orbit_( ip, theta, sline2 )
	sline2_array(ip) = sline2

		dQdT_ave(iy,ip) = 0.0			!Clear matrices dQdT_ave
		do i_fourier=0,n_fourier		!and Te.
		    te(i_fourier,iy,ip)=(0.0,0.0)
		end do
	   end do          !loop in ip

c The annual average Te is calculated next.

	   if(iy .le. iy_last)then
		call qs_new( theta, p_array, pmb, sline2_array, 
     1             T_equil_array, dQdT_ave_array, out)
	   else
		do ip = 1, ncols
		T_equil_array(ip) = te(0, nrows+1-iy, ip)
		dQdT_ave_array(ip) = dQdT_ave(nrows+1-iy, ip)
		end do
	   end if
c The fluctuations in T due to dfdp are calculated in the second year loop:

	    do ip = 1, ncols
		te(0,iy,ip) = dcmplx(T_equil_array(ip), 0.0d0)
		dQdT_ave(iy,ip) = dQdT_ave_array(ip)
		
		call orbit_(ip, theta, sline2)           !to fill sl2 comm blck

		do it=1,n_its				!it is an index for t
							!phi is the year angle

		    t=2*pi*(it-1.0)/n_its	!t is time of year (in rads)

		    T_equil = T_equil_array(ip)
		    call flux2(T_equil,it,ip,theta,solar_heating)

c Use complex array te to store solar heating Fourier representation.
c Notice that this is purely an inverse Fourier transform when our forward
c transform involves only nonnegative orders and taking the real part.

		    do i_fourier = 1, n_fourier		!do integration

			te(i_fourier,iy,ip) = te(i_fourier,iy,ip) -
     1			   solar_heating/(n_its/2.0 * dQdT_ave(iy,ip)) *
     2			   cdexp(dcmplx(0.0d0, -dfloat(i_fourier)*t))

		    end do      !end Fourier accumulation of amplitudes
 
		end do		!end year loop

		call evaluate_pressure(ip,p)
		if(out) write(*,554)iy,nrows,ip,ncols,y,p*p_base
554		format(' iy=',i3,'/',i3,' ip=',i3,'/',i3,
     1			' sin(lat) = ',f8.5,' pressure = ',1pe12.2,/)

		if (out) write( *, 341) ( te(ii,iy,ip), ii = 0, n_fourier)
341		format( 2( 1x, 2( 5x, '(', f8.4, ',', f8.4, ')'), /))

	    end do   !end ip loop
	end do       !end iy loop

c  Find Te_annual and dQdT_annual as a function of pressure.

	do ip=1,ncols
	    te_annual(ip)=0.0
	    dQdT_annual(ip) = 0.0
	    do iy=1,nrows
		te_annual(ip) = te_annual(ip)+te(0,iy,ip)
		dQdT_annual(ip) = dQdT_annual(ip)+dQdT_ave(iy,ip)
	    end do
	    te_annual(ip) = te_annual(ip)/nrows
	    dQdT_annual(ip) = -grav/cp * dQdT_annual(ip)/nrows
c	    call evaluate_pressure(ip, p)
c	    dQdT_maximum = FI * grav/(p * p_base * cp * te_annual(ip) )
c	    if (dQdT_annual(ip) .gt. dQdT_maximum)then
c		dQdT_annual(ip) = dQdT_maximum
c	    endif
	end do

c  Evaluate N (rn) as a function of pressure.

	call evaluate_pressure(2,  p     )
	call evaluate_pressure(1,p_minus)
	do ip = 2, ncols-1
	call evaluate_pressure(ip+1,p_plus)
	    dT=(te_annual(ip+1)-te_annual(ip-1))/2
	    dp=(p_plus - p_minus)/2
		factor = 1. - cp/Rgas*p/te_annual(ip)*(dT/dp)
		if(factor.lt.stability_minimum)then
			factor = stability_minimum
		endif
	    r2=grav**2/cp/te_annual(ip)*factor
	    rn2(ip) = r2 
	    p_minus = p
            p = p_plus
	end do
	rn2(1)=rn2(2)
	rn2(ncols)=rn2(ncols-1)

c  Write to output and end.

	type *
	type *, 'Writing output to ', file
*	call radiate_data( 1, nrows, ncols, p_limits, y_limits, 
*    1     Te, Te_annual, Rn2, dQdT_annual, rotation_rate, 
*    1     radius, P_orb, grav, cp, Rgas, file)

*** Write ascii output file:
         call radiate_out(nrows, ncols, p_limits, y_limits, 
     +     Te, Te_annual, Rn2, dQdT_annual, rotation_rate, 
     +     radius, P_orb, grav, cp, Rgas, file)

	write(*,700)
700	format(' Do you want basic.out written? 1 = yes: ',$)
	read(*,*) ianswer
	if( ianswer.eq.1) then
	open(unit=1,name='basic.out',type='new',carriagecontrol='list')
	  do i=1,ncols
	    call evaluate_pressure(i,p)
            write(1,500) p_base*p, Te_annual(i), rn2(i), dQdT_annual(i),
     1               2.*pi/P_orb/dQdT_annual(i)
          end do
	close(unit=1)
500	format(1x, 1pe12.2, 0pf8.2,1p3e12.2)
	endif


	end
c
c************ANGLE SUBROUTINES**************************************
c
	subroutine angles(alpha2, delta2, epsilon, inclin, omega,
     1		pitwid, alpha1, delta1)
	implicit double precision (a-h, o-z)
	real*8 vold(3), vnew(3), inclin
	pi = 3.14159
c
c	Coordinates of planet axis in equatorial coordinates.
c
	vold(1) = cos(delta1)*cos(alpha1)! x component
	vold(2) = cos(delta1)*sin(alpha1)! y component
	vold(3) = sin(delta1)!             z component
c
c	Transform to ecliptic coordinates.
c
	call rotxyz(0., epsilon, 0., vold, vnew)
c
	do 1 i = 1,3
1	vold(i) = vnew(i)
c
c	Transform to planet orbit coordinates.
c
	angle3 = pitwid-omega
	call rotxyz(omega, inclin, angle3, vold, vnew)
c
c	Express pole position in orbital coordinates.
c
	delta2 = asin( vnew(3) )
	if(vnew(1).eq.0. .and. vnew(2).gt.0.)then
		alpha2 = pi/2.
	elseif(vnew(1).eq.0. .and. vnew(2).lt.0.)then
		alpha2 = 3.*pi/2.
	else
		alpha2 = atan( vnew(2)/vnew(1) )
	endif
	if(alpha2.gt.0. .and. vnew(2).lt.0.)then
		alpha2 = alpha2 + pi
	elseif(alpha2.lt.0. .and. vnew(1).lt.0.)then
		alpha2 = alpha2 + pi
	elseif(alpha2.lt.0. .and. vnew(2).lt.0.)then
		alpha2 = alpha2 + 2*pi
	elseif(alpha2.gt.0. .and. vnew(1).lt.0.)then
		alpha2 = alpha2 + pi
	endif
	return
	end
c
c	rotxyz.for
c
c	Find coordinates of vector with respect to new coordinate
c		system. New system given with respect to the old one
c		by the three angles:
c	a = longitude of line of nodes (ascending),
c	b = inclination,
c	c = longitude from ascending node of new x axis.
c
	subroutine rotxyz(a,b,c,vold,vnew)
	implicit double precision (a-h, o-z)
	real*8 r(3,3)
	real*8 vold(3), vnew(3), vprime(3)
c
c  The first rotation (about z)
c
	call rotxy(a,r)
	do 1 j = 1,3
	vprime(j) = 0.0
	do 1 i = 1,3
1	vprime(j) = r(i,j)*vold(i) + vprime(j)
c
c  The second rotation (about x)
c
	call rotxy(b,r)
	do 2 j=1,3
	vold(j) = 0.
	jp = j-1
	if(jp.lt.1) jp=jp+3
	do 2 i = 1,3
	ip = i-1
	if(ip.lt.1) ip=ip+3
2	vold(j) = r(ip,jp)*vprime(i) + vold(j)
c
c The last rotation (about z)
c
	call rotxy(c,r)
	do 3 j = 1,3
	vnew(j) = 0.0
	do 3 i = 1,3
3	vnew(j) = r(i,j)*vold(i) + vnew(j)
c
	return
	end
c
c
	subroutine rotxy(a,r)
	implicit double precision (a-h, o-z)
	real*8 r(3,3)
	do 2 j=1,3
	do 2 i=1,3
2	r(i,j) = 0.
	c = cos(a)
	s = sin(a)
	do 1 j=1,2
	do 1 i=1,2
	if(i.eq.j)then
		r(i,j) = c
	else
		r(i,j) = (i-j)*s
	endif
1	continue
	r(3,3) = 1.
	return
	end

c
c*******CALCULATION SUBROUTINES**************************************
c
	subroutine t_rad_init_new

c   This is the subroutine which loads the common blocks with parameters
c which remain constant throughout the execution of RADIATE4, QS_NEW, etc.
c It is executed before most other calculations are made.

	implicit double precision (a-h, o-z)
	real*8 mubar, nu
	real*8 Xp_integral(6), X_integral(6), p_hat(101, 6)
	real*8 dndp(101, 6), dlndp(101, 6), N(101, 6), T_initial(101)
	real*8 X( 101, 6), X_rec( 4, 2)

c  X_rec is the variable into which goes the value from the input table.
c  The first subscript indicates the type of gas (methane, ethane, acetylene)
c  while the second subscript indicates the record.  There are only ever two
c  records stored simultaneously, but the first record always corresponds to
c  a lower pressure level than the second record, and the two records are
c  always consecutive.  The output array (the array which is used throughout
c  the rest of the program) is X.  The first index in X indicates the type of
c  gas while the second indicates the pressure level (ip).

	character*79 dummy_string
	dimension nu(6), d(6), S(6), F(3), s1(6)
	real*8 p_limits(2), y_limits(2)
	common /qs_arrays/nu, d, s, F, s1, fdnu1, fdnu2, c1, c2, pe, FI
	common /more_arrays/X, N, p_hat, dndp, dlndp, T_initial
	common /t_rad_params/p_base, radius, rotation_rate, 
     1		sigma, Rgas, grav, cp, mubar, h2_fraction
	common /limits/p_limits, y_limits, nrows, ncols

	data nu/3020.,4220.,5861.,1306.,821.,729./	!in inverse centimeters
	data d/10.5,10.5,10.5,5.3,2.6,2.4/		!   "
	data S/320.,20.,3.,185.,34.,800./		!in cm^-2 * amagat^-1
c	data F/8.834e5,1.614e6,2.716e6/			!in ergs/cm/s
	data F/2.94e5, 3.52e5, 7.31e5/			!in ergs/cm/s
c	data fdnu1,fdnu2/1.277e10,9.270e9/		!in ergs/cm^2/s
	data fdnu1,fdnu2/4.454e9, 2.895e9/		!in ergs/cm^2/s
	data c1,c2/4.8e-5,1.36e-4/			!

	open(unit=1,name='planet.dat',
     1		type='old',readonly)
	read(1,*) radius, rotation_rate, 
     1		sigma, Rgas, grav, cp, mubar, p_eff, T_eff, h2_fraction
	close(unit=1)

c  Now evaluate the scale height in centimeters at standard temperature and
c  pressure.  Then convert the pressure at the base of the atmosphere from
c  cgs units to atmospheres.

	scale_height_STP = (Rgas * (273.0)) /  grav
	p_base_atm = p_base / (1.0E6)

c  In the next section of this subroutine we establish the table of mixing
c  ratios, and calculate all quantities relevant to this table.

	open( unit=1, name='X_table', type='old', readonly)
	read( 1, '( a, /, a )' ) dummy_string, dummy_string

c  Read in the first two records of the mixing ratio input file.  P1 is the
c  pressure value for the first record and P2 is the pressure value for the
c  second record.

	read( 1, *) idummy, p1, (X_rec(i, 1), i = 1, 4)
	read( 1, *) idummy, p2, (X_rec(i, 2), i = 1, 4)

c  Remember that the pressures p1 and p2 are in bars (= 1E6 cgs units).

	do ip = ncols, 1, -1			!from top downward
	    call evaluate_pressure(ip, p)	!returns pressure/p_base
	    if(ip.ne.1)then
		call evaluate_pressure(ip-1,p_minus)
		dp = p_minus - p
	    endif				!dp same units as p

c  Convert pressure from cgs to bars...

	    pressure = p_base * p * 1.0E-6

c  Move in the input data file to the point where out pressure is between
c  p1 and p2.  This is particularly easy considering the input file is sorted
c  in ascending order.

	    do while ( (pressure .lt. p1) .or. (pressure .gt. p2) )
		do i = 1, 4
		    X_rec( i, 1) = X_rec( i, 2)
		end do
		p1 = p2
		read( 1, *) idummy, p2, (X_rec( i, 2), i = 1, 4)
	    end do

c  Now we interpolate to find the mixing ratios.

	    press_interp = (pressure - p1)/(p2 - p1)
	    do igas = 1, 3
		X( ip, igas+3) = X_rec(igas, 1) * (1 - press_interp) + 
     1				 X_rec(igas, 2) * press_interp
	    end do
	    T_initial( ip) = X_rec(4, 1) * (1 - press_interp) + 
     1				 X_rec(4, 2) * press_interp


c  Set mixing ratio elements 1 through 3 to the fourth mixing ratio element.  
c  This is the methane fraction.

	    do i = 1, 3
		X(ip, i) = X(ip, 4)
	    end do

c  While still in the ip loop, we can process some integrations.  In the
c  integrations, all pressure variables are nondimensionalized, and the
c  integration is trapezoidal.  However, the dp in the X_integral is non-
c  dimensionalized by standard pressure, whereas the dp in the Xp_integral
c  is nondimensionalized by pressure at the base of the atmosphere.  Hence
c  the factor (p_base/1.e6).

	    do igas = 1, 6

		Xp_integrand = X(ip, igas) * p
		X_integrand = X(ip, igas)

c  This following condition assumes that the mixing ratios approach constant as
c  the pressure approaches zero within the top layer.

		if (ip .eq. ncols) then
		    delta = p_limits(1)/p_base + dp/2
		    Xp_integral(igas) = X(ip, igas) * delta**2 /2
		    X_integral(igas) =  X_integrand * delta * p_base_atm
		  else
		    Xp_integral(igas) = Xp_integrand * dp + Xp_integral(igas)
		    X_integral(igas) = X_integrand * dp * p_base_atm + 
     1					X_integral(igas)
		  endif

c  It is necessary to store the running but incomplete integral in Xp_integral 
c  and at the same time use Xp_int as a complete integral at the current ip
c  level.  The same goes for the X integrations.

		Xp_int = Xp_integral(igas) - Xp_integrand * dp/2
		X_int = X_integral(igas) - X_integrand * dp/2 * p_base_atm

c  The units on the Curtis_Godson pressure p_hat and on the abundance N are 
c  atmospheres and centimeters, respectively.

		p_hat(ip, igas) = Xp_int / X_int * p_base_atm**2
		N(ip, igas) = X_int * scale_height_STP

c  Now we wish to evaluate the derivative of log(p_hat * Nch4) with respect
c  to p.  The dndp array is also useful.

		dp_hat_dp = ( X_int * X(ip, igas) * p - 
     1                              p_base_atm * Xp_int * X(ip, igas))
     1			/ ( X_int**2 ) * p_base_atm**2 / p_base
		dndp(ip, igas) = X(ip, igas) * scale_height_STP
     1                                    * p_base_atm/ p_base 
		dlndp(ip, igas) = dp_hat_dp / p_hat(ip, igas) + 
     1				dndp(ip, igas) / n(ip, igas)

c  The units on the above three quantities are 1/pressure in cgs units.

	    end do	!igas loop

	end do		!ip loop

	do i = 1, 6
	    s1(i) = d(i) / S(i)
	end do
	pe = p_eff
	FI = sigma * (T_eff**4)

	return
	end
c
	subroutine orbit_( ip, theta, sline2 )
c
c  Evaluate orbital radius r(it), solar declination delta(it) and sl2(it) 
c  for future use in orbital variation computations. Evaluate annual average 
c  of sline2.
c
	parameter (n_zeta = 48, n_its = 48, pi = 3.1415926536)
	implicit double precision (a-h, o-z)
	real*8 r(101), sl2(101), delta(101)
	real*8 mu0,mubar,nu,izeta_to_zeta, left, new_part
	logical first
	dimension A(6),A_over_gamma(6),B(6),s2(6)
	dimension nu(6),d(6),S(6),F(3),s1(6)
	real*8 X(101,6), N(101,6), p_hat(101,6), dndp(101,6), dlndp(101,6),
     1       T_initial(101)
	common /orbit/ r, sl2, delta, it_night_start, it_night_end,
     1		weight_start, weight_end
	common /qs_arrays/nu,d,s,F,s1,fdnu1,fdnu2,c1,c2,pe,FI
	common /more_arrays/X, N, p_hat, dndp, dlndp, T_initial
	common /orbital_params/a_orb, e, phi0, deltam, phi_night_start,
     1             phi_night_end

	izeta_to_zeta = pi/n_zeta

c
c  Integrate over zeta and t to find average sline2 over the year.

	sline2 = 0.
	first = .true.
	it_night_start = n_its
	it_night_end   = 1
	do it = 1, n_its			!it is an index for t
						!phi is the year angle
	    if(first) then

c At first, set phi=0 and the distance from the planet equal to r at phi=0.

		    first = .false.
		    phi = 0.0
		    r_new = a_orb*(1-e*e)/(1+e*cos(phi-phi0))
		    r(it) = r_new
		    r_old = r_new
		else

c Otherwise increment phi by dphi(previous r) and find r_new(new phi).
c Let r be the average of r_new and r_old.

		    dphi = 2*pi*(a_orb/r_old)**2*(1-e*e)**.5/(n_its-1)
		    phi_old = phi
		    phi = phi + dphi
			if((phi.ge.phi_night_start) .and.
     1			      (phi_old.lt.phi_night_start) )then
			     it_night_start = it - 1
		weight_start = (phi_night_start - phi_old)/dphi
			else if((phi.ge.phi_night_end) .and.
     1			      (phi_old.lt.phi_night_end) )then
			     it_night_end = it
		weight_end = (phi - phi_night_end) / dphi
			end if
		    r_old = r_new
		    r_new = a_orb*(1-e*e)/(1+e*cos(phi-phi0))
		    r(it) = (r_new+r_old)/2
		endif

	    steradians = 6.8000e-5/(r(it)*r(it))

	    delta(it) = asin( sin(deltam)*sin(phi) )

	    sl2_save = 0.
	    left = 0.
	    cosine_sunset = -tan(theta)*tan(delta(it))
	    if(cosine_sunset .ge. 1.) then
		go to 95
	    else if(cosine_sunset .le. 1.)then
	  	zeta_sunset = pi
	    else
	        zeta_sunset = acos(cosine_sunset)
	    end if
	    do izeta = 0, n_zeta
		zeta = izeta * izeta_to_zeta
		if(zeta.gt.zeta_sunset)then
			sl2_save = sl2_save + 
     1		   (zeta_sunset-zeta)/izeta_to_zeta * left
			go to 95
		end if
		mu0 = cos(delta(it))*cos(theta)*cos(zeta) 
     1			+ sin(delta(it))*sin(theta)
		if(mu0.le.0.0) then
		   new_part = 0.
		else
		   new_part = 
     1			+ dndp(ip, 4)*(fdnu1*c1*exp(-c1*N(ip,4)/mu0)
     1		        + fdnu2*c2*exp(-c2*N(ip,4)/mu0))
		end if
	    if(izeta .eq.1)then
		left = new_part
		go to 94
	    else 
		sl2_save = sl2_save + (new_part + left)/2
	  	left = new_part
		go to 94
	    end if
94	    continue
	    
				!Sline2 is the evaluation of line 2 in the
				!summary paper (see above).
	    end do	!End of time of day integration.
95	    sl2(it) = steradians * sl2_save / n_zeta
	if(it.eq.1)then
		bigleft = sl2(it)
		go to 96
	else if(it.eq.(it_night_start+1)) then
		sline2 = sline2 + bigleft*weight_start/2
		go to 96
	else if(it.gt.(it_night_start+1) .and. it.lt.it_night_end)then
		go to 96
	else if(it.eq.it_night_end)then
		sline2 = sline2 + sl2(it)*weight_end/2
		bigleft = sl2(it)
		go to 96
	else 
		sline2 = sline2 + (bigleft + sl2(it) )/2
		bigleft = sl2(it)
	end if

96	continue
	end do		!end year loop

	sline2 = sline2 / (n_its-1)		!sline2 is now the average 
						!over the year
	return
	end
c

	subroutine qs_new( theta, p_array, pmb, sline2_array, 
     1        T_array, dQdT_array, out)

c   This is the new function which evaluates the heating/cooling at
c a given latitude in an atmosphere.  This formula was found by Barney
c Conrath and summarized by P. Gierasch.  For a copy of this work,
c ask P. Gierasch.
c   The variables in this function are...
c
c	delta:		solar declination,
c	theta:		latitude (in radians),
c	zeta:		hour angle (in radians) from local noon,
c	mu0:		cosine of angle of incidence,
c	p:		pressure, nondimensionalized by p_base,
c	r:		radial distance from sun (in AU),
c	n_zeta:		number of intervals in zeta integrations,
c	n_its:		number of intervals in annular integrations,
c	accuracy:	measure of matching heating and cooling (in percent),
c	steradians:	solid angle of sun in planet's sky,
c	pressure:	pressure level in units of p_base (cgs most likely),
c	dfdp:		ln of derivative of F with respect to pressure,
c	dtdps:		sigma x the derivative of tau_mean with respect to p,
c	s1,s2:		substitution arrays to minimize calculations.
c	
c   The parameters in the common block are specific to a planet and
c are read in by subroutine t_rad_init_new for use in the main program 
c and this function.

	parameter (pi=3.141592654, small_a=.555556, 
     1			n_zeta = 48, n_its = 48, accuracy=.001)
	implicit double precision (a-h, o-z)
	logical out
	real*8 mu0,mubar,nu,izeta_to_zeta
	real*8 p_limits(2), y_limits(2)
	dimension A(6),A_over_gamma(6),B(6),s2(6)
	dimension nu(6),d(6),S(6),F(3),s1(6)
	real*8 X(101,6), N(101,6), p_hat(101,6), dndp(101,6), dlndp(101,6),
     1       T_initial(101), sline2_array(101)
c  May 88: new arrays for exchange integral formulation.
	dimension p_array(101), pmb(101), T_array(101), heating(101),
     1   cooling(101), dfdp(101)
	dimension heating1(101), cooling1(101), heating2(101),
     1    cooling2(101), dfdp1(101), dfdp2(101),dcoolingdT(101)
	dimension T_low(101), T_high(101), T_new(101),dQdT_array(101)
	common /qs_arrays/nu,d,s,F,s1,fdnu1,fdnu2,c1,c2,pe,FI
	common /t_rad_params/p_base, radius, rotation_rate, 
     1		sigma, Rgas, grav, cp, mubar, h2_fraction
	common /flux_params/tau_mean,t04,t0,dtdps,
     1		steradians,izeta_to_zeta,s2
	common /more_arrays/X, N, p_hat, dndp, dlndp, T_initial
	common /limits/p_limits, y_limits, nrows, ncols

c  The next 35 or so lines of code are preparations for using the subroutine
c  "flux" which determines the heating and cooling at a given y and p in the
c  atmosphere.  This can be altered if a programmer wished to adapt this
c  program for another heating/cooling model on different planets....

c  The remainder of this subroutine is valid for all heating/cooling models
c  dependent on the Planck function.

	do ip = 2, ncols
	  T_low(ip)  = T_initial(ip)
	  T_high(ip) = T_initial(ip) + 0.1
	  T_array(ip) = T_initial(ip)
	end do
	T_low(1) = T_initial(1)
	T_high(1) = T_initial(1)
	T_array(1) = T_initial(1)

	call flux( T_low, pmb, theta, sline2_array, 
     1			heating1, cooling1, dfdp1,dcoolingdT,1, out)

	call flux( T_high, pmb, theta, sline2_array, 
     1			heating2, cooling2, dfdp2,dcoolingdT,0, out)

100	continue
	do ip = 2, ncols-1

	  T_new(ip) = T_low(ip) 
     1    + (heating1(ip) - cooling1(ip)) / dcoolingdT(ip)
							!Extrapolate to a 
	end do						!new temperature
	T_new(ncols) = 2.*T_new(ncols-1) - T_new(ncols-2)
	T_new(1)     = T_initial(1)


	delta_T = 0.
	do ip = 2, ncols-1
	delta_T = max( delta_T, abs(T_new(ip)-T_array(ip)) )
	end do
	if( delta_T .gt. 5.)then
		i_tau = 1
		do ip = 1, ncols
			T_array(ip) = T_new(ip)
		end do
	  else
		i_tau = 0
	end if
								
	call flux(T_new,pmb,theta,sline2_array,heating,cooling,dfdp,
     1		dcoolingdT, i_tau, out)

c  Check for end of iteration.

	error = 0.
	do ip = 2, ncols-1
	error = max( error, abs(heating(ip)-cooling(ip))/heating(ip) )
	end do

	if( error .le. accuracy ) then
		do ip = 2, ncols
			T_array(ip) = T_new(ip)
			T_high(ip) = T_new(ip) + 0.1
			T_initial(ip) = T_new(ip)
		end do
		T_array(1) = T_new(1)
		T_high(1)  = T_new(1)
		call flux(T_high, pmb, theta,sline2_array,
     1			  heating2,cooling2,dfdp,dcoolingdT,0, out)
		T_eff = ( FI / sigma )**0.25
		itest = 0
		do ip = 2, ncols-1
		    if(T_array(ip) .gt. T_eff .and. itest .eq. 0 )then
			isave = ip
		    else
			itest = 1
			dQdT_array(ip)
     1     = ( heating2(ip)-cooling2(ip)-heating(ip)+cooling(ip) )
     1       /( 0.1)
		    end if
		end do
		do ip = 2, isave
			dQdT_array(ip) = dQdT_array(isave+1)
                end do
		dQdT_array(ncols) = dQdT_array(ncols-1)
		dQdT_array(1)     = dQdT_array(2)
		return
	    endif

c  Assign a new boundary.

	do ip = 1, ncols
		T_high(ip) = T_low(ip)
		heating2(ip) = heating1(ip)
		cooling2(ip) = cooling1(ip)
	end do

	do ip = 1, ncols
		T_low(ip) = T_new(ip)
		heating1(ip) = heating(ip)
		cooling1(ip) = cooling(ip)
	end do

	goto 100	!Continue iteration.

	end
c
c
	subroutine flux(T_array, pmb, theta, sline2_array,
     1     heating, cooling, dfdp, d_by_dT_cooling, i_tau, out)

c   This subroutine evaluates the heating and the cooling fluxes and dfdp
c given the temperature (T) and the value for sline2 (see above).  The
c necessary angles are delta and theta, the same as in function qs_new.
c   The inputs (reals) are T, delta, theta, and sline2; the outputs (reals)
c are heating, cooling, and dfdp (which is actually the log of cooling over
c heating).  Dfdp is only ever used to be a weighting factor in the iteration
c portion of the subroutine qs_new.

	parameter (pi=3.1415926536, n_zeta = 48, 
     1			n_its = 48, small_a=.555556)
	implicit double precision (a-h, o-z)
	logical out
	real*8 mu0, mubar, nu, izeta_to_zeta, littlesum, new_part
	real*8 X(101,6), N(101,6), p_hat(101,6), dndp(101,6), dlndp(101,6),
     1       T_initial(101)
	dimension A(6),A_over_gamma(6),B(6),s2(6), dBdT(6)
	dimension nu(6),d(6),S(6),F(3),s1(6)
	real*8 r(101), sl2(101), delta(101)
c  May 88: additions -- 
	real*8 p_limits(2), y_limits(2), left
	real*8 sline1_array(101), sline2_array(101), sline3_array(101),
     1     heating_reverse(101), cooling_reverse(101), 
     1     heating(101), cooling(101), dfdp(101), T_array(101),
     1     T_array_reverse(101), pmb(101), dqdt_array(101),
     1     d_by_dT_sline3_array(101), d_by_dT_cooling(101)
	common /orbit/ r, sl2, delta, it_night_start, it_night_end,
     1		weight_start, weight_end
	common /qs_arrays/nu,d,s,F,s1,fdnu1,fdnu2,c1,c2,pe,FI
	common /t_rad_params/p_base, radius, rotation_rate, 
     1		sigma, Rgas, grav, cp, mubar, h2_fraction
	common /more_arrays/X, N, p_hat, dndp, dlndp, T_initial
	common /flux_params/tau_mean,t04,t0,dtdps,
     1		steradians,izeta_to_zeta,s2
	common /limits/p_limits, y_limits, nrows, ncols
	common /orbital_params/a_orb, e, phi0, deltam, phi_night_start,
     1             phi_night_end

      do ip = 1, ncols         !loop added May 88 

	T = T_array(ip)


	call find_temp_dep(T,nu,A,A_over_gamma,B,dBdT)

c  Integrate to find average sline1 over a day and a year.

	izeta_to_zeta = pi/n_zeta
	sline1 = 0.
	do it = 1, n_its
	steradians = 6.8000E-5 /r(it)/r(it)
	    sum=0.0
	    left = 0.
	    cosine_sunset = -tan(theta)*tan(delta(it))
	    if(cosine_sunset .ge. 1.) then
		go to 105
	    else if(cosine_sunset .le. 1.)then
	  	zeta_sunset = pi
	    else
	        zeta_sunset = acos(cosine_sunset)
	    end if
	    do izeta = 0, n_zeta
		zeta = izeta * izeta_to_zeta
		if(zeta.gt.zeta_sunset)then
			sum = sum + 
     1		   (zeta_sunset-zeta)/izeta_to_zeta * left /2.
			go to 105
		end if
		mu0 = cos(delta(it))*cos(theta)*cos(zeta) 
     1			+ sin(delta(it))*sin(theta)
		if(mu0.le.0.) then
			new_part = 0.
		else

	    do i=1,3		!S2(1:3) is the substitution array for sline1
		s2(i)=s1(i)*mu0/(2 * p_hat(ip,i) * N(ip,i))
	    end do

	    littlesum = 0.				!evaluate line 1
	    do i=1,3
		littlesum = littlesum
     1			+ F(i)*A(i)/(1+sqrt(s2(i)*A_over_gamma(i)))
	    end do
	    new_part = littlesum * mu0 * steradians
		end if
	    if(izeta .eq.1)then
		left = new_part
		go to 94
	    else 
		sum = sum + (new_part + left)/2
	  	left = new_part
		go to 94
	    end if
94	    continue
	end do

105	continue

	if(it.eq.1)then
		bigleft = sum
		go to 96
	else if(it.eq.(it_night_start+1)) then
		sline1 = sline1 + bigleft*weight_start/2
		go to 96
	else if(it.gt.(it_night_start+1) .and. (it.lt.it_night_end))then
		go to 96
	else if(it.eq.it_night_end)then
		sline1 = sline1 + sum*weight_end/2
		bigleft = sum
		go to 96
	else 
		sline1 = sline1 + (bigleft + sum )/2
		bigleft = sum
	end if

96	continue
	end do		!end year loop

	sline1 = sline1 * dlndp(ip,4) /n_zeta/(n_its-1)   

	sline1_array(ip) = sline1

	sline3 = 0.0				!evaluate line 3
	d_by_dT_sline3 = 0.0
	do i = 4, 6

	    s2(i) = s1(i) * small_a / 2 / p_hat(ip,i) / N(ip, i)
	    sline3 = sline3 + dlndp(ip, i) * 
     1		pi*B(i) * A(i) / (1 + sqrt(s2(i) * A_over_gamma(i)))
	    d_by_dT_sline3 = d_by_dT_sline3 + dlndp(ip, i) * 
     1		pi*dBdT(i) * A(i) / (1 + sqrt(s2(i) * A_over_gamma(i)))

	end do
	sline3_array(ip) = sline3
	d_by_dT_sline3_array(ip) = d_by_dT_sline3

      end do           ! end loop in ip, May 88. Have sline1,2,3 arrays now.

c  May 88: now get the hydrogen heating and cooling.
	do ip = 1, ncols
		T_array_reverse(ip) = T_array(ncols+1-ip)
	end do
c  May 88: hydrogen heating subroutine. All variables are passed as 
c  arguments. Arrays are numbered from top of atmosphere downward.

	call h2_heating(ncols, pmb, H2_fraction, rgas, cp, grav, 
     1          T_array_reverse, heating_reverse, cooling_reverse,
     1          dqdt_array, i_tau)

c May 88: dqdt_array is the derivative of the COOLING w/res to T. 
c It is a positive number, and it is in reverse order.

	do ip = 2, ncols-1
	heating(ip) = sline1_array(ip) + sline2_array(ip)
     1		+ heating_reverse(ncols+1-ip)/1000.  !erg/s/cm^2/(dyne/cm^2)
	cooling(ip) = sline3_array(ip) + cooling_reverse(ncols+1-ip)/1000.
	dfdp(ip) = log( abs(cooling(ip) / heating(ip)) )
	d_by_dT_cooling(ip) = dqdt_array(ncols+1-ip)/1000.  !per mb to per cgs
     1			+ d_by_dT_sline3_array(ip)
	if (out) then
	write(6,1111) ip, pmb(ncols+1-ip), T_array(ip), sline1_array(ip), 
     1       sline2_array(ip), sline3_array(ip), 
     1       heating_reverse(ncols+1-ip)/1000., 
     1       cooling_reverse(ncols+1-ip)/1000., dfdp(ip),
     1	     d_by_dT_sline3_array(ip), dqdt_array(ncols+1-ip)/1000., 
     1       d_by_dT_cooling(ip)
	end if
	end do
1111	format( 1x, i2, 1pg9.2, 0pf7.2, 1p6g10.2,
     1       /, 1x, 2x,     9x,     7x, 20x, 1p2g10.2,10x, 1pg10.2)

	return
	end
c
c
	subroutine flux2(T,it,ip,theta,heating)

c   This subroutine does the same thing as "flux" but includes the calculation
c of sline2.  Also, it only returns the heating.  This subroutine is called by
c the main program ("RADIATE4") in order to calculate the higher Fourier terms.

	parameter (pi=3.1415926536, n_zeta = 48)

	implicit double precision (a-h, o-z)
	real*8 r(101), sl2(101), delta(101)
	real*8 mu0, mubar, nu, izeta_to_zeta
	real*8 X(101,6), N(101,6), p_hat(101,6), dndp(101,6), dlndp(101,6),
     1       T_initial(101)
	dimension A(6),A_over_gamma(6),B(6),s2(6),dBdT(6)
	dimension nu(6),d(6),S(6),F(3),s1(6)
	common /orbit/ r, sl2, delta, it_night_start, it_night_end,
     1		weight_start, weight_end
	common /qs_arrays/nu,d,s,F,s1,fdnu1,fdnu2,c1,c2,pe,FI
	common /t_rad_params/p_base, radius, rotation_rate, 
     1		sigma, Rgas, grav, cp, mubar, h2_fraction
	common /more_arrays/X, N, p_hat, dndp, dlndp, T_initial
	common /flux_params/tau_mean,t04,t0,dtdps,
     1		steradians,izeta_to_zeta,s2

	call find_temp_dep(T,nu,A,A_over_gamma,B,dBdT)


c  Integrate to find average sline1 over a day.

	izeta_to_zeta = pi/n_zeta
	sum=0.0
	do izeta=1,n_zeta
	    zeta =  (izeta-.5) * izeta_to_zeta
	    mu0 = cos(delta(it))*cos(theta)*cos(zeta) 
     1		+ sin(delta(it))*sin(theta)
	    if(mu0.le.0.)goto 105

	    do i=1,3		!S2(1:3) is the substitution array for sline1
		s2(i)=s1(i)*mu0/(2 * p_hat(ip,i) * N(ip,4))
	    end do
	    sline1=0				!evaluate line 1
	    do i=1,3
		sline1=sline1+F(i)*A(i)/(1+sqrt(s2(i)*A_over_gamma(i)))
	    end do
	    sline1 = mu0 * sline1
	    sum=sum+sline1
105	end do
	steradians = 6.8000E-5 /r(it)/r(it)
	sline1 = sum * dlndp(ip, 4) / n_zeta *steradians

c  Evaluate average heating over a day.

	heating = sline1 + sl2(it) 

	return
	end
c
c
	subroutine find_temp_dep(T,nu,A,A_over_gamma,B,dBdT)
	parameter (pi=3.1415926536)

c  This subroutine finds the values of the temperature dependent variables.
c  It is meant to be coupled with the flux and flux2 subroutines and should
c  be used only with those subroutines.

	implicit double precision (a-h, o-z)
	real*8 nu
	dimension nu(6),A(6),A_over_gamma(6),B(6),coef_ag(6),coef_a(6)
	dimension dBdT(6)
	data coef_ag/3*5.51111111, 2.31111111, 1.20915033, 1.14814815/
	data coef_a/3*3.04959014, 1.97484177, 1.94267856, 1.67032931/
	data c1,c2/3.74185e-5,1.43883/
	exp_limit=37*log(10.0)

	do i=1,6		!Evaluate A_over_gamma and A
	    A_over_gamma(i)=coef_ag(i)*T
	    A(i)=sqrt(A_over_gamma(i))*coef_a(i)
	end do

	do i=4,6
	    exponent=c2*nu(i)/T
	    if(exponent.ge.exp_limit)then
		    B(i)=0.0
		    dBdT(i) = 0.0
		else
		    B(i)=c1/pi*(nu(i)**3)/(exp(exponent)-1)	!ergs/cm/s

		    dBdT(i)=B(i)*c2*nu(i)/T**2 /(1 - exp(-exponent))

		endif
	end do

	return
	end
c
c
c
	subroutine evaluate_pressure(ip, p)
c
c Returns p = ratio of pressure to p_base.
c
	implicit double precision (a-h, o-z)
	real*8 p_limits(2), y_limits(2)
	common /limits/p_limits, y_limits, nrows, ncols

	p_ln = log( p_limits(2) ) * (ncols-ip)/FLOAT(ncols-1) 
     1       + log( p_limits(1) ) * (ip-1)/FLOAT(ncols-1)
	p = exp( p_ln ) / p_limits(2)
c
	return
	end

	SUBROUTINE RADIATE_DATA (FLAG,nrows,ncols,P_limits,Y_limits,
     .			Te, Te_annual, Rn, dQdT_annual, omega, a, 
     .			P_orb, grav, Cp, Rgas, FILE)
C-----------------------------------------------------------------------
C	This subroutine is designed to write or read the output of 
C	the RADIATE family of programs in a manner compatible with the
C	subsequent use of DIFFEQ and EVALL_ALL.  The write of read
C	option is selected with FLAG.
C	
C			FLAG = 1 --> WRITE variables to FILE
C			FLAG = 0 --> READ variables from FILE
C	
C-----------------------------------------------------------------------
	implicit none

	Integer		nm,n_fourier,nrows,ncols, i,j,k
	Real*8		grav,Cp,Rgas, omega,p_orb,a
	parameter 	(nm=101,n_fourier=6)

	Complex*16		Te(0:n_fourier, nm, nm)
	Real*8		Te_annual(nm),dQdT_annual(nm),Rn(nm)
	Real*8		Y_limits(2),P_limits(2)
	Character*40	FILE
	Byte		FLAG

	if(FLAG) then
		open(unit=2,name=file,type='new')
		write(2,*)nrows,ncols,P_limits,Y_limits
		write(2,*)omega, a, P_orb, grav
		write(2,*)Cp,Rgas
		write(2,555)(((Te(i,j,k),i=0,n_fourier),j=1,nrows),
     .							k=1,ncols)
		write(2,556)(Te_annual(k),Rn(k),dQdT_annual(k),
     .							k=1,ncols)
	else
		open(unit=2,name=file,type='old')
		read(2,*)nrows,ncols,P_limits,Y_limits
		read(2,*)omega, a, P_orb, grav
		read(2,*)Cp,Rgas
		read(2,555)(((Te(i,j,k),i=0,n_fourier),j=1,nrows),
     .							k=1,ncols)
		read(2,556)(Te_annual(k),Rn(k),dQdT_annual(k),
     .							k=1,ncols)
	endif
	close(unit=2)

555	format(<ncols>(<nrows>(<n_fourier+1>(2a8/))))
556	format(<ncols>(3a8,/))

	RETURN
	END


         SUBROUTINE RADIATE_OUT (nrows,ncols,P_limits,Y_limits,
     +              Te, Te_annual, Rn, dQdT_annual, omega, a, 
     +                P_orb, grav, Cp, Rgas, FILE)
C-----------------------------------------------------------------------
C       This subroutine writes the output of 
C       the RADIATE family of programs to an ascii file.  
C       T. Dowling, 11/25/96
C-----------------------------------------------------------------------
         implicit none

         Integer             nm,n_fourier,nrows,ncols,i,j,k
         Real*8              grav,Cp,Rgas,omega,p_orb,a
         parameter (nm=101,n_fourier=6)

         Complex*16     Te(0:n_fourier, nm, nm)
         Real*8             Te_annual(nm),dQdT_annual(nm),Rn(nm)
         Real*8             Y_limits(2),P_limits(2)
         Character*40   FILE

         open(unit=2,name=file,status='unknown')
         write(2,*) n_fourier,nrows,ncols,P_limits,Y_limits
         write(2,*) omega,a,P_orb,grav
         write(2,*) Cp,Rgas
         do  k = 1,ncols
           do j = 1,nrows
*                      \/ 14 = 2*(n_fourier+1)
             write(2,"(14(1X,E11.4))") 
     +            (real(Te(i,j,k)),imag(Te(i,j,k)),i=0,n_fourier)
           enddo
         enddo
         write(2,*) 'Done with Fourier coefficients.'
         do k = 1,ncols
           write(2,"(3(2X,E12.5))") Te_annual(k),Rn(k),dQdT_annual(k)
         enddo
         close(unit=2)

         RETURN
         END

	subroutine h2_heating(nt, pmb, q, rgas, cp, g, T, heating, 
     1		cooling, dqdt_array, i_tau)
c
c  NOTE: Within this subroutine, the height index counts from top down.
c
	implicit double precision (a-h, o-z)
	real*8 mu,nu
	dimension pmb(100),t(100),fp(100), heating(100), cooling(100),
     .	fpara(100),taua(50,100),dtaua(50,100), dqdt_array(100),
     .	tau(100),dtaudp(100),xnu(50),qabv(100),qblw(100),
     .	w(50),qdeep(100)
	save dlnp, xnu, taua, dtaua
	data xnumin/10./, delnu/40./, numax/25/, fch4/0./, npara/0/
	parameter (b1=1.1927e-5,b2=1.4394,pi=3.14159)
	bf(x,y) = b1*x**3/(exp(b2*x/y) - 1.)
	dbdt(x,y) = bf(x,y)*(b2*x/y**2)/
     .	(1. - exp(-b2*x/y))
	alpha = rgas/cp
c Calculate tau and dtaudp arrays.  First index is frequency and
c second index is atmospheric level.

      if(i_tau.eq.0) go to 9
	dlnp = (log(pmb(nt)) - log(pmb(1)) ) / (nt-1)
	do 8 i=1,numax
	xnu(i) = xnumin + (i - 1.)*delnu
	nu = xnu(i)
	call depth(nu,nt,t,pmb,g,q,npara,fpara,dlnp,pmb(nt),fch4,
     .	tau,dtaudp)
	do 8 j=1,nt
	taua(i,j) = tau(j)
	dtaua(i,j) = dtaudp(j) 
c	write(*,1001) i, j, pmb(j), taua(i,j), dtaua(i,j)
c1001	format(1x, 2i3, 1p3g14.3)
8	continue
9       continue
c
c Calculate radiation from deep atmosphere.
c
	do 11 k=2,numax-1
11	w(k) = delnu
	w(1) = .5*delnu
	w(numax) = .5*delnu
	do 12 i=1,nt-1
	sum1 = 0.
	sum2 = 0.
	do 13 k=1,numax
	sum1 = sum1 + dtaua(k,i)*
     .	bf(xnu(k),t(nt))*e2(taua(k,nt) - taua(k,i))*w(k)
13	sum2 = sum2 + w(k)*dtaua(k,i)*dbdt(xnu(k),t(nt))*
     .  e3(taua(k,nt) - taua(k,i))/dtaua(k,nt)
12	qdeep(i) = 2.*pi*sum1 + 2.*pi*t(nt)*alpha*sum2/pmb(nt)
c
c Calculate heating contributions from other layers.
c
	call exch(nt,xnu,numax,delnu,dtaua,taua,t,qabv,qblw)
	do 50 i=2,nt-1
	qh2 = qabv(i) + qblw(i) + qdeep(i)
	heating(i) = qh2
c	write(*,1000) qabv(i), qblw(i), qdeep(i), heating(i)
c1000	format(1x,1p4g12.2)
	q0 = 0.
	dqdt = 0.
	do 51 k=1,numax
	dqdt = dqdt + dbdt(xnu(k),t(i))*dtaua(k,i)*w(k)
51	q0 = q0 + bf(xnu(k),t(i))*dtaua(k,i)*w(k)
	q0 = 4.*pi*q0
	dqdt = dqdt*4.*pi
	cooling(i) = q0
	dqdt_array(i) = dqdt
c	write(5,1000) m,i,j,pmb(i),delt,t(i),q0,qdeep(i)
c1000	format(' ',2x,3(i2,2x),5(e11.4,2x))
50	continue
	return
	end
c*******************************************************************
c*******************************************************************
	subroutine depth(nu,nt,t,p,g,q,npara,fpara,dlnp,
     .  pmax,fch4,tau,dtaudp)
c This subroutine calculates the optical thickness of a h2-he
c atmosphere from a given temperature profile.
	implicit double precision (a-h, o-z)
	dimension t(100),p(100),fpara(100),a(100),b(100),
     .	tau(100),dummy(1),dtaudp(100)
	real*8 nu
	data t0,p0,rt0/273.18,1013.25,2.271e+10/
	parameter (b1=1.1927e-5,b2=1.4394)
	bf(y,z) = b1*z**3/(exp(b2*z/y) - 1.)
	tbf(x,z) = b2*z/(log(1. + (b1*z**3/x)))
c
c Calculate optical depth profile.
c
	wt = 4.0026 - (1.9868 - 12.040*fch4)*q
	qhe = 1. - q*(1. - fch4)
	fac = (rt0*t0*dlnp)/(p0*p0*g*wt*2.)
c	write(5,200)
200	format(' ','Temperature from rad')
c	do 75 i=1,nt
c75	write(5,201) t(i)
201	format(' ',5x,e11.4)
	call xkh2(nu,nt,t,dummy,a,npara,fpara)
	call xkhe(nu,nt,t,dummy,b,npara,fpara)
	tau(1) = (q*q*a(1) + q*qhe*b(1))*t0*rt0*p(1)*p(1)/
     .  (t(1)*g*2.*wt*p0*p0)
	do 1 i=2,nt
	tau(i) = tau(i-1) + fac*((q*q*a(i-1)+q*qhe*b(i-1))*
     .	p(i-1)**2/t(i-1) + (q*q*a(i)+q*qhe*b(i))*p(i)**2/t(i))
	dtaudp(i) = (q*q*a(i) +q*qhe*b(i))*fac*2.*p(i)/
     .	(t(i)*dlnp)
1	continue
	return
	end
c ****************************************************************
c ****************************************************************
	subroutine exch(nt,nu,numax,delnu,dtaua,taua,t,qabv,qblw)
	implicit double precision (a-h, o-z)
	dimension dtaua(50,100),taua(50,100),qabv(100),qblw(100),
     .  w(50),t(100)
	real*8 nu(50)	
	parameter (b1=1.1927e-5,b2=1.4394)
	bf(x,y) = b1*x**3/(exp(b2*x/y) - 1.)
	pi = 3.14159
c
c Calculate quadrature weights for frequency integration.
c
	do 1 k=2,numax - 1
1	w(k) = delnu
	w(1) = .5*delnu
	w(numax) = .5*delnu
c
c Initialize frequency summation.
c
	do 2 j=1,nt
	qabv(j) = 0.
2	qblw(j) = 0.
c
c Begin frequency loop on k.
c
	do 3 k=1,numax
c
c Calculate exchange integrals for upper most pressure.
c
	sum = 0.
	do 4 j=1,nt - 1
	bf1 = bf(nu(k),t(j))
	bf2 = bf(nu(k),t(j+1))
	e21 = e2(taua(k,j) - taua(k,1))
	e22 = e2(taua(k,j+1) - taua(k,1))
4	sum = sum + .5*(bf1 + bf2)*(e22 - e21)
	qblw(1) = qblw(1) + 2.*pi*dtaua(k,1)*sum*w(k)
	qabv(1) = 0.
c
c Calculate exchange integrals for levels 2,nt-1.
c
	do 5 i=2,nt-1
c Upper exchange integral***************************************
	sum = 0.
	do 6 j=1,i-1
	bf1 = bf(nu(k),t(j))
	bf2 = bf(nu(k),t(j+1))
	e21 = e2(taua(k,i) - taua(k,j+1))
	e22 = e2(taua(k,i) - taua(k,j))
	e31 = e3(taua(k,i) - taua(k,j+1))
	e32 = e3(taua(k,i) - taua(k,j))
	dbdtau = (bf2 - bf1)/(taua(k,j+1) - taua(k,j))
6	sum = sum + bf2*e21 - bf1*e22 + dbdtau*(e32 - e31)
	qabv(i) = qabv(i) + 2.*pi*dtaua(k,i)*sum*w(k)
c Lower exchange integral***************************************
	sum = 0.
	do 7 j=i,nt-1
	e21 = e2(taua(k,j+1) - taua(k,i))
	e22 = e2(taua(k,j) - taua(k,i))
	e31 = e3(taua(k,j+1) -taua(k,i))
	e32 = e3(taua(k,j) - taua(k,i))
	bf1 = bf(nu(k),t(j))
	bf2 = bf(nu(k),t(j+1))
	dbdtau = (bf2 - bf1)/(taua(k,j+1) - taua(k,j))
7	sum = sum + bf1*e22 - bf2*e21 + dbdtau*(e32 - e31)
5	qblw(i) = qblw(i) + 2.*pi*dtaua(k,i)*sum*w(k)
c
c Calculate exchange integrals for lowermost level, i = nt.
c
	qblw(nt) = 0.
	sum = 0.
	do 8 j=1,nt-1
	bf1 = bf(nu(k),t(j))
	bf2 = bf(nu(k),t(j+1))
	e21 = e2(taua(k,nt) - taua(k,j+1))
	e22 = e2(taua(k,nt) - taua(k,j))
	e31 = e3(taua(k,nt) - taua(k,j+1))
	e32 = e3(taua(k,nt) - taua(k,j))
	dbdtau = (bf2 - bf1)/(taua(k,j+1) - taua(k,j))
8	sum = sum + bf2*e21 - bf1*e22 + dbdtau*(e32 - e31)
	qabv(nt) = qabv(nt) + 2.*pi*dtaua(k,nt)*sum*w(k)
c
c End of frequency integration loop (on k).
c
3	continue
	return
	end
c*****************************************************************

c*****************************************************************
    
	SUBROUTINE XKHE(NU,NT,T,DUMMY,XK,NPARA,FPARA)
C
C     XKHE IS CALLED BY TH2HE TO COMPUTE XKH2, A FACTOR IN THE H2-HE
C     COLLISION-INDUCED ABSORPTION.
C
C     THIS VERSION INCLUDES RESULTS OF RECENT LOW-TEMPERATURE FITS
C     BY G. BIRNBAUM ET AL (NBS), AS IMPLEMENTED BY VIRGIL KUNDE
C     AND GORDON BJORAKER, AUGUST 1982.
C
C     THIS VERSION ALLOWS FOR A NON-EQUILIBRIUM PARA-HYDROGEN FRACTION.
C
C     LAST CHANGED BY JOHN HORNSTEIN   CSC   SEP 07,1982
C
C  SIX PARAMETERS ARE USED TO FIT THE H2-HE OPACITY FROM 0 TO 1000 CM-1:
C  QUADS, ISOTRP, TAU1(QUAD), TAU2(QUAD), TAU1(ISOTRP), & TAU2(ISOTRP).
C
C  QUADS & ISOTRP ARE STRENGTHS (IN KELVIN-ANGSTROM**6)
C  OF THE QUADRUPOLAR INDUCED DIPOLES AND ISOTROPIC
C  OVERLAP INDUCED DIPOLES RESPECTIVELY.
C
C  FOR BOTH QUAD & ISOTROPIC PARTS THERE ARE
C  TIME PARAMETERS TAU1 & TAU2:
C  TAU1 CONTROLS HALF WIDTH NEAR LINE CENTER
C  TAU2 CONTROLS EXPONENTIAL DECAY OF WINGS
C
C  G(J) ARE STAT WTS: FOR EVEN ROT STATES G(J)=1; ODD ROT STATES G(J)=3
C  (ODD FOR ORTHO, EVEN FOR PARA).
c#######################################################################
c########################################################################
	implicit double precision (a-h, o-z)
      REAL*8 NU,NSQR,K,ISOTRP
      DIMENSION T(100),XK(100),XJ(10),G(10),A2(100),RHO(100,10),E(10),
     .          WN(10),WW(10),CG(10),BH(100),QUADS(100),ISOTRP(100),
     .          TAU1(2,100),TAU2(2,100),DUMMY(1),FPARA(100)
C
c	He parameters from G. Bachet, Collision Induced Spectra of the
c	H2-He Interaction from 79K to 248K between 200 and 700 cm-1,
c	preprint, 1987.  
      DATA JMAX1/8/,JMAX2/8/,XNLOS/2.687E19/,HCK/1.43879/,
     . PI/3.141593/,HBAR/1.05450E-27/,C/2.997925E+10/,K/1.38054E-16/,
     . SQ0/16.93/,BSQ/0.5/,T1Q0/8.045/,BT1Q/-0.50/,T2Q0/2.007/,
     . BT2Q/-0.50/,SI0/24.4917/,BSI/1./,T1I0/1.8593/,BT1I/-0.5/,
     . T2I0/12.4244/,BT2I/-0.5/,T0/77.4/
C
C
C***********************************************************************
C
c      NPARA = 0
      DO 10 J = 1,JMAX2
        XJ(J) = FLOAT(J-1)
        G(J) = 1.0
        IF (MOD(J,2) .EQ. 0) G(J) = 3.0
 10     E(J) = H2ENER(0.0,XJ(J))
C  CG(J) = (2*J+1) * (CLEPSCH-GORDAN COEFF <J 2 J'>)**2
      DO 20  J = 1,JMAX1
        CG(J) = (2.0*XJ(J)+1.0) * (3.0*(XJ(J)+1.0)*(XJ(J)+2.0)) /
     .        (2.0*(2.0*XJ(J)+1.0)*(2.0*XJ(J)+3.0))
        WN(J) = E(J+2) - E(J)
 20     WW(J) = 2.0 * PI * C * WN(J)
      DO 90  IT = 1,NT
        BH(IT) = HBAR / (K*T(IT))
C  EVALUATE 6 PARAMETERS AT DESIRED TEMP USING POWER LAW.
C  CONSTANTS USED IN TEMP POWER LAW ARE FROM COHEN ET AL (1982)
C  EQUATIONS 14 - 19
        QUADS(IT) = SQ0 * (T(IT)/T0)**BSQ
        ISOTRP(IT) = SI0 * (T(IT)/T0)**BSI
C     TAU1(1,IT) = TAU1 (QUAD)    TAU1(2,IT) = TAU1 (ISOTROPIC)
        TAU1(1,IT) = T1Q0 * (T(IT)/T0)**BT1Q * 1.0E-14
        TAU1(2,IT) = T1I0 * (T(IT)/T0)**BT1I * 1.0E-14
C     TAU2(1,IT) = TAU2 (QUAD)    TAU2(2,IT) = TAU2 (ISOTROPIC)
        TAU2(1,IT) = T2Q0 * (T(IT)/T0)**BT2Q * 1.0E-14
        TAU2(2,IT) = T2I0 * (T(IT)/T0)**BT2I * 1.0E-14
C     COMPUTE THE FULL PARTITION FUNCTION Z, WHICH IS USED WHEN
C     ORTHO AND PARA HYDROGEN ARE REGARDED AS DIFFERENT STATES
C     OF THE SAME SPECIES. THIS IS CONVENIENT FOR THE EQUILIBRIUM
C     RATIO OF ORTHO TO PARA.
        Z = 0.0
        DO 50  J = 1,JMAX2
          RHO(IT,J) = EXP(-1.*HCK*E(J)/T(IT))
50        Z = Z + (2.0*XJ(J)+1.0)*G(J)*RHO(IT,J)
        IF (NPARA .LE. 0)  GO TO 54
C     COMPUTE RHO FOR A NON-EQUILIBRIUM PARA-HYDROGEN FRACTION.
C     ZPARA AND ZORTHO ARE THE PARTITION FUNCTIONS USED IN THIS
C     CASE, WHERE ORTHO AND PARA ARE REGARDED AS DISTICT SPECIES.
C     THE NUCLEAR SPIN DEGENERACIES CANCEL OUT IN THIS CASE.
C     THE NEW RHO EQUALS RHO(PARA) WHEN XJ IS EVEN (INDEX J IS ODD)
C     AND EQUALS RHO(ORTHO) WHEN XJ IS ODD.
        ZPARA = 0.
        ZORTHO = 0.
        DO 1000  J=1,JMAX2,2
          ZPARA = ZPARA + (2.*XJ(J) + 1.)*RHO(IT,J)
          JJ = J + 1
1000      ZORTHO = ZORTHO + (2.*XJ(JJ) + 1.)*RHO(IT,JJ)
        FORTHO = 1. - FPARA(IT)
        DO 3000  J=1,JMAX2,2
          RHO(IT,J) = FPARA(IT)*RHO(IT,J)/ZPARA
          JJ = J + 1
3000      RHO(IT,JJ) = FORTHO*RHO(IT,JJ)/ZORTHO
        GO TO 57
C  EVALUATE EQUILIBRIUM HYDROGEN STATISTICAL WEIGHTS
54      DO 55 J = 1,JMAX2
55        RHO(IT,J) = G(J)*RHO(IT,J) / Z
57      A2(IT) = 0.0
        DO 60 J = 1,JMAX1
60        A2(IT) = A2(IT) + XJ(J)*(XJ(J)+1.0)*(2.0*XJ(J)+1.0)*RHO(IT,J)/
     .                  ((2.0*XJ(J)-1.0)*(2.0*XJ(J)+3.0))
 90     CONTINUE
      W = 2.0 * PI * C * NU
      DO 200  IT = 1,NT
C     CALCULATE F(W) QUADRUPOLE
C  TRANSLATIONAL PART
        FQ =A2(IT) *  GAMFCN(W,T(IT),TAU1(1,IT),TAU2(1,IT))
C  ROTATIONAL PART
        DO 125  J = 1,JMAX1
          FQ = FQ + CG(J) * (RHO(IT,J)*GAMFCN(W-WW(J),T(IT),TAU1(1,IT),
     .          TAU2(1,IT))+RHO(IT,J+2)*GAMFCN(W+WW(J),T(IT),TAU1(1,IT),
     .          TAU2(1,IT)))
 125      CONTINUE
C     CALCULATE F(W) ISOTROPIC OVERLAP
          FIO =  GAMFCN(W,T(IT),TAU1(2,IT),TAU2(2,IT))
          XK(IT) = 4.14911E-8 * W * (1.0-EXP(-BH(IT)*W)) *
     .         (QUADS(IT)*FQ + ISOTRP(IT)*FIO)
 200    CONTINUE
      RETURN
      END
      SUBROUTINE XKH2(NU,NT,T,DUMMY,XK,NPARA,FPARA)
C
C     XKH2 IS CALLED BY TH2HE TO COMPUT XK1, A FACTOR IN THE
C     H2-H2 COLLISION-INDUCED ABSORPTION.
C
C     THIS VERSION INCLUDES RESULTS OF RECENT LOW-TEMPERATURE FITS
C     BY G. BIRNBAUM ET AL (NBS), AS IMPLEMENTED BY VIRGIL KUNDE
C     AND GORDON BJORAKER, AUGUST 1982.
C
C     THIS VERSION ALLOWS FOR A NON-EQUILIBRIUM PARA-HYDROGEN FRACTION.
C
C     LAST CHANGED BY JOHN HORNSTEIN   CSC   FEB 28,1983
C     THIS CHANGE DELETED THE DOUBLE-TRANSITIONS, WHICH HAD BEEN
C     INCORRECTLY FORMULATED IN BACHET ET AL. THE DELETION IS BY
C     COMMENTING OUT WITH A "CD".
C
C  THREE PARAMETERS ARE USED TO FIT THE H2-H2 OPACITY FROM
C  0 TO 2000 CM-1:  STREN, TAU1, & TAU2.  THE TEMPERATURE
C  DEPENDENCE OF EACH PARAMETER IS HANDLED DIFFERENTLY.
C  STREN WAS FIT TO EXPERIMENTAL VALUES IN DORE ET AL AND
C  BACHET ET AL (1982) USING A LINEAR TEMPERATURE RELATION.
C  TAU1 & TAU2 FOLLOW A POWER LAW OF THE FORM
C  TAU = TAU0 * (T/T0)**BT AS SUGGESTED BY DORE ET AL.
C  CONSTANTS FOR TAU1 AND TAU2 ARE FROM DORE ET AL
C
C  PARAMETER DESCRIPTION
C
C  STRENGTH PARAMETER:  STREN     (SEE BACHET ET AL EQUATION 4B)
C  STREN IS REALLY S/KB, TO AVOID LARGE POWERS OF 10.
C
C  STREN = 6 * <A**2> * OMEGA**2 * I8(R/SIGMA) / K
C  (UNITS:  KELVIN*ANGSTROM**6)
C
C  <A**2> IS MEAN SQUARE POLARIZABILITY   UNIT:  ANGSTROM**6
C  OMEGA  IS QUADRUPOLE MOMENT            UNIT:  ESU*CM**2
C  R IS DISTANCE, SIGMA IS MOLECULAR SEPARATION, X = R/SIGMA
C  I8(X) = 4*PI*INT(X**-8*EXP(-4.*E/(K*T)*(X**-12-X**-6))*X**2*DX)
C  E & SIGMA ARE PARAMETERS OF LENNARD-JONES INTERMOLECULAR POTENTIAL
C  INT(...DX) IS INTEGRAL 0 TO INFINITY
C
C  TIME PARAMETERS:  TAU1 & TAU2
C
C  TAU1    CONTROLS HALF WIDTH NEAR LINE CENTER
C  TAU2    CONTROLS EXPONENTIAL DECAY OF WINGS
C
C
C  DOUBLE TRANSITIONS ARE INCLUDED.
C  THE ABSORPTION DUE TO DOUBLE TRANSITIONS IS PROPORTIONAL TO C2
C  C2 = 4./45. * KAPPA**2
C  KAPPA = (APL-APP)/(1./3. * (APL+2.*APP))
C  APL & APP ARE POLARIZABILITIES PARALLEL
C  AND PERPENDICULAR TO INTERNUCLEAR AXIS
C  KAPPA = .375 FOR GROUND VIB STATE & J=0,1 OR 2 SEE KOLOS & WOLNIEWICZ
C  J. CHEM. PHYS. 46, 1426 (1967) TABLE 3
c######################################################################
c######################################################################
	implicit double precision (a-h, o-z)
      REAL*8  NU, NSQR, K
      LOGICAL*4  ILIST
      DIMENSION T(100),XK(100),XJ(10),G(10),A2(100),RHO(100,10),E(10),
     A   DUMMY(1),WN(10),WW(10),CG(10),BH(100),STREN(100),
     B   TAU1(100),TAU2(100),FPARA(100)
      DATA JMAX1/8/,JMAX2/8/,
     .   PI/3.141593/,HBAR/1.05450E-27/,C/2.997925E+10/,K/1.38054E-16/,
     .   HCK/1.43879/,XNLOS/2.687E19/,S0/178./,DSDT/.4091/,TAU10/4.68/,
     .   BT1/-.605/,TAU20/2.23/,BT2/-.607/,T0/273.15/,C2/.0125/
C
C
C***********************************************************************
C
C
c      NPARA = 0
      ILIST = F
      IF (NT .GT. 100) GO TO 900
C  G(J) ARE STATISTICAL WTS:
C  G(J)=1 EVEN ROTATIONAL STATES,
C  G(J)=3 ODD  ROTATIONAL STATES.
      DO 10  J = 1,JMAX2
        XJ(J) = FLOAT(J-1)
        G(J) = 1.0
        IF (MOD(J,2) .EQ. 0)  G(J) = 3.0
 10     E(J) = H2ENER(0.0,XJ(J))
C CG(J) = (2*J+1) * (CLEBSCH-GORDAN COEFF <J 2 J'> )**2
      DO 20  J = 1,JMAX1
        CG(J) = (2.0*XJ(J)+1.0) * (3.0*(XJ(J)+1.0)*(XJ(J)+2.0)) /
     .          (2.0*(2.0*XJ(J)+1.0)*(2.0*XJ(J)+3.0))
        WN(J) = E(J+2) - E(J)
 20     WW(J) = 2.0 * PI * C * WN(J)
C     LIST HEADING FOR ORTHO AND PARA PROFILES:
      IF ( (NPARA .LE. 0) .OR. (.NOT. ILIST))  GO TO 7000
c      WRITE(6,5000)
5000  FORMAT(//,' EQUILIBRIUM AND ACTUAL FRACTIONS OF PARA-',
     .      ' AND ORTHO-HYDROGEN:',/,' LAYER',7X,'FPARA(EQU)',
     .      7X,'FPARA(HERE)',7X,'FORTHO(EQU)',7X,'FORTHO(HERE)')
7000  DO 90 IT = 1,NT
        BH(IT) = HBAR / (K*T(IT))
C     EVALUATE    STREN, TAU1, TAU2 AT DESIRED T
        STREN(IT) = (S0 + DSDT*T(IT))
        TAU1(IT) = TAU10 * (T(IT)/T0)**BT1 * 1.0E-14
        TAU2(IT) = TAU20 * (T(IT)/T0)**BT2 * 1.0E-14
C     COMPUTE THE FULL PARTION FUNCTION Z, USED IN EQUILIBRIUM,
C     WHERE ORTHO AND PARA ARE CONVENIENTLY TREATED AS DIFFERENT
C     STATES OF THE SAME SPECIES.
        Z = 0.0
        DO 50 J = 1,JMAX2
          RHO(IT,J) = EXP(-1.*HCK*E(J)/T(IT))
 50       Z = Z + (2.0*XJ(J)+1.0)*G(J)*RHO(IT,J)
        IF (NPARA .LE. 0)  GO TO 54
C     COMPUTE THE PARTITION FUNCTIONS ZPARA AND ZORTHO USED FOR
C     NON-EQUILIBRIUM RATIOS, WHERE IT IS CONVENIENT TO TREAT
C     ORTHO AND PARA AS DISTINCT SPECIES. THE NUCLEAR SPIN
C     FACTORS G(J) CANCEL OUT IN THIS CASE.
        ZPARA = 0.
        ZORTHO = 0.
        DO 1000  J=1,JMAX2,2
          ZPARA = ZPARA + (2.*XJ(J) + 1.)*RHO(IT,J)
          JJ = J + 1
1000      ZORTHO = ZORTHO + (2.*XJ(JJ) + 1.)*RHO(IT,JJ)
C     COMPUTE AND LIST THE EQUILIBRIUM AND ACTUAL PROFILES:
        FEPARA = ZPARA/Z
        FEORTH = 3.*ZORTHO/Z
        FORTHO = 1. - FPARA(IT)
        IF (ILIST)  WRITE(6,2000)  IT, FEPARA, FPARA(IT), FEORTH, FORTHO
2000    FORMAT(' ',I4,2F15.5,10X,2F15.5)
C     FORM A NEW RHO WHICH EQUALS RHO(PARA) WHEN XJ IS EVEN
C     (INDEX J IS ODD) AND EQUALS RHO(ORTHO) WHEN XJ IS ODD.
      DO 3000  J=1,JMAX2,2
        RHO(IT,J) = FPARA(IT)*RHO(IT,J)/ZPARA
        JJ = J + 1
3000    RHO(IT,JJ) = FORTHO*RHO(IT,JJ)/ZORTHO
      GO TO 57
C  EQUILIBRIUM HYDROGEN STATISTICAL WEIGHTS
54      DO 55  J = 1,JMAX2
55        RHO(IT,J) = G(J)*RHO(IT,J) / Z
57      A2(IT) = 0.0
        DO 60  J = 1,JMAX1
60        A2(IT) = A2(IT) + XJ(J)*(XJ(J)+1.0)*(2.0*XJ(J)+1.0)*RHO(IT,J)/
     .                  ((2.0*XJ(J)-1.0)*(2.0*XJ(J)+3.0))
 90   CONTINUE
      W = 2.0 * PI * C * NU
      DO 200  IT = 1,NT
        DBL = 0.
        FR = 0.
C    EVALUATE TRANSLATIONAL PART OF SHAPE FACTOR F
        FT = A2(IT) * GAMFCN(W,T(IT),TAU1(IT),TAU2(IT))
        DO 125 J = 1,JMAX1
C     EVALUATE THE ROTATIONAL PART OF F
          FR = FR + CG(J) * (RHO(IT,J)*GAMFCN(W-WW(J),T(IT),TAU1(IT),
     .         TAU2(IT))+RHO(IT,J+2)*GAMFCN(W+WW(J),T(IT),TAU1(IT),
     .         TAU2(IT)))
C         DBL = DBL + CG(J) * (RHO(IT,J) + RHO(IT,J+2))
 125  CONTINUE
C  ADD ON PART OF DOUBLE TRANSITION OPACITY (BACHET ET AL EQN 11)
CD    FR = FR * (1.0 + C2 * A2(IT))
C  MORE DOUBLE TRANS  (BACHET ET AL EQN 12)
CD    FT = FT * (1.0 + C2 * A2(IT))
C  AND STILL MORE DOUBLE TRANS (BACHET ET AL EQN 13)
CD    FT = FT + C2 * DBL*DBL*GAMFCN(W,T(IT),TAU1(IT),TAU2(IT))
      F = FT + FR
C  EVALUATE & ADD ON DOUBLE TRANSITIONS (BACHET ET AL EQN 14)
CD    DO 130 J1=1,4
CD    DO 130 J2=1,4
CD    IF (J1.NE.(J1+2).OR.J2.NE.(J1+2))
CD   A  F = F + C2*RHO(IT,J1)*CG(J1)*RHO(IT,J2)*CG(J2)
CD   B  * GAMFCN(W-WW(J1)-WW(J2),T(IT),TAU1(IT),TAU2(IT))
CD    IF ((J1+2).NE.(J2+2).OR.J2.NE.J1)
CD   A  F = F + C2*RHO(IT,J1+2)*CG(J1)*RHO(IT,J2)*CG(J2)
CD   B  * GAMFCN(W+WW(J1)-WW(J2),T(IT),TAU1(IT),TAU2(IT))
CD    IF ((J1+2).NE.J2.OR.(J2+2).NE.J1)
CD   A  F = F + C2*RHO(IT,J1+2)*CG(J1)*RHO(IT,J2+2)*CG(J2)
CD   B  * GAMFCN(W+WW(J1)+WW(J2),T(IT),TAU1(IT),TAU2(IT))
CD    IF (J1.NE.J2.OR.(J2+2).NE.(J1+2))
CD   A  F = F + C2*RHO(IT,J1)*CG(J1)*RHO(IT,J2+2)*CG(J2)
CD   B  * GAMFCN(W-WW(J1)+WW(J2),T(IT),TAU1(IT),TAU2(IT))
CD130 CONTINUE
      XK(IT) =  2.07455e-8 * W * (1.0-EXP(-BH(IT)*W)) * STREN(IT)*F
200   CONTINUE
      RETURN
 900  WRITE(6,905)  NT
 905  FORMAT (1H0,'*** ERROR *** S/R XKH2, NT(MAX) = 50, NT=',I2)
      STOP
      END

      double precision FUNCTION H2ENER(V,J)
C     H2ENER SUBROUTINE OF THE INV PROGRAM. COMPUTES THE ENERGY
C     (IN CM-1) OF A VIBRATION-ROTATION STATE OF A HYDROGEN MOLECULE.
C     THE VIBRATION QUANTUM NUMBER IS V, AND THE QUANTUM NUMBER FOR
C     THE RIGID BODY ANGULAR MOMENTUM IS J (A REAL*4 QUANTITY).
C     THE FIRST LINE OF THE FORMULA FOR E IS THE CONTRIBUTION FROM
C     PURE VIBRATION, INCLUDING ANHARMONIC TERMS. THE OTHER LINES
C     ACCOUNT FOR COUPLED VIBRATION AND ROTATION, INCLUDING
C     CENTRIFUGAL DISTORTION. (THE (J(J+1))**2 AND (J(J+1))**3
C     PROVIDE FOR CENTRIFUGAL DISTORTION; THE VP AND VP**2 IN THE
C     ROTATION TERMS PROVIDE FOR COUPLING BETWEEN VIBRATION AND
C     ROTATION.)
C     THE FORMULA OF COHEN AND BIRNBAUM(1981),
C       NU = 59.3392*(J(J+1))  - 0.04599*(J(J+1))**2
C            + 0.000052*(J(J+1))**3 CM-1,
C     IS A SPECIAL CASE OF THE FORMULA USED HERE, OBTAINED BY
C     SETTING V=0 (IE., VP = 1/2); THE VP TERMS IN THE VIBRATION-
C     ROTATION CONTRIBUTIONS CORRECT THE INITIAL TERMS TO
C     PRODUCE THE COEFFICIENTS IN THE COHEN AND BIRNBAUM FORMULA.
C     FOR THE COLD ATMOSPHERES OF THE OUTER PLANETS, THE SIGNIF-
C     ICANTLY POPULATED LEVELS HAVE V=0.
C
C***********************************************************************
C
c########################################################################
c########################################################################
	implicit double precision (a-h, o-z)
      REAL*8 J,JP
C
      VP = V + 0.5
      JP = J + 1.0
      E = 4400.39*VP - 120.815*VP**2 + 0.7242*VP**3 +
     A    (60.841 - 3.0177*VP + 0.0286*VP**2)*J*JP -
     B    (0.04684 - 0.00171*VP + 3.1E-05*VP**2)*J**2*JP**2 +
     C    5.2E-05*J**3*JP**3 - 2170.08
      H2ENER = E
      RETURN
      END
      double precision FUNCTION GAMFCN(W,T,TAU1,TAU2)
      implicit double precision (a-f,o-z)
C
C     COMPUTES THE LINE SHAPE FOR PRESSURE-INDUCED H2-H2 AND H2-HE
C     TRANSITIONS, FROM THE SEMI-EMIRICAL FORMULAE OF BIRNBAUM ET AL.
C     SEE EG., GEORGE BIRNBAUM AND E. RICHARD COHEN, CANADIAN JOURNAL
C     OF PHYSICS, VOL. 54, 593 (1976).
C     NOTE THAT "GAMFCN" IS A POOR NAME FOR THIS ROUTINE; THE GAMMA
C     OF BIRNBAUM AND COHEN IS NOT THE USUAL "GAMMA FUNCTION".
C
C
C***********************************************************************
C
c#######################################################################
c#######################################################################
      DATA HK/7.638315E-12/,PI/3.141593/,TAU10/0.0/,TAU20/0.0/
C     NOTE: HK = 1.05450E-27 / 1.38054E-16
      IF (TAU1 .NE. TAU10) GO TO 10
      IF (TAU2 .EQ. TAU20) GO TO 20
 10   TAU12 = TAU1 * TAU1
      TAU22 = TAU2 * TAU2
      HBH = 0.5 * HK / T
      Z2 = SQRT(TAU22 + HBH**2) / TAU1
      TAU10 = TAU1
      TAU20 = TAU2
 20   WSQR = W * W
      Z = SQRT(1.0+WSQR*TAU12) * Z2
      IF (Z .LE. 1.0) GO TO 50
C     COMPUTE K1 BESSEL FUNCTION USING POLYNOMIAL APPROXIMATION
      A = 1.0 / Z
      BK1 = 1.253314 + .4699927*A
      B = A * A
      BK1 = BK1 - .1468583*B
      B = B * A
      BK1 = BK1 + .1280427*B
      B = B * A
      BK1 = BK1 - .1736432*B
      B = B * A
      BK1 = BK1 + .2847618*B
      B = B * A
      BK1 = BK1 - .4594342*B
      B = B * A
      BK1 = BK1 + .6283381*B
      B = B * A
      BK1 = BK1 - .6632295*B
      B = B * A
      BK1 = BK1 + .5050239*B
      B = B * A
      BK1 = BK1 - .2581304*B
      B = B * A
      BK1 = BK1 + .7880001E-01*B
      B = B * A
      BK1 = BK1 - .1082418E-01*B
      BK1 = EXP(-Z) * BK1 * SQRT(A)
      GO TO 100
C     COMPUTE K1 BESSEL FUNCTION USING SERIES EXPANSION
 50   A = 0.5 * Z
      B = .5772157 + LOG(A)
      C = A * A
      BK1 = 1.0/Z + A*(B-0.5)
      A = A * C
      BK1 = BK1 + A*.2500000E+00*(0.5+(B-1.500000)*2.0)
      A = A * C
      BK1 = BK1 + A*.2777777E-01*(0.5+(B-1.833333)*3.0)
      A = A * C
      BK1 = BK1 + A*.1736110E-02*(0.5+(B-2.083333)*4.0)
      A = A * C
      BK1 = BK1 + A*.6944439E-04*(0.5+(B-2.283333)*5.0)
      A = A * C
      BK1 = BK1 + A*.1929009E-05*(0.5+(B-2.449999)*6.0)
      A = A * C
      BK1 = BK1 + A*.3936752E-07*(0.5+(B-2.592855)*7.0)
      A = A * C
      BK1 = BK1 + A*.6151173E-09*(0.5+(B-2.717855)*8.0)
 100  CONTINUE
      GAMFCN = TAU1/PI * EXP(TAU2/TAU1+HBH*W) * Z*BK1 / (1.0+WSQR*TAU12)
      RETURN
      END

	double precision function e2(x)
c#################################################################
c#################################################################
	implicit double precision (a-h, o-z)
	if(x.eq.0.)go to 100
	if(x.le.1.0) then
	a10 = -.57721566
	a11 =  .99999193
	a12 = -.24991055
	a13 =  .05519968
	a14 = -.00976004
	a15 =  .00107857
	E1 = -log(x) + a10 + a11*x + a12*x*x + a13*x*x*x
     .	+ a14*x*x*x*x + a15*x*x*x*x*x
	else
	a21 = 2.334733
	a22 =  .250621
	b21 = 3.330657
	b22 = 1.681534
	E1 = (exp(-x)/x)*(x*x + a21*x + a22)/(x*x + b21*x
     .	+ b22)
	endif
	E2 = exp(-x) - x*E1
	E3 = (exp(-x) - x*E2)/2.
	return
100	e2 = 1.
	return
	end
c
	double precision function e3(x)
c#################################################################
C#################################################################
	implicit double precision (a-h, o-z)
	if(x.eq.0.)go to 100
	if(x.le.1.0) then
	a10 = -.57721566
	a11 =  .99999193
	a12 = -.24991055
	a13 =  .05519968
	a14 = -.00976004
	a15 =  .00107857
	E1 = -log(x) + a10 + a11*x + a12*x*x + a13*x*x*x
     .	+ a14*x*x*x*x + a15*x*x*x*x*x
	else
	a21 = 2.334733
	a22 =  .250621
	b21 = 3.330657
	b22 = 1.681534
	E1 = (exp(-x)/x)*(x*x + a21*x + a22)/(x*x + b21*x
     .	+ b22)
	endif
	E2 = exp(-x) - x*E1
	E3 = (exp(-x) - x*E2)/2.
	return
100	e2 = 1.
	e3 = 0.5
	return
	end

