/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * s_epic_functions.c  * * * * * * * * * * * * * * * * 
 *                                                                           *
 *       Single-processor specific functions.                                *
 *                                                                           *
 *       T. Dowling, R. LeBeau                                               *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

/*==========================  bc_ji() =======================================*/

void bc_ji(double *pt,
           int    index,
           int    k) {
  int  
    j,nj,
    i,ni;
  double
    *buffji;

  nj = grid.nj;
  ni = grid.ni;

#if defined(EPIC_AVS)
  if (index > NO_INDEX && index < MAX_NVARS) {
    /*  prognostic variable case, stored as AVSfield */
    for (j = 0; j <= nj+1; j++) {
      /* zonal periodicity */
      FIELD(var.field,var.index[index],k,j,0) = 
        FIELD(var.field,var.index[index],k,j,ni);
      FIELD(var.field,var.index[index],k,j,ni+1) = 
        FIELD(var.field,var.index[index],k,j,1);
    }
    if (strcmp(grid.geometry,"f-plane")      == 0 &&
        strcmp(grid.f_plane_map,"cartesian") == 0) {
      /* doubly periodic */
      for (i = 0; i <= ni+1; i++) {
        FIELD(var.field,var.index[index],k,   0,i) = 
          FIELD(var.field,var.index[index],k,nj,i);
        FIELD(var.field,var.index[index],k,nj+1,i) = 
          FIELD(var.field,var.index[index],k, 1,i);
      }
    }
  }
  else {
#endif
    /* Stored as ordinary vector. */
    /* Align buffji to beginning of JI plane:  */
    buffji  = pt-(IPAD+JPAD*Iadim+KPAD*Nelem2d);
    buffji += (k-Kshift)*Nelem2d;
    /* zonal periodicity */
    for (j = 0; j <= nj+1; j++) {
      BUFFJI(j,   0) = BUFFJI(j,ni);
      BUFFJI(j,ni+1) = BUFFJI(j, 1);
    }
    if (strcmp(grid.geometry,"f-plane")      == 0 &&
        strcmp(grid.f_plane_map,"cartesian") == 0) {
      /* doubly periodic */
      for (i = 0; i <= ni+1; i++) {
        BUFFJI(   0,i) = BUFFJI(nj,i);
        BUFFJI(nj+1,i) = BUFFJI(1, i);
      }
    }
#if defined(EPIC_AVS)
  }
#endif

}

/*========================== end bc_ji() ====================================*/

/*========================== write_ep() =====================================*/

void write_ep(planetspec *planet)
{
  int
    K,J,I;
  double
    ep_flux_lat,
    ep_flux_theta,
    p_avg,fp_avg,fpe_avg,
    field[4],
    pressure,
    temperature,temp_avg,
    u_avg,
    uoup,fpe,fgibb;
  char
    outfile[FILE_STR];
  FILE
    *outdat;     

  sprintf(outfile,EPIC_PATH"/tmp/ep_flux%05d-%02d:%02d:%02d.dat",
          var.time[0] / 86400 + var.time[1]*365,   /* days    */
         (var.time[0] /(60*60))  % 24,             /* hours   */
         (var.time[0] / 60)      % 60,             /* minutes */
          var.time[0]            % 60);            /* seconds */
  outdat = fopen(outfile,"w");

  /* Capitalize name for printing */
  planet->name[0] = (char)toupper((int)planet->name[0]);
  fprintf(outdat," %s\n",planet->name);
  /* Return name to lower case */
  planet->name[0] = (char)tolower((int)planet->name[0]);

  fprintf(outdat," nk nj ni secs years \n");
  fprintf(outdat," %2d %2d %2d %7d %3d \n",
                  grid.nk,grid.nj,grid.ni,var.time[0],var.time[1]);
  fprintf(outdat,"  theta range      lat range    lon range  \n");
  fprintf(outdat," %4.2f %8.2f   %4.1f %4.1f   %4.1f %4.1f \n",
                  grid.theta[2*KLAST_ACTIVE],grid.theta[2*1],
                  grid.lat[2*0],grid.lat[2*(grid.nj+1)],
                  grid.lon[2*1],grid.lon[2*(grid.ni+1)]);
  fprintf(outdat," equatorial radius \n");
  fprintf(outdat," %e \n",planet->re);
  fprintf(outdat," chem_on[FPARA_INDEX] \n");
  fprintf(outdat," %d \n",var.chem_on[FPARA_INDEX]);
  fprintf(outdat," hasten_factor \n");
  fprintf(outdat," %3.0f\n",grid.hasten);
  fprintf(outdat,"lat[deg] theta[K] p_avg[mbar]    flux_lat  flux_theta flux_div[m/s^2]"
                 "  field1   field2 \n");

  for (K = 1; K <= KLAST_ACTIVE; K++) {
    for (J = 0; J <= grid.nj; J++) {
      ep_flux_lat   = .5*(EP_FLUX_Y(K,J)+EP_FLUX_Y(K,J+1));
      if (K > 1) {
        ep_flux_theta = .5*(EP_FLUX_Z(K,J)+EP_FLUX_Z(K-1,J));
      }
      else {
        ep_flux_theta = .5*(EP_FLUX_Z(K,J)+.0);
      }
      p_avg = 0.;
      for (I = 1; I <= grid.ni; I++) {
        p_avg += get_p(planet,P_INDEX,2*K,J,I);
      }
      p_avg /= 100.*(double)grid.ni;
      if (var.chem_on[FPARA_INDEX] == 1) {
        fp_avg   = 0.;
        fpe_avg  = 0.;
        u_avg    = 0.;
        temp_avg = 0.;
        for (I = 1; I <= grid.ni; I++) {
          fp_avg      += FPARA(K,J,I);
          pressure     = get_p(planet,P_INDEX,2*K,J,I);
          temperature  = return_temp(planet,FPARA(K,J,I),pressure,grid.theta[2*K]);
          temp_avg   += temperature;
          return_enthalpy(planet,FPARA(K,J,I),pressure,temperature,&fgibb,&fpe,&uoup);
          fpe_avg += fpe;
          u_avg  += U(K,J,I);
        }
        fp_avg   /= grid.ni;
        fpe_avg  /= grid.ni;
        u_avg    /= grid.ni;
        temp_avg /= grid.ni;
        /****
        field[2] = fpe_avg-fp_avg;
        *****/
        field[1] = temp_avg;
        field[2] = u_avg;
      }
      else {
        field[1] = 0.;
        field[2] = 0.;
      }
      fprintf(outdat," %6.2f %8.2f %10.3f    %10.3e %10.3e %10.3e %10.3e %10.3e \n",
                      grid.lat[2*J+1],grid.theta[2*K],p_avg,
                      ep_flux_lat,ep_flux_theta,EP_FLUX_DIV(K,J),
                      field[1],field[2]);
    }
  }
  fclose(outdat);

  return;
}

/*========================== end write_ep() =================================*/

/*========================== uvp_out() ======================================*/

void uvp_out(planetspec *planet)
{
  /*
   *  Writes u,v,p,q to stdout as ascii floats.
   */
  int
    K,J,I;
  static int
    initialized=0;
  static double
    *q;
 
  if (!initialized) {
    q = dvector(0,Nelem2d-1);
    initialized = 1;
  }

  printf("%d %d %d\n",grid.nk,grid.nj,grid.ni);
  printf("%d %d %d\n",(int)TIME,grid.dt,grid.cfl_dt);
  for (K = 1; K <= KLAST_ACTIVE; K++) {
    /* calculate q */
    potential_vorticity(planet,q,K,1);
    for (J = JLO; J <= JHI; J++) {
      for (I = ILO; I <= IHI; I++) {
        printf("%11.3e %11.3e %11.3e %11.3e\n",
                (float)U(K,J,I),(float)V(K,J,I),(float)P(K,J,I),(float)Q(J,I));
      }
    }
  }

  return;
}

/*========================== end of uvp_out() ===============================*/


/* * * * * * * * * * * * * end of s_epic_functions.c * * * * * * * * * * * * */ 
