/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * s_epic_main.c * * * * * * * * * * * * * * * * 
 *                                                                           *
 *     Explicit Planetary Isentropic-Coordinate (EPIC)                       *
 *     atmospheric model for Jupiter, Saturn, Uranus, and Neptune.           * 
 *                                                                           *
 *     Timothy E. Dowling                                                    *
 *                                                                           *
 *     Contributions from A. Fischer, E.E. Charrette, J. Harrington,         *
 *     R. Lebeau, C. Santori.                                                *
 *                                                                           *
 *     C version.                                                            *
 *     AVS module version for a Unix single processor.                       * 
 *                                                                           * 
 *     The scheme follows Hsu and Arakawa (1990) "Numerical modeling         *
 *     of the atmosphere with an isentropic vertical coordinate,"            *
 *     Monthly Weather Review, 118: 1933-1959, and Arakawa and Lamb (1981)   *
 *     "A potential enstrophy and energy conserving scheme for the shallow   *
 *     water equations,"  Monthly Weather Review, 109: 18-36.                *
 *                                                                           *
 *     The model has nk vertical layers.  The vertical coordinate is         *
 *     potential temperature, theta. The top layer is denoted by k = 1.      * 
 *     Pressure is defined at the layer interfaces.                          *
 *     For gas-giant planets, variables in the bottom layer (k = nk) are     *
 *     specified externally.                                                 *
 *                                                                           *
 *     The velocities u and v are marched forward in                         *
 *     time using the 3rd-order Adams-Bashforth method.  For a               *
 *     discussion of this timestep see Durran (1991): "The                   *
 *     third-order Adams-Bashforth method: an attractive alternative         * 
 *     to leapfrog time differencing," Monthly Weather Review, 119:          *
 *     702-720.                                                              *
 *                                                                           *
 *     The small longitudinal grid spacings near the poles violate the       *
 *     CFL numerical stability criterion, but the instability is removed     *
 *     by filtering the tendencies.                                          *
 *                                                                           *
 *     Time is in seconds; length is in meters.                              *
 *                                                                           *
 *     The input file epic.nc is generated by epic_initial.c.                *
 *     Output files are labeled by the time.                                 *
 *                                                                           *
 *     NOTE: The code subset #if defined(EPIC_READ) is used to               *
 *     make epic_read.mod.                                                   *
 *                                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

extern chem_element
  *Element;

     /*
      * Function prototypes:
      */
int AVSepic(void);

/*======================= AVSepic(), the AVS description function ===========*/
/*          NOTE: The order of the ports must match the creation order       */

int AVSepic(void)
{
  int       
    output1,        /*  Output port descriptor                       */
    output2,        /*  Output port descriptor                       */
    output3,        /*  Output port descriptor                       */
    output4,        /*  Output port descriptor                       */
    output5,        /*  Output port descriptor                       */
    wd1,            /*  Widget descriptor                            */
    wd2;            /*  Widget descriptor                            */
#if !defined(EPIC_READ)
  int
    wd5,            /*  Widget descriptor                            */
    wd6,            /*  Widget descriptor                            */
    wd7,            /*  Widget descriptor                            */
    wd8;            /*  Widget descriptor                            */
#endif
  int
    wd3,            /*  Widget descriptor                            */
    wd4;            /*  Widget descriptor                            */

  /* Set the module name and type */
#if defined(EPIC_READ)
  AVSset_module_name("EPIC Read",MODULE_DATA);
#else
  AVSset_module_name("EPIC Single",MODULE_DATA);
#endif

  /* Create output ports */
  output1 = AVScreate_output_port("Planet structure","struct planetspec"      );
  output2 = AVScreate_output_port("Domain structure","struct domainspec"      );
  output3 = AVScreate_output_port("Variables",       "field 3D double uniform");
  output4 = AVScreate_output_port("Time",            "struct timespec"        );
  output5 = AVScreate_output_port("Diagnostics",     "field 3D double uniform");

  /*
   * NOTE: on the single-processor AVS version of the model, an extra output
   *       port, "Diagnostics", is added. This port holds three KJ planes that
   *       can be used for EP fluxes, etc.
   */

  AVSautofree_output(output1);  
  AVSautofree_output(output2);  
  AVSautofree_output(output3);
  AVSautofree_output(output4);
  AVSautofree_output(output5);

  /* 
   * Add paramter widgets. They appear on the AVS control panel in the order added.
   * If you renumber, make sure to change the AVSadd_parameter argument as well. 
   */
  wd1 = AVSadd_parameter("Status:","string","Initializing...",0,0); 
        AVSadd_parameter_prop(wd1, "width", "integer", 4);
  wd2 = AVSadd_parameter("Data file:","string",EPIC_PATH"/bin/epic.nc",0,0);
        AVSadd_parameter_prop(wd2,"width","integer",4);
#if !defined(EPIC_READ)
  /* widgets will appear in order declared */
  wd5 = AVSadd_parameter("Backup every...","integer",2000,0,1000000);
        AVSconnect_widget(wd5,"typein_integer");
  wd6 = AVSadd_parameter("Save every...","integer",1000000,0,1000000);
        AVSconnect_widget(wd6,"typein_integer");
  wd7 = AVSadd_parameter("Output every...", "integer",100,1,100000);
        AVSconnect_widget(wd7,"typein_integer");
  wd8 = AVSadd_parameter("Sleep","boolean",1,0,1);
        AVSadd_parameter_prop(wd8,"width","integer",4);
#endif
  wd3 = AVSadd_parameter("GO","oneshot",0,0,0);
        AVSadd_parameter_prop(wd3,"width","integer",4);
  wd4 = AVSadd_parameter("Exit","oneshot",0,0,0);
        AVSadd_parameter_prop(wd4,"width","integer",4);
  /* 
   *  Input user-defined data types:
   */
  AVSload_user_data_types(EPIC_PATH"/include/avs_epic.h");

  return(1);
}

/*======================= end of AVSepic() ===================================*/

/*======================= main() =============================================*/

  /* 
   *  NOTE: AVScorout_init() requires main() to have the standard arguments
   *  main(argc,argv), or the module will not show up on the network palette. 
   */

main (int   argc,
      char *argv[])
{
  /*
   * NOTE: Structures planet, grid, var, domain, and diag are 
   *       declared globally in epic.h.
   */
  char   
    inmem[  FILE_STR],     /*  memory for infile                         */
    statmem[FILE_STR],     /*  memory for status string                  */
    *infile,               /*  input file                                */
    *status,               /*  status string                             */
    buf[80];               /*  Temporary string buffer                   */
  timespec
    *ptr_time;             /*  pointer to AVS timespec struct            */
  int
    epic_go,               /*  AVS widget oneshot - Signal to go         */
    epic_exit,             /*  AVS widget oneshot - Signal exit          */
    k,j,i,                 /*  Integer counters                          */
    io_type;
  long
    offset;                /*  byte offset in input file                 */
  double
    *buffji;
#if !defined(EPIC_READ)
  char
    outmem[ FILE_STR],     /*  memory for outfile                        */ 
    savmem[ FILE_STR],     /*  memory for save file                      */
    *outfile,              /*  output file                               */
    *savdir,               /*  savdir                                    */
    str_itout[16];         /*  itime%itout string                        */
  int
    epic_sleep;            /*  AVS widget boolean - sleep                */
  long
    count;                 /*  output-frame counter                      */
  int
    K,J,I;
  double
    ep_flux_lat,
    ep_flux_theta,
    p_avg,q_avg,fpara_avg,null;
#endif

  infile  = inmem;
  status  = statmem;

#if !defined(EPIC_READ)
  outfile = outmem;
  savdir  = savmem;
#endif

  /* First, initialize AVS coroutine */
  AVScorout_init(argc,argv,AVSepic);

  /* AVS memory allocations: */
  planet   = (planetspec *)AVSdata_alloc("struct planetspec", 0);
  domain   = (domainspec *)AVSdata_alloc("struct domainspec", 0);
  ptr_time = (timespec   *)AVSdata_alloc("struct timespec",   0);

  /* Wait for "epic_go" before reading parameters and starting epic */
  mark_beginning:
  AVSmodify_parameter("Status:", AVS_VALUE, 
		      "Waiting for GO",0,0);

  do {
#if defined(EPIC_READ)
    /* arguments must match order in description function */
    AVScorout_input(&status,&infile,&epic_go,&epic_exit);
    /* Set grid.itback != 0, since it is part of a denominator in epic_view.c */
    grid.itback = 10;
    grid.itsave = 10;
    grid.itout  = 10;
#else
    /* arguments must match order in description function */
    AVScorout_input(&status,&infile,&(grid.itback),&(grid.itsave),&(grid.itout),
                    &epic_sleep,&epic_go,&epic_exit);
#endif
    if (epic_exit) goto exit_point;
  } while (epic_go != 1);

  if (epic_exit) goto exit_point;

  AVSmodify_parameter("Status:",AVS_VALUE,"Reading EPIC data",0,0);

  /* 
   * Read infile to determine model size, allocate memory for arrays,
   * and read in rest of data:
   */
  var_read(planet,infile,0,VIA_FILE,INIT_DATA);
  make_arrays();
  var_read(planet,infile,0,VIA_FILE,NONINIT_DATA);

  /* Allocate JI plane buffer */
  buffji = dvector(0,Nelem2d-1);

  /* The first frame */
  AVSmodify_parameter("Status:", AVS_VALUE,"First frame",0,0);

  /* Output first frame: */
  grid_to_domain(domain,1);
  AVSmodify_parameter("Status:", AVS_VALUE,"Data output",0,0);

  /* Bring the AVS timespec port up to date before output */
  ptr_time->secs  = var.time[0];
  ptr_time->years = var.time[1];

  /* Wait for flow executive to finish other tasks before output: */
  AVScorout_exec();
  AVScorout_output(planet,domain,var.field,ptr_time,diag.field);
  AVSmodify_parameter("Status:", AVS_VALUE, "Output first frame",0,0);

  /*
   *  NOTE: The rest of the code is not needed for epic_read.mod
   */
#if defined(EPIC_READ)
  AVSmodify_parameter("GO",AVS_VALUE,0,0,0);
  AVScorout_wait();
  goto mark_beginning;
#else

  /* Arguments must match order in description function: */
  AVScorout_input(&status,&infile,&(grid.itback),&(grid.itsave),&(grid.itout),
                  &epic_sleep,&epic_go,&epic_exit);
  if (epic_exit) goto exit_point;

  /* cd to infile directory so saved files go there */
  strcpy(savdir,infile);
  *(strrchr(savdir, '/')) = '\0';
  if (chdir(savdir) == -1) {
    fprintf(stderr, "Couldn't cd to save directory.\n");
    perror(savdir);
  }  

  /* 
   * Main loop:
   */
  count = 0;
  grid.itime = 0;
  while(1) {
    if ((grid.itime)%(grid.itout) == 0) {
      /* arguments must match order in description function */
      AVScorout_input(&status,&infile,&(grid.itback),&(grid.itsave),&(grid.itout),
                    &epic_sleep,&epic_go,&epic_exit);
    }

    while (epic_sleep) {
      AVSmodify_parameter("Status:", AVS_VALUE, "Sleeping...",0,0);
      AVScorout_wait();
      /* arguments must match order in description function */
      AVScorout_input(&status,&infile,&(grid.itback),&(grid.itsave),&(grid.itout),
                      &epic_sleep,&epic_go,&epic_exit);
      if (epic_exit) {
	goto exit_point;
      }
    } 

    if (epic_exit) {
      AVSmodify_parameter("Status:", AVS_VALUE, "EXITING...",0,0);
      goto exit_point;
    }

    /* Save data when requested. */
    if ( (grid.itime)%(grid.itsave) == 0 && grid.itime > 0) {
      /* Wait for flow executive to finish other tasks */
      AVScorout_exec();

      sprintf(outfile,"epic%05d-%02d:%02d:%02d.nc",
              var.time[0] / 86400 + var.time[1]*365,  /* days    */
             (var.time[0] /(60*60))  % 24,            /* hours   */
             (var.time[0] / 60)      % 60,            /* minutes */
              var.time[0]            % 60);           /* seconds */
      /* zero indicates no expansion in zonal direction */
      var_write(planet,outfile,0,VIA_FILE,ALL_DATA);
    } 

    /* Write out backup frame periodically; always have a good one. */
    if (  (grid.itime)%(grid.itback) == 0  && grid.itime > 0) {
      /* Wait for flow executive to finish other tasks */
      AVScorout_exec();

      sprintf(outfile,"epic_back1.nc");
      if (0 != rename(outfile,"epic_back2.nc") ) {
        if (errno != ENOENT){
          perror(outfile);
        } 
        else {
          errno = 0;
        }
      }
      var_write(planet,outfile,0,VIA_FILE,ALL_DATA);

      /* 
       * Write out EP flux data to a file in tmp directory: 
       */
      write_ep(planet);
    }


    /* Send data out periodically */
    if ((grid.itime)%(grid.itout) == 0 && grid.itime > 0) {
      sprintf(buf, "Output of frame %6d", count);
      AVSmodify_parameter("Status:",AVS_VALUE,buf,0,0);
      grid_to_domain(domain,1);

      /* Bring the AVS timespec port up to date before output */
      ptr_time->secs  = var.time[0];
      ptr_time->years = var.time[1];
      /* Wait for flow executive to finish other tasks before output: */
      AVScorout_exec();
      AVScorout_output(planet,domain,var.field,ptr_time,diag.field);

      /* Increment output-frame counter */
      count++;
    }

    sprintf(buf, "Frame %d: %d",count,grid.itime%grid.itout);
    if ((grid.itime)%10 == 0) {
      /* update counter in status window every 10 counts */
      AVSmodify_parameter("Status:",AVS_VALUE,buf,0,0);
    }

    /* Take a step */
    /* Wait for flow executive to finish other tasks before next timestep: */
    AVScorout_exec();
    timestep();

    /* increment itime: */
    (grid.itime)++;


  } /* End of main loop */    
 
#endif

 /* Mark exit_point */
 exit_point:
 free_arrays(planet);
 free_dvector(buffji,0,Nelem2d);
 AVSmodify_parameter("Status:",AVS_VALUE,"EXIT COMPLETE",0,0);
 exit(1);

}

/*======================= end of main()  ====================================*/

/* * * * * * * * * * * *  end of s_epic_main.c * * * * * * * * * * * * * * * */







