/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * epic_output.c * * * * * * * * * * * * * * * * 
 *                                                                     *
 *    AVS output for the EPIC atmospheric model                        *
 *                                                                     *
 *    A.Fischer, T. Dowling, C. Santori                                * 
 *                                                                     *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h> 

/*
 *  Definitions 
 */
#define    X_COORD      0
#define    Y_COORD      1
#define    Z_COORD      2
#define    RED          0
#define    GREEN        1
#define    BLUE         2

typedef float  
  FLOAT3[3];

/* ----------------------   make_layers_list()   --------------------- */

void make_layers_list(planetspec    *planet,
                      GEOMedit_list *list,
                      listspec      *info)

{
  int                
    ni,nj,nk,
    small_ni,small_nj,
    i,j,k,
    small_i,small_j,
    count,
    recount,
    num,
    uscale,
    iswap,
    uzero;
  int
    *ugraph_l,
    *ugraph_h;
  FLOAT3             
    *verts, 
    *colors;
  AVSfield_double
    *color;
  char               
    name[30];
  double              
    hue,sat,val,
    red,green,blue,
    lat,lon,r,
    u_avg,zscale;
  GEOMobj            
    *obj = GEOM_NULL; 

  ni = grid.ni;
  nj = grid.nj;
  nk = grid.nk;

  uscale  = info->wind_scale;
  color   = info->color;

  if (ni == 1) {
    /* sweep out in zonal direction for plotting */
    small_ni = (int)((double)64/info->downsize);
  }
  else {
    small_ni = (int)((double)ni/info->downsize);
  }
  small_nj = (int)((double)nj/info->downsize);

  /*
   *  Allocate memory for verts and colors:
   */
  verts  = (FLOAT3 *)calloc((small_ni)*(small_nj-JLO+1),sizeof(FLOAT3));
  colors = (FLOAT3 *)calloc((small_ni)*(small_nj-JLO+1),sizeof(FLOAT3));
  if (info->wind){
    ugraph_l = (int *)calloc((small_nj+2),sizeof(int));
    ugraph_h = (int *)calloc((small_nj+2),sizeof(int));
  }

  /* Loop through layers */
  for (k = 1; k <= KLAST_ACTIVE; k++) {
    /* Make table for u vs. lat graph */
    if (info->wind){
      uzero = .5 * small_ni;
      for (small_j = JLO; small_j <= small_nj; small_j++){
	j = (int)(.5+ (double)((small_j-1)*(nj-1))/(small_nj-1) ) + 1;
	u_avg = 0.;
	for (i = 0; i < ni; i++){
	  u_avg += U(k,j,i);
	}
	u_avg /= ni;
	if (info->polar) u_avg *= (90. - grid.lat[2*JLO+1]) / (91. - grid.lat[2*j+1]);
	ugraph_h[small_j] = (int)(uzero + uscale * u_avg * small_ni / 50000.);
      }
      ugraph_h[JLO-1] = ugraph_h[small_nj+1] = uzero;
      for (small_j = JLO; small_j <= small_nj; small_j++){
	ugraph_h[small_j] = .5*(ugraph_h[small_j+1] + ugraph_h[small_j]);
	ugraph_l[small_j] = ugraph_h[small_j - 1];
      }
      for (small_j = JLO; small_j <= small_nj; small_j++){
	if (ugraph_l[small_j] > ugraph_h[small_j]){
	  iswap = ugraph_l[small_j];
	  ugraph_l[small_j] = ugraph_h[small_j];
	  ugraph_h[small_j] = iswap;
	}
	while (ugraph_l[small_j] > small_ni){
	  ugraph_l[small_j] -= small_ni;
	  ugraph_h[small_j] -= small_ni;
	}
      }
    }

    zscale = (double)info->height_scale/50.;
    count  = 0;
    for (small_i = 1; small_i <= small_ni; small_i++) {
      if (ni == 1) {
        i = 1;
      }
      else {
        i = NINT((double)small_i*info->downsize);
      }
      for (small_j = JLO; small_j <= small_nj; small_j++) {
        j = NINT((double)small_j*info->downsize);
	/* Color data */
        /* If necessary, average color variable onto p-grid */
        if (strcmp(info->color_input,"potential vorticity") == 0) {
          num = (int)( (fabs(E3D(color,k,j,i  )+E3D(color,k,j+1,  i)
                            +E3D(color,k,j,i+1)+E3D(color,k,j+1,i+1))*.25 
                            - info->color_min[k])
		      /(info->color_max[k]-info->color_min[k])*255. );
        }
        else {
          num = (int)( fabs(E3D(color,k,j,i)-info->color_min[k])
		      /(info->color_max[k]-info->color_min[k])*255. );
        }
	num = (num < 0) ? 0 : num;
	num = (num >= (info->colormap)->size) ? (info->colormap)->size-1 : num;
	hue = (info->colormap)->hue[       num];
	sat = (info->colormap)->saturation[num];
	val = (info->colormap)->value[     num];
        /* light north pole */
        if(info->light_north) {
          if(small_i == 1 && small_j > small_nj/2) {
            sat = 0.;
          }
        }

	/* draw wind profile */
	if(info->wind){
	  if (small_i == uzero ||
	      (small_i >= ugraph_l[small_j] && small_i <= ugraph_h[small_j]) ||
	      (ugraph_h[small_j] >= small_ni && small_i <= ugraph_h[small_j] - small_ni)){
	    val = 1.;
	    hue = 0.2;
	    sat = 1.;
	  }
	}
	hsv_to_rgb(hue,sat,val,&red,&green,&blue);
	colors[count][RED  ] = red;
	colors[count][GREEN] = green;
	colors[count][BLUE ] = blue;

	/* Verts data */
	if(P(k,j,i) <= 0.) {
	  fprintf(stderr,"Warning: epic_output:p(%2d,%2d,%2d)=%9.1e ",
                          k,j,i,P(k,j,i));
        }
        verts[count][Z_COORD] = -(log10(P(k,j,i))-4.)*zscale;
        if (ni == 1) {
          lon = -180.+360.*((double)small_i-1.)/((double)small_ni);
        }
        else {
          lon = grid.lon[2*i+1];
        }
        lat = grid.lat[2*j+1];
        if (strcmp(grid.geometry,"globe") == 0) {
	  if (info->polar){
	    verts[count][X_COORD] = cos(lon*DEG)*(90.-lat)/90.;
	    verts[count][Y_COORD] = sin(lon*DEG)*(90.-lat)/90.;
	  }
	  else{
	    verts[count][X_COORD] = (lon)/90.;
	    verts[count][Y_COORD] = (lat)/90.;
	  }
	}
        else if (strcmp(grid.geometry,"f-plane") == 0) {
          if (strcmp(grid.f_plane_map,"cartesian") == 0) {
	    verts[count][X_COORD] = (lon)/90.;
	    verts[count][Y_COORD] = (lat)/90.;
          }
          else if (strcmp(grid.f_plane_map,"polar") == 0) { 
            /* back off from i degeneracy at poles for ease of plotting */
            if (lat >=  90.) lat = 90.-1.e-5;
            r = (90.-lat)/90.;
            verts[count][X_COORD] = r*cos((lon-90.)*DEG);
            verts[count][Y_COORD] = r*sin((lon-90.)*DEG);
          }
          else {
            fprintf(stderr,"epic_output: Unrecognized f_plane_map \n");
            exit(1);
          }
        }
        else {
          fprintf(stderr,"epic_output: Unrecognized geometry \n");
          exit(1);
        }

	count++;
      }
    }

    /* Create geometry */
    obj = GEOMcreate_mesh(NULL,(float *)verts,(small_nj-JLO+1),(small_ni),GEOM_COPY_DATA);
    GEOMadd_float_colors(obj,(float *)colors, (small_nj-JLO+1)*(small_ni),GEOM_COPY_DATA);
    GEOMgen_normals(obj,0);
    GEOMcvt_mesh_to_polytri(obj,GEOM_SURFACE|GEOM_WIREFRAME);
    
    /* Put geometry in edit list and destroy geom */
    sprintf(name, "Layer%d", k);
    GEOMedit_geometry(*list,name,obj);
    GEOMedit_parent(*list,name,"Layers");
    GEOMdestroy_obj(obj);
    
  } 
  /* end of k loop */

  if (info->wind){
    free(ugraph_l);
    free(ugraph_h);
  }
  free(verts);
  free(colors);

}

/* -----------------------   make_sphere_list()   --------------------------- */

void make_sphere_list(planetspec    *planet,
                      GEOMedit_list *list,
                      listspec      *info)
{
  int                
    ni,nj,nk,
    small_ni,small_nj,
    i,j,k,
    small_i,small_j,
    count,recount,
    num,itmp,
    uscale,
    iswap,
    uzero;
  int
    *ugraph_l,
    *ugraph_h;
  double              
    re,rp,
    hue,sat,val,
    red,green,blue,
    lat,lon,g,r,
    u_avg,
    *log_press, 
    p_south, p_north,
    *buffji,buff_n,buff_s,temp;
  AVSfield_double
    *color;
  FLOAT3             
    *verts, 
    *colors;
  char               
    name[30];
  GEOMobj           
    *obj = GEOM_NULL;

  ni = grid.ni;
  nj = grid.nj;
  nk = grid.nk;

  color   = info->color;

  small_ni = (int)((double)(grid.ni)/info->downsize);
  small_nj = (int)((double)(grid.nj)/info->downsize);

  re = planet->re;
  rp = planet->rp;

  g = 3.0/re;

  /* Select layer */
  k = info->layer;

  /*  
   * Allocate memory for verts and colors.
   * (small_ni+1) is 1,small_ni plus periodicity step.
   * (small_nj+2) is 0,small_nj+1 (pole to pole).
   */
  verts  = (FLOAT3 *) calloc((small_ni+1)*(small_nj+2),sizeof(FLOAT3));
  colors = (FLOAT3 *) calloc((small_ni+1)*(small_nj+2),sizeof(FLOAT3));

  /* allocate memory */
  log_press = dvector(0,Nelem2d-1);
  buffji    = dvector(0,Nelem2d-1);

  if (info->wind){
    ugraph_l = (int *) calloc((small_nj+2),sizeof(int));
    ugraph_h = (int *) calloc((small_nj+2),sizeof(int));
  }
  
  /* average pressure onto q-grid, take log: */

  p_south = 0.;
  p_north = 0.;
  for (i = 1; i <= ni; i++) {
    for (j = 1; j <= nj; j++) {
      LOG_PRESS(j,i) = log( (P(k,j,i  )+P(k,j-1,i  )
                            +P(k,j,i-1)+P(k,j-1,i-1))*.25 );
    }
    p_south += P(k,0,i);
    p_north += P(k,nj,i);
  }
  /* periodicity */
  for (j = 1; j <= nj; j++) {
    LOG_PRESS(j,ni+1) = LOG_PRESS(j,1);
  }
  p_south = log(p_south/ni);
  p_north = log(p_north/ni);
  for (i = 1; i <= ni+1; i++) {
    LOG_PRESS(   0,i) = p_south;
    LOG_PRESS(nj+1,i) = p_north;
  }

  /* If necessary, average color variable onto q-grid. */
  if (strcmp(info->color_input,"potential vorticity") != 0) {
    buff_n = 0.;
    buff_s = 0.;
    for (i = 1; i <= ni; i++) {
      for (j = 1; j <= nj; j++) {
        BUFFJI(j,i) = .25*(E3D(color,k,j,i  )+E3D(color,k,j-1,i  )
                          +E3D(color,k,j,i-1)+E3D(color,k,j-1,i-1));
      }
      buff_s += BUFFJI( 0,i);
      buff_n += BUFFJI(nj,i);
    }

    /* periodicity */
    for (j = 1; j <= nj; j++) {
      BUFFJI(j,ni+1) = BUFFJI(j,1);
    }
    buff_s /= ni;
    buff_n /= ni;
    for (i = 1; i <= ni+1; i++) {
      BUFFJI(   0,i) = buff_s;
      BUFFJI(nj+1,i) = buff_n;
    }
  }
  else {
    for (j = JLO; j <= nj+1; j++) {
      for (i = 1; i <= ni; i++) {
	itmp = i-Ishift+(j-Jshift)*Iadim;
	buffji[itmp] = E3D(color,k,j,i);
      }
      itmp = 1-Ishift+(j-Jshift)*Iadim;
      temp = buffji[itmp];
      BUFFJI(j,ni+1) = temp;
    }
  }

  /* Make table for u vs. lat graph */
  
  if (info->wind){
    
    uzero = .5 * small_ni;
    for (small_j = JLO; small_j <= small_nj; small_j++){
      j = (int)(.5+ (double)((small_j-1)*(nj-1))/(small_nj-1) ) + 1;
      u_avg = 0.;
      for (i = 0; i < ni; i++){
	u_avg += U(k, j, i);
      }
      u_avg /= ni;
      ugraph_h[small_j] = (int)(uzero + uscale * u_avg / 100);
    }
    ugraph_h[JLO-1] = ugraph_h[small_nj+1] = uzero;
    for (small_j = JLO; small_j <= small_nj; small_j++){
      ugraph_h[small_j] = .5 * (ugraph_h[small_j+1] + ugraph_h[small_j]);
      ugraph_l[small_j] = ugraph_h[small_j - 1];
    }
    for (small_j = JLO; small_j <= small_nj; small_j++){
      if (ugraph_l[small_j] > ugraph_h[small_j]){
	iswap = ugraph_l[small_j];
	ugraph_l[small_j] = ugraph_h[small_j];
	ugraph_h[small_j] = iswap;
      }
      while (ugraph_l[small_j] > small_ni){
	ugraph_l[small_j] -= small_ni;
	ugraph_h[small_j] -= small_ni;
      }
    }
  }

  count = 0;
  for (small_i = 1; small_i <= small_ni; small_i++) {
    i = NINT((double)small_i*info->downsize);
    for (small_j = 0; small_j <= small_nj+1; small_j++) {
      j = NINT((double)small_j*info->downsize);
      /* Color data */
      num = (int)(  (fabs(BUFFJI(j,i))-info->color_min[k])
		  /(info->color_max[k]-info->color_min[k])*255. );
      num = (num < 0) ? 0 : num;
      num = (num >= (info->colormap)->size) ? (info->colormap)->size-1 : num;
      hue = (info->colormap)->hue[num];
      sat = (info->colormap)->saturation[num];
      val = (info->colormap)->value[num];
      /* light north pole */
      if(info->light_north) {
        if(small_i==1 && small_j>small_nj/2) {
          sat = 0.;
        }
      }
      /* draw wind profile */
      if(info->wind){
	if (small_i == uzero ||
	    (small_i >= ugraph_l[small_j] && small_i <= ugraph_h[small_j]) ||
	    (ugraph_h[small_j] >= small_ni && small_i <= ugraph_h[small_j] - small_ni)){
	  val = 1.;
	  hue = .2;
	  sat = 1.;
	}
      }
      hsv_to_rgb(hue,sat,val,&red,&green,&blue);
      colors[count][RED  ] = red;
      colors[count][GREEN] = green;
      colors[count][BLUE ] = blue;
      
      /* Verts data */
      lat = grid.lat[2*j];
      /* back off from i degeneracy at poles for ease of plotting */
      if (lat >=  90.) lat =  90.-1.e-5;
      if (lat <= -90.) lat = -90.+1.e-5;

      lon = grid.lon[2*i];
      /* cylindrical radius: */
      r = g*re/sqrt( 1. + pow((rp/re)*tan(lat*DEG), 2.) );
      /* 
       * Modulate radius with (-log p).
       */
      r *= 1.+( -LOG_PRESS(j,i)-(-p_south) )
           /fabs(p_south)*(double)info->height_scale/50.;

      verts[count][X_COORD] = r*sin(lon*DEG);
      verts[count][Y_COORD] = r*(rp/re)*(rp/re)*tan(lat*DEG);
      verts[count][Z_COORD] = r*cos(lon*DEG);

      count++;
    }
  }
  /* 
   * Periodicity: 
   */
  recount = 0;
  for (small_j = 0; small_j <= small_nj+1; small_j++) {
    colors[count][RED  ] = colors[recount][RED  ];
    colors[count][GREEN] = colors[recount][GREEN];
    colors[count][BLUE ] = colors[recount][BLUE ];
      
    verts[count][X_COORD] = verts[recount][X_COORD];
    verts[count][Y_COORD] = verts[recount][Y_COORD];
    verts[count][Z_COORD] = verts[recount][Z_COORD];

    count++;
    recount++;
  }

  /* Create geometry */
  obj = GEOMcreate_mesh(NULL,(float *)verts,small_nj+2,  small_ni+1, GEOM_COPY_DATA);
  GEOMadd_float_colors(obj,(float *)colors,(small_nj+2)*(small_ni+1),GEOM_COPY_DATA);
  GEOMgen_normals(obj,0);
  GEOMcvt_mesh_to_polytri(obj,GEOM_SURFACE|GEOM_WIREFRAME);
  
  /* Put geometry in edit list and destroy geom */
  sprintf(name,"Sphere",k);
  GEOMedit_geometry(*list, name, obj);
  GEOMdestroy_obj(obj);
  
  /* Free memory */

  free(verts);
  free(colors);
  if (info->wind){
    free(ugraph_l);
    free(ugraph_h);
  }
  free_dvector(log_press,0,Nelem2d-1);
  free_dvector(buffji,   0,Nelem2d-1);

  return;
}

/*---------------------- hsv_to_rgb() ------------------------------------*/

#include <stdio.h>
#include <math.h>

void hsv_to_rgb (double h,  double s,  double v,
		 double *r, double *g, double *b)
/*
 * Function to convert HSV values to RGB
 * All values are in range [0., 1.]
 *
 * 1 August 1991, A. Fischer
 * Adapted from "Computer Graphics: Principles and Practice, 
 * 2nd Ed." by Foley, van Dam, Feiner & Hughes, 1990, page 593.
 */
{
  int   
    i;
  double 
    f,p,q,t;

  if (s == 0.) {	/* The color is on the black- 	*/
   *r = v;              /* and-white center line	*/
   *g = v;
   *b = v;
  } else {
    if (h == 1.) h = 0.;
    h = h * 6.;	/* hue is now in [0., 6.)               */
    i = (int)floor(h);	/* floor returns largest integer <= h   */
    f = h - (double)i;	/* f is the fractional part of h.       */
    p = v * (1 - s);
    q = v * (1 - (s * f));
    t = v * (1 - (s * (1 - f)));
    switch (i) {
      case 0: 
        *r = v, *g = t, *b = p;
        break;
      case 1: 
        *r = q, *g = v, *b = p;
        break;
      case 2: 
        *r = p, *g = v, *b = t;
        break;
      case 3: 
        *r = p, *g = q, *b = v;
        break;
      case 4: 
        *r = t, *g = p, *b = v;
        break;
      case 5: 
        *r = v, *g = p, *b = q;
    }
  }
}

/*--------------------- end of epic_output.c ---------------------------*/
