/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*  
 *  AVS module to receive EPIC model data from model host 
 *  via network sockets.
 *
 *  R. LeBeau, T. Dowling  7/96
 *
 */

#include <epic.h>

extern chem_element
  *Element;

/* prototypes */
int AVSreceive(void);

/*======================= AVSreceive(), the AVS description function ===========*/

int AVSreceive(void)
{
  int       
    output1,    /*  Output port descriptor                           */
    output2,    /*  Output port descriptor                           */
    output3,    /*  Output port descriptor                           */
    output4,    /*  Output port descriptor                           */
    wd[50],     /*  Widget descriptors                               */
    iwd;

  /* Set the module name and type */
  AVSset_module_name("EPIC Receive", MODULE_DATA);
  
  /* Create ports */
  output1 = AVScreate_output_port("Planet structure", "struct planetspec"      );
  output2 = AVScreate_output_port("Domain structure", "struct domainspec"      );
  output3 = AVScreate_output_port("Variables",        "field 3D double uniform");
  output4 = AVScreate_output_port("Time",             "struct timespec"        );

  AVSautofree_output(output1);  
  AVSautofree_output(output2);  
  AVSautofree_output(output3);
  AVSautofree_output(output4);

  /* Create widgets */
  iwd = -1;
  wd[++iwd] = AVSadd_parameter("Status:","string","Waiting for launch",NULL,NULL); 
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("Model user","string",USER,NULL,NULL);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("Model host","string",EPIC_MPI_HOST,NULL,NULL);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("Infile","string","./epic.nc",NULL,NULL);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("Number of nodes","integer",1,1,512);
  AVSconnect_widget(wd[iwd],"typein_integer");

  wd[++iwd] = AVSadd_parameter("Backup interval:","integer",2000,0,1000000);
  AVSconnect_widget(wd[iwd],"typein_integer");

  wd[++iwd] = AVSadd_parameter("Save interval:","integer",1000000,0,1000000);
  AVSconnect_widget(wd[iwd],"typein_integer");

  wd[++iwd] = AVSadd_parameter("View interval:","integer",100,1,1000000);
  AVSconnect_widget(wd[iwd],"typein_integer");

  wd[++iwd] = AVSadd_parameter("Socket port:","string","waiting",NULL,NULL);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("Launch MPI","oneshot",0,NULL,NULL);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("Launch single","oneshot",0,NULL,NULL);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("sleep","boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  wd[++iwd] = AVSadd_parameter("kill","boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  /* 
   *  Input user-defined data types:
   */
  AVSload_user_data_types(EPIC_PATH"/include/avs_epic.h");

  return(1);
}
 
/*======================= end of AVSreceive() ==================================*/

/*======================= main() ===============================================*/

main(int   argc,
     char *argv[])
{ 
  char   
    statmem[64],          /*  AVS widget string - status                 */
    usermem[64],          /*  AVS widget string - user on remote machine */
    modelmem[FILE_STR],   /*  AVS widget string - computer running model */
    progmem[FILE_STR],    /*  AVS widget string - prog to exec           */
    inmem[FILE_STR],      /*  AVS widget string - input file             */
    sockmem[16],          /*  AVS widget string - socket port            */
    sysstr[FILE_STR],     /*  system command                             */
    buf[80],              /*  Temporary string buffer                    */
    *status     = statmem,
    *model_user = usermem,
    *model_host = modelmem,
    *infile     = inmem,
    *sockstr    = sockmem; 
  timespec
    *ptr_time;
  int    
    count,                 /*  main loop counter                          */
    num_nodes,             /*  AVS widget integer - num_nodes             */
    launch_mpi,            /*  AVS widget oneshot - launch signal         */
    launch_single,         /*  AVS widget oneshot - launch signal         */
    sleep,                 /*  AVS widget boolean - sleep process         */
    kill,                  /*  AVS widget boolean - kill model            */
    socket_port;           /*  AVS widget integer - socket port           */
  int
    sock;
  FILE
    *tmpfile;
  /* 
   * The following are part of DEBUG_MILESTONE statements: 
   */
  int
    idbms=0;
  char
    dbmsname[]="epic_receive";

  /* 
   * Initialize coroutine 
   */
  AVScorout_init(argc,argv,AVSreceive);

  planet   = (planetspec *)AVSdata_alloc("struct planetspec", 0);
  domain   = (domainspec *)AVSdata_alloc("struct domainspec", 0);
  ptr_time = (timespec   *)AVSdata_alloc("struct timespec",   0);

  /* 
   * Wait for "Launch" before reading parameters and starting epic model 
   */
  AVSmodify_parameter("Status:", AVS_VALUE,"Waiting for launch",NULL,NULL);
  do {
    AVScorout_input(&status,&model_user,&model_host,&infile,
                    &num_nodes,&grid.itback,&grid.itsave,&grid.itout,
                    &sockstr,&launch_mpi,&launch_single,&sleep,&kill);
    if (kill) goto exit_point;
  } while (launch_mpi != 1 && launch_single != 1);

  AVSmodify_parameter("Status:",AVS_VALUE,"Creating socket",NULL,NULL);
  create_socket(&sock,&socket_port);

  sprintf(buf,"%d",socket_port);
  AVSmodify_parameter("Socket port:",AVS_VALUE,buf,NULL,NULL); 

  if (launch_mpi) {
    /*
     * Launch epic on MPI host:
     */
    if (strcmp(EPIC_VIEW_HOST,model_host) != 0) {
      /* Remote machine */
      sprintf(sysstr,"ssh -n -l %s %s 'cd $EPIC_PATH/bin; "
                     "mpi_launch.csh %d %d %d %d %d %s %s' &",
              model_user,model_host,num_nodes,grid.itback,grid.itsave,grid.itout,
              socket_port,EPIC_VIEW_HOST,infile);
    }
    else {
      sprintf(sysstr,"cd $EPIC_PATH/bin; "
                     "mpi_launch.csh %d %d %d %d %d %s %s &",
                      num_nodes,grid.itback,grid.itsave,grid.itout,
                      socket_port,EPIC_VIEW_HOST,infile);
    }
  }
  else if (launch_single) {
    /*
     * Launch epic on single processor without MPI:
     */
    if (strcmp(EPIC_VIEW_HOST,model_host) != 0) {
      /* Remote machine */
      sprintf(sysstr,"ssh -n -l %s %s 'cd $EPIC_PATH/bin; "
                     "single_launch.csh %d %d %d %d %s %s' &",
              model_user,model_host,grid.itback,grid.itsave,grid.itout,
              socket_port,EPIC_VIEW_HOST,infile);
    }
    else {
      sprintf(sysstr,"cd $EPIC_PATH/bin;"
                     "single_launch.csh %d %d %d %d %s %s &",
              grid.itback,grid.itsave,grid.itout,
              socket_port,EPIC_VIEW_HOST,infile);
    }
  }
  else {
    fprintf(stderr,"Error: epic_receive, unknown launch. \n");
    exit(1);
  }
  system(sysstr);
  sprintf(buf,"EPIC on %s",model_host);
  AVSmodify_parameter("Status:",AVS_VALUE,buf);

  /* 
   * Get the model size and declare arrays: 
   */
  count=0;
  var_read(planet,infile,sock,VIA_SOCKET,INIT_DATA);
  AVSmodify_parameter("Status:",AVS_VALUE,"Make arrays");
  make_arrays();

  /* Get the initial frame from model host: */
  AVSmodify_parameter("Status:",AVS_VALUE,"Receiving initial frame");

  var_read(planet,infile,sock,VIA_SOCKET,UVP_DATA);
  grid_to_domain(domain,1);

  /* Output first frame: */
  AVSmodify_parameter("Status:",AVS_VALUE,"Waiting...",NULL,NULL);
  AVSmodify_parameter("Status:",AVS_VALUE,"Data output",NULL,NULL);

  /* Bring the AVS timespec port up to date before output */
  ptr_time->secs  = var.time[0];
  ptr_time->years = var.time[1];

  /* Wait for flow executive to finish other tasks before output: */
  AVScorout_exec();
  AVScorout_output(planet,domain,var.field,ptr_time);

  AVSmodify_parameter("Status:", AVS_VALUE, "Output first frame",NULL,NULL);

  /* 
   * Main loop of program: 
   */
  count = 1;
  while(1) {
    AVScorout_input(&status,&model_user,&model_host,&infile,
                    &num_nodes,&grid.itback,&grid.itsave,&grid.itout,
                    &sockstr,&launch_mpi,&launch_single,&sleep,&kill); 
    if (kill) goto exit_point;
    while (sleep) {
      AVSmodify_parameter("Status:",AVS_VALUE,"Sleeping",NULL,NULL);
      AVScorout_wait();
      AVScorout_input(&status,&model_user,&model_host,&infile,
                      &num_nodes,&grid.itback,&grid.itsave,&grid.itout,
                      &sockstr,&launch_mpi,&launch_single,&sleep,&kill);
      if (kill) goto exit_point;
    }
   
    /* Read the next frame of data */
    sprintf(buf,"Waiting for frame %d",count);
    AVSmodify_parameter("Status:",AVS_VALUE,buf,NULL,NULL);

    var_read(planet,infile,sock,VIA_SOCKET,UVP_DATA);
    grid_to_domain(domain,1);

    /* Output to AVS */
    sprintf(buf, "Output of frame %6d", count);
    AVSmodify_parameter("Status:",AVS_VALUE,buf,NULL,NULL);

    /* Bring the AVS timespec port up to date before output */
    ptr_time->secs  = var.time[0];
    ptr_time->years = var.time[1];
    /* Wait for flow executive to finish other tasks before output: */
    AVScorout_exec();
    AVScorout_output(planet,domain,var.field,ptr_time);

    /* Increment counter */
    count++;    
  } 

  /*
   * EXIT POINT
   */
  exit_point:

  /* Read the final frame of data */

  var_read(planet,infile,sock,VIA_SOCKET,UVP_EXIT);
  grid_to_domain(domain,1);

  /* Output to AVS */
  sprintf(buf, "Final frame %6d", count);
  AVSmodify_parameter("Status:",AVS_VALUE,buf,NULL,NULL);

  /* Bring the AVS timespec port up to date before output */
  ptr_time->secs  = var.time[0];
  ptr_time->years = var.time[1];
  /* Wait for flow executive to finish other tasks before output: */
  AVScorout_exec();
  AVScorout_output(planet,domain,var.field,ptr_time);

  free_arrays(planet);

  exit(1);
}

/*======================= end of main()  ====================================*/






