/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                             *
 *   epic_view.c                                                               *
 *                                                                             *
 *   AVS subroutine module to convert the EPIC                                 *
 *   variables into AVS geometries.                                            *
 *                                                                             *
 *   A. Fischer, T. Dowling                                                    *
 *                                                                             *
 *   NOTE: The order of the port arguments in the computation function must    *
 *         match the creation order in the description function.               *
 *                                                                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <epic.h>

#define HUMIDITY_TYPE_SPECIFIC 0
#define HUMIDITY_TYPE_RELATIVE 1
#define HUMIDITY_TYPE_COMBINED 2

/*
 * Prototypes:
 */
/* Description function */
int view_description(void);     

/* 
 * Prototype for computation function.
 * NOTE: Order of arguments must be: input ports, output ports, widgets.
 */
int view_computation(
  planetspec      *planet,       /* Input ports */
  domainspec      *domain,
  AVSfield_double *vars,
  timespec        *ptr_time,
  AVScolormap     *colormap,
  GEOMedit_list   *layers_list,  /* Output ports */
  GEOMedit_list   *sphere_list,
  AVSfield_double **right_field,
  AVSfield_double **left_field,
  int             light_north,   /* Widgets */
  int             layers,
  int             sphere,
  int             fileout,
  int             write_qbzd,
  int             write_heating,
  int             write_uvp,
  int             layers_polar,
  int             wind_l,
  int             wind_s,
  int             calc_min_Ri,
  int             wind_scale,
  int             layers_height_scale,
  int             sphere_height_scale,
  float           *delta_color_scale,
  int             layers_downsize,
  int             sphere_downsize,
  int             sphere_layer,
  char            *str_Ri,
  char            *str_CFL,
  int             humidity_type,
  char            *right_field_input,
  char            *left_field_input);

/* Global variables */
static char
  str_Ri[64],               /* memory for Min Ri string              */
  str_CFL[64],              /* memory for CFL timestep               */
  right_field_init[64],     /* memory for right output field init    */
  left_field_init[64],      /* memory for left  output field init    */
  default_field_list[512],  /* memory for default output field list  */
  right_field_list[512],    /* memory for right output field list    */
  left_field_list[512];     /* memory for left  output field list    */

void AVSinit_modules()
{
  /* Invoke EPIC View module from description function */
  AVSmodule_from_desc(view_description);
}

/* -------------------   Description function   ---------------------------- */
/*     This function describes the attributes of the ports and widgets.      */

view_description()
{
  int    
    output[4],        /* Output port descriptors                             */
    wd[50],           /* widget numbers                                      */
    iwd;              /* widget index                                        */

  /* Set module name and type */
  AVSset_module_name("EPIC View", MODULE_MAPPER);

  /* 
   * NOTE: The following must be in the order of the arguments 
   * in the computation function.
   */
  /* Create input ports */
  AVScreate_input_port("Planet structure", "struct planetspec", REQUIRED);
  AVScreate_input_port("Domain structure", "struct domainspec", REQUIRED);
  AVScreate_input_port("Variables",        "field 3D double",   REQUIRED);
  AVScreate_input_port("Time",             "struct timespec",   REQUIRED);
  AVScreate_input_port("colormap",         "colormap",          OPTIONAL);
  
  /* Create output ports */
  output[0] = AVScreate_output_port("layers geometry", "geom");
  output[1] = AVScreate_output_port("sphere geometry", "geom");
  output[2] = AVScreate_output_port("right output field & file output",  "field");
  output[3] = AVScreate_output_port("left output field & geometry color","field");

  AVSautofree_output(output[0]);  
  AVSautofree_output(output[1]);  
  AVSautofree_output(output[2]);
  AVSautofree_output(output[3]);

  /* Create widgets */
  iwd = -1;

  wd[++iwd] = AVSadd_parameter("light north",  "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:light north\" -w toggle -p \"$Module\""
    " -xy 10,10 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("layers output","boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:layers output\" -w toggle -p \"$Module\""
    " -xy 128,10 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("sphere output","boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:sphere output\" -w toggle -p \"$Module\""
    " -xy 10,32 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("file output",  "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:file output\" -w toggle -p \"$Module\""
    " -xy 128,32 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("write q,B,z,d","boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:write q,B,z,d\" -w toggle -p \"$Module\""
    " -xy 10,54 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("write heat",   "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:write heat\" -w toggle -p \"$Module\""
    " -xy 128,54 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("write u,v,p",  "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:write u,v,p\" -w toggle -p \"$Module\""
    " -xy 10,76 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("polar layers", "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:polar layers\" -w toggle -p \"$Module\""
    " -xy 128,76 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("graph u on layers",  "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:graph u on layers\" -w toggle -p \"$Module\""
    " -xy 10,98 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("graph u on sphere",  "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:graph u on sphere\" -w toggle -p \"$Module\""
    " -xy 128,98 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("Calculate Min Ri",   "boolean",0,0,1);
  AVSadd_parameter_prop(wd[iwd],"layout","string_block",
    "manipulator \"$Module:Calculate Min Ri\" -w toggle -p \"$Module\""
    " -xy 10,120 -wh 116,22");

  wd[++iwd] = AVSadd_parameter("u scale", "integer",50,0,500);
  AVSconnect_widget(wd[iwd],"islider");

  wd[++iwd] = AVSadd_parameter("layers height scale", "integer", 50,  0, 500);
  AVSconnect_widget(wd[iwd],"islider");

  wd[++iwd] = AVSadd_parameter("sphere height scale", "integer", 50,  0, 500);
  AVSconnect_widget(wd[iwd],"islider");

  wd[++iwd] = AVSadd_float_parameter("delta color scale",1.,FLOAT_UNBOUND,FLOAT_UNBOUND);
  AVSconnect_widget(wd[iwd],"typein_real");

  wd[++iwd] = AVSadd_parameter("layers downsize",     "integer",  1,  1,  32);
  AVSconnect_widget(wd[iwd],"islider");

  wd[++iwd] = AVSadd_parameter("sphere downsize",     "integer",  1,  1,  32);
  AVSconnect_widget(wd[iwd],"islider");

  wd[++iwd] = AVSadd_parameter("sphere layer",        "integer",  1,  1,   3);
  AVSconnect_widget(wd[iwd],"islider");

  strcpy(str_Ri,"(k,j,i) =              ");
  wd[++iwd] = AVSadd_parameter("Min Ri","string",str_Ri,NULL,NULL);

  strcpy(str_CFL,"(k,j,i) =               ");
  wd[++iwd] = AVSadd_parameter("CFL dt","string",str_CFL,NULL,NULL);

  wd[++iwd] = AVSadd_parameter("humidity type","tristate",HUMIDITY_TYPE_RELATIVE,0,2);
  AVSadd_parameter_prop(wd[iwd],"width","integer",4);

  /* memory for strings defined above */
  strcpy(default_field_list,"potential vorticity!angular momentum!"
                            "N^2!temperature!delta temperature");
  strcpy(right_field_init,"potential vorticity");
  strcpy(right_field_list,default_field_list);
  wd[++iwd] = AVSadd_parameter("right output field & file output","choice",
                                right_field_init,right_field_list,"!");
  AVSconnect_widget(wd[iwd],"choice_browser");

  strcpy(left_field_init,"potential vorticity");
  strcpy(left_field_list,default_field_list);
  wd[++iwd] = AVSadd_parameter("left output field & geometry color","choice",
                                left_field_init,left_field_list,"!");
  AVSconnect_widget(wd[iwd],"choice_browser");

  /* computation function */
  AVSset_compute_proc(view_computation);

  /* User data types */
  AVSload_user_data_types(EPIC_PATH"/include/avs_epic.h");
}  

/* -----------------------   Computation function   ------------------------- */

#define NUM_HEAT      5
#define NUM_QBZD      5
#define AVG_HEAT(k,l) avg_heat[l+(k-KLO)*NUM_HEAT]
#define HUMIDITY(j,i) humidity[i+(j)*Iadim-Shift2d]

int view_computation(
  planetspec      *planet,      /* Input ports */
  domainspec      *domain,
  AVSfield_double *vars,
  timespec        *ptr_time,
  AVScolormap     *colormap,
  GEOMedit_list   *layers_list, /* Output ports */
  GEOMedit_list   *sphere_list,
  AVSfield_double **right_field,
  AVSfield_double **left_field,
  int             light_north,  /* Widgets */
  int             layers, 
  int             sphere, 
  int             fileout,
  int             write_qbzd,
  int             write_heating,
  int             write_uvp,
  int             layers_polar,
  int             wind_l,
  int             wind_s,
  int             calc_min_Ri,
  int             wind_scale,
  int             layers_height_scale, 
  int             sphere_height_scale, 
  float           *delta_color_scale,
  int             layers_downsize,
  int             sphere_downsize,
  int             sphere_layer,
  char            *str_Ri,
  char            *str_CFL,
  int             humidity_type,
  char            *right_field_input,
  char            *left_field_input)
{
  listspec
    laymem,*layerinfo,      /*  geometry information              */
    sphmem,*sphereinfo;
  int
    nk,nj,ni,
    l,k,j,i,ii,
    kmin,jmin,imin,
    dims[3],
    index,
    rt_index=-1,
    lt_index=-1,
    nkm1,nkm1ni,
    shift,flag;
  static int
    prev_time[2] = {-INT_MAX,-INT_MAX},
    prev_size[3] = { INT_MAX, INT_MAX, INT_MAX},
    state_change = TRUE,
    made_arrays  = FALSE;
  static char
    prev_name[64];
  long
    offset;
  double
    *q,*mont,*ptr,
     pv,hqy,beta,
     kin,exner,bern,z,
     d_th0,n2jp1,
     uu,vv,pp,
    *heat;
  static double 
    *color_min,             /*  Min color variable by layer       */
    *color_max,             /*  Max color variable by layer       */
    *avg_field,             /*  average color-variable value      */
    *buff[2],               /*  x-y plane working memory          */
    *row1,*row2,            /*  working row arrays                */
    *avg_heat,              /*  average heat                      */
    *humid,                 /*  humidity in K column              */
    *cloud_density;         /*  cloud density [kg/m^3]            */
  double
    *buffji,                /*  generic pointer to x-y memory     */
    *humidity,              /*  humidity in J,I plane             */
    *h;                     /*  h = -dp/dtheta                    */
  double
    Ri,min_Ri,dudtheta,     /*  Used to calculate Richardson num. */
    temperature,            /*  temperature                       */
    pressure,               /*  pressure                          */
    density,                /*  density                           */
    mu,                     /*  molar mass                        */
    fpara,                  /*  fraction of para hydrogen         */
    fpe,uoup,fgibb,         /*  equil fp, etc                     */
    theta,                  /*  potential temperature             */
    mn_2jp1,mn_2j_inv,      /*  map factors for kin               */
    mn_2jp2_inv,dx,         /*  map factors for kin               */
    radius,                 /*  radius                            */
    tempmin, tempmax,       /*  temporary max/mins                */
    cpr,                    /*  reference cp                      */
    tmp;
  static double
    prev_x_h2=DBL_MAX,      /*  used to trigger thermo_setup()    */
    prev_x_he=DBL_MAX,      /*  used to trigger thermo_setup()    */
    prev_x_3 =DBL_MAX;      /*  used to trigger thermo_setup()    */
  char
    outfile_str[128],       /*  string for output file name       */
    field_type[64],         /*  output field description          */
    header[64],             /*  char string buffer                */
    humidity_str[32],       /*  string for type of humidity       */
    err_str[256];           /*  error message string              */
  FILE
    *outfile,               /*  pointer to output file            */
    *qbzd_bin,
    *heat_bin,
    *qbzd_text,
    *heat_text,
    *uvp_text;
  int
    idbms=0;
  char
    dbmsname[]="epic_view_computation";

  if (planet->x_h2 != prev_x_h2 ||
      planet->x_he != prev_x_he ||
      planet->x_3  != prev_x_3) {
    prev_x_h2 = planet->x_h2;
    prev_x_he = planet->x_he;
    prev_x_3  = planet->x_3;
   /*
    * Set up thermodynamics subroutines:
    */
    thermo_setup(planet,&cpr);
  }

  nk = domain->nk;
  nj = domain->nj;
  ni = domain->ni;

  layerinfo  = &laymem;
  sphereinfo = &sphmem;

  /* 
   * Modify widgets:
   */
  strcpy(right_field_list,default_field_list);
  strcpy(left_field_list, default_field_list);

  for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
    if (domain->chem_on[index]) {
      if (index == FPARA_INDEX) {
        sprintf(header,"!%s!equil %s",
                domain->chem_name[index],domain->chem_name[index]);
      }
      else {
        sprintf(header,"!%s",domain->chem_name[index]);
      }
      strcat(right_field_list,header);
      strcat(left_field_list, header);
    }
  }

  AVSmodify_parameter("right output field & file output",AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
                       right_field_input,right_field_list,"!");
  AVSmodify_parameter("left output field & geometry color",AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
                       left_field_input, left_field_list, "!");
  AVSmodify_parameter("sphere layer",AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
                       sphere_layer,1,domain->klast_active);

  switch (humidity_type) {
    case HUMIDITY_TYPE_SPECIFIC:
      sprintf(humidity_str,"specific humidity");
      break;
    case HUMIDITY_TYPE_RELATIVE:
      sprintf(humidity_str,"relative humidity");
      break;
    case HUMIDITY_TYPE_COMBINED:
      sprintf(humidity_str,"total cloud density");
      break;
  }
  AVSmodify_parameter_prop("humidity type","title","string",humidity_str);

  /* 
   * Set up var so that prognostic-variable macros (ie, P(k,j,i)) will work: 
   */
  var.field   = vars;
  var.time[0] = ptr_time->secs;
  var.time[1] = ptr_time->years;
  for (index = 0; index < MAX_NVARS; index++) {
    var.index[index]   = domain->index[index];
    var.chem_on[index] = domain->chem_on[index];
    strcpy(var.chem_name[index],domain->chem_name[index]);
  }

  /* Determine if model has changed */
  if (strcmp(planet->name,prev_name) != 0 ||
      var.time[0] != prev_time[0] ||
      var.time[1] != prev_time[1] ||
      grid.ni     != prev_size[0] ||
      grid.nj     != prev_size[1] ||
      grid.nk     != prev_size[2]) {

    state_change = TRUE;

    strcpy(prev_name,planet->name);
    prev_time[0] = var.time[0];
    prev_time[1] = var.time[1];
    prev_size[0] = grid.ni;
    prev_size[1] = grid.nj;
    prev_size[2] = grid.nk;
  }

  if (state_change == TRUE &&
      made_arrays  == TRUE) {
    /* Free allocated memory: */

    free_dvector(cloud_density,0,grid.nk);
    free_dvector(humid,        0,grid.nk);
    free_dvector(avg_heat,     0,NUM_HEAT*KLAST_ACTIVE-1);
    free_dvector(row2,         0,NUM_QBZD*KLAST_ACTIVE-1);
    free_dvector(row1,         0,NUM_HEAT*KLAST_ACTIVE-1);
    free_dvector(buff[1],      0,Nelem2d-1);
    free_dvector(buff[0],      0,Nelem2d-1);
    free_dvector(avg_field,    1,nk);
    free_dvector(color_max,    1,KLAST_ACTIVE);
    free_dvector(color_min,    1,KLAST_ACTIVE);
    free_arrays(planet);

    made_arrays = FALSE;
  }

  if (state_change == TRUE &&
      made_arrays  == FALSE) {

    /* Allocate memory */
    make_arrays(planet,domain);

    color_min     = dvector(1,KLAST_ACTIVE);
    color_max     = dvector(1,KLAST_ACTIVE);
    avg_field     = dvector(1,nk);
    buff[0]       = dvector(0,Nelem2d-1);
    buff[1]       = dvector(0,Nelem2d-1);
    row1          = dvector(0,NUM_HEAT*KLAST_ACTIVE-1);
    row2          = dvector(0,NUM_QBZD*KLAST_ACTIVE-1);
    avg_heat      = dvector(0,NUM_HEAT*KLAST_ACTIVE-1);
    humid         = dvector(0,grid.nk);  
    cloud_density = dvector(0,grid.nk);    

    made_arrays = TRUE;

    /* Store heavily used diagnostic variables: */
    store_temp_dens_p(planet,JLO,JHI,ILO,IHI);
  }

  dims[0] = IADIM;
  dims[1] = JADIM;
  dims[2] = KADIM;
  /* 
   *  Allocate AVSfield memory for the output fields: 
   */
  if (strcmp(grid.geometry,"f-plane")  == 0 &&
      strcmp(grid.f_plane_map,"polar") == 0) {
    double
      r,lon,lat;

    sprintf(field_type,"field 3D scalar double irregular");
    /* output ports are doubly indirected (ie, use *right_field instead of right_field): */
    *right_field = (AVSfield_double *) AVSdata_alloc(field_type,dims);
    *left_field  = (AVSfield_double *) AVSdata_alloc(field_type,dims);
    /* 
     * Specify coordinate mapping (use q grid):
     */
    for (k = KLO-KPAD; k <= KHI+KPAD; k++) {
      for (j = JLO-JPAD; j <= JHI+JPAD; j++) {
        if (j == -1) {
          /* give consistent geometry to this unused row */
          lat = grid.lat[2*0]-grid.dlt;
        }
        else if (j == nj+1) {
          /* back off from i degeneracy at pole for ease of plotting */
          lat = 90.-1.e-5;
        }
        else {
          lat = grid.lat[2*j];
        }
        r = (90.-lat)/90.;
        for (i = ILO-IPAD; i <= IHI+IPAD; i++) {
          if (ni == 1) {
            lon = -180.+360.*((double)i-1.)/((double)ni);
          }
          else {
            lon = grid.lon[2*i];
          }
          ECOORD_X(*right_field,k,j,i) = .5*r*cos((lon-90.)*DEG);
          ECOORD_X(*left_field, k,j,i) = .5*r*cos((lon-90.)*DEG);

          ECOORD_Y(*right_field,k,j,i) = .5*r*sin((lon-90.)*DEG);
          ECOORD_Y(*left_field, k,j,i) = .5*r*sin((lon-90.)*DEG);

          ECOORD_Z(*right_field,k,j,i) = 2.*(float)(nk-k)/(nk-1);
          ECOORD_Z(*left_field, k,j,i) = 2.*(float)(nk-k)/(nk-1);
        }
      }
    }
  }
  else {
    sprintf(field_type,"field 3D scalar double uniform");
    *right_field = (AVSfield_double *) AVSdata_alloc(field_type,dims);
    *left_field  = (AVSfield_double *) AVSdata_alloc(field_type,dims);
  }

    /* 
     *  Create geometries.
     */
  /* Initialize the edit lists */
  *layers_list = GEOMinit_edit_list(*layers_list);
  *sphere_list = GEOMinit_edit_list(*sphere_list);

  /* 
   * Calculate Min Ri:
   */
  if (calc_min_Ri) {
    min_Ri = min_richardson(planet,&kmin,&jmin,&imin);
    if (min_Ri == DBL_MAX) {
      sprintf(str_Ri,"infinity");
    }
    else {
      sprintf(str_Ri,"(%d,%d,%d) = %-9.2f",kmin,jmin,imin,min_Ri);
    }
    AVSmodify_parameter("Min Ri",AVS_VALUE,str_Ri,NULL,NULL);
  }

  /* Post CFL dt */
  if (grid.cfl_dt > 0.) {
    sprintf(str_CFL,"%d",grid.cfl_dt);
    AVSmodify_parameter("CFL dt",AVS_VALUE,str_CFL,NULL,NULL);
  }

  if (write_qbzd) {
    /*
     * Write q,B,hqy,z,density to text files for each layer, labeled by theta:
     */

    nkm1   = KLAST_ACTIVE;
    nkm1ni = nkm1*grid.ni;
    shift  = KLO+ILO*nkm1+JLO*nkm1ni;
    /* 
     * The values are not calculated in a column order that is convenient
     * for plotting.  Rather than declaring 3D arrays here, 
     * use a random-access binary disk file to store the values
     * as they are calculated, and then convert the file to ascii.
     */ 
    qbzd_bin = fopen(EPIC_PATH"/tmp/qbzd.bin","wb");

    q = buff[0];
    h = buff[1];

    /* Calculate and store potential vorticity, pv, and hqy: */
    for (k = KLAST_ACTIVE; k >= 1; k--) {
      potential_vorticity(planet,q,k,1);
      for (j = JLO; j <= JHI; j++) {
        for (i = ILO; i <= IHI; i++) {
          H(j,i) = get_h(planet,2*k,j,i);
        }
      }
      /* No need to apply BC2D to H here */

      for (j = JLO; j <= JHI; j++) {
        n2jp1 = grid.n[2*j+1];
        for (i = ILO; i <= IHI; i++) {
          /* average q onto B grid */
          pv = .25*(Q(j,i)+Q(j,i+1)+Q(j+1,i)+Q(j+1,i+1));

          /* Seek to first slot: */
          offset = (long)(((k+i*nkm1+j*nkm1ni-shift)*NUM_QBZD+0)*sizeof(double));
          fseek(qbzd_bin,offset,SEEK_SET);
          fwrite(&pv,sizeof(double),1,qbzd_bin);

          hqy = H(j,i)*.5*(Q(j+1,i)+Q(j+1,i+1)
                          -Q(j,  i)-Q(j,  i+1))*n2jp1;

          /* Seek to third slot: */
          offset = (long)(((k+i*nkm1+j*nkm1ni-shift)*NUM_QBZD+2)*sizeof(double));
          fseek(qbzd_bin,offset,SEEK_SET);
          fwrite(&hqy,sizeof(double),1,qbzd_bin);
        }
      }
    }
    /* 
     * Calculate Bernoulli streamfunction, B, altitude, z, and density: 
     */
    mont = buff[0];
    mont_nk(planet,mont); 
    for (k = KLAST_ACTIVE; k >= 1; k--) {
      if (k > 1) {
        d_th0 = grid.theta[2*k]-grid.theta[2*k+2];
      }
      else {
        /* special delta-theta in top layer: */
        d_th0 = grid.theta[2*k]-grid.theta[2*k+2];
      }
      for (j = JLO; j <= JHI; j++) {
        mn_2jp1     =    (grid.mn)[2*j+1];
        mn_2j_inv   = 1./(grid.mn)[2*j  ];
        mn_2jp2_inv = 1./(grid.mn)[2*j+2];
        for (i = ILO; i <= IHI; i++) {
          kin =  (    U(k,j,  i+1)*U(k,j,  i+1)+
                      U(k,j,  i  )*U(k,j,  i  ) 
            +mn_2jp1*(V(k,j+1,i  )*V(k,j+1,i  )*mn_2jp2_inv 
                     +V(k,j,  i  )*V(k,j,  i  )*mn_2j_inv  ) )*.25;
          theta       = grid.theta[2*k+1];
          temperature = T2(k,j,i);
          exner       = (planet->cp)*temperature/theta;
          if (k < grid.nk) {
            /* k = nk case (terrestrial planets) is set before entering loop */
            MONT(j,i)  += exner*d_th0;
          }
          bern        = MONT(j,i)+kin; 

          /* Seek to second slot: */
          offset = (long)(((k+i*nkm1+j*nkm1ni-shift)*NUM_QBZD+1)*sizeof(double));
          fseek(qbzd_bin,offset,SEEK_SET);
          fwrite(&bern,sizeof(double),1,qbzd_bin);

          /* Calculate altitude, z, and density: */
          fpara       = get_chem(planet,FPARA_INDEX,2*k,j,i);
          pressure    = P1(   k,j,i);
          temperature = T(    k,j,i);
          z           = (MONT(j,i)-return_enthalpy(planet,fpara,pressure,
                                   temperature,&fgibb,&fpe,&uoup))/planet->g;
          /* Seek to fourth slot: */
          offset = (long)(((k+i*nkm1+j*nkm1ni-shift)*NUM_QBZD+3)*sizeof(double));
          fseek(qbzd_bin,offset,SEEK_SET);
          fwrite(&z,sizeof(double),1,qbzd_bin);

          density = RHO(k,j,i);
          /* Seek to fifth slot: */
          offset = (long)(((k+i*nkm1+j*nkm1ni-shift)*NUM_QBZD+4)*sizeof(double));
          fseek(qbzd_bin,offset,SEEK_SET);
          fwrite(&density,sizeof(double),1,qbzd_bin);
        }
      }
    }
    fclose(qbzd_bin);

    /* 
     * Convert the binary file to an ascii file, with the ordering
     * in columns that are convenient for plotting.
     */
    qbzd_bin  = fopen(EPIC_PATH"/tmp/qbzd.bin","rb");
    
    for (k = 1; k <= KLAST_ACTIVE; k++) {
      sprintf(outfile_str,EPIC_PATH"/tmp/qbzd%05.0f_%05d.dat",
              grid.theta[2*k],
              var.time[0]/86400+var.time[1]*365);     
      qbzd_text = fopen(outfile_str,"w");

      fprintf(qbzd_text," system: %s  t = %d s, %d yrs \n",
                         planet->name,var.time[0],var.time[1]);
      fprintf(qbzd_text," nk = %d nj = %d ni = %d theta = %5.0f\n",
                         grid.nk,grid.nj,grid.ni,grid.theta[2*k]);
      fprintf(qbzd_text," Units: lat and lon in degrees, else in mks \n");
      /* Write NUM_QBZD+2 instead of NUM_QBZD to count lon,beta columns: */
      fprintf(qbzd_text," %d\n",NUM_QBZD+2);
      /* The ()'s are so that the preceeding symbol may be located, ie lon: */
      fprintf(qbzd_text,"   lat    lon()     beta() ");
      theta = grid.theta[2*k];
      fprintf(qbzd_text,"    q(%05.0f)     B(%05.0f)    hqy(%05.0f)"
                           "   z(%05.0f)    dens(%05.0f)\n",
                         theta,theta,theta,theta,theta);

      fclose(qbzd_text);
    }

    for (j = JLO; j <= JHI; j++) {
      beta  = (grid.f[2*j+2]-grid.f[2*j])*grid.n[2*j+1];
      for (i = ILO; i <= IHI; i++) {
        fread(row2,sizeof(double),NUM_QBZD*KLAST_ACTIVE,qbzd_bin);
        ptr = row2;
        for (k = 1; k <= KLAST_ACTIVE; k++) {
          sprintf(outfile_str,EPIC_PATH"/tmp/qbzd%05.0f_%05d.dat",
                  grid.theta[2*k],
                  var.time[0]/86400+var.time[1]*365);     
          qbzd_text = fopen(outfile_str,"a");
          if (!qbzd_text) {
            sprintf(err_str,"Error: epic_view: Failed to open qbzd_text[%d] %s\n",
                    k,outfile_str);
            perror(err_str);
            exit(1);
          }

          fprintf(qbzd_text," %6.1f %6.1f %12.5e", 
                             grid.lat[2*j+1],grid.lon[2*i+1],beta);
          for (l = 0; l < NUM_QBZD; l++) {
            fprintf(qbzd_text," %12.5e",*(ptr++));
          }
          fprintf(qbzd_text,"\n");
          fclose(qbzd_text);
        }
      }
    }
    fclose(qbzd_bin);

    /* Remove temporary binary file: */
    system("rm "EPIC_PATH"/tmp/qbzd.bin");
  }

  if (write_heating || 
      time_mod(var.time,domain->itback*domain->dt) == 0 ) {
    /*
     * Write heating profiles to a text file.
     */
    heat = buff[0];

    /* 
     * Calling heating() from EPIC_VIEW generates 
     * EPIC_PATH/tmp/heat.bin.
     *
     * NOTE: It is assumed here that the function heating() called from 
     * EPIC_VIEW is the same one that was called while running the model.
     */
    for (k = KLO; k <= KLAST_ACTIVE; k++) {
      heating(planet,heat,k);
    }

    /*
     * Convert heat.bin to a text file with columns that are
     * convenient for plotting:
     */
    heat_bin = fopen(EPIC_PATH"/tmp/heat.bin","rb");
    if (!heat_bin) {
      perror("epic_view: heat.bin rb");
    }

    for (k = KLO; k <= KLAST_ACTIVE; k++) {
      sprintf(outfile_str,EPIC_PATH"/tmp/heat%07.2f_%05d",
              grid.theta[2*k],var.time[0]/86400+var.time[1]*365);     
      heat_text = fopen(outfile_str,"w");
      if(!heat_text) {
        sprintf(err_str,"Error: epic_view: Failed to open heat_text[%d] %s\n",
                k,outfile_str);
        perror(err_str);
        exit(1);
      }
      fprintf(heat_text," system: %s  t = %d s, %d yrs \n",
                          planet->name,var.time[0],var.time[1]);
      fprintf(heat_text," nk = %d nj = %d ni = %d theta = %7.2f\n",
                            grid.nk,grid.nj,grid.ni,grid.theta[2*k]);
      fprintf(heat_text," Units are mks.\n");
      fprintf(heat_text," %d\n",NUM_HEAT);
      fprintf(heat_text,"   lat   ");

      theta = grid.theta[2*k];
      fprintf(heat_text,"cnd(%05.0f)    fp(%05.0f)   nwt(%05.0f)"
                           "   adj(%05.0f)   tot(%05.0f)\n",
                         theta,theta,theta,theta,theta);
      fclose(heat_text);
    }

    for (j = grid.jlo; j <= grid.nj; j++) {
      for (k = KLO; k <= KLAST_ACTIVE; k++) {
        for (l = 0; l < NUM_HEAT; l++) {
          AVG_HEAT(k,l) = 0.;
        }
      }
      /* Read heat components, zonally average: */
      for (i = 1; i <= grid.ni; i++) {
        fread(row1,sizeof(double),NUM_HEAT*KLAST_ACTIVE,heat_bin);
        ptr = row1;
        for (k = KLO; k <= KLAST_ACTIVE; k++) {
          for (l = 0; l < NUM_HEAT; l++) {
            AVG_HEAT(k,l) += *(ptr++);
          }
        }
      }
      for (k = KLO; k <= KLAST_ACTIVE; k++) {
        sprintf(outfile_str,EPIC_PATH"/tmp/heat%07.2f_%05d",
                grid.theta[2*k],var.time[0]/86400+var.time[1]*365);     
        heat_text = fopen(outfile_str,"a");
        if(!heat_text) {
          sprintf(err_str,"Error: epic_view: Failed to open heat_text[%d] %s\n",
                  k,outfile_str);
          perror(err_str);
          exit(1);
        }

        fprintf(heat_text," %6.1f",grid.lat[2*j+1]);
        for (l = 0; l < NUM_HEAT; l++) {
          AVG_HEAT(k,l)  /= grid.ni;
          fprintf(heat_text," %12.5e",AVG_HEAT(k,l));
        }
        fprintf(heat_text,"\n");
        fclose(heat_text);
      }
    }

    /* Remove heat.bin: */
    fclose(heat_bin);
    system("rm "EPIC_PATH"/tmp/heat.bin");
  }

  if (write_uvp || 
      time_mod(var.time,domain->itback*domain->dt) == 0) {

    for (k = KLO; k <= KLAST_ACTIVE; k++) {
      sprintf(outfile_str,EPIC_PATH"/tmp/uvp%07.2f_%05d",
              grid.theta[2*k],var.time[0] / 86400 + var.time[1]*365); 
      uvp_text = fopen(outfile_str,"w");
      if (!uvp_text) {
        sprintf(err_str,"Error: epic_view: Failed to open uvp_text[%d] %s\n",
                k,outfile_str);
        perror(err_str);
        exit(1);
      }

      fprintf(uvp_text," system: %s  t = %d s, %d yrs \n",
                     planet->name,var.time[0],var.time[1]);
      fprintf(uvp_text," nk = %d nj = %d ni = %d theta = %7.2f\n",
                           grid.nk,grid.nj,grid.ni,grid.theta[2*k]);
      fprintf(uvp_text," u[m/s]  v[m/s]  p[mbar] \n");
      fprintf(uvp_text," 3\n");
      fprintf(uvp_text,"   lat   ");
      theta = grid.theta[2*k];
      fprintf(uvp_text,"   u(%05.0f)     v(%05.0f)     p(%05.0f) \n",
                          theta,theta,theta);
      fclose(uvp_text);
    }

    for (j = grid.jlo; j <= grid.nj; j++) {
      for (k = KLO; k <= KLAST_ACTIVE; k++) {
        sprintf(outfile_str,EPIC_PATH"/tmp/uvp%07.2f_%05d",
                grid.theta[2*k],var.time[0] / 86400 + var.time[1]*365); 
        uvp_text = fopen(outfile_str,"a");
        if (!uvp_text) {
          sprintf(err_str,"Error: epic_view: Failed to open uvp_text[%d] %s\n",
                  k,outfile_str);
          perror(err_str);
          exit(1);
        }
        fprintf(uvp_text," %6.1f",grid.lat[2*j+1]);
        /* Compute zonal averages, average onto h grid */
        uu = 0.;
        vv = 0.;
        pp = 0.;
        for (i = 1; i <= grid.ni; i++) {
          uu += U(k,j,i);
          vv += .5*(V(k,j,i)+V(k,j+1,i));
          pp += .01*P1(k,j,i);
        }
        uu /= grid.ni;
        vv /= grid.ni;
        pp /= grid.ni;
        fprintf(uvp_text," %12.5e %12.5e %12.5e\n",uu,vv,pp);
        fclose(uvp_text);
      }
    }
  }

  /* 
   * Calculate right and left output field variables.
   * NOTE: The left field is used for geometry color.
   */
  memset(buff[0],0,Nelem2d*sizeof(double));

  if (strcmp(right_field_input,"potential vorticity") == 0 ||
      strcmp(left_field_input, "potential vorticity") == 0) {
    for (k = 1; k <= KLAST_ACTIVE; k++) {
      potential_vorticity(planet,buff[0],k,1);
      buffji = buff[0];
      if (strcmp(right_field_input,"potential vorticity") == 0) {
        for (j = JLO; j <= JHI+1; j++) {
          for (i = 1; i <= ni; i++) {
            E3D(*right_field,k,j,i) = BUFFJI(j,i);
          }
          /* periodicity */
          E3D(*right_field,k,j,   0) = E3D(*right_field,k,j,ni);
          E3D(*right_field,k,j,ni+1) = E3D(*right_field,k,j, 1);
        }
      }
      if (strcmp(left_field_input,"potential vorticity") == 0) {
        color_min[k] =  DBL_MAX;
        color_max[k] = -DBL_MAX;
        for (j = JLO; j <= JHI+1; j++) {
          for (i = 1; i <= ni; i++) {
            E3D(*left_field,k,j,i) = BUFFJI(j,i);
            color_min[k] = MIN(color_min[k],BUFFJI(j,i));
            color_max[k] = MAX(color_max[k],BUFFJI(j,i));
          }
          /* periodicity */
          E3D(*left_field,k,j,   0) = E3D(*left_field,k,j,ni);
          E3D(*left_field,k,j,ni+1) = E3D(*left_field,k,j, 1);
        }
        if (planet->omega >= 2.*M_PI/(2.*24.*60.*60.)) {
          /* 
           * NOTE: Use predetermined range for q for rapidly rotating planets,
           * autostretch otherwise. (Probably should add an autostrech widget.)
           */
          color_min[k] = domain->qmin[k];
          color_max[k] = domain->qmax[k];
        }
      }
    }
  } 

  if (strcmp(right_field_input,"angular momentum") == 0 ||
      strcmp(left_field_input, "angular momentum") == 0) {
    double
      min_ang_momentum=+DBL_MAX,
      max_ang_momentum=-DBL_MAX;
    for (k = 1; k <= KLAST_ACTIVE; k++) {
      angular_momentum(planet,buff[0],k);
      buffji = buff[0];
      if (strcmp(right_field_input,"angular momentum") == 0) {
        for (j = JLO; j <= JHI+1; j++) {
          for (i = 1; i <= ni; i++) {
            E3D(*right_field,k,j,i) = BUFFJI(j,i);
          }
          /* periodicity */
          E3D(*right_field,k,j,   0) = E3D(*right_field,k,j,ni);
          E3D(*right_field,k,j,ni+1) = E3D(*right_field,k,j, 1);
        }
      }
      if (strcmp(left_field_input,"angular momentum") == 0) {
        for (j = JLO; j <= JHI+1; j++) {
          for (i = 1; i <= ni; i++) {
            E3D(*left_field,k,j,i) = BUFFJI(j,i);
            min_ang_momentum = MIN(min_ang_momentum,BUFFJI(j,i));
            max_ang_momentum = MAX(max_ang_momentum,BUFFJI(j,i));
          }
          /* periodicity */
          E3D(*left_field,k,j,   0) = E3D(*left_field,k,j,ni);
          E3D(*left_field,k,j,ni+1) = E3D(*left_field,k,j, 1);
        }
      }
    }
    if (strcmp(left_field_input,"angular momentum") == 0) {
      /* Use global range for color min, max: */
      for (k = 1; k <= KLAST_ACTIVE; k++) {
        color_min[k] = min_ang_momentum;
        color_max[k] = max_ang_momentum;
      }
    }
  } 

  if (strcmp(right_field_input,"temperature")       == 0 ||
      strcmp(right_field_input,"delta temperature") == 0 ||
      strcmp(right_field_input,"N^2")               == 0 ||
      strcmp(left_field_input, "temperature")       == 0 ||
      strcmp(left_field_input, "delta temperature") == 0 ||
      strcmp(left_field_input, "N^2")               == 0) {
    h = buff[0];
    for (k = 1; k <= KLAST_ACTIVE; k++) {
      for (j = JLO; j <= JHI; j++) {
        for (i = ILO; i <= IHI; i++) {
          H(j,i) = get_h(planet,2*k+1,j,i);
        }
      }
      /* No need to apply BC2D to H here */

      avg_field[k] = 0.;
      for (j = JLO; j <= JHI; j++) {
        for (i = 1; i <= ni; i++) {
          temperature = T2(  k,j,i);
          density     = RHO2(k,j,i);
          if (strcmp(right_field_input,"temperature")     == 0 ||
              strcmp(right_field_input,"delta temperature") == 0) {
            E3D(*right_field,k,j,i) = temperature;
            avg_field[k] += temperature; 
          }
          else if (strcmp(right_field_input,"N^2") == 0) {
            E3D(*right_field,k,j,i) = (planet->g)*density
                                     /(grid.theta[2*k+1]*H(j,i));
          }
          if (strcmp(left_field_input,"temperature")       == 0 ||
              strcmp(left_field_input,"delta temperature") == 0) {
            E3D(*left_field,k,j,i) = temperature;
          }
          else if (strcmp(left_field_input,"N^2") == 0) {
            E3D(*left_field,k,j,i) = (planet->g)*density
                                     /(grid.theta[2*k+1]*H(j,i));
          }
        }
        /* periodicity */
        E3D(*right_field,k,j,   0) = E3D(*right_field,k,j,ni);
        E3D(*right_field,k,j,ni+1) = E3D(*right_field,k,j, 1);

        E3D(*left_field, k,j,   0) = E3D(*left_field, k,j,ni);
        E3D(*left_field, k,j,ni+1) = E3D(*left_field, k,j, 1);
      }

      if (strcmp(left_field_input,"temperature") == 0) {
        color_min[k] = domain->tmin[k];
        color_max[k] = domain->tmax[k];
      }
      else if (strcmp(left_field_input,"N^2") == 0) {
        color_min[k] = 0.;
        color_max[k] = 1.e-3;
      }

      if (strcmp(right_field_input,"delta temperature") == 0 ||
          strcmp(left_field_input, "delta temperature") == 0) {
        avg_field[k] /= ni*(nj-JLO+1);
        if (strcmp(right_field_input,"delta temperature") == 0) {
          for (j = JLO; j <= nj; j++) {
            for (i = 1; i <= ni; i++) {
              E3D(*right_field,k,j,i) -= avg_field[k];
            }
            /* periodicity */
            E3D(*right_field,k,j,   0) = E3D(*right_field,k,j,ni);
            E3D(*right_field,k,j,ni+1) = E3D(*right_field,k,j, 1);
          }
        }
        if (strcmp(left_field_input,"delta temperature") == 0) {
          color_min[k] = -(*delta_color_scale);
          color_max[k] =  (*delta_color_scale);
          for (j = JLO; j <= nj; j++) {
            for (i = 1; i <= ni; i++) {
              E3D(*left_field,k,j,i) -= avg_field[k];
            }
            /* periodicity */
            E3D(*left_field,k,j,   0) = E3D(*left_field,k,j,ni);
            E3D(*left_field,k,j,ni+1) = E3D(*left_field,k,j, 1);
          }
        }
      }
    }

    if (fileout) {
      /*
       * Output delta temperature data to file:
       */
      sprintf(outfile_str,EPIC_PATH"/tmp/delta_temp%05-%02d:%02d:%02d.dat",
              var.time[0] / 86400 + var.time[1]*365, /* days    */
              var.time[0] /(60*60) % 24,             /* hours   */
              var.time[0] / 60     % 60,             /* minutes */
              var.time[0]          % 60);            /* seconds */
      outfile = fopen(outfile_str,"w");
      /* print pressure reference level (mbar) */
      fprintf(outfile,"   km   ");
      for (k = 1; k <= KLAST_ACTIVE; k++) {
        fprintf(outfile,"%12.5e ",grid.p_avg[k]/100.);
      }
      fprintf(outfile,"\n");
      /* print delta temperature data */
      if (strcmp(grid.f_plane_map,"cartesian") == 0) {
        j = nj/2;
        for (i = ni/2; i <= ni; i++) {
          radius = (double)(i-ni/2)/grid.m[2*j+1]/1.e+3;
          fprintf(outfile,"%7.1f ",radius);
          for (k = 1; k <= KLAST_ACTIVE; k++) {
            fprintf(outfile,"%12.5e ",E3D(*right_field,k,j,i));
          }
          fprintf(outfile,"\n");
        }
      }
      else if (strcmp(grid.f_plane_map,"polar") == 0) {
        i = 1;
        for (j = nj; j >= 0; j--) {
          radius = grid.f_plane_half_width*(1.-grid.lat[2*j+1]/90.)/1.e+3;
          fprintf(outfile,"%7.1f ",radius);
          for (k = 1; k <= KLAST_ACTIVE; k++) {
            fprintf(outfile,"%12.5e ",E3D(*right_field,k,j,i));
          }
          fprintf(outfile,"\n");
        }
      }
      fclose(outfile);
    }
  } 

  if (strcmp(right_field_input,"equil fpara") == 0 ||
      strcmp(left_field_input, "equil fpara") == 0) {
    for (k = 1; k <= KLAST_ACTIVE; k++) {
      for (j = JLO; j <= nj; j++) {
        for (i = 1; i <= ni; i++) {
          fpara       = get_chem(planet,FPARA_INDEX,2*k+1,j,i);
          temperature = T2(k,j,i);
          return_enthalpy(planet,fpara,P(k,j,i),temperature,&fgibb,&fpe,&uoup);
          if (strcmp(right_field_input,"equil fpara") == 0) {
            E3D(*right_field,k,j,i) = fpe;
          }
          if (strcmp(left_field_input, "equil fpara") == 0) {
            E3D(*left_field,k,j,i)  = fpe;
          }
        }
        /* periodicity */
        if (strcmp(right_field_input,"equil fpara") == 0) {
          E3D(*right_field,k,j,   0) = E3D(*right_field,k,j,ni);
          E3D(*right_field,k,j,ni+1) = E3D(*right_field,k,j, 1);
        }
        if (strcmp(left_field_input,"equil fpara")  == 0) {
          E3D(*left_field,k,j,   0) = E3D(*left_field,k,j,ni);
          E3D(*left_field,k,j,ni+1) = E3D(*left_field,k,j, 1);
        }
      }
    }
  }

  for (index = FPARA_INDEX; index <= LAST_HUMIDITY; index++) {
    if (strcmp(right_field_input,domain->chem_name[index]) == 0) {
      rt_index = index;
    }
    if (strcmp(left_field_input,domain->chem_name[index])  == 0) {
      lt_index = index;
    }
  }

  humidity = buff[0];
  buffji   = buff[1];

  if (rt_index >= FPARA_INDEX && rt_index <= LAST_HUMIDITY) {
    for (k = 1; k <= nk; k++) {
      if (humidity_type == HUMIDITY_TYPE_SPECIFIC) {
        for (j = JLO; j <= JHI; j++) {
          for (i = ILO; i <= IHI; i++) {
            HUMIDITY(j,i) = FIELD(var.field,var.index[rt_index],k,j,i);
          }
        }
      }
      else if (humidity_type == HUMIDITY_TYPE_RELATIVE) {
        relative_humidity(planet,rt_index,humidity,k);
      }
      else if (humidity_type == HUMIDITY_TYPE_COMBINED) {
        /* Determine cloud density. */
        for (j = JLO; j <= JHI; j++) {
          for (i = ILO; i <= IHI; i++) {
            /* 
             * Initialize to 1.e-30 instead of 0. to make 
             * plotting log easier. 
             */
            HUMIDITY(j,i) = 1.e-30;
          }
        }
        for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
          if (domain->chem_on[index]) {
            relative_humidity(planet,index,buffji,k);
            for (j = JLO; j <= JHI; j++) {
              for (i = ILO; i <= IHI; i++) {
                if (BUFFJI(j,i) > .99) {
                  /* In a cloud: */
                  HUMIDITY(j,i) += FIELD(var.field,var.index[index],k,j,i)*
                                   RHO(k,j,i);
                }
              }
            }
          }
        }
      }
      else {
        fprintf(stderr,"Error: epic_view: unknown humidity_type %d \n",humidity_type);
        exit(1);
      }
      /* Assign humidity data to right field: */
      for (j = JLO; j <= nj; j++) {
        for (i = 1; i <= ni; i++) {
          E3D(*right_field,k,j,i) = HUMIDITY(j,i);
        }
      }
      /* periodicity */
      for (j = JLO; j <= nj; j++) {
        E3D(*right_field,k,j,   0) = E3D(*right_field,k,j,ni);
        E3D(*right_field,k,j,ni+1) = E3D(*right_field,k,j, 1);
      }
      /* Record column data: */
      humid[k]         = HUMIDITY(JLO,ILO);
      cloud_density[k] = RHO(k,JLO,ILO)*FIELD(var.field,var.index[rt_index],k,JLO,ILO);
      if (humidity_type == HUMIDITY_TYPE_RELATIVE) {
        if (humid[k] < .99) {
          /* Set to 1.e-30 instead of 0. to make plotting log easier. */
          cloud_density[k] = 1.e-30;
        }
      }
    }  
    /* Write humidity column data: */
    sprintf(outfile_str,EPIC_PATH"/tmp/rt_field_humidity.dat");
    print_humidity_column(planet,outfile_str,rt_index,humidity_str,
                          humid,cloud_density,JLO,ILO);
  }

  if (lt_index >= FPARA_INDEX && lt_index <= LAST_HUMIDITY) {
    for (k = 1; k <= nk; k++) {
      if (humidity_type == HUMIDITY_TYPE_SPECIFIC) {
        for (j = JLO; j <= JHI; j++) {
          for (i = ILO; i <= IHI; i++) {
            HUMIDITY(j,i) = FIELD(var.field,var.index[lt_index],k,j,i);
          }
        }
      }
      else if (humidity_type == HUMIDITY_TYPE_RELATIVE) {
        relative_humidity(planet,lt_index,humidity,k);
      }
      else if (humidity_type == HUMIDITY_TYPE_COMBINED) {
        /* Determine cloud density. */
        for (j = JLO; j <= JHI; j++) {
          for (i = ILO; i <= IHI; i++) {
            /* 
             * Initialize to 1.e-30 instead of 0. to make 
             * plotting log easier. 
             */
            HUMIDITY(j,i) = 1.e-30;
          }
        }
        for (index = FIRST_HUMIDITY; index <= LAST_HUMIDITY; index++) {
          if (domain->chem_on[index]) {
            relative_humidity(planet,index,buffji,k);
            for (j = JLO; j <= JHI; j++) {
              for (i = ILO; i <= IHI; i++) {
                if (BUFFJI(j,i) > .99) {
                  /* In a cloud: */
                  HUMIDITY(j,i) += FIELD(var.field,var.index[index],k,j,i)*
                                   RHO(k,j,i);
                }
              }
            }
          }
        }
      }
      else {
        fprintf(stderr,"Error: epic_view: unknown humidity_type %d \n",humidity_type);
        exit(1);
      }
      /* Assign humidity data to left field: */
      for (j = JLO; j <= nj; j++) {
        for (i = 1; i <= ni; i++) {
          E3D(*left_field,k,j,i) = HUMIDITY(j,i);
        }
      }
      /* periodicity */
      for (j = JLO; j <= nj; j++) {
        E3D(*left_field,k,j,   0) = E3D(*left_field,k,j,ni);
        E3D(*left_field,k,j,ni+1) = E3D(*left_field,k,j, 1);
      }
      /* Record column data: */
      humid[k]         = HUMIDITY(JLO,ILO);
      cloud_density[k] = RHO(k,JLO,ILO)*FIELD(var.field,var.index[lt_index],k,JLO,ILO);
      if (humidity_type == HUMIDITY_TYPE_RELATIVE) {
        if (humid[k] < .99) {
          /* Set to 1.e-30 instead of 0. to aid log plots. */
          cloud_density[k] = 1.e-30;
        }
      }
    }
    /* Write humidity column data: */
    sprintf(outfile_str,EPIC_PATH"/tmp/lt_field_humidity.dat");
    print_humidity_column(planet,outfile_str,lt_index,humidity_str,
                          humid,cloud_density,JLO,ILO);
  }

  /* Create output geometries */

  if (layers) {
    layerinfo->color_input  = left_field_input;
    layerinfo->color        = *left_field;
    layerinfo->color_min    = color_min;
    layerinfo->color_max    = color_max;
    layerinfo->colormap     = colormap;
    layerinfo->height_scale = layers_height_scale;
    layerinfo->downsize     = layers_downsize;
    layerinfo->light_north  = light_north;
    layerinfo->wind         = wind_l;
    layerinfo->wind_scale   = wind_scale;
    layerinfo->polar        = layers_polar;
    make_layers_list(planet,layers_list,layerinfo);
  }

  if (strcmp(grid.geometry,"globe") == 0) {
    if (sphere) {
      sphereinfo->color_input  = left_field_input;
      sphereinfo->color        = *left_field;
      sphereinfo->color_min    = color_min;
      sphereinfo->color_max    = color_max;
      sphereinfo->colormap     = colormap;
      sphereinfo->layer        = sphere_layer;
      sphereinfo->height_scale = sphere_height_scale;
      sphereinfo->downsize     = sphere_downsize;
      sphereinfo->light_north  = light_north;
      sphereinfo->wind         = wind_s;
      sphereinfo->wind_scale   = wind_scale;
      make_sphere_list(planet,sphere_list,sphereinfo);
    }
  }

  /* Avoid recalculating variables that have not changed. */
  state_change = FALSE;

  return 1;
} 
/* ------------------------  end of view_computation() -------------------------- */





