/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 *	AVS subroutine module to create a label geometry
 *	from a timespec input.
 *
 *	A. Fischer, 7 June 91
 *      T. Dowling, 11 Oct 94, 10 Dec 96
 */

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <stdio.h>
#include <string.h>
#include <avs_epic.h>

#define YEAR 31536000

/* Prototypes: */
int compute(planetspec    *planet,
            domainspec    *domain,
            timespec      *ptr_time, 
            GEOMedit_list *output,    /* Pointer to edit list */
            int            alignment, 
            int            hhmmss,
            int            season,
            int            white, 
            int            font, 
            float         *ref_x, 
            float         *ref_y, 
            float         *ref_z, 
            float         *point);

AVSinit_modules()
{
  int 
    timedisplay();  /* Description function */

  /* Invokes timedisplay module from description function: */
  AVSmodule_from_desc(timedisplay);
}


timedisplay()	/* This is the description function */
{
  int	
    compute(),    /* Computation function */
    out_port,
    font,white,hhmmss,season,
    ref_x,ref_y,ref_z,
    point,
    position;

  AVSset_module_name("Time display", MODULE_MAPPER);

  AVScreate_input_port("Planet","struct planetspec",OPTIONAL);
  AVScreate_input_port("Domain","struct domainspec",OPTIONAL);
  AVScreate_input_port("Time",  "struct timespec",  REQUIRED);

  out_port = AVScreate_output_port("Label", "geom");

  position = AVSadd_parameter("alignment",     "tristate",0,NULL,NULL);
  hhmmss   = AVSadd_parameter("display hhmmss","boolean", 0,   0,   1);
  season   = AVSadd_parameter("planet years",  "boolean", 0,   0,   1);
  white    = AVSadd_parameter("color",         "boolean", 1,   0,   1);
  font     = AVSadd_parameter("Font",          "integer", 3,   0,  21);
  ref_x    = AVSadd_float_parameter("x", -0.9, -1., 1.);
  ref_y    = AVSadd_float_parameter("y", -0.76, -1., 1.);
  ref_z    = AVSadd_float_parameter("z",  4., -1., 5.);
  point    = AVSadd_float_parameter("size", 0.1, 0., 1.);

  (void)AVSconnect_widget(ref_x, "slider");
  (void)AVSconnect_widget(ref_y, "slider");
  (void)AVSconnect_widget(ref_z, "slider");
  (void)AVSconnect_widget(point, "slider");
  (void)AVSconnect_widget(font, "islider");
	
  AVSautofree_output(out_port);
  AVSset_compute_proc(compute);
}

/* This is the computation function */
int compute(planetspec    *planet,
            domainspec    *domain,
            timespec      *ptr_time, 
            GEOMedit_list *output,    /* Pointer to edit list */
            int            alignment, 
            int            hhmmss,
            int            season,
            int            white, 
            int            font, 
            float         *ref_x, 
            float         *ref_y, 
            float         *ref_z, 
            float         *point)
{
  char            
    timestr[81],   
    seasonstr[81],
    planet_letter[16];
  int		
    label_flags = 0,
    font_number,title, 
    background,drop, 
    align, stroke;
  float		
    ref_pt0[3],
    ref_pt1[3],
    offset[3],
    height,
    color[3];
  double
    years,
    planet_years;
  GEOMobj
    *obj = GEOM_NULL;

  font_number =	font;
  title       = 1;               /* Title label */
  background  = 0;               /* No background */
  drop        = 0;               /* No dropshadow */
  stroke      = 0;               /* Not implemented */
  if (alignment == 0) align = GEOM_LABEL_LEFT;
  if (alignment == 1) align = GEOM_LABEL_CENTER;
  if (alignment == 2) align = GEOM_LABEL_RIGHT;

  ref_pt0[0] = *ref_x;
  ref_pt0[1] = *ref_y; 
  ref_pt0[2] = *ref_z;
  ref_pt1[0] = ref_pt0[0];
  ref_pt1[1] = ref_pt0[1]-(*point)*1.5;
  ref_pt1[2] = ref_pt0[2];
  offset[0]  = 0.0; 
  offset[1]  = 0.0; 
  offset[2]  = 0.0;
  height     = *point;
  color[0]   = (float)white;
  color[1]   = (float)white;
  color[2]   = (float)white;

  if (hhmmss) {
    sprintf(timestr, "%5dyr %3dd %2dh %2dm %2ds",
           ptr_time->years,                  /* years   */ 
  	   ptr_time->secs / 86400,           /* days    */
          (ptr_time->secs /(60*60))  % 24,   /* hours   */
          (ptr_time->secs / 60)      % 60,   /* minutes */
	   ptr_time->secs            % 60);  /* seconds */
  }
  else {
    years = (double)ptr_time->years+
            (double)ptr_time->secs/(double)YEAR;
    sprintf(timestr,"%9.3fyr",years);
  }

  if (season) {
    if (!planet && !domain) {
      sprintf(seasonstr,"(need planet & domain connections)");
    }
    else {
      strcpy(planet_letter,planet->name);
      planet_letter[0] = (char)toupper((int)planet_letter[0]);
      planet_letter[1] = '\0';
      planet_years = ((double)ptr_time->years+
                      (double)ptr_time->secs/(double)YEAR)/
                      planet->orbit_period*domain->hasten;
      if (domain->hasten != 1.) {
        sprintf(seasonstr,"%9.3f%syr  hasten = %1.0f",
                           planet_years,planet_letter,domain->hasten);
      }
      else {
        sprintf(seasonstr,"%9.3f%syr",
                           planet_years,planet_letter);
      }
    }
  }
  else {
    sprintf(seasonstr," ");
  }

  label_flags = GEOMcreate_label_flags(font_number,title,background,
                                       drop,align,stroke);
  obj = GEOMcreate_label(GEOM_NULL,label_flags);
  GEOMadd_label(obj,timestr,  ref_pt0,offset,height,color,label_flags);
  GEOMadd_label(obj,seasonstr,ref_pt1,offset,height,color,label_flags);

  *output = GEOMinit_edit_list(*output);
  GEOMedit_geometry(*output,"Label",obj);
  GEOMdestroy_obj(obj);

  return(1);
}
