/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                 *
 * Copyright (C) 1998 Timothy E. Dowling                           *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 * A copy of this License is in the file:                          *
 *   $EPIC_PATH/License.txt                                        *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            *
 *                                                                 *
 * You should have received a copy of the GNU General Public       *
 * License along with this program; if not, write to the Free      *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,         *
 * Boston, MA  02111-1307, USA.                                    *
 *                                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**************************   video_capture.c   ******************************
 *                                                                           *
 *   Albert Fischer                                                          *
 *   11 January 1993                                                         *
 *                                                                           *
 *   AVS coroutine module                                                    *
 *   Input:   AVS image   Output: To videotape (none in AVS)                 *
 *                                                                           *
 *   Captures AVS images of size 640 x 484 (generally from the geometry      *
 *   viewer module) and writes them to the frame buffer in Flolab's          *
 *   Video/Scan, then tells the MII to record it at the current location     *
 *   and advance to record again.                                            *
 *                                                                           *
 *   The first part of the program readies the Video/Scan and MII for        *
 *   frame capturing, and runs independently of AVS's flow executive.        *
 *   The second part waits for a frame to capture, then writes it, and       *
 *   runs under the control of the flow executive.  The flow executive       *
 *   then serves as the synchronizer between the geometry output module      *
 *   (the model) and the video capture module.                               *
 *                                                                           *
 *   Updated 10 March 1993: Includes a slider for duration of each edit,     *
 *   controlling the number of frames put down.  The number can be           *
 *   negative, which puts frames down in reverse order on the tape.          *
 *                                                                           *
 *   jh/af 16 April 1993: Slider max extended to 1000, radio buttons added   *
 *   to control recording.  If set to "No", it will not write to the frame   *
 *   buffer or record.  If set to "Once", it will write and record once,     *
 *   then return to the "No" setting.  If set to "Record", it captures all   *
 *   images to video.                                                        *
 *                                                                           *
 *   td 1 Dec 1997: Slider max reduced to 100.                               *
 *                                                                           *
 *****************************************************************************/
#include <stdio.h>

#include <avs/avs.h>
#include <avs/flow.h>
#include <avs/field.h>

#include "/fvl/vscan/include/std_defs.h"
#include "/fvl/vscan/include/vscan.h"

/*
 *  Definitions 
 */
#define          SYNCHRONOUS      1
#define          ASYNCHRONOUS     0

#define SLEEP  "Sleep"
#define CAP    "Capture once"
#define ONCE   "Record once"
#define ALWAYS "Record always"
#define SEP    "!"
/*
 *  Function declarations
 */
void video_capture();
void vscan_prime();
void vlan_prime();
void vlan_perform();
int  edit_status();
int  vlan_send();


/* -------------   video_capture()  AVS description function   ------------- */
void video_capture()
{
  int  status, duration;  /* Widget descriptors */ 
  char a[50];

  a[0] = '\0';

  /* Set module name and type */
  AVSset_module_name("Video capture", MODULE_RENDER);

  /* Create input port */
  AVScreate_input_port("Video image", "field 2D 4-vector byte", REQUIRED);

  /* Create status widget */
  status = AVSadd_parameter("Status:", "string", 
			    "INITIALIZING: Please wait...", NULL, NULL);
  AVSadd_parameter_prop(status, "width", "integer", 4);

  /* Create duration widget */
  duration = AVSadd_parameter("Duration (frames)", "integer", 1, -5, 100);
  AVSconnect_widget(duration, "islider");

  /* Create record widget */
  strcat(a, SLEEP);
  strcat(a, SEP);
  strcat(a, CAP);
  strcat(a, SEP);
  strcat(a, ONCE);
  strcat(a, SEP);
  strcat(a, ALWAYS);
  AVSadd_parameter("Record", "choice", SLEEP, a, SEP);
}

/* ----------------------------   main()   --------------------------------- */
main (int   argc, 
      char *argv[])
{ 
  vscan_struct_t      *vscan;       /*  V/SCAN structure             */
  AVSfield_char       *avs_image;   /*  AVS image input as field     */
  vscan_pixel_t       color;
  vscan_pixel_t       vs_image      /*  V/S image output             */
                       [VSCAN_NTSC_FB_HEIGHT]
                       [VSCAN_NTSC_FB_WIDTH];
  int                 i;            /*  horizontal counting integer  */
  int                 j;            /*  vertical counting integer    */
  int                 height;       /*  height for fb write          */
  int                 start_line;   /*  beginning line for fb write  */
  int                 x_min, x_max; /*  extent of image              */
  int                 y_min, y_max; /*  extent of image              */
  char                status[80];   /*  used to modify status widget */
                                    /*  Not used to read from status */
  int                 n;            /*  number of frames             */
  char                *record[15];  /*  record state                 */
  int                 record_flag;  /*  Flags record or not          */

  /* Initialize with AVS, running independently */
  AVScorout_init(argc, argv, video_capture);
  AVScorout_set_sync(ASYNCHRONOUS);

  /* Initialize Video/Scan and get the vscan structure */
  vscan = vscan_open("/dev/cam", VSCAN_NTSC);
  if (vscan == NULL) {
    AVSfatal("Can't open Video/Scan.");
    exit(0);
  }
  if (vscan->error != VSCAN_ERROR_OK) {
    AVSfatal("Error opening Video/Scan.");
    exit(0);
  }

  /* Freeze the frame buffer and clear it */
  AVSmodify_parameter("Status:", AVS_VALUE, "Clearing FB", NULL, NULL);
  vscan_prime(vscan);
  
  /* Open communications to MII on V-LAN, set up editing run */
  AVSmodify_parameter("Status:", AVS_VALUE, "Talking to VTR", NULL, NULL);
  vlan_prime(vscan);

  /* Now change to synchronous execution */
  AVScorout_set_sync(SYNCHRONOUS);

  while (1) {

    AVSmodify_parameter("Status:", AVS_VALUE, "Ready for input...", NULL,NULL);
    AVScorout_wait();
    AVScorout_input(&avs_image, status, &n, record);
    AVSmodify_parameter("Status:", AVS_VALUE, "Received Image", NULL, NULL);

    if ( 0 == strcmp(*record, SLEEP) ){
	continue;
    }else if ( 0 == strcmp(*record, ONCE) ){ 
	AVSmodify_parameter("Record", AVS_VALUE, SLEEP, NULL, NULL);
	strcpy(*record, SLEEP);
	record_flag = 1;
    }else if ( 0 == strcmp(*record, CAP) ){
	AVSmodify_parameter("Record", AVS_VALUE, SLEEP, NULL, NULL);
	strcpy(*record, SLEEP);
	record_flag = 0;
    } 
    if ( 0 == strcmp(*record, ALWAYS) ){
      record_flag = 1;
    }


    /* Put AVS image in V/S image, centered top/bottom, left/right if
     * the AVS image is smaller, or taking upper left if AVS is bigger */
    x_min = 0;
    x_max = VSCAN_NTSC_FB_WIDTH;
    y_min = 0;
    y_max = VSCAN_NTSC_FB_HEIGHT;
    if ( MAXX(avs_image) < VSCAN_NTSC_FB_WIDTH ) {
      x_min = (int)( (float)(VSCAN_NTSC_FB_WIDTH - MAXX(avs_image)) / 2. );
      x_max = x_min + MAXX(avs_image);
    }
    if ( MAXY(avs_image) < VSCAN_NTSC_FB_HEIGHT ) {
      y_min = (int)( (float)(VSCAN_NTSC_FB_HEIGHT - MAXY(avs_image)) / 2. );
      y_max = y_min + MAXY(avs_image);
    }
    for (i = 0; i < VSCAN_NTSC_FB_WIDTH; i++) {
      for (j = 0; j < VSCAN_NTSC_FB_HEIGHT; j++) {
	color.rgb.red   = 0;
	color.rgb.green = 0;
	color.rgb.blue  = 0;
	vs_image[j][i] = color;
      }
    }
    for (i = x_min; i < x_max; i++) {
      for (j = y_min; j < y_max; j++) {
	color.rgb.red   = *( I2DV(avs_image, i - x_min, j - y_min) + 1 );
	color.rgb.green = *( I2DV(avs_image, i - x_min, j - y_min) + 2 );
	color.rgb.blue  = *( I2DV(avs_image, i - x_min, j - y_min) + 3 );
	vs_image[j][i] = color;
      }
    }

    /* Write to the frame buffer */
    if ( n != 0 ){
	AVSmodify_parameter("Status:", AVS_VALUE, "Writing to FB", NULL, NULL);
	height = VSCAN_NTSC_FB_HEIGHT;
	start_line = 0;
	vscan_fb_write(vscan, vs_image, height, start_line);

	/* Perform the edit */
	if (record_flag) {
	  vlan_perform(vscan, n);
	}
    }
  }
}
/* ---------------------------   end of main()   --------------------------- */


/*============================   vscan_prime()   ===========================*/
/* Freezes and clears the frame buffer                                      */

void vscan_prime(vscan)
vscan_struct_t *vscan;
{

   char             control;      /* control character for V/SCAN     */
   char             mode;         /* mode character for V/SCAN        */
   vscan_pixel_t    color;        /* color pixel                      */
   boolean          error;        /* return from V/SCAN calls         */

              /* 
	       *   Reset Video/Scan 
	       */
   (void)vscan_reset (vscan);
              /* 
	       *   Freeze frame buffer and decompress it 
	       */
   mode = VSCAN_NO_COMPRESS |
          VSCAN_OUTPUT_BANK_1;
   error = vscan_set_mode (vscan, mode);
   if (error == TRUE) {
     AVSfatal("vscan_set_mode failed in vscan_prime()");
     exit(0);
   } 
   control = VSCAN_FB_FREEZE;
   error = vscan_set_control (vscan, control);
   if (error == TRUE) {
     AVSfatal("vsan_set_control failed in vscan_prime()");
     exit(0);
   }
              /* 
	       *   Clear frame buffer 
	       */
   color.rgb.red   = 20;
   color.rgb.green = 0;
   color.rgb.blue  = 50;
   error = vscan_fb_clear (vscan, &color, 1024, 0);
   if (error == TRUE) {
     AVSfatal("Frame buffer clear failed in vscan_prime()");
     exit(0);
   }

}



/*=========================   vlan_prime()   ==============================*/
/* Opens communication with V-LAN, and prepares for an editing run         */

void vlan_prime(vscan)
vscan_struct_t *vscan;
{
  char        buf[80];

      /*   
       * Tell V-LAN receiver to execute downloaded code 
       This is an undocumented kluge, but it is the only
       way to get the vlan going that I've found  -AF */
   vlan_send (vscan, "     GO\r");
  
   /* Set code mode select to time code */
   vlan_send (vscan, "CS1\r");

   /* Clear all edit registers */
   vlan_send (vscan, "CL\r");
   
   /* Set to edit all tracks */
   vlan_send (vscan, "TSV\r");
   
   /* Set initial inpoint to current point */
   vlan_send (vscan, "EI\r");
   
   /* Move tape to preroll point after each edit */
   vlan_send (vscan, "EC GP\r");
}


/*==========================   vlan_perform()   ===========================*/
/* Peform edit and return when deck is done */

void vlan_perform (vscan, n)
vscan_struct_t *vscan;
int            n;                 /* Number of frames to put down     */
{

   int              edit_retval;  /* return from V-LAN calls          */
   char             buf[80];      /* Buffer to send on V-LAN          */

   /* Set duration and increment to n frames */
   sprintf (buf, "SD%d\r", abs(n));
   vlan_send(vscan, buf);
   sprintf (buf, "AI%d\r", n);
   vlan_send(vscan, buf);

   /* Peform edit */
   vlan_send (vscan, "PF\r");
   
   /* Check status and wait until complete */
   do {
     edit_retval = edit_status(vscan);
     if (edit_retval == -1) {
       AVSfatal("Error: Edit aborted.");
       exit(0);
     }
   } while (edit_retval != 0);

}   



/*===========================   edit_status()   ============================*/
/* Checks edit status, and exits if the VTR is not set to remote control    */
/* Returns 0 if editing is done, -1 if aborted, and 1 otherwise             */

int edit_status(vscan)
vscan_struct_t *vscan;
{

   int             vlan_return,
                   i;
   unsigned char   data;
   char            buf[255];
   int             length;
   int             retval;
   char            status[80];

   /* Request Edit Status over V-LAN*/
   sprintf(buf, "ES\r");
   length = strlen(buf);
   vlan_return = vscan_vlan_write (vscan, buf, length);
   if (vlan_return < 0) {
     AVSfatal("Error requesting status in edit_status()");
     exit(0);
   }

   /* Read return */
   i = 0;
   do {
     vlan_return = vscan_vlan_read(vscan, &data, 1);
     if (data >= ' ' && data <= '~')
        status[i++] = data;
     if (data == '\r')
       status[i++] = '\0';
     if (vlan_return < 0) {
       AVSfatal("Error reading status in edit_status()");
       exit(0);
     }
   } while (data != '\r');

   AVSmodify_parameter("Status:", AVS_VALUE, status, NULL, NULL);
   
   /* Set return value based on return of edit status request */
   retval = 1;
   if (0 == strncmp(status, "DONE", 4))
     retval = 0;
   if (0 == strncmp(status, "ABORTED", 7))
     retval = -1;
   if (0 == strncmp(status, "LOCAL", 5)) {
     AVSfatal("ERROR: VTR not set to remote control");
     exit(0);
   }

   return(retval);

}



/*===========================   vlan_send()   ==============================*/
/*  Sends V-LAN codes to the tape deck and checks response for errors       */

int vlan_send(vscan, buf)
vscan_struct_t *vscan;
char           buf[256];
{

  int		length;         /* length of character buffer             */
  int		vlan_return;    /* Error flag                             */
  unsigned char	data;           /* char return from vlan_read             */
  
  /* Send code to VTR over VLAN */
  length = strlen(buf);
  vlan_return = vscan_vlan_write(vscan, buf, length);
  if (vlan_return <=0) {
    AVSfatal("Error writing to V-LAN in vlan_send().  Code: %s", buf);
    exit(0);
  }
  
  /* Read return and exit if there is an error */
  do {
    vlan_return = vscan_vlan_read(vscan, &data, 1);
    if (vlan_return < 0) {
      AVSfatal("Error reading V-LAN return in vlan_send()");
      exit(0);
    }
  } while (data != '\r');
  
  return vlan_return;

}





    









