%
% Script performs a diagnostic on EPIC data at specified time increments and
%  displays the diagnostics verses time on a figure.  Multiple curves and/or diagnostics
%  can be displayed on the same graph.  Each curve is defined in structures with the format:
%
% Primary Fields:
%
%      curve(:).name              = name of curve
%      curve(:).nc_directory      = directory containing extract file(s)
%      curve(:).process_type      = 'MAX_CHANNEL_VECTOR'     , 'MAX_CHANNEL_SCALAR'            , 
%                                   'MIN_CHANNEL_SCALAR'     , 'ABS_MAX_CHANNEL_SCALAR'        ,
%                                   'GLOBAL_MEAN_VS_PRESSURE', 'EQ_POLE_DIFFERENCE_VS_PRESSURE', 
%                                or 'TIME_AVERAGE_EPIC_DATA'
%      curve(:).channel           = [min_val max_val]
%      curve(:).variables(1).name = 'u';
%      curve(:).variables(2).name = 'v';
%      curve(:).extract_it        = 'first', 'last', 'all', or <array>
%      curve(:).extract_ik        = 'top', 'bottom', 'all', or <array>
%      curve(:).convert_Pa_to_mbar = TRUE;
%
% Optional Fields:
%
%      curve_options(:).x_data_scale  = 1.0;
%      curve_options(:).y_data_scale  = 1.0;
%      curve_options(:).line_type     = '-k'
%      curve_options(:).line_size     = 1
%      curve_options(:).marker_size   = 8
%      curve_options(:).time_units    = 'decades', 'years', 'days', 'hours', 'minutes', or 'seconds';
%      curve_options(:).time_adjust   = <same units as "time_units">
%
%
% Other input options are:
%     fig_width     = [420];
%     AxisFontSize  = 15;
%   
%     Title         = '0^o < latitude < 90^o'
%     DisplayLegend = TRUE or FALSE
%     DisplayBox    = TRUE or FALSE
%     DisplayGrid   = TRUE or FALSE
%     XLabel        = 'year'
%     YLabel        = 'Maximum Velocity  (m / s)'
%   
%     XLimits       = [ xmin xmax ]
%     YLimits       = [ ymin ymax ]
%     
%     xtick         = <array>
%     xticklabel    = <array>
%     ytick         = <array>
%     yticklabel    = <array>
%     
%     ReverseXaxis  = TRUE or FALSE
%     ReverseYaxis  = TRUE or FALSE
%     
%     SquareAxis    = TRUE or FALSE
%     
%     postPlot      = 'grid on';
%     postPlot      = 'reverseFigAndAxesColors';
%     
%     bmp_filename  = 'PlotEpicDiagnostic';
%     output_file_format = 'bmp' or 'eps;
%     
%     
% This script is most effective when used in "external" mode.  The input parameters listed above
%  should be place in a sepparate script along with the lines:
%
%     .   
%     .   
%     .   
%
%     %
%     % Process Data and Generate Graph
%     %
%     load_input_externally       = true;
%     load_comparisons_externally = false;   % (set to true to use saved data)
%     produce_plot_externally     = false;   % (set to true to turn off plot option)
%     
%     PlotEpicDiagnostic
%
%

EpicNC_initialize


%%
%% ---------------------------------- INPUT ---------------------------------- 
%%

if SetInput==TRUE
  fig_width    = [420];
  AxisFontSize  = 15;

  Title         = '0^o < latitude < 90^o';
  DisplayLegend = TRUE;
  DisplayBox    = TRUE;
  %XLabel        = 'year';
  %YLabel        = 'Maximum Velocity  (m / s)';

  %XLimits       = [ 100 900 ];
  %YLimits       = 
  %
  %LogXData      = TRUE or FALSE
  LogYData      = TRUE;
  %
  %xtick         = 
  %xticklabel    = 
  %ytick         = 
  %yticklabel    = 
  %
  %ReverseXaxis  = TRUE or FALSE
  ReverseYaxis  = TRUE;
  %
  %SquareAxis    = TRUE or FALSE
  %
  %postPlot      = 'grid on';
  %postPlot      = 'reverseFigAndAxesColors';
  %
  %bmp_filename  = 'PlotEpicDiagnostic';
  %


  checkEPICncPATHs
  nc=1;

  curve(nc).name                   = 'Topo';
  curve(nc).nc_directory           = [ EPIC_DATA_PATH slash 'Venus' slash 'experiment09' ];
  curve(nc).nc_file_name           = 'extract.nc';
  curve(nc).process_type           = 'GLOBAL_MEAN_VS_PRESSURE';
  curve(nc).channel                = [0.0  90.0];
  curve(nc).variables(1).name      = 't2';
  curve(nc).calc_variables(1).name = 'p2';
  curve(nc).extract_it             = 3;
  curve(nc).extract_ik             = 'all';
  curve_options(nc).line_type      = '-k';
  curve_options(nc).line_size      = 1;
  curve_options(nc).marker_size    = 8;
  curve_options(nc).time_units     = 'years';
  curve_options(nc).time_adjust    = 2.99801;  %( same units as "time_units" )
  nc=nc+1;


  %curve(nc).name                   = 'Flat';
  %curve(nc).nc_directory           = [ EPIC_DATA_PATH slash 'Venus_Flat' slash 'experiment07' ];
  %curve(nc).nc_file_name           = 'extract.nc';
  %curve(nc).process_type           = 'GLOBAL_MEAN_VS_PRESSURE';
  %curve(nc).channel                = [0.0  90.0];
  %curve(nc).variables(1).name      = 't3';
  %curve(nc).calc_variables(1).name = 'p3';
  %curve(nc).extract_it             = 3;
  %curve(nc).extract_ik             = 'all';
  %curve_options(nc).line_type      = '--k';
  %curve_options(nc).line_size      = 1;
  %curve_options(nc).marker_size    = 8;
  %nc=nc+1;

end

%%
%% ---------------------------------- DATA PROCESSING ---------------------------------- 
%%
if SetOutput==FALSE
  if length(who('save_processed_data_filepath'))==0
    error([' Missing key variable "save_processed_data_filepath".'])
  end
  % Before loading the mat file, save the current work space in case
  %  it contains modifications to the variables in save_processed_data_filepath.
  current_work_space = 'tmp_save_file';
  save_work_space = ['save ' current_work_space];
  work_space = who;
  for iws=1:length(work_space)
    this_var = work_space(iws);
    this_var_name = this_var{1};
    if not( strcmp(this_var_name, 'curve') )
      save_work_space = [ save_work_space ' ' this_var_name ];
    end
  end
  eval(save_work_space)
  eval(['load ' save_processed_data_filepath])
  eval(['load ' current_work_space])
  eval(['delete ' current_work_space '.mat'])

else
  if length(who('curve'))==0
    error(' Input structure "curve" does not exist.')
  elseif length(curve)==0
    error(' Input structure "curve" has zero length.')
  end
  if length(who('curve_options'))==0
    curve_options(1:length(curve)) = -1;
  end
  
  %
  % Process each curve
  %
  for ic=1:length(curve)
    time_dependent_curve=TRUE;
    if strcmp( curve(ic).process_type, 'GLOBAL_MEAN_VS_PRESSURE' )         ...
     | strcmp( curve(ic).process_type, 'EQ_POLE_DIFFERENCE_VS_PRESSURE' )  ...
      time_dependent_curve=FALSE;
    end

    fprintf(['\n Processing ' curve(ic).name ' : '])

  
    %
    % Get Extract file paths
    %
    clear nc_file_directory tmp_file_name nc_file_name nc_filepath
    if isfield( curve(ic), 'nc_directory' )
      if length( curve(ic).nc_directory ) > 0
        nc_file_directory       = curve(ic).nc_directory;
      end
    end
    if length(who('nc_file_directory'))==0
      nc_file_directory = [ EPIC_DATA_PATH slash curve(ic).planet slash curve(ic).experiment ];
      curve(ic).nc_directory = nc_file_directory;
    end
    if isfield( curve(ic), 'nc_file_name' )
      if length( curve(ic).nc_file_name ) > 0
        nc_filepath.name = [ nc_file_directory slash curve(ic).nc_file_name ];
      end
    else
      tmp_file_name           = getFileNames( 'extract', '.nc', nc_file_directory);
      if strcmp( tmp_file_name(1).name, 'extract.nc' )
        nc_file_name          = [ tmp_file_name(2:length(tmp_file_name))  tmp_file_name(1) ];
      else
        nc_file_name          = tmp_file_name;
      end
      for inc=1:length(nc_file_name)
      nc_filepath(inc).name = [ nc_file_directory slash nc_file_name(inc).name ];
      end
    end
  
    %
    % Get remaining info necessary to extract data
    %
    clear extract_it extract_ik var_names
    for ivn=1:length( curve(ic).variables )
      var_names(ivn).name = curve(ic).variables(ivn).name;
    end
    extract_var_names = var_names;
    if isfield( curve(ic), 'calc_variables' )
      for ivn=1:1:length( curve(ic).calc_variables )
        extract_var_names(length(var_names)+ivn).name = curve(ic).calc_variables(ivn).name; 
      end
    end
    extract_it = curve(ic).extract_it;
    extract_ik = curve(ic).extract_ik;
  
    if strcmp(extract_it,'first')
      clear extract_it
      extract_it = 1;
    elseif strcmp(extract_it,'last')
      clear extract_it
      extract_it = getNumberSnapShots( nc_filepath );
    elseif strcmp(extract_it,'all')  
      clear extract_it
      extract_it = 1:getNumberSnapShots( nc_filepath );
    elseif isnumeric(extract_it)
      extract_it = sort(extract_it);
    else
      error('  Invalid input:  extract_it')
    end

  
    %
    % Set Time Units and Adjustment
    %
    clear time_units time_adjust
    time_units = 'years';
    if isfield(curve_options(ic),'time_units')
      if length( curve_options(ic).time_units ) > 0
        time_units = curve_options(ic).time_units;
      end
    end
    time_adjust = 0.0;
    if isfield(curve_options(ic),'time_adjust')
      if length( curve_options(ic).time_adjust ) > 0
        time_adjust = curve_options(ic).time_adjust;
      end
    end
  
    if strcmp(time_units,'decades')
      clear time_units;                time_units = 1.0/3.1556925e8;
    elseif strcmp(time_units,'years')
      clear time_units;                time_units = 1.0/3.1556925e7;
    elseif strcmp(time_units,'days')
      clear time_units;                time_units = 1.0/8.64e4;
    elseif strcmp(time_units,'hours')
      clear time_units;                time_units = 1.0/3600;
    elseif strcmp(time_units,'minutes')
      clear time_units;                time_units = 1.0/60;
    elseif strcmp(time_units,'seconds')
      clear time_units;                time_units = 1.0;
    else
      error([' time_units = ' time_units ' is not supported ' ...
             ' (valid types are: decades, years, days, hours, minutes, or seconds)']);
    end
  
  
    %
    % Extract and Process each data point individually to avoid
    %  memory overload.
    %
    for it=1:length(extract_it)
      if time_dependent_curve==TRUE
        if (it>1)
          for ib=1:13
            fprintf('\b');
          end
        end
        fprintf('%3d%% complete', round((it-1)/length(extract_it)*100) );
      end
  
      %
      % Extract Data
      %
      clear epic_vars 
      if time_dependent_curve==TRUE
        [epic_vars, dimensions, nc_file] = ExtractEpicNCdata( nc_filepath, extract_var_names, extract_it(it), extract_ik );
      else
        [epic_vars, dimensions, nc_file] = ExtractEpicNCdata( nc_filepath, extract_var_names, extract_it    , extract_ik );
      end

      % CONVERT PASCALS TO MBAR 
      if isfield( curve(ic), 'convert_Pa_to_mbar' )
        if curve(ic).convert_Pa_to_mbar
          for iv=1:length(epic_vars)
            if strcmp( epic_vars(iv).units, 'Pa' )
              epic_vars(iv).data = epic_vars(iv).data .* 0.01;
              epic_vars(iv).units = 'mb';
            end
          end
        end
      end
    
      %
      % PROCESS DATA
      %
      if length(epic_vars)==0
        it=length(extract_it)+1;
      else
        % ---------------------------------------------------------------------------------

        if strcmp( curve(ic).process_type, 'MAX_CHANNEL_VECTOR' )
           if not( isfield(curve(ic), 'channel') )
             error(['  structure "curve" is missing the field "channel".'])
           elseif length(curve(ic).channel)==0
             error(['  structure "curve" is missing the field "channel".'])
           else
             clear channel
             channel = curve(ic).channel;
           end

           [max_vel min_vel]    = getChannelMaxMin( nc_file(1).pntr, epic_vars, var_names, channel, 'VECTOR2D' );
    
           curve(ic).x_data(it) = epic_vars(1).time * time_units - time_adjust;
           curve(ic).y_data(it) = max_vel;

        % ---------------------------------------------------------------------------------

        elseif strcmp( curve(ic).process_type, 'MAX_CHANNEL_SCALAR' )
           if not( isfield(curve(ic), 'channel') )
             error(['  structure "curve" is missing the field "channel".'])
           elseif length(curve(ic).channel)==0
             error(['  structure "curve" is missing the field "channel".'])
           else
             clear channel
             channel = curve(ic).channel;
           end
           if length(var_names)>1
             error(['  structure "var_names" must have only 1 entry for MAX_CHANNEL_SCALAR.'])
           end

           [max_vel min_vel]    = getChannelMaxMin( nc_file(1).pntr, epic_vars, var_names, channel, 'SCALAR' );
    
           curve(ic).x_data(it) = epic_vars(1).time * time_units - time_adjust;
           curve(ic).y_data(it) = max_vel;

        % ---------------------------------------------------------------------------------

        elseif strcmp( curve(ic).process_type, 'MIN_CHANNEL_SCALAR' )
           if not( isfield(curve(ic), 'channel') )
             error(['  structure "curve" is missing the field "channel".'])
           elseif length(curve(ic).channel)==0
             error(['  structure "curve" is missing the field "channel".'])
           else
             clear channel
             channel = curve(ic).channel;
           end
           if length(var_names)>1
             error(['  structure "var_names" must have only 1 entry for MIN_CHANNEL_SCALAR.'])
           end

           [max_vel min_vel]    = getChannelMaxMin( nc_file(1).pntr, epic_vars, var_names, channel, 'SCALAR' );
    
           curve(ic).x_data(it) = epic_vars(1).time * time_units - time_adjust;
           curve(ic).y_data(it) = min_vel;

        % ---------------------------------------------------------------------------------

        elseif strcmp( curve(ic).process_type, 'ABS_MAX_CHANNEL_SCALAR' )
           if not( isfield(curve(ic), 'channel') )
             error(['  structure "curve" is missing the field "channel".'])
           elseif length(curve(ic).channel)==0
             error(['  structure "curve" is missing the field "channel".'])
           else
             clear channel
             channel = curve(ic).channel;
           end
           if length(var_names)>1
             error(['  structure "var_names" must have only 1 entry for ABS_MAX_CHANNEL_SCALAR.'])
           end

           [max_vel min_vel]    = getChannelMaxMin( nc_file(1).pntr, epic_vars, var_names, channel, 'SCALAR' );
    
           curve(ic).x_data(it) = epic_vars(1).time * time_units - time_adjust;
           curve(ic).y_data(it) = max(abs([max_vel min_vel]));

        % ---------------------------------------------------------------------------------

        elseif strcmp( curve(ic).process_type, 'GLOBAL_MEAN_VS_PRESSURE' )
           if length(var_names)>1
             error(['  structure "var_names" must have only 1 entry for GLOBAL_MEAN_VS_PRESSURE.'])
           end

           global_means = getGlobalMeanVsPressure( nc_file(1).pntr, epic_vars, var_names );
    
           curve(ic).x_data = global_means.y_data;
           if isfield(global_means, 'x_data_2')
             curve(ic).y_data = global_means.x_data_2;
           else
             curve(ic).y_data = global_means.x_data_3;
           end

        % ---------------------------------------------------------------------------------

        elseif strcmp( curve(ic).process_type, 'EQ_POLE_DIFFERENCE_VS_PRESSURE' )
           if length(var_names)>1
             error(['  structure "var_names" must have only 1 entry for EQ_POLE_DIFFERENCE_VS_PRESSURE.'])
           end

           differentials = getEqPoleDiffVsPressure( nc_file(1).pntr, epic_vars, var_names );
    
           curve(ic).x_data = differentials.y_data;
           if isfield(differentials, 'x_data_2')
             curve(ic).y_data = differentials.x_data_2;
           else
             curve(ic).y_data = differentials.x_data_3;
           end

        % ---------------------------------------------------------------------------------

        elseif strcmp( curve(ic).process_type, 'TIME_AVERAGE_EPIC_DATA' )
           %for iv=1:length(var_names)
           %  epic_vars(iv).data = epic_vars(iv).data ./ length(extract_it);
           %end

           if it==1
             ave_epic_vars = epic_vars;
           else
             for iv=1:length(var_names)
               ave_epic_vars(iv).data = ave_epic_vars(iv).data + epic_vars(iv).data;
             end
           end

           if it==length(extract_it)
             for iv=1:length(var_names)
               ave_epic_vars(iv).data = ave_epic_vars(iv).data ./ length(extract_it);
             end
             clear epic_vars
             epic_vars = ave_epic_vars;
           end

           ProducePlot = FALSE;

        % ---------------------------------------------------------------------------------

        else
          error([' process_type = "' curve(ic).processs_type '" is not supported.'])
        end
      end
  
      %
      % Close nc files
      %
      for inc=1:length(nc_file)
        close( nc_file(inc).pntr );
      end

      if time_dependent_curve==FALSE
        it=length(extract_it)+1;
      end
    end % loop over extract_it
    if time_dependent_curve==TRUE
      for ib=1:13
        fprintf('\b');
      end
      fprintf('done\n');
    end
  
  end  % loop over "curve"
  
  save_data_dir = [ curve(1).nc_directory slash 'MatFiles' ];
  if isdir(save_data_dir)==FALSE
     mkdir(save_data_dir);
  end

  if length(who('save_processed_data_filename'))==0
    save_processed_data_filepath = [save_data_dir slash 'PlotEpicDiagnostic'];
  else
    save_processed_data_filepath = [save_data_dir slash save_processed_data_filename];
  end
  fprintf(['\n Saving processed data to file ' save_processed_data_filepath '.mat .....\n'])
  clear nc_file
  eval([' save ' save_processed_data_filepath ])
  
end % end SetOutput==TRUE


%%
%% ---------------------------------- PLOT DATA ---------------------------------- 
%%
if ProducePlot==TRUE
  fprintf('\n Displaying Figure .....\n')

  %
  % Display Figure
  %
  figure
  if length(who('fig_pos')) == FALSE
     fig_pos   = [ 50, 375 ];
  end
  if length(who('fig_width')) == FALSE
     fig_width = 420;
  end
  fig_pos(3:4) = [ fig_width(1) fig_width(length(fig_width)) ];
  set(gcf,'Position', fig_pos);

  fcolor = 'w';
  if length(who('fig_color'))
    fcolor = fig_color;
  end
  set(gcf,'Color',fcolor);

  hold on

  %
  % Display Data on Figure
  %
  for ic=1:length(curve)
    clear x_data y_data line_type
    x_data = curve(ic).x_data;
    y_data = curve(ic).y_data;

    if isfield( curve_options(ic), 'x_data_scale' )
      if length( curve_options(ic).x_data_scale ) > 0
        x_data = x_data .* curve_options(ic).x_data_scale;
      end
    end
  
    if isfield( curve_options(ic), 'y_data_scale' )
      if length( curve_options(ic).y_data_scale ) > 0
        y_data = y_data .* curve_options(ic).y_data_scale;
      end
    end
  
    line_type = '-';
    if isfield( curve_options(ic), 'line_type' )
      if length(curve_options(ic).line_type) > 0
        line_type = curve_options(ic).line_type;
      end
    end
  
    curve(ic).line_id = plot( x_data, y_data, line_type );
  end 

  %
  % Post Plotting Functions 
  %
  if length(who('AxisFontSize'))
    set(gca,'FontSize', AxisFontSize * min(fig_width)/420 );
  end
   
  if length(who('Title'))
    title( Title );
  end
   
  if length(who('XLabel'))
    xlabel( XLabel );
  end
   
  if length(who('YLabel'))
    ylabel( YLabel );
  end
  
  if length(who('XLimits'))
    cur_axis = axis;
    cur_axis(1) = XLimits(1);
    cur_axis(2) = XLimits(2);
    axis( cur_axis );
  end
  
  if length(who('YLimits'))
    cur_axis = axis;
    cur_axis(3) = YLimits(1);
    cur_axis(4) = YLimits(2);
    axis( cur_axis );
  end
   
  if length(who('LogXData'))
    if LogXData
      set(gca, 'XScale', 'log');
    end
  end

  if length(who('LogYData'))
    if LogYData
      set(gca, 'YScale', 'log');
    end
  end

  revX=FALSE;
  if length(who('ReverseXaxis'))
    revX = ReverseXaxis;
  end
  revY=FALSE;
  if length(who('ReverseYaxis'))
    revY = ReverseYaxis;
  end
  if revX & not(revY)
     view(180,-90);
  elseif not(revX) & revY
     view(0,-90);
  elseif revX & revY
     view(180,90);
  end
  
  if length(who('xtick'))
    set(gca,'XTick',xtick);
  end
  if length(who('ytick'))
    set(gca,'YTick',ytick);
  end
  if length(who('xticklabel'))
    set(gca,'XTickLabel',xticklabel);
  end
  if length(who('yticklabel'))
    set(gca,'YTickLabel',yticklabel);
  end
  
  if length(who('DisplayBox'))
    if DisplayBox
      box on
    else
      box off
    end
  end
  
  if length(who('DisplayGrid'))
    if DisplayGrid
      grid on
    else
      grid off
    end
  end
  
  if length(who('SquareAxes'))
     if SquareAxes
        fpos = get(gcf,'Position');
        apos = get(gca,'Position');
        if fpos(3) > fpos(4)
            apos(3) = apos(4)*fpos(4) / fpos(3);
        else
            apos(4) = apos(3)*fpos(3) / fpos(4);
        end
        xbuff = 0.035 * fpos(3)/560;
        apos(1) = 0.5*(1.0 - apos(3)) + xbuff;
        set(gca,'Position',apos);
     end
  end
  
  if length(who('DisplayLegend'))
    if DisplayLegend
       if length(who('LegendLocation'))
          leg_loc = LegendLocation;
       else
          leg_loc = 'Best';
       end
       Legend = 'leg_handle = legend(gca, ';
       for ic=1:length(curve)
          Legend = [ Legend '''' curve(ic).name ''',' ]; 
       end
       Legend = [ Legend  ' ''Location'', ''' leg_loc ''' ); ' ];
       eval( Legend );
       legend boxoff
    end
  end
  
  for ic=1:length(curve)
    if isfield( curve_options(ic), 'line_size' )
     if length( curve_options(ic).line_size ) > 0
       set( curve(ic).line_id, 'LineWidth', curve_options(ic).line_size * min(fig_width)/420 );
     end
    end

    if isfield( curve_options(ic), 'marker_size' )
     if length( curve_options(ic).marker_size ) > 0
       set( curve(ic).line_id, 'MarkerSize', curve_options(ic).marker_size * min(fig_width)/420 );
     end
    end
  end
  
  if length(who('postPlot'))
    eval( postPlot );
  end


  %
  % Write figure to file
  %
  if length( who('bmp_filename') ) == 1
     write_data_dir = [ curve(1).nc_directory slash 'Figures' ];
     if isdir(write_data_dir)==FALSE
        mkdir(write_data_dir);
     end
     output_filepath = [write_data_dir slash bmp_filename];
     %fprintf(['writing file ' output_filepath '.bmp .....\n'])
     %writeFig2Bitmap( output_filepath );

     file_format='bmp';
     if length(who('output_file_format'))
       file_format = output_file_format;
     end
     if strcmp( file_format, 'bmp' )
       unix(['rm -f ' output_filepath '.bmp ']);
       fprintf(['\tWriting bmp file ' output_filepath '.bmp\n'])
       writeFig2Bitmap( output_filepath );

     elseif strcmp( file_format, 'eps' )
       unix(['rm -f ' output_file_name '.eps ']);
       fprintf(['\tWriting eps file ' output_filepath '.eps\n'])
       writeFig2Eps( output_filepath );

     else
       fprintf([' output_file_format="' file_format '" is not supported.'])
     end
  end

end % if ProducePlot==TRUE

fprintf('\ndone\n')

