
function [epic_vars, dimensions, nc_file] = ExtractEpicNCdata( nc_filepath, extract_var, extract_it, extract_ik )
%
% function [epic_vars, dimensions, nc_file] = ExtractEpicNCdata( nc_filepath, extract_var, extract_it, extract_ik )
%
%
% INPUT:
%
%       nc_filepath : String or struct of the form nc_filepath(:).name specifying the .nc filepath(s).
%
%       extract_var : Structure containing the names of variables to be extracted from the nc file.
%                     Format is extract_var(:).name = "variable names" .
%                     If a variable name does not exist, it is skipped.
%                     A dummy variable containing zeros can be set using the variable names "empty_u",
%                     "empty_v", etc.
%
%       extract_it  : Array containing the time steps to extract.
%                     Omitting this argument will extract all time steps.
%                     May also pass in the strings 'first', 'last', or 'all'.
%
%       extract_ik  : Array containing the depth layers to extract.
%                     Omitting this argument will extract all depth layers.
%                     May also pass in the strings 'top', 'bottom', or 'all'.
%
%
% OUTPUT:
%
%       epic_vars   : Variables extracted from the nc file.
%                     Format is epic_vars(:).name
%                               epic_vars(:).units
%                               epic_vars(:).dim(:).name
%                               epic_vars(:).data(it,ik,ilat,ilon)
%                               epic_vars(:).tlo,  epic_vars(:).thi
%                               epic_vars(:).klo,  epic_vars(:).khi
%                               epic_vars(:).jlo,  epic_vars(:).jhi
%                               epic_vars(:).ilo,  epic_vars(:).ihi
%
%       dimensions  : All possible variable dimensions in the nc file.
%                     Format is dimensions(:).name
%                               dimensions(:).data
%
%       nc_file     : Tag of the opened netcdf file.
%
%
%
% NOTE: The t and k indices are for indexing RETRIEVED data.  They are the same as the models 
%       t and k indices only when all times and all depths are retrieved, respectively.  The 
%       dimensions output is NOT altered to give only the indices in extract_it and extract_ik
%

TRUE=1; FALSE=0;

if nargin<4
  extract_ik = -1;
end

if nargin<3
  extract_it = -1;
end

if nargin<2
  error('Input missing (variable names)')
end

if nargin<1
  error('Input missing (1. nc filepath,  2. variable names)')
end

if isfield(extract_var,'name') == false
  error('Variable names must be given in a structure of the form extract_var(:).name')
end



%
% Create the list of nc filepaths
%
if ischar(nc_filepath)
   nc_filepath_list.name = nc_filepath;
else
   if isfield(nc_filepath,'name')==FALSE
      error(' parameter "nc_filepath" must be a string indicating the filepath, or a structure nc_filepath(:).name')
   end
   nc_filepath_list = nc_filepath;
end

num_files = length( nc_filepath_list );
for inc=1:num_files 
  [invalid_nc_file tmp] = unix(['ls ' nc_filepath_list(inc).name]);
  if invalid_nc_file
    error([' Can not locate ' nc_filepath_list(inc).name])
  end

  if nc_filepath_list(inc).name(1) == '~';
    numchars = length( nc_filepath_list(inc).name );
    if numchars==1
      nc_filepath_list(inc).name = getEnvVariable('HOME');
    else
      nc_filepath_list(inc).name = [ getEnvVariable('HOME') nc_filepath_list(inc).name(2:numchars) ];
    end
  end
end

%
% Open File
%
max_nc_length = 100000;

for inc=1:num_files 
   nc_path = nc_filepath_list(inc).name;
   nc_file(inc).pntr = netcdf( nc_path, 'nowrite' );
   
   if length(nc_file(inc).pntr) > max_nc_length
     %error('  nc file is longer than the maximum allowed (file might not exist)')
     error(['  nc file "' nc_path '" does not exist'])
   end
end

%
% Extract Dimensions and make sure all dimensions (except time) agree
%
dimensions = getNCdimensions( nc_file(1).pntr );
if length( dimensions ) < 1
   error(['  nc file "' nc_filepath_list(1).name '" is dimensionless'])
end
for inc=2:1:num_files 
   dims = getNCdimensions( nc_file(inc).pntr );
   if length( dims ) < 1
     error(['  nc file "' nc_filepath_list(inc).name '" is dimensionless'])
   end

   % verify dimensions match
   dims_agree = TRUE;
   if length(dims) ~= length(dimensions)
      dims_agree = FALSE;
   else
      for in=1:length(dims)
         if strcmp( dims(in).name, 'time' ) == FALSE
            diff = abs( (dims(in).data - dimensions(in).data) / max( dims(in).data, 100*eps ) );
            if diff > 10^-6
               dims_agree = FALSE;
            end
         else
            dimensions(in).data = cat(1, dimensions(in).data, dims(in).data );
         end
      end
   end
   if dims_agree==FALSE
      error([' dimensions do not agree in "' nc_filepath_list(1).name '" and "' nc_filepath_list(inc).name '"'])
   end
end
time_dim_exists = FALSE;
for idim=1:length(dimensions)
  if strcmp( dimensions(idim).name, 'time' )
     time_dim_exists = TRUE;
  end
end


%
% Prepare extract_it and extract_ik
%
if strcmp(extract_it,'first')
  clear extract_it
  nc_file    = nc_file(1);
  num_files  = 1;
  extract_it = 1;
elseif strcmp(extract_it,'last')
  clear extract_it
  nc_file    = nc_file(num_files);
  num_files  = 1;
  extract_it = length( var(nc_file.pntr,'time')  );
elseif strcmp(extract_it,'all')  |   extract_it==-1
  clear extract_it
  extract_it = 1:length( var(nc_file(1).pntr,'time')  );
  for inc=2:1:num_files
     num_new_it = length( var(nc_file(inc).pntr,'time') );
     first_it = extract_it(length(extract_it)) + 1;
     extract_it(length(extract_it)+1 : length(extract_it)+num_new_it) = first_it : first_it+num_new_it-1;
  end
elseif isnumeric(extract_it)
  extract_it = sort(extract_it);
else
  error('  Invalid input:  extract_it')
end

if strcmp(extract_ik,'top')
  clear extract_ik
  extract_ik = 1;
elseif strcmp(extract_ik,'bottom')
  clear extract_ik
  extract_ik = length( var(nc_file(1).pntr,'sigmatheta_u')  );
elseif strcmp(extract_ik,'all')  |   extract_ik==-1
  clear extract_ik
  extract_ik = 1:length( var(nc_file(1).pntr,'sigmatheta_u')  );
elseif isnumeric(extract_ik)
  extract_ik = sort(extract_ik);
else
  error('  Invalid input:  extract_ik')
end



%
% Find NC files where ALL VARIABLES EXIST
%
for inc=1:length( nc_file )
   all_variables_exist=TRUE;
   for iv=1:length( extract_var )
      clear variable vname
      vname = extract_var(iv).name;
      variable = var( nc_file(inc).pntr, vname );
      if length(variable)==0
         all_variables_exist=FALSE;
      end
   end

   if all_variables_exist
      if length(who('valid_nc'))==0
         valid_nc = inc;
      else
         valid_nc(length(valid_nc)+1) = inc;
      end
   end
end

if length(who('valid_nc'))==0
  if length( nc_file ) == 1
    error(' ALL variables do not exist in nc file' )
  else
    error(' at least one nc file must contain ALL extraction variables')
  end
end


%
% Find the last valid time index of each NC file 
%   (removes overlaps in time indexing)
%
for inc=length(valid_nc)-1 : -1 : 1
   clear time_it
   time_it = getNCvardata( nc_file(valid_nc(inc)).pntr, 'time' );
   last_valid_file_it(inc) = length( time_it );

   clear time_itp1
   time_itp1 = getNCvardata( nc_file(valid_nc(inc+1)).pntr, 'time' );

   if time_it(1) > time_itp1(1)
      error('  extract files must be given sequentially in time')
   end 

   clear igt
   igt = find( time_it(length(time_it)) >= time_itp1 );
   if length(igt)>0
      last_valid_file_it(inc) = length( time_it ) - max(igt);
      if last_valid_file_it(inc) < 1
         valid_nc(inc) = [];
         last_valid_file_it(inc) = [];
      end
   end
end
clear time_it
if time_dim_exists
  time_it = getNCvardata( nc_file(valid_nc(length(valid_nc))).pntr, 'time' );
  last_valid_file_it( length(valid_nc) ) = length( time_it );
else
  time_it = 1;
  last_valid_file_it = 1;
end

%for inc=1:length(valid_nc)
%   time_it = getNCvardata( nc_file(valid_nc(inc)).pntr, 'time' );
%   fprintf('  time 1, valid, end = %f, %f, %f\n',  ...
%              time_it(1)/31.4e6, time_it(last_valid_file_it(inc))/31.4e6, time_it(length(time_it))/31.4e6 );
%end


%
% Extract Variables
%
v_cnt=1;
for iv=1:length( extract_var )
   clear vname

   empty_var = FALSE;
   if length( strmatch('empty_', extract_var(iv).name) ) > 0
      vname = extract_var(iv).name( length('empty_')+1:length(extract_var(iv).name) );
      empty_var = TRUE;
   else
      vname = extract_var(iv).name;
   end

   data_found = FALSE; it_cnt=1; 
   for vnc=1:length(valid_nc)
      inc=valid_nc(vnc);
      clear variable var_size nc_time

      if time_dim_exists
         nc_time  = getNCvardata( nc_file(inc).pntr, 'time');
      else
         nc_time = 0.0;
      end

      variable = var( nc_file(inc).pntr, vname );
      var_size = size(variable);

      if vnc==1
         % get variable size
         if length(var_size) ~= 4  &  length(var_size) ~= 2
           error(['  "' extract_var(iv).name '" is not a data variable'])
         end
   
         % set variable name and units
         epic_vars(v_cnt).name  = remove_sloppy_characters( name( variable ) );
         epic_vars(v_cnt).units = remove_sloppy_characters( variable.units(:) );
   
         % set variable dimensions (names only)
         if length(var_size)==4
           epic_vars(v_cnt).dim = getNCdimensions( variable, TRUE );
         elseif length(var_size)==2
           dim_tmp = getNCdimensions( variable, TRUE );
           epic_vars(v_cnt).dim(1).name = 'NA';
           epic_vars(v_cnt).dim(2).name = 'NA';
           epic_vars(v_cnt).dim(3).name = dim_tmp(1).name;
           epic_vars(v_cnt).dim(4).name = dim_tmp(2).name;
         end

         first_it=1;
      else
         first_it = 1;
         for tmp=1:vnc-1
           first_it = first_it + last_valid_file_it(tmp);
         end
      end
%first_it, var_size(1)
      % set data
      if length( var_size ) == 4
         for it=1:length(extract_it), if first_it<=extract_it(it)  &  extract_it(it)<=var_size(1)+first_it
            nc_file_it = extract_it(it) - first_it+1;
%last_valid_file_it(vnc)
            if nc_file_it <= last_valid_file_it(vnc)
              ik_cnt=1;
              for ik=1:length(extract_ik), if extract_ik(ik) <= var_size(2)
                 data_found = TRUE;
     
                 epic_vars(v_cnt).data( it_cnt, ik_cnt, 1:var_size(3), 1:var_size(4) ) ...
                    = variable( nc_file_it, extract_ik(ik), :,:);
                    %= variable( extract_it(it)-first_it+1, extract_ik(ik), :,:);
                 
                 if empty_var==TRUE
                    epic_vars(v_cnt).data( it_cnt, ik_cnt, :,: ) = 0.0;
                 end
  
                 epic_vars(v_cnt).time( it_cnt ) = nc_time( nc_file_it );
                 %epic_vars(v_cnt).time( it_cnt ) = nc_time( extract_it(it)-first_it+1 );
     
                 ik_cnt=ik_cnt+1;
              end, end  % loop over ik
              it_cnt=it_cnt+1;
            end   % if nc_file_it is valid
         end, end  % loop over it
      elseif length( var_size ) == 2
         data_found = TRUE;
   
         epic_vars(v_cnt).data( 1, 1, 1:var_size(1), 1:var_size(2) ) ...
                       = variable( :,: );

         epic_vars(v_cnt).time( 1 ) = nc_time( 1 );
               
         if empty_var==TRUE
            epic_vars(v_cnt).data( 1, 1, :,: ) = 0.0;
         end
      end

   end  % loop over valid nc files

   % remove variable if no data was extracted
   if data_found == FALSE
      epic_vars(v_cnt) = [];

   % store data in epic_vars
   else
      if length(var_size)==4
         IHI = var_size(4);
         JHI = var_size(3);
         KHI = ik_cnt-1;
         THI = it_cnt-1;
      elseif length(var_size)==2
         IHI = var_size(2);
         JHI = var_size(1);
         KHI = 1;
         THI = 1;
      end
      epic_vars(v_cnt).ilo  = 1;
      epic_vars(v_cnt).ihi  = IHI;
      epic_vars(v_cnt).jlo  = 1;
      epic_vars(v_cnt).jhi  = JHI;

      epic_vars(v_cnt).klo  = 1;
      epic_vars(v_cnt).khi  = KHI;         % (see NOTE on t and k indexing)
      epic_vars(v_cnt).tlo  = 1;                
      epic_vars(v_cnt).thi  = THI;         

      v_cnt = v_cnt+1;
   end


end % loop over variable names

clear variable

%if length(epic_vars)==0
%  error('  Data was not extracted.')
%end


%
% Close the nc file, if necessary
%
if nargout<3
  for inc=1:length(nc_file)
     close(nc_file(inc).pntr)
  end
end




return

