function [var_max var_min ] = getChannelMaxMin( nc_file_input, epic_vars, vars, channel, var_type, extract_it, extract_ik )
%
% function [var_max var_min] = getChannelMaxMin( nc_file, epic_vars, vars, channel, var_type, extract_it, extract_ik )
%
%  Computes the maximum and minimum of epic_vars within a specified channel of latitude.  The variable type 
%  is specified with "var_type" and may be set to SCALAR or VECTOR2D.  The names of variables are specified 
%  in "vars" which should have the following format:
%
%
%    ________________SCALAR____________________      _______________VECTOR2D_________________
%           vars(1).name = <scalar 1>                vars(1).name = <u component of vector 1>
%           vars(2).name = <scalar 2>                vars(2).name = <v component of vector 1>
%           vars(3).name = <scalar 3>                vars(3).name = <u component of vector 2>
%           vars(4).name = <scalar 4>                vars(4).name = <v component of vector 2>
%                        .                                        .
%                        .                                        .
%                        .                                        .
%
%
%  If there are N scalars specified by vars, there will be N values returned in var_max and var_min 
%  (i.e., one max and one min for each scalar).  If there are N/2 vectors specifed by vars, there will be 
%  N/2 values returned in var_max and var_min (i.e., one max and one min for each vector).
%

%
% set defaults
%
FALSE=0; TRUE=1;
if nargin<7
  extract_ik = 'all';
end
if nargin<6
  extract_it = 'last';
end
if nargin<5
  var_type = 'SCALAR';
end
if nargin<4
  channel = [-90.0  90.0];
  warning([' Input "channel" is missing.  Using default value of [-90.0 90.0]'])
end
if nargin<3
  error([' Missing input:  getMaxChannelValue( nc_file, epic_vars, vars, channel, var_type, extract_it, extract_ik )'])
end

%
% check for valid input
%
if length( epic_vars ) == 0 
  error([' epic_vars is empty'])
end

if isfield(vars,'name')==FALSE
  error([' Input "vars" must have the format vars(:).name .'])
end

if length(channel) ~= 2
  error([' Input "channel" must specify the latitudinal range of the channel in the format channel=[lat_min lat_max].'])
end

if strcmpi(var_type,'SCALAR')==0  &  strcmpi(var_type,'VECTOR2D')==0  &  strcmpi(var_type,'VECTOR3D')
  error([' "var_type" is invalid.  Use "SCALAR", "VECTOR2D", or "VECTOR3D".'])
end
if strcmpi(var_type, 'VECTOR2D')
  if mod(length(vars), 2) > 0
    error([' The number of variables (' num2str(length(vars)) ') must be a multiple of 2 for var_type="' var_type '".'])
  end
end
if strcmpi(var_type, 'VECTOR3D')
  if mod(length(vars), 3) > 0
    error([' The number of variables (' num2str(length(vars)) ') must be a multiple of 3 for var_type="' var_type '".'])
  end
end

if ischar( nc_file_input )
  nc_file = netcdf( nc_file_input );
else
  if isfield( nc_file_input, 'pntr' )
    nc_file = nc_file_input(1).pntr;
  else
    nc_file = nc_file_input;
  end
end

if strcmp(extract_it,'first')
  clear extract_it
  extract_it = epic_vars(1).tlo;
elseif strcmp(extract_it,'last')
  clear extract_it
  extract_it = epic_vars(1).thi;
elseif strcmp(extract_it,'all')  
  clear extract_it
  extract_it = epic_vars(1).tlo : epic_vars(1).thi;
elseif isnumeric(extract_it)
  extract_it = sort(extract_it);
else
  error('  Invalid input:  extract_it')
end

if strcmp(extract_ik,'top')
  clear extract_ik
  extract_ik = epic_vars(1).klo;
elseif strcmp(extract_ik,'bottom')
  clear extract_ik
  extract_ik = epic_vars(1).khi;
elseif strcmp(extract_ik,'all')  |   extract_ik==-1
  clear extract_ik
  extract_ik = epic_vars(1).klo : epic_vars(1).khi;
elseif isnumeric(extract_ik)
  extract_ik = sort(extract_ik);
else
  error('  Invalid input:  extract_ik')
end


%
% get indices cooresponding to variable names
%
for iv=1:length(vars)
  var_found=FALSE;
  for iev=1:length(epic_vars)
    if strcmp(vars(iv).name, epic_vars(iev).name)
      var_found=TRUE;
      vars(iv).id = iev;
    end
  end
  if var_found==FALSE
    error([' Could not find variable "' vars(iv).name '" in epic_vars.'])
  end
end


%
% get max_vals : SCALAR
%
if strcmpi( var_type, 'SCALAR' )
  var_max(1:length(vars)) = -10e10;
  var_min(1:length(vars)) =  10e10;
  for iv=1:length(vars)
    clear latitude below_channel above_channel
    latitude = getNCvardata( nc_file, epic_vars( vars(iv).id ).dim(3).name );
    if latitude(1)                > channel(2) ...
     | latitude(length(latitude)) < channel(1)
      error([' Invalid latitude range specified in input "channel".'])
    end
    below_channel = find(latitude<channel(1));
    above_channel = find(latitude>channel(2));
    if length(below_channel)>0
      jmin = below_channel(length(below_channel))+1;
    else
      jmin=epic_vars( vars(iv).id ).jlo; 
    end
    if length(above_channel)>0
      jmax = above_channel(1)-1;
    else
      jmax=epic_vars( vars(iv).id ).jhi; 
    end
    if jmin>jmax
      error([' Input "channel" is too narrow.'])
    end

    for it=1:length(extract_it)
      t=extract_it(it);
      for ik=1:length(extract_ik)
        k=extract_ik(ik);
        var_max(iv) = max( var_max(iv), max(max( epic_vars(vars(iv).id).data(t,k,jmin:jmax,:) )) );
        var_min(iv) = min( var_min(iv), min(min( epic_vars(vars(iv).id).data(t,k,jmin:jmax,:) )) );
      end
    end
  end
end


%
% get max_vals : VECTOR2D
%
if strcmpi( var_type, 'VECTOR2D' )
  var_max(1:length(vars)/2) = -10e10;
  var_min(1:length(vars)/2) =  10e10;
  for iiv=1:length(vars)/2
    iv = 2*iiv-1;
    clear latitude
    latitude = getNCvardata( nc_file, epic_vars( vars(iv).id ).dim(3).name );
    if latitude(1)                > channel(2) ...
     | latitude(length(latitude)) < channel(1)
      error([' Invalid latitude range specified in input "channel".'])
    end
    below_channel = find(latitude<channel(1));
    above_channel = find(latitude>channel(2));
    if length(below_channel)>0
      jmin = below_channel(length(below_channel))+1;
    else
      jmin=epic_vars( vars(iv).id ).jlo; 
    end
    if length(above_channel)>0
      jmax = above_channel(1)-1;
    else
      jmax=epic_vars( vars(iv).id ).jhi; 
    end
    if jmin>jmax
      error([' Input "channel" is too narrow.'])
    end

    if strcmp( epic_vars( vars(iv  ).id ).dim(3).name,  ...
               epic_vars( vars(iv+1).id ).dim(3).name )==FALSE

      epic_vars = remapEpicVar( epic_vars, nc_file,              ...
                                epic_vars( vars(iv  ).id ).name, ...
                                epic_vars( vars(iv+1).id ).name );
    end
  
    u_data = epic_vars( vars(iv  ).id ).data;
    v_data = epic_vars( vars(iv+1).id ).data;
    for it=1:length(extract_it)
      t=extract_it(it);
      for ik=1:length(extract_ik)
        k=extract_ik(ik);
        clear mag_data
        mag_data = sqrt( u_data(t,k,jmin:jmax,:).^2.0 + v_data(t,k,jmin:jmax,:).^2.0 );
        var_max(iv) = max( var_max(iv), max(max(mag_data)) );
        var_min(iv) = min( var_max(iv), min(min(mag_data)) );
      end
    end
  end
end


%
% get max_vals : VECTOR3D
%
if strcmpi( var_type, 'VECTOR3D' )
  error([' Option var_type="VECTOR3D" is not yet supported.'])
end




if ischar( nc_file_input )
  close(nc_file)
end

