function [EP_FLUX_Y, EP_FLUX_Z, EP_FLUX_DIV, EP_THETA, EP_THETA2] = ...
         getEPfluxVector( epic_vars, nc_file, input_it )
%
% function [EP_FLUX_Y, EP_FLUX_Z, EP_FLUX_DIV, EP_THETA, EP_THETA2] 
%                          =  getEPfluxVector( epic_vars, nc_file, input_it )
%
%
%  Returns the Eliassen-Palm flux vector and its divergence.  Output data 
%    is located on constant theta surfaces and has the following units and 
%    coordinates :
%
%   ___Component__|___units___|__vert coord__|__horiz coord__|
%   |  EP_FLUX_Y  |  N/(m K)  |   EP_THETA2  |     lat_v     |
%   |             |           |              |               |
%   |  EP_FLUX_Z  |  N/(m K)  |   EP_THETA   |    lat_hdry   |
%   |             |           |              |               |
%   | EP_FLUX_DIV |   m/s^2   |   EP_THETA2  |    lat_hdry   |
%
%
%  NOTE:  EP_THETA(K) = 0.5*( EP_THETA2(K-1)+EP_THETA2(K) ),  when K>2
%         (the K=1 layer is a little screwy)
%
%  The input argument epic_vars must contain the following variables :
%
%    u
%    v
%    hdry
%    theta
%    theta2
%    mont2
%    hdry3   
%    p3      
%    heat3   
%    exner3  
%

TRUE=1;  FALSE=0;

if nargin<3
  input_it = 'last';
end

if nargin<2
  error(' Missing input:  epic_vars, nc_file, it')
end

DEG = pi/180.0;

if ischar(input_it)
  if strcmp(input_it,'first')
     it = 1;
  elseif strcmp(input_it,'last')
     it = epic_vars(1).thi - epic_vars(1).tlo + 1;
  %elseif strcmp(input_it,'all')
  %   it = 1 : (epic_vars(iv).thi - epic_vars(iv).tlo + 1);
  else
     error([' Parameter it = "' input_it '" is not permitted'])
  end
else
  it = input_it;
end


PRINT_DIAGS = TRUE;


%
% Define grid range in ZERO index space.
%
grid = getNCgridObject( nc_file );
ILO    = 1;
IHI    = grid.ni;
JLO    = 1;
JFIRST = grid.jfirst+1;
JHI    = grid.nj+1;

if grid.ni==1
   error('EP flux requires grid.ni > 1')
end


%
% Set names of variables used in the calculations,
%  and check to see if they exist.
%
iv=1;
EP_Y_dep_vars_k(iv).name = 'u';            iv=iv+1;
EP_Y_dep_vars_k(iv).name = 'v';            iv=iv+1;
EP_Y_dep_vars_k(iv).name = 'hdry';         iv=iv+1;

iv=1;
EP_Z_dep_vars_k(iv).name = 'u';            iv=iv+1;
EP_Z_dep_vars_k(iv).name = 'mont2';        iv=iv+1;

iv=1;
EP_Z_dep_vars_kph(iv).name = 'hdry3';      iv=iv+1;
EP_Z_dep_vars_kph(iv).name = 'p3';         iv=iv+1;
EP_Z_dep_vars_kph(iv).name = 'heat3';      iv=iv+1;
EP_Z_dep_vars_kph(iv).name = 'exner3';     iv=iv+1;

dep_vars = [EP_Y_dep_vars_k EP_Z_dep_vars_k EP_Z_dep_vars_kph];
for iv=1:length( dep_vars )
   iev=1;
   var_exists=FALSE;
   while var_exists==FALSE  &  iev<=length(epic_vars)
      if strcmp( dep_vars(iv).name, epic_vars(iev).name )
         var_exists = TRUE;
      else
         iev=iev+1;
      end
   end
   if var_exists==FALSE
      error(['  Variable "' dep_vars(iv).name '" does not exist in epic_vars'])
   end
end


%=============================== EP_FLUX_Y =============================== 

EP_Y_vars = epic_vars;

%
%  Remap variables required for EP_FLUX_Y to constant theta surfaces located in the u,v,h layers
%
if PRINT_DIAGS==TRUE
   fprintf('\nRemapping EP_FLUX_Y variables to "in the k layer" theta surfaces ....\n')
end
for iv=1:length(EP_Y_dep_vars_k)
   [EP_Y_vars, theta2_coord] = remapEpicVarToNewSurfaces( EP_Y_vars, nc_file,                           ...
                                                          EP_Y_dep_vars_k(iv).name, 'theta2', 'theta2', ...
                                                          FALSE, input_it );
                                                          %EP_Y_dep_vars(iv).name, 'theta2', 'theta', ...
                                                          %TRUE, input_it );
end                                                    

KLAST_ACTIVE_EPY = length( find(theta2_coord>0) );

%
% Get variables 
%
for iv=1:length(EP_Y_vars)
   if strcmp( EP_Y_vars(iv).name, 'u' )
      U(:,:,:) = EP_Y_vars(iv).data(it,:,:,:);
      size_u = size(U);
      U(:,:,size_u(3)+1) = U(:,:,1);
      clear size_u
   end

   if strcmp( EP_Y_vars(iv).name, 'v' )
      V(:,:,:) = EP_Y_vars(iv).data(it,:,:,:);
   end

   if strcmp( EP_Y_vars(iv).name, 'hdry' )
      H(:,:,:) = EP_Y_vars(iv).data(it,:,:,:);
   end
end

clear EP_Y_vars


%
% Print coordinates of EP_Y data interpolated to k surfaces
%
if PRINT_DIAGS
   fprintf('\n  theta2_coord:\n')
   %fprintf('      ')
   for ik=1:KLAST_ACTIVE_EPY
     fprintf(['    '  num2str(theta2_coord(ik))])
   end
   fprintf('\n\n')

   fprintf('  theta2_coord mid points:\n')
   fprintf('      ')
   for ik=1:KLAST_ACTIVE_EPY-1
      fprintf([ '    ' num2str( 0.5*(theta2_coord(ik)+theta2_coord(ik+1)) ) ])
   end
   fprintf('\n')
end


%
% Calculate EP_FLUX_Y
%
if PRINT_DIAGS==TRUE
   fprintf('\nCalculating EP_FLUX_Y ....\n')
end
  for K=1:KLAST_ACTIVE_EPY
    %for jj = JFIRST : JHI
      %Jstag = 2*jj+1;
      %J = jj+1;
    for J = JFIRST : JHI

      %if (grid.ni > 1 ) 
        VH_AVG(J) = 0.0;
        U_AVG( J) = 0.0;
        for I = ILO : 1 : IHI
          VH(J,I)    = V(K,J,I)*(H(K,J,I)+H(K,J-1,I));
          VH_AVG(J)  = VH_AVG(J)  +  VH(J,I);
          U_AVG( J)  =  U_AVG(J)  +  (U(K,J,I)+U(K,J,I+1)+U(K,J-1,I)+U(K,J-1,I+1));
        end
        % NOTE: only works if i-direction is not decomposed. 
        VH_AVG(J) = 2.0 * VH_AVG(J) / grid.ni;
        U_AVG( J) =        U_AVG(J) / grid.ni;

        %
        % Calculate meridional component of EP flux:
        %
        EP_FLUX_Y(K,J) = 0.0;
        for I = ILO : 1 : IHI
          EP_FLUX_Y(K,J) =  EP_FLUX_Y(K,J)        ...
                         +  (VH(J,I)-VH_AVG(J))*  ...
                            ((U(K,J,I)+U(K,J,I+1)+U(K,J-1,I)+U(K,J-1,I+1))-U_AVG(J));
        end
        % NOTE: Only works if i-direction not decomposed. 
        EP_FLUX_Y(K,J) = - EP_FLUX_Y(K,J) / ( 4.0*grid.ni*grid.m(2*J-1)*grid.dln*DEG );
                                          %   ^ accounts for averaging
      %else 
      %  % For ni = 1 case, store v in place of ep_flux_y: 
      %  EP_FLUX_Y(K,J) = V(K,J,ILO);
      %end

    end  % loop over J
  end  % loop over K

  EP_FLUX_Y(:,1) = 0.0;
  EP_FLUX_Y(:,JHI+1) = 0.0;
  %size( EP_FLUX_Y )

clear U V H VH_AVG U_AVG

%=============================== EP_FLUX_Z =============================== 

EP_Z_vars = epic_vars;

%
%  Remap variables required for EP_FLUX_Z to theta surfaces located on on u,v,h interfaces
%
if PRINT_DIAGS==TRUE
   fprintf('\nRemapping EP_FLUX_Z variables to k+1/2 theta surfaces ....\n')
end
% first remap variables originally defined in the k layer
for iv=1:length(EP_Z_dep_vars_k)
   [EP_Z_vars, theta_coord] = remapEpicVarToNewSurfaces( EP_Z_vars, nc_file,                           ...
                                                         EP_Z_dep_vars_k(iv).name, 'theta2', 'theta2', ...
                                                         TRUE, input_it );
                                                         %EP_Z_dep_vars(iv).name, 'theta2', 'theta', ...
                                                         %FALSE, input_it );
end

% next remap variables originally defined on the k layer interfaces
for iv=1:length(EP_Z_dep_vars_kph)
   [EP_Z_vars, theta_coord] = remapEpicVarToNewSurfaces( EP_Z_vars, nc_file,                            ...
                                                         EP_Z_dep_vars_kph(iv).name, 'theta', 'theta2', ...
                                                         TRUE, input_it );
                                                         %EP_Z_dep_vars(iv).name, 'theta', 'theta', ...
                                                         %FALSE, input_it );
end

KLAST_ACTIVE_EPZ = length( find(theta_coord>0) );


%
% Get variables 
%
for iv=1:length(EP_Z_vars)
   %if grid.ni>1
     if strcmp( EP_Z_vars(iv).name, 'u' )
        U(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);
        size_u = size(U);
        U(:,:,size_u(3)+1) = U(:,:,1);
        clear size_u
     end
  
     if strcmp( EP_Z_vars(iv).name, 'hdry3' )
        H(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);
     end
  
     if strcmp( EP_Z_vars(iv).name, 'p3' )
        P(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);
     end
  
     if strcmp( EP_Z_vars(iv).name, 'mont2' )
        MONT(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);
        size_m = size(MONT);
        MONT(:,:,size_m(3)+1) = MONT(:,:,1);
        size_m(3) = size_m(3)+1;
        for ii=size_m(3)+1 : -1 : 2
           MONT(:,:,ii) = MONT(:,:,ii-1);
        end
        size_m(3) = size_m(3)+1;
        MONT(:,:,1) = MONT(:,:,size_m(3)-1);
        MONT_periodic_xoffset=1;
     end
  
   %  if strcmp( EP_Z_vars(iv).name, 'w3' )
   %     W(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);
   %  end

   %elseif grid.ni==1
     if strcmp( EP_Z_vars(iv).name, 'heat3' )
        HEAT(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);   % ( J/kg/K )
     end

     if strcmp( EP_Z_vars(iv).name, 'exner3' )        
        EXNER(:,:,:) = EP_Z_vars(iv).data(it,:,:,:);  % ( J/kg/s )
     end
   %end
end

clear EP_Z_vars


%
% Print coordinates of EP_Z data interpolated to k+1/2 surfaces
%
if PRINT_DIAGS
   fprintf('\n  theta_coord:\n')
   fprintf('      ')
   for ik=1:KLAST_ACTIVE_EPZ
     fprintf(['    '  num2str(theta_coord(ik))])
   end
   fprintf('\n')

   %fprintf('  theta_coord mid points:\n')
   %fprintf('      ')
   %for ik=1:KLAST_ACTIVE_EPZ-1
   %   fprintf([ '    ' num2str( 0.5*(theta_coord(ik)+theta_coord(ik+1)) ) ])
   %end
   %fprintf('\n\n')
end


%
% Calculate EP_FLUX_Z
%
if PRINT_DIAGS==TRUE
   fprintf('\nCalculating EP_FLUX_Z ....\n')
end
  g = getNCattribute(nc_file, 'planet_g');

  for K = 1 : KLAST_ACTIVE_EPZ
    %if grid.ni > 1
      % 
      % Start calculation of z-component of EP flux.
      %
      % Because avg(mont_x) = 0., avg(p'*mont_x') = avg(p*mont_x).
      %
      % NOTE: For EP_FLUX_Z, k = nk case effectively uses p(nk+.5)*mont(nk) 
      % instead of p(nk+.5)*.5*[mont(nk+1)+mont(nk)].
      %
      for J = JLO : JHI
        EP_FLUX_Z(K+1,J) = 0.0;
        dx2_inv = ( grid.m(2*J) )^2.0;  %  (removal of ()^2 and divide by 2 for central diffs occurs later)
        for I = ILO : IHI
          Im = I + MONT_periodic_xoffset;
          EP_FLUX_Z(K+1,J)  = EP_FLUX_Z(K+1,J)  ...
                            + P(K,J,I)/g        ...
                            * (MONT(K,J,Im+1)-MONT(K,J,Im-1))*dx2_inv;  
        end
      end
  
      %
      % Finish calculation of vertical component of EP flux.
      %
      clear WH;
      %WH(:,:) = W(K,:,:) .* H(K,:,:);
      WH(:,:) = (HEAT(K,:,:) ./ EXNER(K,:,:)) .* H(K,:,:);
      for J=JLO : JHI
          WH_AVG(J) = 0.0;
          U_AVG( J) = 0.0;
          for I = ILO : 1 : IHI
            WH_AVG(J) = WH_AVG(J)  +  WH(J,I);
            U_AVG( J) = U_AVG( J)  +  U(K,J,I)+U(K,J,I+1);
          end
          % NOTE: only works if i-direction not decomposed: 
          WH_AVG(J) = WH_AVG(J) / grid.ni;
          U_AVG( J) = U_AVG( J) / grid.ni; % (divide by 2 for average occurs below)
          for I=ILO : 1 : IHI
            EP_FLUX_Z(K+1,J) =  EP_FLUX_Z(K+1,J)                                   ...
                             -  (WH(J,I)-WH_AVG(J))*                               ...
                                (U(K,J,I)+U(K,J,I+1)-U_AVG(J));
          end
          % NOTE: only works if i-direction not decomposed: 
          EP_FLUX_Z(K+1,J) =   EP_FLUX_Z(K+1,J) / ( 2.0 * grid.ni * grid.m(2*J)*grid.dln*DEG );
                               %                    ^ d(MONT)/dx and U_AVG
                               %                          ^ average EP_FLUX_Z
      end                      %                                    ^----- a cos(lat) -----^

    %else  
    %  % 
    %  % For ni = 1 case, store w in place of ep_flux_z. 
    %  %
    %  EP_FLUX_Z(K+1,:) = HEAT(K,:,ILO) ./ EXNER(K,:,ILO);
    %end

  end % loop over K

  EP_FLUX_Z(1,:) = 0.0;
  for K=length(theta_coord)+1 : -1 : 2
     theta_coord(K) = theta_coord(K-1);
  end
  theta_coord(1) = -1;

  %size( EP_FLUX_Z )

  clear U H P MONT W HEAT EXNER WH_AVG U_AVG


%=============================== DIVERGENCE( EP_FLUX_Y, EP_FLUX_Z ) =============================== 

if PRINT_DIAGS==TRUE
   fprintf('\nCalculating EP_FLUX_DIV ....\n')
end

if KLAST_ACTIVE_EPZ <= KLAST_ACTIVE_EPY
   KLAST_ACTIVE_EPY  = KLAST_ACTIVE_EPZ - 1;
else
   KLAST_ACTIVE_EPZ  = KLAST_ACTIVE_EPY + 1;
end

if KLAST_ACTIVE_EPZ ~= KLAST_ACTIVE_EPY+1
  warning(['  EP_FLUX_Y layers = ' num2str(KLAST_ACTIVE_EPY) ',  EP_FLUX_Z layers = ' num2str(KLAST_ACTIVE_EPZ)])
end

%size( EP_FLUX_Y )
%size( EP_FLUX_Z )

%if (grid.ni > 1)
   %
   % Get mass variable
   %
   for iv=1:length(epic_vars)
      if strcmp( epic_vars(iv).name, 'hdry' )
         H(:,:,:) = epic_vars(iv).data(it,:,:,:);
      end
   end

  % 
  % Compute divergence of EP flux:
  %
  for K = 1 : 1: KLAST_ACTIVE_EPY
    for J = JLO : 1 : JHI
      % NOTE: only works if i-direction not decomposed: 
      H_AVG(J) = mean( H(K,J,:) );
    end

    if (K > 1) 
      thetaT = theta_coord(K);
    else 
      % special delta-theta in top layer: 
      thetaT = theta2_coord(K);
    end

    if not( thetaT>=theta2_coord  &  theta2_coord(K)>=theta_coord(K+1) )
       error([' coordinate mismatch at K=' num2str(K) ' :  theta(K)=' num2str(thetaT) ...
              ', theta2(K)=' num2str(theta2_coord(K)) ', theta(K+1)=' num2str(theta_coord(K+1)) ])
    end
    
    % store theta coordinates for output
    EP_THETA (K)   = thetaT;
    EP_THETA (K+1) = theta_coord(K+1);
    EP_THETA2(K)   = theta2_coord(K);

    d_th1_inv    = 1.0 / ( thetaT - theta_coord(K+1) );

    for J=JLO : 1 : JHI
      EP_FLUX_DIV(K,J) = (EP_FLUX_Z(K,J)-EP_FLUX_Z(K+1,J))*d_th1_inv;

      mn_J    = grid.mn(2*J);
      if ( J == JLO ) 
        m_Jmh_inv = 0.0;               
      else  
        m_Jmh_inv = 1.0 / grid.m(2*J-1);
      end
      if ( J == JHI )   
        m_Jph_inv = 0.0;
      else  
        m_Jph_inv = 1.0 / grid.m(2*J+1);
      end
      EP_FLUX_DIV(K,J) = EP_FLUX_DIV(K,J)                             ...
                       + mn_J * ( (EP_FLUX_Y(K,J+1)*m_Jph_inv         ...
                                  -EP_FLUX_Y(K,J  )*m_Jmh_inv  ) );

      % NOTE: Normalize ep_flux_div to have units of acceleration: 
      EP_FLUX_DIV(K,J) = EP_FLUX_DIV(K,J) *  grid.m(2*J)*grid.dln*DEG/H_AVG(J);
    end
  end
%end
%  
%if (grid.ni == 1)  
%  % 
%  % For ni = 1 case, store -dudt in place of ep_flux_div:
%  %
%  for K = 1 : 1: KLAST_ACTIVE_EPY
%    for J = JLO : 1 : JHI
%      EP_FLUX_DIV(K,J) = -DUDT(IT_ZERO,K,J,ILO);
%    end
%  end
%end

%size( EP_FLUX_DIV )

if PRINT_DIAGS==TRUE
   fprintf('\nEP_FLUX Calculations Complete\n')
end

