function [rm_vars, new_surf_coord] = remapEpicVarToNewSurfaces( epic_vars, nc_file, ...
                                                                var_name, surf_var_name, new_surf_var_name, ...
                                                                def_khalf, input_extract_it ) 
%
%function [rm_vars, new_surf_coord] = remapEpicVarToNewSurfaces( epic_vars, nc_file, ...
%                                                                var_name, surf_var_name, new_surf_var_name, ...
%                                                                def_khalf, input_extract_it) 
% Input: 
%          epic_vars          :  previously extracted epic variables
%
%          nc_file            :  the file from which epic_vars was extracted
%
%          var_name           :  name of variable to interpolate
%
%          surf_var_name      :  name of the surface variable with the same sigmatheta coordinates as var_name
%
%          new_surf_var_name  :  name of the variable that defines the new surfaces to which var_name
%                                is to be interpolated
%
%          def_khalf          :  TRUE  => interpolate to the interfaces of new_surf_var_name
%                                FALSE => interpolate to the layers of new_surf_var_name (default)
%
%          input_extract_it   :  time indices to extract.  (default is 'last')
%
%
%
% NOTE:  surf_var_name and new_surf_var_name should refer to the same physical quantity.  For exammple,
%           a remap of variable 'u' to constant 'theta' surfaces would look like:
%
%           [rm_vars, new_surf_coord] = remapEpicVarToNewSurfaces( epic_vars, nc_file, 'u', 'theta2', 'theta' );
%
%        Routine will also remap the variable horizontally, if necessary.
%
%
% Output:
%          rm_vars            :  identicle to epic_vars, but with var_name remapped to the new surfaces
%
%          new_surf_coord     :  coordinates of the newly mapped var_name
%
%

TRUE = 1;  FALSE = 0;

if nargin<7
   input_extract_it = 'last';
end
if nargin<6
   def_khalf = FALSE;
end
if nargin<5
   error('Missing input :  epic_vars, nc_file, var_name, surf_var_name, new_surf_var_name, def_khalf, input_extract_it ')
end



%surf_var_name = 'theta2';

%
% Find new surface variable
%
insrf = -1;
for iv=1:length(epic_vars)
   if strcmp( epic_vars(iv).name, new_surf_var_name )
      insrf = iv;
   end
end
if insrf==-1
   error(['Variable "' new_surf_var_name '" does not exist in epic_vars'])
end


%
% Find old surface variable (i.e., has sigmatheta coordinates equal to those of var_name)
%
isrf = -1;
for iv=1:length(epic_vars)
   if strcmp( epic_vars(iv).name, surf_var_name )
      isrf = iv;
   end
end
if isrf==-1
   error(['Variable "' surf_var_name '" does not exist in epic_vars'])
end


%
% Find variable to remap
%
iv = -1;
for itmp=1:length(epic_vars)
   if strcmp( epic_vars(itmp).name, var_name )
      iv = itmp;
   end
end
if iv==-1
   error(['Variable "' var_name '" does not exist in epic_vars'])
end


%
% Set extract_it
%
if ischar(input_extract_it)
  if strcmp(input_extract_it,'first')
     extract_it = 1;
  elseif strcmp(input_extract_it,'last')
     extract_it = epic_vars(iv).thi - epic_vars(iv).tlo + 1;
  elseif strcmp(input_extract_it,'all')
     extract_it = 1 : (epic_vars(iv).thi - epic_vars(iv).tlo + 1);
  else
     error([' Parameter extract_it = ' extract_it ' is not permitted'])
  end
else
  extract_it = input_extract_it;
end
   

%
% Make sure the remap variable and its surface variable
%  have the same sigmatheta coordinates.
%
sg_v = getNCvardata(nc_file,   epic_vars(iv).dim(2).name );
sg_s = getNCvardata(nc_file, epic_vars(isrf).dim(2).name );
if min( sg_v==sg_s ) == FALSE
   error([' Variable "' surf_var_name '" must have the same sigmatheta coordinates as variable "' var_name '"' ])
end


%
% Determine the deepest layer where the surface variable is monotonic for all (ii,jj) .
%
surf_var = epic_vars(insrf).data;
N = size( surf_var );
ikmid = round( N(2)/2 );
num_k_nsv = N(2);
if num_k_nsv<2
   error([' Remapping requires at least 2 depth layers in epic_vars'])
end
shallowest_k(1:length(extract_it),1:N(3),1:N(4)) = N(2)-1;
   deepest_k(1:length(extract_it),1:N(3),1:N(4)) = 2;
for tn=1:length(extract_it)
   it = extract_it(tn);
   if it>N(1)
      error(['Time index ' num2str(it) ' does not exist in epic_vars'])
   end

   for jj=1:N(3)
      for ii=1:N(4)

%         ik=2;  deepest_k_found = FALSE;
%         while deepest_k_found==FALSE  &  ik<=num_k_nsv
%            if surf_var(it,ik,jj,ii) < surf_var(it,ik-1,jj,ii)
%               deepest_k(it,jj,ii) = ik;
%            else
%               deepest_k_found = TRUE;
%            end
%            ik = ik+1;
%         end

        clear col 
        col = surf_var(it,:,jj,ii);
        %for ik=1:N(2)-1
        for ik=1:length(col)-1
          dcol(ik) = ( col(ik+1)-col(ik) ) / abs( col(ik+1)-col(ik) );
        end
        dcoltop = dcol;
        dcolbot = dcol;
        dcoltop(  ikmid:N(2)-1) = dcol(ikmid);  % (assumes col is monotonic at ikmid)
        dcolbot(1:ikmid       ) = dcol(ikmid);

        clear topbad botbad
        topbad = find( dcoltop ~= dcol(ikmid) );
        botbad = find( dcolbot ~= dcol(ikmid) );

        if length( topbad ) == 0
          shallowest_k(it,jj,ii) = 1;
        else
          shallowest_k(it,jj,ii) = topbad(length(topbad))+1;
        end

        if length( botbad ) == 0
          deepest_k(it,jj,ii) = N(2);
        else
          deepest_k(it,jj,ii) = botbad(1);
        end

      end
   end

   iktop(tn) = max( max( shallowest_k(it,:,:) ) );
   ikbot(tn) = min( min(    deepest_k(it,:,:) ) );
end

%
% Determine the deepest layer where the new surface variable is monotonic for all (ii,jj) .
%
%new_surf_var = epic_vars(insrf).data;
%N = size( new_surf_var );
%num_k_nsv = N(2);
%if num_k_nsv<2
%   error(['Theta remapping requires at least 2 depth layers in epic_vars'])
%end
%deepest_k(1:length(extract_it),1:N(3),1:N(4)) = 2;
%for tn=1:length(extract_it)
%   it = extract_it(tn);
%   if it>N(1)
%      error(['Time index ' num2str(it) ' does not exist in epic_vars'])
%   end
%
%   for jj=1:N(3)
%      for ii=1:N(4)
%
%         ik=2;  deepest_k_found = FALSE;
%         while deepest_k_found==FALSE  &  ik<=num_k_nsv
%            if new_surf_var(it,ik,jj,ii) < new_surf_var(it,ik-1,jj,ii)
%               deepest_k(it,jj,ii) = ik;
%            else
%               deepest_k_found = TRUE;
%            end
%            ik = ik+1;
%         end
%
%      end
%   end
%
%   ikbot(tn) = min( min( deepest_k(it,:,:) ) );
%end

%
% Define constant surfaces as layer means
%
new_surf_var = epic_vars(insrf).data;
for tn=1:length(extract_it)
   it = extract_it(tn);

   if (iktop(tn)>1)
     new_surf_coord(tn,1:iktop(tn)-1) = -1;
   end
   new_surf_coord(tn,iktop(tn))  = min(min( new_surf_var(it,iktop(tn),:,:) ));
   for ik=iktop(tn)+1 : 1 : ikbot(tn)-1
     new_surf_coord(tn,ik)       = getEpicVarLayerMean( epic_vars, new_surf_var_name, nc_file, ik, extract_it(tn) );
   end
   new_surf_coord(tn,ikbot(tn))  = max(max( new_surf_var(it,ikbot(tn),:,:) ));
   if ikbot(tn)<N(2)
     new_surf_coord(tn,ikbot(tn)+1:N(2)) = -1;
   end

end

%new_surf_coord(1,:)


%
% Reset coordinates to half surfaces
%
if def_khalf    
   for tn=1:length(extract_it)
      for k=iktop(tn):ikbot(tn)-1
         avg_coord(tn,k) = 0.5 * ( new_surf_coord(tn,k) + new_surf_coord(tn,k+1) );
      end
      for ik=ikbot(tn) : 1 : num_k_nsv-1
         avg_coord(tn,ik) = -1;
      end
      ikbot(tn) = ikbot(tn)-1;
   end

   clear new_surf_coord
   new_surf_coord = avg_coord;
   num_k_nsv = num_k_nsv - 1;
end
for tn=1:length(extract_it)
  positive_is_up(tn) = TRUE;
  if new_surf_coord(tn,iktop(tn)+1) > new_surf_coord(tn,iktop(tn))
    positive_is_up(tn) = FALSE;
  end
end

%
% Remap old surface variable horizontally, if necessary
%
if not(strcmp( epic_vars(iv).dim(3).name, epic_vars(isrf).dim(3).name )) ...
 | not(strcmp( epic_vars(iv).dim(4).name, epic_vars(isrf).dim(4).name )) 

   rm_vars = remapEpicVar( epic_vars, nc_file, surf_var_name, epic_vars(iv).name );

else
   rm_vars  = epic_vars;
end

surf_var = rm_vars(isrf).data;

%
% re-adjust top and bottom new_surf_coord so that coordinates are bound vertically by the interpolation data
%
for tn=1:length(extract_it)
  it = extract_it(tn);

  if positive_is_up(tn)
    min_rm_coord                  = min( min( surf_var(it,1,:,:) ) );
    new_surf_coord(tn,1)          = min( new_surf_coord(tn,1), min_rm_coord );

    max_rm_coord                  = max( max( surf_var(it,ikbot(tn),:,:) ) );
    new_surf_coord(tn,ikbot(tn))  = max( new_surf_coord(tn,ikbot(tn)), max_rm_coord );
  else
    min_rm_coord                  = max( max( surf_var(it,iktop(tn),:,:) ) );
    new_surf_coord(tn,iktop(tn))  = max( new_surf_coord(tn,iktop(tn)), min_rm_coord );

    max_rm_coord                  = min( min( surf_var(it,ikbot(tn),:,:) ) );
    new_surf_coord(tn,ikbot(tn))  = min( new_surf_coord(tn,ikbot(tn)), max_rm_coord );
  end

  % make sure new surface coordinates are still monotonic
  clear col 
  col = new_surf_coord(tn,:);
  %for ik=1:N(2)-1
  for ik=1:length(col)-1
    dcol(ik) = ( col(ik+1)-col(ik) ) / abs( col(ik+1)-col(ik) );
  end
  dcoltop = dcol;
  dcolbot = dcol;
  dcoltop(  ikmid:N(2)-1) = dcol(ikmid);  % (assumes col is monotonic at ikmid)
  dcolbot(1:ikmid       ) = dcol(ikmid);

  clear topbad botbad
  topbad = find( dcoltop ~= dcol(ikmid) );
  botbad = find( dcolbot ~= dcol(ikmid) );

  if length( topbad ) == 0
    iktop(tn) = max(iktop(tn),1);
  else
    iktop(tn) = topbad(length(topbad))+1;
    new_surf_coord(tn,1:iktop(tn)-1) = -1;
  end

  if length( botbad ) == 0
    ikbot(tn) = min(ikbot(tn),N(2));
    %ikbot(tn) = N(2);
  else
    ikbot(tn) = botbad(1);
    %new_surf_coord(tn,ikbot(tn)+1:length(new_surf_coord(tn,:))) = -1;
    new_surf_coord(tn,ikbot(tn)+1:N(2)) = -1;
  end

end
  
 
%
% make sure the horizontal coordinates of var_name and surf_var_name are the same
%
lon_v = getNCvardata(nc_file,  epic_vars(iv).dim(3).name );
lon_s = getNCvardata(nc_file,  rm_vars(isrf).dim(3).name );
lat_v = getNCvardata(nc_file,  epic_vars(iv).dim(4).name );
lat_s = getNCvardata(nc_file,  rm_vars(isrf).dim(4).name );
if min( lon_v==lon_s )==FALSE  |  min( lat_v==lat_s )==FALSE
   error(['  Horizontal coordinates of "' var_name '" and "' surf_var_name '" are not equal'])
end


%
% Interpolate variable onto constant surfaces
%
var_data = epic_vars(iv).data;
rm_data  = var_data;

for tn=1:length(extract_it)
   it=extract_it(tn);

   for ik=1 : 1 : iktop(tn)-1
      rm_data(it,ik,:,:) = -1;
   end

   for ik=ikbot(tn)+1 : 1 : num_k_nsv
      rm_data(it,ik,:,:) = -1;
   end

   for ik=iktop(tn):ikbot(tn)
      surfC = new_surf_coord(tn,ik);
      for jj=1:N(3)
         for ii=1:N(4)
            clear ike
            ike = find( new_surf_coord(tn,ik) == surf_var(it,:,jj,ii) );
            if length(ike)>0
               ikT = ike;   ikB = ike;
               wT = 1.0;
               wB = 0.0;
            else

               ilt = find( surf_var(it,:,jj,ii) < surfC );
               igt = find( surf_var(it,:,jj,ii) > surfC );
               if length(igt)==0
                  error(['  Can not interpolate because the new surface coordinate, ' num2str(surfC) ...
                            ', is greater than the original surface data, ' num2str( max(surf_var(it,:,jj,ii)) )])
               end
               if length(ilt)==0
                  error(['  Can not interpolate because the new surface coordinate, ' num2str(surfC) ...
                            ', is less than the original surface data, ' num2str( min(surf_var(it,:,jj,ii)) )])
               end

               surfT = 10e10;  surfB = -10e10;
               for itmp=1:length(ilt)
                  surfB = max( surfB, surf_var(it,ilt(itmp),jj,ii) );
               end
               ikB = find( surfB == surf_var(it,:,jj,ii) );
               for itmp=1:length(igt)
                  surfT = min( surfT, surf_var(it,igt(itmp),jj,ii) );
               end
               ikT = find( surfT == surf_var(it,:,jj,ii) );
               wT = (surfC  -  surfB)  / (surfT-surfB);
               wB = (surfT  -  surfC ) / (surfT-surfB);
            end
            
            rm_data(it,ik,jj,ii) = wB*var_data(it,ikB,jj,ii)  +  wT*var_data(it,ikT,jj,ii);

         end
      end
   end

end


%
% Set output variable
%
rm_vars(iv).data        = [];
rm_vars(iv).data        = rm_data;
rm_vars(iv).dim(2).name = [surf_var_name '_surfaces'];

rm_vars(isrf) = epic_vars(isrf);  % IMPORTANT!!!  (Reset the old coordinate variable in case it was remapped horizontally)


