function setFigureMargins( rows_cols, margins, ratio_height_to_width, scale, scale_type )
%
% function setFigureMargins( rows_cols, margins, ratio_height_to_width, scale, scale_type )
%
% Sets margins and gaps for figures with an arbitrary number of subplots.
%
% Input Parameters:  
%                  rows_cols    -  1 entry for square numbers of subplots, 2 entries otherwise.
%
%                  margins.L    -  Percentage of figure taken up by the left margin.
%                  margins.R    -  Percentage of figure taken up by the right margin.
%                  margins.Xgap -  Percentage of figure used to sepparate subplot columns.
%                  margins.B    -  Percentage of figure taken up by the bottom margin.
%                  margins.T    -  Percentage of figure taken up by the top margin.
%                  margins.Ygap -  Percentage of figure used to sepparate subplot rows..
%
%                  ratio_height_to_width -  Ratio of each subplot's physical dimensions 
%                                           (i.e., NOT a percentage ratio).
%
%                  scale        -  Number describing either the figure width, figure height,
%                                  subplot width, or subplot height.
%
%                  scale_type   -  Spedifies which parameter "scale" is representing.
%                                  Valid options are 'FIG_WIDTH' (default), 'FIG_HEIGHT', 
%                                  'AXES_WIDTH', or 'AXES_HEIGHT'.
%

%
% Check for valid input
%
if nargin<4
  error(' Missing input : rows_cols, margins, ratio_height_to_width, scale, scale_type.')
end
if nargin<5
  scale_type = 'FIG_WIDTH';
end

if not(strcmp(scale_type,'FIG_WIDTH' )) &  not(strcmp(scale_type,'FIG_HEIGHT' ))   ...  
 & not(strcmp(scale_type,'AXES_WIDTH')) &  not(strcmp(scale_type,'AXES_HEIGHT'))  
  if not(ischar( scale_type ))
    scale_type = num2str( scale_type );
  end
  error([' scale_type = "' scale_type '" is not supported.  Valid types are FIG_WIDTH, FIG_HEIGHT, AXES_WIDTH, and AXES_HEIGHT.'])
end

if scale<= 0.
  error(' Input parameter "scale" must be greater than 0.0.')
end

if length(rows_cols)==1
  rows_cols(2) = rows_cols(1);
end

if length(rows_cols)~=2
  error(' Input parameter "rows_cols" must have length 1 or 2.')
end

if not(isfield(margins, 'L'))
  error(' Input parameter "margins" is missing the field "L".')
end
if not(isfield(margins, 'R'))
  error(' Input parameter "margins" is missing the field "R".')
end
if not(isfield(margins, 'T'))
  error(' Input parameter "margins" is missing the field "T".')
end
if not(isfield(margins, 'B'))
  error(' Input parameter "margins" is missing the field "B".')
end
if rows_cols(2)>1
  if not(isfield(margins, 'Xgap'))
    error(' Input parameter "margins" is missing the field "Xgap".')
  end
else
  margins.Xgap = 1;
end
if rows_cols(1)>1
  if not(isfield(margins, 'Ygap'))
    error(' Input parameter "margins" is missing the field "Ygap".')
  end
else
  margins.Ygap = 1;
end

if ratio_height_to_width <= 0.0
  error(' Input parameter "ratio_height_to_width" must be greater than 0.0.')
end


%
% Put input into a nice format
%
m(1,1:2) = [ margins.L    margins.R    ];
m(2,1:2) = [ margins.B    margins.T    ];
gap      = [ margins.Xgap margins.Ygap ];
ratio    = [ ratio_height_to_width  1.0/ratio_height_to_width ];
rc       = [ rows_cols(2) rows_cols(1) ];
if strfind(scale_type,'WIDTH')
  d1 = 1;  d2 = 2;
else
  d1 = 2;  d2 = 1;
end

%
% Set Figure Atributes 
%
fpos = get(gcf,'Position');
if strfind( scale_type, 'FIG' )
       width(d1) = ( 1.0 - gap(d1)*(rc(d1)-1) - m(d1,1) - m(d1,2) ) / rc(d1);
  real_width(d1) = width(d1) * scale;
  real_width(d2) = real_width(d1)*ratio(d1);
      fpos(2+d1) = scale;
      fpos(2+d2) = real_width(d2)*rc(d2) / (1.0 - gap(d2)*(rc(d2)-1) - m(d2,1) - m(d2,2));
       width(d2) = real_width(d2) / fpos(2+d2);
else
  real_width(d1) = scale;
  real_width(d2) = real_width(d1)*ratio(d1);
      fpos(2+d1) = real_width(d1)*rc(d1) / (1.0 - gap(d1)*(rc(d1)-1) - m(d1,1) - m(d1,2));
      fpos(2+d2) = real_width(d2)*rc(d2) / (1.0 - gap(d2)*(rc(d2)-1) - m(d2,1) - m(d2,2));
       width(d1) = real_width(d1) / fpos(2+d1);
       width(d2) = real_width(d2) / fpos(2+d2);
end


%
% Reset Figure Attributes
%
ax_id = get(gcf,'Children');
for iax=length(ax_id):-1:1
  if strcmp( get(ax_id(iax),'Type'), 'uimenu' )
    ax_id(iax) = [];
  end
  if strcmp( get(ax_id(iax),'Tag'), 'Colorbar' )
    ax_id(iax) = [];
  end
end

if length(ax_id) ~= rc(1)*rc(2)
  error([' Found ' num2str(length(ax_id)) ' axes instead of ' num2str(rc(1)) 'x' num2str(rc(2)) '=' num2str(rc(1)*rc(2)) '.'])
end

set(gcf,'Position', fpos);
apos(1:4)=0;
apos(3:4) = [ width(1) width(2) ];
iax=length(ax_id);
for J=1:rc(2)
  apos(2) = margins.B + (width(2)+margins.Ygap)*(J-1);
  for I=1:rc(1)
    apos(1) = margins.L + (width(1)+margins.Xgap)*(I-1);
    set( ax_id(iax), 'Position', apos );
    iax=iax-1;
  end
end



