
function [thermo, cpr_out] = thermo_setup( planet )
%
% This function initializes the thermodynamics functions.
% Adapted from Peter Gierasch's Fortran subroutines setup(),
% numbers(), h2properties(), hydrogen(), theta2t_table() and trgrid().
%
% NOTE: cpr_out is the low-temperature-limit of cp/rgas.
% 
% (code translated from epic_funcs_diag.c) 
% 

% the following are taken from epic.h
M_LN2        = 0.69314718055994530942;
M_PI         = 3.14159265358979323846;
K_B          = 1.3806503e-23         ;  % Boltzmann constant, J/K 
H_PLANCK     = 6.62606876e-34        ;  % Planck constant, J s 
M_PROTON     = 1.67262158e-27        ;  % proton mass, kg 

% the following are taken from epic_datatypes.h
NDIM_THERMO  =  16                   ;
THLO_THERMO  =  20.0                 ;
THHI_THERMO  =  600.0                ;
CPRH2        =  2.5                  ;  % nondim low T ref. cp for H_2 
CPRHE        =  2.5                  ;  % nondim low T ref. cp for He 
CPR3         =  3.5                  ;  % nondim low T ref. cp for non H_2,He component 
MDIM_THERMO  =  128                  ;

  jmax = 50;
  ndegeneracy = [3.0, 1.0];

  xh2      = planet.x_h2;
  xhe      = planet.x_he;
  x3       = planet.x_3;
  if (planet.x_h2 > 0)  
    cpr_out = (CPRH2*xh2+CPRHE*xhe+CPR3*x3)/(xh2+xhe+x3);
  else  
    cpr_out = planet.cpr;
  end
  cpr = cpr_out;

  % 
  % Calculate hydrogen (H_2) properties.
  % The subscript "o" refers to ortho-hydrogen, "p" to para-hydrogen.
  %
  % In Gierasch's Fortran code, this segment is contained in the subroutine
  % h2properties().
  % 

  % 
  % Reference temperature and pressure used to define the mean 
  % potential temperature.
  % See Dowling et al (1998), Appendix A.
  % 
  t0 = 1.0;
  p0 = 1.0e+5;

  %
  % See Dowling et al (1998), eq. (A.12).
  % 
  c1    = log(K_B*t0/p0 * (2.0*M_PI*(M_PROTON/H_PLANCK)*(K_B*t0/H_PLANCK))^1.5);
  c2    = log(9.0);
  p     = p0;
  theta = 87.567;

  %
  % The array a[0:7] has entries
  %  a[0]:      equilibrium para fraction
  %  a[1],a[2]: ortho, para rotational internal energy per particle over K_B
  %  a[3],a[4]: ortho, para rotational cp per particle, units K_B
  %  a[5],a[6]: ortho, para rotational -Helmholtz free energy per particle over K_B*T
  %  a[7]:      equilibrium H2 (converting) cp per particle, units K_B
  % 

  %for (i = 0; i < MDIM_THERMO; i++)  
  for ii=0:1:(MDIM_THERMO-1)
    I=ii+1;
    temperature = 500.0*(ii+1)/MDIM_THERMO;
    if (temperature < 10.0)  
      %
      % Real hydrogen is not an ideal gas at this low-T limit.
      % These values are placeholders to avoid blow-ups during testing.
      % 
      ho       = CPRH2*temperature;
      hp       = CPRH2*temperature;
      pottempo = temperature;
      pottempp = temperature;
      a(1)     = 1.0;
      a(2)     = 175.1340;
      a(3)     = 0.0;
      a(4)     = 0.0;
      a(5)     = 0.0;
      a(6)     = 0.0;
      a(7)     = 0.0; 
      a(8)     = 0.0;
      ff       = 0.0;
    else  
      %
      % In Gierasch's Fortran, this segment is contained in the subroutine
      % hydrogen().
      % 
      y = theta/temperature;
      y = min(y,30.0);
      %for (n = 0; n < 2; n++)  
      %  for (m = 0; m < 3; m++)  
      %    z[m][n] = 0.0;
      %  end
      %end
      for n = 0:1
        N=n+1;
        for m = 0:2
          M=m+1;
          z(M,N) = 0.0;
        end
      end
      %for (j = 1; j <= jmax; j++)  
      for jj = 1 : jmax
        J=jj+1;
        jn(1) = 2*jj-1;
        jn(2) = jn(1)-1;
        %for (n = 0; n < 2; n++)  
        for n = 0 : 1
          N=n+1;
          term = ndegeneracy(N)*(2*jn(N)+1)*exp(-jn(N)*(jn(N)+1)*y);
          %for (m = 0; m < 3; m++)  
          for m = 0:2
            M=m+1;
            z(M,N) = z(M,N) + term;
            if (m < 2)  
              term = term * jn(N)*(jn(N)+1);
            end
          end
        end
        if (jj > 1 & term < 1.0e-20) 
          break;
        end
      end
      %den  = z[0][0]+z[0][1];
      den  = z(1,1)+z(1,2);

      a(1) = z(1,2)/den;
      for n = 0:1
        N=n+1;
        a(N+1) = theta*z(2,N)/z(1,N); 
        a(N+3) = y*y*(z(1,N)*z(3,N)-z(2,N)*z(2,N))/(z(1,N)*z(1,N));
        a(N+5) = log(z(1,N));
        %a(N+1) = theta*z[1][n]/z[0][n]; 
        %a(N+3) = y*y*(z[0][n]*z[2][n]-z[1][n]*z[1][n])/(z[0][n]*z[0][n]);
        %a(N+5) = log(z[0][n]);
      end
      a(8) = (1.0-a(1))*a(4)+a(1)*a(5)                               ...
            +(a(3)-a(2))*y/temperature*                              ...
                     (z(2,2)*z(1,1)-z(1,2)*z(2,1))/(den*den);
      %a(8) = (1.0-a[0])*a[3]+a[0]*a[4]
      %      +(a[2]-a[1])*y/temperature*
      %               (z[1][1]*z[0][0]-z[0][1]*z[1][0])/(den*den);

      %
      % End of segment contained in Gierasch's Fortran subroutine hydrogen().
      % 
      ho = a(2)+CPRH2*temperature-2.0*theta;
      hp = a(3)+CPRH2*temperature;
      %
      % entropies normalized at p0, T-->0, per particle divided by K_B
      % 
      so = -log(p/p0)+2.5*log(temperature)                           ...
                     +1.5*M_LN2+c1+(ho+2.0*theta)/temperature+a(6);
      sp = -log(p/p0)+2.5*log(temperature)                           ...
                     +1.5*M_LN2+c1+hp/temperature+a(7);
      %
      % potential temperatues, equal T as T-->0
      % 
       pottempo = exp(0.4*(so-c2-1.5*M_LN2-c1-2.5));
       pottempp = exp(0.4*(sp   -1.5*M_LN2-c1-2.5));
      %
      % curly F, equals -free energy difference, normalized at T=0
      % 
      ff = -(so-c2-1.5*M_LN2-c1-2.5-ho/temperature)          ...
            +(sp-1.5*M_LN2-c1-2.5-hp/temperature);
      ff = ff * temperature;
    end

    %
    % Save T, ortho and para enthalpies (offset so h(T=0)=0, ortho and
    % para entropies, ortho and para potential temperatures, and curly F.
    % Units are per particle, divided by Boltzmann constant, K_B.  Potential
    % temperatures and curly F are degrees K and degrees K per particle
    % over K_B.
    % 
    temp(I)= temperature;
    tho(I) = pottempo;
    thp(I) = pottempp;

    thermo.array(1,I)       = ho;
    thermo.array(2,I)       = hp;
    thermo.array(3,I)       = ff;
    thermo.array(4,I)       = a(1);
    thermo.array(5,I)       = a(2)-a(3);
    thermo.t_grid(I)        = temperature;
    thermo.theta_array(1,I) = pottempo;
    thermo.theta_array(2,I) = pottempp;
  end
  %
  % End of segment contained in Gierasch's Fortran subroutine h2properties().
  % 

  %
  % In Gierasch's Fortran, this segment is contained in the subroutine
  % theta2t_table().
  % 

  %for (m = 0; m < MDIM_THERMO; m++)  
  for m = 0 : MDIM_THERMO-1
    M=m+1;
    thermo.theta_grid(M) = (THLO_THERMO*(MDIM_THERMO-(m+1))         ...
                           +THHI_THERMO*(m))/(MDIM_THERMO-1);
  end
  %for (n = 0; n < NDIM_THERMO; n++)  
  for n = 0 : NDIM_THERMO-1
    N=n+1;
    thermo.fpdat(N) = (n)/(NDIM_THERMO-1);
    %for (i = 0; i < MDIM_THERMO; i++)  
    for ii = 0 : MDIM_THERMO-1
      I=ii+1;
      thetaln = ( planet.x_h2*CPRH2*((1.0-thermo.fpdat(N))*log(tho(I))         ...
                                     +(   thermo.fpdat(N))*log(thp(I)))        ...
                +(planet.x_he*CPRHE                                            ...
                + planet.x_3 *CPR3 )*log(temp(I)) )/cpr;
      thvector(I) = exp(thetaln);
      tvector(I)  = temp(I);
    end
    %
    % In Gierasch's Fortran, this segment is contained in the subroutine
    % trgrid().
    % 
    %for (j = 0; j < MDIM_THERMO; j++)  
    for jj = 0 : MDIM_THERMO-1
      J=jj+1;
      aa(J) = tvector(J);
    end
    tvector(MDIM_THERMO) = aa(MDIM_THERMO);
    %for (i = 0; i < MDIM_THERMO; i++)  
      %for (j = 1; j < MDIM_THERMO; j++)  
    for ii = 0 : MDIM_THERMO-1
      I=ii+1;
      for jj = 1 : MDIM_THERMO-1
        J=jj+1;
        if (thvector(J) >= thermo.theta_grid(I))  
          break;
        end
      end
      tvector(I) = aa(J-1)+(aa(J)-aa(J-1))*                                     ...
                         (thermo.theta_grid(I)-thvector(J-1))/                  ...
                         (thvector(J)         -thvector(J-1));
    end
    %
    % End of segment contained in Gierasch's Fortran subroutine trgrid().
    % 

    %for (m = 0; m < MDIM_THERMO; m++)  
    for m = 0 : MDIM_THERMO-1
      M=m+1;
      thermo.t(N,M) = tvector(M);
    end
  end
  %
  % End of segment contained in Gierasch's Fortran subroutine theta2t_table().
  % 

  return;
end
