/**********************************************************************
 *
 * mexgateway.c
 *
 * This file functions as the mex-file entry point.  The intended mexnc
 * operation is gleaned from the first argument, and then we transfer
 * control to the source file that handles either the NetCDF-2 or
 * NetCDF-3 API.
 *
 *********************************************************************/

/*
 * $Id: mexgateway.c,v 1.26 2005/12/07 14:10:43 johnevans007 Exp $
 * */

# include <ctype.h>
# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <string.h>

# include "netcdf.h"

# include "mex.h"

# include "mexnc.h"



op    *opname2opcode ( const mxArray *, int, int );
void   get_mexnc_info ( int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[], OPCODE );	


void mexFunction ( int nlhs, mxArray *plhs[], int nrhs, const mxArray	*prhs[] ) {

	op   *nc_op;

	/*
	 * loop index
	 * */
	int              j;



	char	error_message[1000];



	/*
	 * The Id tag tells which version of this particular file we have.
	 *
	 * The Name tag tells which release of mexnc was checked out via CVS
	 * and compiled (but only if the release was specifically named).  
	 * By running the command
	 *
	 * strings mexnc.mex* | grep "Name:"
	 *
	 * one should clearly see the release number.
	 * 
	 * */
	static char *cvsid="$Id: mexgateway.c,v 1.26 2005/12/07 14:10:43 johnevans007 Exp $";
	static char *cvsid_name="$Name: mexnc-2_0_15_1 $";



	/*	Disable the NC_FATAL option from ncopts.	*/
	
	if (ncopts & NC_FATAL)	{
		ncopts -= NC_FATAL;
	}
	
	/*	Display usage if less than one input argument.	*/
	
	if (nrhs < 1)	{
	
		Usage();
		
		return;
	}


	/*
	 * Make sure the first argument is not the empty set.
	 * */
	if ( single_matrix_input_is_empty_set ( prhs[0] ) ) {
		mexErrMsgTxt ( "cannot have empty set in first input position.\n\0" );
	}

	
	nc_op = opname2opcode ( prhs[0], nlhs, nrhs );
	

	/*
	 * Now make sure that none of the other arguments are the
	 * empty set.  We need to know the name of the netcdf operation
	 * before we can do this, since a few of the netcdf-2 functions
	 * do actually allow for the empty set.  If there are any illegal 
	 * empty set arguments, then an exception is thrown.
	 *
	 * */
	check_other_args_for_empty_set ( nc_op, prhs, nrhs );


	
	/*	Perform the NetCDF operation.	*/
	
	switch ( nc_op->opcode)	{

		case GET_MEXNC_INFO:
            		plhs[0] = mexncCreateDoubleScalar ( MEXNC_VERSION_NUM );
            		plhs[1] = mxCreateString ( MEXNC_RELEASE_NAME );
            		plhs[2] = mexncCreateDoubleScalar ( MEXNC_RELEASE_DATE );
			break;

		
		/*
		 * Handle the NetCDF-3 operations
		 * */
		case ABORT:
		case CLOSE:
		case COPY_ATT: 
		case _CREATE:
		case CREATE:
		case DEF_DIM:
		case DEF_VAR:
		case DEL_ATT:
		case _ENDDEF:
		case END_DEF:
		case ENDDEF:
		case GET_ATT_DOUBLE:
		case GET_ATT_FLOAT:
		case GET_ATT_INT:
		case GET_ATT_SHORT:
		case GET_ATT_SCHAR:
		case GET_ATT_UCHAR:
		case GET_ATT_TEXT:
		case GET_VAR_DOUBLE:
		case GET_VAR_FLOAT:
		case GET_VAR_INT:
		case GET_VAR_SHORT:
		case GET_VAR_SCHAR:
		case GET_VAR_UCHAR:
		case GET_VAR_TEXT:
		case GET_VAR1_DOUBLE:
		case GET_VAR1_FLOAT:
		case GET_VAR1_INT:
		case GET_VAR1_SHORT:
		case GET_VAR1_SCHAR:
		case GET_VAR1_UCHAR:
		case GET_VAR1_TEXT:
		case GET_VARA_DOUBLE:
		case GET_VARA_FLOAT:
		case GET_VARA_INT:
		case GET_VARA_SHORT:
		case GET_VARA_SCHAR:
		case GET_VARA_UCHAR:
		case GET_VARA_TEXT:
		case GET_VARS_DOUBLE:
		case GET_VARS_FLOAT:
		case GET_VARS_INT:
		case GET_VARS_SHORT:
		case GET_VARS_SCHAR:
		case GET_VARS_UCHAR:
		case GET_VARS_TEXT:
		case GET_VARM_DOUBLE:
		case GET_VARM_FLOAT:
		case GET_VARM_INT:
		case GET_VARM_SHORT:
		case GET_VARM_SCHAR:
		case GET_VARM_UCHAR:
		case GET_VARM_TEXT:
		case INQ:
		case INQ_NDIMS:
		case INQ_NVARS:
		case INQ_NATTS:
		case INQ_ATT:
		case INQ_ATTID:
		case INQ_ATTLEN:
		case INQ_ATTNAME:
		case INQ_ATTTYPE:
		case INQ_DIM:
		case INQ_DIMID:
		case INQ_DIMNAME:
		case INQ_DIMLEN:
		case INQ_LIBVERS:
		case INQ_VAR:
		case INQ_VARNAME:
		case INQ_VARTYPE:
		case INQ_VARNDIMS:
		case INQ_VARDIMID:
		case INQ_VARNATTS:
		case INQ_UNLIMDIM:
		case INQ_VARID:
		case _OPEN:
		case OPEN:
		case PUT_ATT_DOUBLE:
		case PUT_ATT_FLOAT:
		case PUT_ATT_INT:
		case PUT_ATT_SHORT:
		case PUT_ATT_SCHAR:
		case PUT_ATT_UCHAR:
		case PUT_ATT_TEXT:
		case PUT_VAR_DOUBLE:
		case PUT_VAR_FLOAT:
		case PUT_VAR_INT:
		case PUT_VAR_SHORT:
		case PUT_VAR_SCHAR:
		case PUT_VAR_UCHAR:
		case PUT_VAR_TEXT:
		case PUT_VARA_DOUBLE:
		case PUT_VARA_FLOAT:
		case PUT_VARA_INT:
		case PUT_VARA_SHORT:
		case PUT_VARA_SCHAR:
		case PUT_VARA_UCHAR:
		case PUT_VARA_TEXT:
		case PUT_VARS_DOUBLE:
		case PUT_VARS_FLOAT:
		case PUT_VARS_INT:
		case PUT_VARS_SHORT:
		case PUT_VARS_SCHAR:
		case PUT_VARS_UCHAR:
		case PUT_VARS_TEXT:
		case PUT_VARM_DOUBLE:
		case PUT_VARM_FLOAT:
		case PUT_VARM_INT:
		case PUT_VARM_SHORT:
		case PUT_VARM_SCHAR:
		case PUT_VARM_UCHAR:
		case PUT_VARM_TEXT:
		case PUT_VAR1_DOUBLE:
		case PUT_VAR1_FLOAT:
		case PUT_VAR1_INT:
		case PUT_VAR1_SHORT:
		case PUT_VAR1_SCHAR:
		case PUT_VAR1_UCHAR:
		case PUT_VAR1_TEXT:
		case REDEF:
		case RENAME_ATT:
		case RENAME_DIM:
		case RENAME_VAR:
		case SET_FILL:
		case STRERROR:
		case SYNC:
			handle_netcdf3_api ( nlhs, plhs, nrhs, prhs, nc_op );	
			break;
			
			
			
			

			
		/*
		 * Ok these are all the NetCDF 2.4 API calls.  Keep'em locked 
		 * away in the attic.
		 * */
		case ATTCOPY:
		case ATTDEL:
		case ATTGET:
		case ATTINQ:
		case ATTNAME:
		case ATTRENAME:
		case ATTPUT:
		case DIMDEF:
		case DIMID:
		case DIMINQ:
		case DIMRENAME:
		case ENDEF:
		case ERR:
		case INQUIRE:
		case PARAMETER:
		case RECPUT:
		case RECGET:
		case RECINQ:
		case SETFILL:
		case SETOPTS:
		case TYPELEN:
		case VARCOPY:
		case VARDEF:
		case VARGET:
		case VARGET1:
		case VARGETG:
		case VARID:
		case VARINQ:
		case VARPUT:
		case VARPUT1:
		case VARPUTG:
		case VARRENAME:

			handle_netcdf2_api ( nlhs, plhs, nrhs, prhs, nc_op );	
			break;




		
		default:
		
			sprintf ( error_message, "unhandled opcode %d, %s, line %d, file %s\n", nc_op->opcode, nc_op->opname, __LINE__, __FILE__ );
			mexPrintf ( error_message );
			for ( j = 0; j < nlhs; ++j ) {
				plhs[j] = mexncCreateDoubleScalar ( mxGetNaN () );
			}
			break;
	}
	
	return;
}




/*
 * This function transforms the implied name of a netcdf function 
 * (such as "nc_open") into an enumerated type (such as OPEN) that
 * is more readily handled by switch statements.  We also check to 
 * make sure that the right number of inputs and outputs are specified.
 * */
op *opname2opcode ( const mxArray *opname_matrix, int nlhs, int nrhs ) {

	OPCODE   opcode;
	char    *opname;
	int      j;

	/*
	 * Used to access the opcode portions.
	 * */
	char		*	p;

	char	error_message[1000];

	opname = Mat2Str(opname_matrix);

	/*	Convert the operation name to its opcode.	*/
	for (j = 0; j < strlen(opname); j++)	{
		opname[j] = (char) tolower((int) opname[j]);
	}
	p = opname;
	if (strncmp(p, "nc", 2) == 0)	{	/*	Trim away "nc".	*/
		p += 2;
	}
	
	j = 0;
	opcode = NONE;
	while (ops[j].opcode != NONE)	{
		if (!strcmp(p, ops[j].opname))	{
			opcode = ops[j].opcode;
			if (ops[j].nrhs > nrhs)	{
				sprintf ( error_message, "MEXNC: opname = %s, too few input arguments (%d), we need at least %d.\n", opname, nrhs, ops[j].nrhs );
				mexErrMsgTxt(error_message);
			}
			else if (0 && ops[j].nlhs > nlhs)	{	/*	Disabled.	*/
				sprintf ( error_message, "MEXNC: opname = %s, too few output arguments (%d), we need at least %d.\n", opname, nlhs, ops[j].nlhs );
				mexErrMsgTxt(error_message);
			}
			break;
		}
		else	{
			j++;
		}
	}
	
	if (opcode == NONE)	{
		sprintf ( error_message, "MEXNC: opname = %s, no such operation.\n", opname );
		mexErrMsgTxt(error_message);
	}
	
	return ( & ops[j] );


}










/*
 * single_matrix_input_is_empty_set
 * 
 * Passing in the empty set as an input will generally cause 
 * segmentation faults.  This can actually arise more than 
 * one would think, especially as matlab guis fall into the hands of
 * the unwise.
 * */
int single_matrix_input_is_empty_set ( const mxArray *input ) {

	/*
	 * These are the number of rows and columns of the input.
	 * If it is zero, then we know we have the empty set.
	 * */
	int m, n;

	m = mxGetM ( input );
	n = mxGetN ( input );

	if ( (m*n) == 0 ) {

		/*
		 * Yeah it's the empty set.
		 * */
		return ( 1 );

	} else {

		/*
		 * No it is not the empty set.
		 * */
		return ( 0 );

	}
}







/*
 * After the first argument has been checked to see that it is not
 * the empty set, the op name can be extracted.  If the op name is
 * known, then we can intelligently check the other input arguments
 * to make sure that none of them are the empty set except where
 * specifically allowed.  The reason for the allowed cases aren't 
 * really explained anywhere.  And unfortunately there's already
 * code out there that makes use of this.  What a terrific 
 * situation.  Sloppy.  Damned sloppy.
 * 
 * The empty string is considered different than [], I suppose, so
 * we skip that.
 *
 * */
void check_other_args_for_empty_set ( op *nc_op, const mxArray *mx_input[], int num_inputs ) {

	/*
	 * If we encounter the empty set where it is illegal, say so here.
	 * */
	char error_msg[500];
	char warning_msg[500];

	/*
	 * Shortcut to the matrix data
	 * */
	double *pr;


	/*
	 * These are the number of rows and columns of the input.
	 * If it is zero, then we know we have the empty set.
	 * */
	int m, n;


	/*
	 * Loop index for matlab array inputs.
	 * */
	int i;

	for ( i = 1; i < num_inputs; ++i ) {


		/*
		 * An empty string is sometimes allowed.
		 * */
		if ( mxIsChar ( mx_input[i] ) ) {
			continue;
		}


		if ( ( nc_op->opcode == ATTPUT ) && ( i == 6 ) ) {
			continue;
		}





		if ( ( nc_op->opcode == DEF_VAR ) && ( i == 5 ) ) {
			continue;
		}

		if ( ( nc_op->opcode == VARDEF ) && ( i == 5 ) ) {
			continue;
		}






		if ( ( nc_op->opcode == VARGETG ) && ( i == 6 ) ) {
			continue;
		}
		if ( ( nc_op->opcode == VARPUTG ) && ( i == 6 ) ) {
			continue;
		}
		if ( ( nc_op->opcode == VARPUTG ) && ( i == 7 ) ) {
			continue;
		}

		if ( ( nc_op->opcode == VARPUT1 ) && ( i == 4 ) ) {
			continue;
		}





		if ( ( nc_op->opcode == VARPUT ) && ( i == 5 ) ) {
			continue;
		}

		/*
		 * The test we use is whether the matrix size is zero.
		 * If so, then assume that we've got [] as an input.
		 * */
		pr = mxGetPr ( mx_input[i] );
		if ( pr == NULL ) {
			/*
			 * Yeah it's the empty set.
			 * */
			sprintf ( error_msg, "%s:  cannot have empty set in input position %d.\n\0", nc_op->opname, i+1 );
			mexErrMsgTxt ( error_msg );
		}

	}




}


